/*
 * Back-end misc module
 * See "gdiff.h" for the details of data structure.
 * This module should be independent from GUI frontend.
 *
 * Copyright INOUE Seiichiro <inoue@ainet.or.jp>, licensed under the GPL.
 */
#include <sys/stat.h>
#include <unistd.h>
#include <glib.h>
#include "gdiff.h"


/* NOTE:
 * Most of the routines doesn't modify DiffFiles argument.
 * However, I can't specify 'const' modifier, because dfiles_get_fileinfo() requires 'not-const' argument.
 * Refer to "diffdata.c" about why dfiles_get_fileinfo() does.
 */

/**
 * dfiles_get_special_status:
 * See "gdiff.h" about enum FilesSpecialStatus.
 * Input:
 * (const)DiffFiles *dfiles;
 * Output:
 * Return value: FilesSpecialStatus.
 **/
FilesSpecialStatus
dfiles_get_special_status(DiffFiles *dfiles)
{
	const FileInfo *fi1 = dfiles_get_fileinfo(dfiles, FIRST_FILE, FALSE);
	const FileInfo *fi2 = dfiles_get_fileinfo(dfiles, SECOND_FILE, FALSE);
	
	if (dfiles->binary)
		return BINARY_FILES;
	if (fi1->fname == NULL) 
		return ONLY_FILE2_EXISTS;
	if (fi2->fname == NULL)
		return ONLY_FILE1_EXISTS;
	if (fi1->f_dir == TRUE) {
		/* double check */
		if (fi2->f_dir == FALSE) {
			g_warning("something wrong in dfiles_get_special_status()\n");
		}
		return DIRECTORIES;
	}
	if (dfiles->dlines_list == NULL)
		return IDENTICAL_FILES;

	return DIFFERENT_FILES;
}

/**
 * dfiles_has_file_modified:
 * Check the files have been modified.
 * If a file has been modified by the other process, mmap'ed memory is also modified.
 * Some actions depend on mmap'ed memory image, so this check is necessary.
 * Only mtime is insufficient, because you can change mtime to arbitrary value, e.g. by 'touch' command.
 * Input:
 * (const)DiffFiles *dfiles;
 * Output:
 * Return value;
 **/
FileModStatue
dfiles_has_file_modified(DiffFiles *dfiles)
{
	const FileInfo *fi1 = dfiles_get_fileinfo(dfiles, FIRST_FILE, FALSE);
	const FileInfo *fi2 = dfiles_get_fileinfo(dfiles, SECOND_FILE, FALSE);
	struct stat sb;

	/* This function doesn't matter unless they are comparable files. */
	if (dfiles_get_special_status(dfiles) != DIFFERENT_FILES)
		return MOD_DONTCARE;
	
	if (stat(fi1->fname, &sb) == -1) {
		g_warning("%s: stat in dfiles_is_file_modified()", fi1->fname);
		return MOD_ERROR;
	}
	if (sb.st_mtime > fi1->mtime || sb.st_ctime > fi1->ctime)
		return MOD_MODIFIED;

	if (stat(fi2->fname, &sb) == -1) {
		g_warning("%s: stat in dfiles_is_file_modified()", fi2->fname);
		return MOD_ERROR;
	}
	if (sb.st_mtime > fi2->mtime || sb.st_ctime > fi2->ctime)
		return MOD_MODIFIED;
	
	return MOD_NOMODIFIED;
}


/**
 * dfiles_get_merged_nlines:
 * Return the number of lines of merged buffers.
 * Input:
 * (const)DiffFiles *files;
 * Output:
 * DiffFiles *files; Could be updated (by calling dfiles_get_fileinfo()). But theoretically, not updated here.
 * Return value; the number of lines of merged buffers.
 **/
int
dfiles_get_merged_nlines(DiffFiles *files)
{
	const FileInfo *fi1 = dfiles_get_fileinfo(files, FIRST_FILE, TRUE);
	int merged_nline;
	GList *node;/* node of DiffLines' list */
	DiffLines *dlines;
	
	merged_nline = fi1->nline;/* first file */
	/* second file */
	node = files->dlines_list;
	while (node) {
		dlines = node->data;
		if (dlines->difftype == F2ONLY || dlines->difftype == CHANGE) {
			merged_nline += (dlines->between[SECOND_FILE].end - dlines->between[SECOND_FILE].begin + 1);
		}
		node = g_list_next(node);
	}
	
	return merged_nline;
}

/**
 * dfiles_calc_total_nlines:
 * Calculate the tolal number of different lines, and return it.
 * Input:
 * const DiffFiles *dfiles;
 * WhichFile n;
 * Output:
 * Return value; The tolal number of different lines.
 **/
int
dfiles_calc_total_nlines(const DiffFiles *dfiles, WhichFile n)
{
	GList *list;/* Doubly linked list of DiffLines */
	int ret_nline = 0;
	
	for (list = dfiles->dlines_list; list; list = list->next) {
		DiffLines *dlines = list->data;
		ret_nline += dlines->between[n].end - dlines->between[n].begin + 1; 
	}
	return ret_nline;
}


/**
 * dfiles_get_max_nlines:
 * Compare the number of lines of two files, and return one of the bigger.
 * Input:
 * (const)DiffFiles *dfiles;
 * Output:
 * Return value; The number of lines of the bigger.
 **/
int
dfiles_get_max_nlines(DiffFiles *dfiles)
{
	const FileInfo *fi1 = dfiles_get_fileinfo(dfiles, FIRST_FILE, TRUE);
	const FileInfo *fi2 = dfiles_get_fileinfo(dfiles, SECOND_FILE, TRUE);

	return (fi1->nline > fi2->nline) ? fi1->nline : fi2->nline;
}



/**
 * dfiles_get_firstl:
 * Get first different lines.
 **/
const GList*
dfiles_get_firstl(const DiffFiles *dfiles, const GList *cur_dlines_list)
{
	if (cur_dlines_list)
		return g_list_first((GList*)cur_dlines_list);
	else
		return dfiles->dlines_list;
}

/**
 * dfiles_get_lastl:
 **/
const GList*
dfiles_get_lastl(const DiffFiles *dfiles, const GList *cur_dlines_list)
{
	if (cur_dlines_list)
		return g_list_last((GList*)cur_dlines_list);
	else
		return g_list_last(dfiles->dlines_list);
}

/**
 * dfiles_find_nextl:
 * Using the current difference, find the next diff part, and return it.
 * The DiffLines is an internal data, so it shouldn't be modified by the caller.
 * Input:
 * const DiffFiles *dfiles;
 * const GList *cur_dlines_list;
 * Output:
 * Return value;
 **/
const GList*
dfiles_find_nextl(const DiffFiles *dfiles, const GList *cur_dlines_list)
{
	if (cur_dlines_list)
		return g_list_next((GList*)cur_dlines_list);
	else
		return dfiles->dlines_list;/* first one */
}

/**
 * dfiles_find_prevl:
 * Using the current difference, find the previous diff part, and return it.
 * The DiffLines is an internal data, so it shouldn't be modified by the caller.
 * See dfiles_find_next() about Input and Output.
 **/
const GList*
dfiles_find_prevl(const DiffFiles *dfiles, const GList *cur_dlines_list)
{
	if (cur_dlines_list)
		return g_list_previous((GList*)cur_dlines_list);
	else
		return g_list_last(dfiles->dlines_list);
}

/**
 * dfiles_find_rel_nextl:
 * Using current line number, find the relative next diff part, and return DiffLines.
 * The DiffLines is an internal data, so it shouldn't be modified by the caller.
 * Input:
 * const DiffFiles *dfiles;
 * WhichFile n; FIRST_FILE or SECOND_FILE.
 * int cur_line; The current line number.
 * Output:
 * Return value;
 **/
const DiffLines*
dfiles_find_rel_nextl(const DiffFiles *dfiles, WhichFile n, int cur_line)
{
	GList *list;/* Doubly linked list of DiffLines */
	
	for (list = dfiles->dlines_list; list; list = list->next) {
		DiffLines *dlines = list->data;
		if (dlines->between[n].begin > cur_line) {
			return dlines;
		}
	}
	return NULL;
}

/**
 * dfiles_find_rel_prevl:
 * Using current line number, find the relative previous diff part, and return DiffLines.
 * The DiffLines is an internal data, so it shouldn't be modified by the caller.
 * See dfiles_find_nextl() about Input and Output.
 **/
const DiffLines*
dfiles_find_rel_prevl(const DiffFiles *dfiles, WhichFile n, int cur_line)
{
	GList *list;/* Doubly linked list of DiffLines */
	DiffLines *prev_one = dfiles->dlines_list->data;
	
	for (list = dfiles->dlines_list; list; list = list->next) {
		DiffLines *dlines = list->data;
		if (dlines->between[n].end > cur_line) {
			return prev_one;
		}
		prev_one = dlines;
	}
	return NULL;
}



/**
 * dfiles_get_firstl_in_merge:
 * See dfiles_find_nextl_in_merge() about sum_f2.
 **/
const GList*
dfiles_get_firstl_in_merge(const DiffFiles *dfiles, const GList *cur_dlines_list, int *sum_f2)
{
	GList *first_list = dfiles->dlines_list;
	DiffLines *dlines = first_list->data;

	*sum_f2 += dlines->between[SECOND_FILE].end - dlines->between[SECOND_FILE].begin + 1;
	return first_list;
}

/**
 * dfiles_get_lastl_in_merge:
 * See dfiles_find_nextl_in_merge() about sum_f2.
 **/
const GList*
dfiles_get_lastl_in_merge(const DiffFiles *dfiles, const GList *cur_dlines_list, int *sum_f2)
{
	GList *first_list = dfiles->dlines_list;
	GList *last_list;
	GList *list;/* Doubly linked list of DiffLines */

	/* This doesn't matter in fact... */
	if (cur_dlines_list)
		last_list = g_list_last((GList*)cur_dlines_list);
	else
		last_list = g_list_last(first_list);
	
	for (list = first_list; list; list = list->next) {
		DiffLines *dlines = list->data;
		*sum_f2 += dlines->between[SECOND_FILE].end - dlines->between[SECOND_FILE].begin + 1;
	}
	return last_list;
}

/**
 * dfiles_find_nextl_in_merge:
 * Using the current difference, find the next diff part in merged files, and return it.
 * The DiffLines is an internal data, so it shouldn't be modified by the caller.
 * Input:
 * const DiffFiles *dfiles;
 * const GList *cur_dlines_list;
 * Output:
 * int *sum_f2; The sum of SECOND_FILE specific lines before the diff part.
 * Return value;
 **/
const GList*
dfiles_find_nextl_in_merge(const DiffFiles *dfiles, const GList *cur_dlines_list, int *sum_f2)
{
	GList *list;/* Doubly linked list of DiffLines */
	GList *next_list;

	if (cur_dlines_list)
		next_list = g_list_next((GList*)cur_dlines_list);
	else
		next_list = dfiles->dlines_list;/* first one */

	for (list = dfiles->dlines_list; list && list != next_list; list = list->next) {
		DiffLines *dlines = list->data;
		*sum_f2 += dlines->between[SECOND_FILE].end - dlines->between[SECOND_FILE].begin + 1;
	}
	return next_list;
}

/**
 * dfiles_find_rel_prevl_in_merge:
 * Using the current difference, find the relative previous diff part in merged files, and return it.
 * The DiffLines is an internal data, so it shouldn't be modified by the caller.
 * See dfiles_find_nextl_in_merge() about Input and Output.
 **/
const GList*
dfiles_find_prevl_in_merge(const DiffFiles *dfiles, const GList *cur_dlines_list, int *sum_f2)
{
	GList *list;/* Doubly linked list of DiffLines */
	GList *prev_list;

	if (cur_dlines_list)
		prev_list = g_list_previous((GList*)cur_dlines_list);
	else
		prev_list = g_list_last(dfiles->dlines_list);
	
	for (list = dfiles->dlines_list; list && list != prev_list; list = list->next) {
		DiffLines *dlines = list->data;
		*sum_f2 += dlines->between[SECOND_FILE].end - dlines->between[SECOND_FILE].begin + 1;
	}
	return prev_list;
}

/**
 * dfiles_find_rel_nextl_in_merge:
 * Using current line number, find the relative next diff part in merged files, and return DiffLines.
 * The DiffLines is an internal data, so it shouldn't be modified by the caller.
 * Input:
 * const DiffFiles *dfiles;
 * int cur_line; The current line number.
 * Output:
 * int *sum_f2; The sum of SECOND_FILE specific lines before the diff part.
 * Return value;
 **/
const DiffLines*
dfiles_find_rel_nextl_in_merge(const DiffFiles *dfiles, int cur_line, int *sum_f2)
{
	GList *list;/* Doubly linked list of DiffLines */
	
	for (list = dfiles->dlines_list; list; list = list->next) {
		DiffLines *dlines = list->data;
		*sum_f2 += dlines->between[SECOND_FILE].end - dlines->between[SECOND_FILE].begin + 1;
		if (dlines->between[FIRST_FILE].begin + *sum_f2 > cur_line) {
			return dlines;
		}
	}
	return NULL;
}

/**
 * dfiles_find_rel_prevl_in_merge:
 * Using current line number, find the relative previous diff part in merged files, and return DiffLines.
 * The DiffLines is an internal data, so it shouldn't be modified by the caller.
 * See dfiles_find_nextl_in_merge() about Input and Output.
 **/
const DiffLines*
dfiles_find_rel_prevl_in_merge(const DiffFiles *dfiles, int cur_line, int *sum_f2)
{
	GList *list;/* Doubly linked list of DiffLines */
	DiffLines *prev_one = dfiles->dlines_list->data;
	
	for (list = dfiles->dlines_list; list; list = list->next) {
		DiffLines *dlines = list->data;
		*sum_f2 += dlines->between[SECOND_FILE].end - dlines->between[SECOND_FILE].begin + 1;
		if (dlines->between[FIRST_FILE].end + *sum_f2 > cur_line) {
			return prev_one;
		}
		prev_one = dlines;
	}
	return NULL;
}
