/* eblook.c - interactive EB interface command
 *
 * Copyright (C) 1997,1998,1999 Keisuke Nishida <knishida@ring.gr.jp>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this software; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <errno.h>

#if STDC_HEADERS
# include <stdlib.h>
# include <string.h>
#else
# ifndef HAVE_STRCHR
#  define strchr index
#  define strrchr rindex
# else
char *strchr (), *strrchr ();
# endif
#endif

#include "getopt.h"

#include "eb/eb.h"
#include "eb/text.h"
#include "eb/font.h"
#include "eb/appendix.h"
#include "eb/error.h"

#ifndef BUFSIZ
#define BUFSIZ	1024
#endif

#define MAX_HIT_SIZE	256
#define MAX_TEXT_SIZE	8192

#define USER_INIT_FILE  "~/.eblookrc"

/*
 * String A-list
 */
typedef struct _StringAlist {
  char                 *key;
  char                 *value;
  struct _StringAlist  *next;
} StringAlist;

/*
 * Internal functions
 */
char *read_command EB_P ((char *, int, FILE *));
int excute_command EB_P ((char *));
int parse_command_line EB_P ((char *, char *[]));

void command_book EB_P ((int, char *[]));
void command_info EB_P ((int, char *[]));
void command_list EB_P ((int, char *[]));
void command_select EB_P ((int, char *[]));
void command_subinfo EB_P ((int, char *[]));
void command_copyright EB_P ((int, char *[]));
void command_menu EB_P ((int, char *[]));
void command_search EB_P ((int, char *[]));
void command_content EB_P ((int, char *[]));
void command_font EB_P ((int, char *[]));
void command_show EB_P ((int, char *[]));
void command_set EB_P ((int, char *[]));
void command_unset EB_P ((int, char *[]));
void command_help EB_P ((int, char *[]));

int check_book EB_P (());
int check_subbook EB_P (());

int internal_set_font EB_P ((EB_Book *, char *));
int parse_dict_id EB_P ((char *, EB_Book *));
int parse_entry_id EB_P ((char *, EB_Position *));

int search_pattern EB_P ((EB_Book *, EB_Appendix *, char *, int, int));
int insert_content EB_P ((EB_Book *, EB_Appendix *, EB_Position *, int, int));
int insert_font EB_P ((EB_Book *, const char *));
int insert_font_list EB_P ((EB_Book *));

int hook_font EB_P ((EB_Book *, EB_Appendix *, char *, EB_Hook_Code, int, const int *));
int hook_newline EB_P ((EB_Book *, EB_Appendix *, char *, EB_Hook_Code, int, const int *));
int hook_stopcode EB_P ((EB_Book *, EB_Appendix *, char *, EB_Hook_Code, int, const int *));
int hook_tags EB_P ((EB_Book *, EB_Appendix *, char *, EB_Hook_Code, int, const int *));

void show_version EB_P ((void));
void show_help EB_P ((void));
void show_try_help EB_P ((void));
void set_error_message EB_P (());
void unset_error_message EB_P (());

StringAlist *salist_set EB_P ((StringAlist *, const char *, const char *));
char *salist_ref EB_P ((StringAlist *, const char *));

int eblook_read EB_P ((char *));
void eblook_write EB_P ((const char *, FILE *));
int jis2euc EB_P ((unsigned char *));
int sjis2euc EB_P ((unsigned char *));
void write_euc2jis EB_P ((const unsigned char *, FILE *));
void write_euc2sjis EB_P ((const unsigned char *, FILE *));

#define variable_set(key, value) \
     variable_alist = salist_set (variable_alist, key, value)
#define variable_ref(key) \
     salist_ref (variable_alist, key)

/*
 * Constants
 */
const char *program_name = PACKAGE;
const char *program_version = VERSION;
const char *default_prompt = "eblook> ";
const char *default_method = "glob";

/*
 * Internal variables
 */
const char *invoked_name;
enum {JIS, SJIS, EUC} kanji_code, default_code;

EB_Book current_book;
EB_Appendix current_appendix;

EB_Hookset text_hookset;
StringAlist *variable_alist = NULL;
int last_search_begin = 0;
int last_search_length = 0;
int (*last_search_function) EB_P ((EB_Book *, const char *)) = NULL;

/*
 * Interactive command table
 */
struct {
  const char *name;
  const char *option_string;
#ifdef __STDC__
  void (*func) (int, char *[]);
#else /* not __STDC__ */
  void (*func) ();
#endif /* not __STDC__ */
  const char *help;
} command_table[] = {
  {"book", "[directory [appendix]]", command_book, "Set a book directory.\n"},
  {"info", "", command_info, "Show information of the selected book.\n"},
  {"list", "", command_list, "List all dictionaries in the selected book.\n"},
  {"select", "subbook", command_select, "Select a subbook.\n"},
  {"subinfo", "", command_subinfo, "Show information of the selected subbook.\n"},
  {"copyright", "", command_copyright, "Show copyright of the selected subbook.\n"},
  {"menu", "", command_menu, "Show the menu of the selected subbook.\n"},
  {"search", "pattern [offset]", command_search, "Search for a word\n"},
  {"content", "entry [offset]", command_content, "Display contents of entry.\n"},
  {"font", "[id]", command_font, "Display the bitmap of gaiji.\n"},
  {"show", "[variable]", command_show, "Show the value of variables.\n"},
  {"set", "variable value", command_set, "Set a variable to the value.\n"},
  {"unset", "varialbe...", command_unset, "Unset variables.\n"},
  {"help", "", command_help, "Show this message.\n"},
  {"quit", "", NULL, "Quit program.\n"},
  {NULL, NULL, NULL, NULL}
};

/*
 * Text hooks
 */
EB_Hook text_hooks[] = {
  {EB_HOOK_NARROW_JISX0208, eb_hook_euc_to_ascii},
  {EB_HOOK_NARROW_FONT,     hook_font},
  {EB_HOOK_WIDE_FONT,	    hook_font},
  {EB_HOOK_NEWLINE,         hook_newline},
  {EB_HOOK_STOPCODE,        hook_stopcode},
  {EB_HOOK_BEGIN_PICTURE,   hook_tags},
  {EB_HOOK_END_PICTURE,     hook_tags},
  {EB_HOOK_BEGIN_MENU,      hook_tags},
  {EB_HOOK_END_MENU,        hook_tags},
  {EB_HOOK_BEGIN_SOUND,     hook_tags},
  {EB_HOOK_END_SOUND,       hook_tags},
  {EB_HOOK_BEGIN_REFERENCE, hook_tags},
  {EB_HOOK_END_REFERENCE,   hook_tags},
  {EB_HOOK_NULL, NULL},
};

static const char *short_options = "hqv";
static struct option long_options[] = {
  {"help",         no_argument,       NULL, 'h'},
  {"no-init-file", no_argument,       NULL, 'q'},
  {"version",      no_argument,       NULL, 'v'},
  {NULL,           no_argument,       NULL, 0}
};


int
main (argc, argv)
     int argc;
     char *const *argv;
{
  int optch;
  int no_init = 0;
  char buff[BUFSIZ];
  const char *book, *appendix, *s;
  FILE *fp;

  invoked_name = argv[0];
#ifdef _WIN32
  default_code = SJIS;
#else
  default_code = EUC;
#endif

  /* parse command line options */
  while ((optch = getopt_long(argc, argv, short_options, long_options, NULL))
	 != EOF) {
    switch (optch) {
    case 'h':
      show_help ();
      exit (0);
    case 'v':
      show_version ();
      exit (0);
    case 'q':
      no_init = 1;
      break;
    default:
      show_try_help ();
      exit (1);
    }
  }

  /* check the rest arguments */
  book = appendix = NULL;
  switch (argc - optind) {
  case 2:
    appendix = argv[optind + 1];
  case 1:
    book = argv[optind];
  case 0:
    break;

  default:
    fprintf (stderr, "%s: too many arguments\n", invoked_name);
    show_try_help ();
    exit (1);
  }

  /* initialize variables */
  eb_initialize (&current_book);
  eb_initialize_appendix (&current_appendix);
  eb_initialize_hookset (&text_hookset);
  eb_set_hooks (&text_hookset, text_hooks);

  variable_set ("prompt", default_prompt);
  variable_set ("search-method", default_method);

  sprintf (buff, "%d", MAX_HIT_SIZE);
  variable_set ("max-hits", buff);

  sprintf (buff, "%d", MAX_TEXT_SIZE);
  variable_set ("max-text", buff);

  sprintf (buff, "%s %s (with EB %d.%d)", program_name, program_version,
	   EB_VERSION_MAJOR, EB_VERSION_MINOR);
  variable_set ("version", buff);

  /* load init file */
  if (!no_init) {
    if (!strncmp (USER_INIT_FILE, "~/", 2)) {
      strcpy (buff, getenv ("HOME"));
      strcat (buff, USER_INIT_FILE + 1);
    } else {
      strcpy (buff, USER_INIT_FILE);
    }
    if ((fp = fopen (buff, "r")) != NULL)
      while (read_command (buff, BUFSIZ, fp) != NULL)
	if (!excute_command (buff))
	  break;
  }

  /* set book and appendix */
  if (book && eb_bind (&current_book, book) < 0) {
    printf ("Warning: invalid book directory: %s\n", book);
    set_error_message ();
  }
  if (appendix && eb_bind_appendix (&current_appendix, appendix) < 0) {
    printf ("Warning: invalid appendix directory: %s\n", appendix);
    set_error_message ();
  }

  /* check the book directory */
  if (!eb_is_bound (&current_book))
    puts ("Warning: you should specify a book directory first");

  /* enter command loop */
  while (1) {
    /* kanji code */
    kanji_code = default_code;
    if ((s = variable_ref ("kanji-code")) != NULL) {
      if (strcmp (s, "JIS") == 0)
	kanji_code = JIS;
      else if (strcmp (s, "SJIS") == 0)
	kanji_code = SJIS;
      else if (strcmp (s, "EUC") == 0)
	kanji_code = JIS;
      else {
	printf ("Invalid kanji code: %s\n", s);
	variable_set ("kanji-code", NULL);
      }
    }

    /* prompt */
    if ((s = variable_ref ("prompt")) == NULL)
      s = default_prompt;
    fputs (s, stdout);
    fflush (stdout);

    /* read and excute */
    unset_error_message ();
    if (read_command (buff, BUFSIZ, stdin) == NULL)
      break;
    if (!excute_command (buff))
      break;
  }

  return 0;
}

char *
read_command (command_line, size, stream)
     char *command_line;
     int size;
     FILE *stream;
{
  char *p;

  /* read string */
  if (fgets (command_line, size, stream) == NULL)
    return NULL;

  /* delete '\n' */
  if ((p = strchr (command_line, '\n')) != NULL) {
    *p = '\0';
    if (eblook_read (command_line) < 0) {
      puts ("Invalid input characters");
      command_line[0] = '\0';
    }
  } else {
    puts ("Input is too long");
    while (fgets (command_line, BUFSIZ, stdin) != NULL &&
	   strchr (command_line, '\n') == NULL);
    command_line[0] = '\0';
  }
  return command_line;
}

int
excute_command (command_line)
     char *command_line;
{
  int i, argc;
  char *argv[BUFSIZ / 2];			/* xxx: no good? */

  argc = parse_command_line (command_line, argv);

  /* if input is empty, do nothing */
  if (argc == 0)
    return 1;

  /* if input is "quit", we should quit */
  if (strcmp (argv[0], "quit") == 0)
    return 0;

  /* otherwise, search command and execute */
  for (i = 0; command_table[i].name != NULL; i++) {
    if (strcmp (argv[0], command_table[i].name) == 0) {
      command_table[i].func (argc, argv);
      return 1;
    }
  }
  if (command_table[i].name == NULL)
    printf ("Unkown command: %s\n", argv[0]);
  return 1;
}

int
parse_command_line (command_line, argv)
     char *command_line;
     char *argv[];
{
  int num;
  int reserved, in_quote;
  char *p;

  /* devide string into tokens by white spaces */
  num = reserved = in_quote = 0;
  for (p = command_line; *p != '\0'; p++) {
    switch (*p) {
    case '"':
      if (!reserved) {
	argv[num++] = p;
	reserved = 1;
      }
      strcpy (p, p + 1);
      p--;
      in_quote = !in_quote;
      break;

    case ' ':
    case '\t':
      if (!in_quote) {
	*p = '\0';
	reserved = 0;
      }
      break;

    case '\\':
      strcpy (p, p + 1);
    default:
      if (!reserved) {
	argv[num++] = p;
	reserved = 1;
      }
    }
  }

  return num;
}


void
command_book (argc, argv)
     int argc;
     char *argv[];
{
  switch (argc) {
  case 3:
    if (eb_bind_appendix (&current_appendix, argv[2]) < 0) {
      printf ("Invalid appendix directory: %s\n", argv[2]);
      set_error_message ();
    }
  case 2:
    if (eb_bind (&current_book, argv[1]) < 0) {
      printf ("Invalid book directory: %s\n", argv[1]);
      set_error_message ();
    }
    break;

  case 1:
    if (eb_is_bound (&current_book)) {
      printf ("book\t%s\n", eb_path (&current_book));
      if (eb_is_appendix_bound (&current_appendix))
	printf ("appendix\t%s\n", eb_appendix_path (&current_appendix));
    } else {
      puts ("No book is specified");
    }
    break;

  default:
    printf ("%s: too many arguments\n", argv[0]);
  }
}

void
command_info (argc, argv)
     int argc;
     char *argv[];
{
  if (argc > 1) {
    printf ("%s: too many arguments\n", argv[0]);
    return;
  }

  if (check_book ()) {
    int subcount;
    EB_Disc_Code disccode;
    EB_Character_Code charcode;

    /* disc type */
    if ((disccode = eb_disc_type (&current_book)) >= 0) {
      fputs (" disc type: ", stdout);
      puts ((disccode == EB_DISC_EB) ? "EB/EBG/EBXA" : "EPWING");
    }

    /* character code */
    if ((charcode = eb_character_code (&current_book)) >= 0) {
      fputs (" character code: ", stdout);
      puts ((charcode == EB_CHARCODE_JISX0208) ? "JIS X 0208" : "ISO 8859-1");
    }

    /* the number of dictionrys */
    if ((subcount = eb_subbook_count (&current_book)) >= 0)
      printf (" the number of dictionries: %d\n", subcount);
  }
}

void
command_list (argc, argv)
     int argc;
     char *argv[];
{
  if (argc > 1) {
    printf ("%s: too many arguments\n", argv[0]);
    return;
  }

  if (check_book ()) {
    int i, num;
    const char *s;
    EB_Subbook_Code list[EB_MAX_SUBBOOKS];

    if ((num = eb_subbook_list (&current_book, list)) < 0)
      goto error;

    for (i = 0; i < num; i++) {
      printf ("%2d. ", i + 1);

      if ((s = eb_subbook_directory2 (&current_book, list[i])) == NULL)
	goto error;
      printf ("%s\t", s);

      if ((s = eb_subbook_title2 (&current_book, list[i])) == NULL)
	goto error;
      eblook_write (s, stdout);
      fputc ('\n', stdout);
    }
    return;

  error:
    printf ("An error occured in command_list: %s\n", eb_error_message ());
    set_error_message ();
    return;
  }
}

void
command_select (argc, argv)
     int argc;
     char *argv[];
{
  switch (argc) {
  case 1:
    eb_unset_subbook (&current_book);
    eb_unset_appendix_subbook (&current_appendix);
    return;

  case 2:
    if (check_book ())
      if (parse_dict_id (argv[1], &current_book))
	if (eb_is_appendix_bound (&current_appendix))
	  eb_set_appendix_subbook (&current_appendix, eb_subbook (&current_book));
    return;

  default:
    printf ("%s: too many arguments\n", argv[0]);
  }
}

void
command_subinfo (argc, argv)
     int argc;
     char *argv[];
{
  if (argc > 1) {
    printf ("%s: too many arguments\n", argv[0]);
    return;
  }
    
  if (check_subbook ()) {
    int i, num;
    const char *s;
    EB_Font_Code list[EB_MAX_FONTS];

    /* title */
    if ((s = eb_subbook_title (&current_book)) != NULL) {
      fputs (" title: ", stdout);
      eblook_write (s, stdout);
      fputc ('\n', stdout);
    }

    /* directory */
    if ((s = eb_subbook_directory (&current_book)) != NULL)
      printf (" directory: %s\n", s);

    /* search methods */
    fputs (" search methods:", stdout);
    if (eb_have_word_search (&current_book))
      fputs (" word", stdout);
    if (eb_have_endword_search (&current_book))
      fputs (" endword", stdout);
    if (eb_have_exactword_search (&current_book))
      fputs (" exactword", stdout);
    if (eb_have_keyword_search (&current_book))
      fputs (" keyword", stdout);
    if (eb_have_multi_search (&current_book))
      fputs (" multi", stdout);
    if (eb_have_menu (&current_book))
      fputs (" menu", stdout);
    if (eb_have_graphic_search (&current_book))
      fputs (" graphic", stdout);
    fputc ('\n', stdout);

    /* font size */
    fputs (" font sizes:", stdout);
    num = eb_font_list (&current_book, list);
    for (i = 0; i < num; i++)
      printf (" %d", list[i]);
    fputc ('\n', stdout);
  }
}

void
command_copyright (argc, argv)
     int argc;
     char *argv[];
{
  if (argc > 1) {
    printf ("%s: too many arguments\n", argv[0]);
    return;
  }

  if (check_subbook ()) {
    EB_Position pos;
    if (eb_copyright (&current_book, &pos) < 0)
      puts ("Current dictionry has no copyright information.");
    else
      insert_content (&current_book, &current_appendix, &pos, 0, 0);
  }
}

void
command_menu (argc, argv)
     int argc;
     char *argv[];
{
  if (argc > 1) {
    printf ("%s: too many arguments\n", argv[0]);
    return;
  }

  if (check_subbook ()) {
    EB_Position pos;
    if (eb_menu (&current_book, &pos) < 0)
      puts ("Current dictionry has no menu.");
    else
      insert_content (&current_book, &current_appendix, &pos, 0, 0);
  }
}

void
command_search (argc, argv)
     int argc;
     char *argv[];
{
  int begin, length;
  char *pattern;

  begin = 1;
  pattern = variable_ref ("max-hits");
  length = pattern ? atoi (pattern) : 0;
  pattern = NULL;

  switch (argc) {
  case 3:
    begin = atoi (argv[2]);
  case 2:
    pattern = argv[1];
  case 1:
    if (check_subbook ())
      search_pattern (&current_book, &current_appendix, pattern, begin, length);
    return;

  default:
    printf ("%s: too many arguments\n", argv[0]);
  }
}

void
command_content (argc, argv)
     int argc;
     char *argv[];
{
  int begin, length;
  char *s;
  EB_Position pos;

  begin = 1;
  s = variable_ref ("max-text");
  length = s ? (atoi (s) / EB_SIZE_PAGE) : 0;

  switch (argc) {
  case 1:
    printf ("%s: too few arguments\n", argv[0]);
    return;

  case 4:
    length = atoi (argv[3]);
  case 3:
    begin = atoi (argv[2]);
  case 2:
    if (check_subbook ()) {
      if (parse_entry_id (argv[1], &pos))
	insert_content (&current_book, &current_appendix, &pos, begin, length);
    }
    return;

  default:
    printf ("%s: too many arguments\n", argv[0]);
  }
}

void
command_font (argc, argv)
     int argc;
     char *argv[];
{
  if (argc > 2) {
    printf ("%s: too many arguments\n", argv[0]);
    return;
  }

  if (check_subbook () && internal_set_font (&current_book, NULL)) {
    if (argc == 1)
      insert_font_list (&current_book);
    else
      insert_font (&current_book, argv[1]);
  }
}

void
command_show (argc, argv)
     int argc;
     char *argv[];
{
  char *s;
  StringAlist *var;
  switch (argc) {
  case 1:
    /*
     * Show all variables and their values
     */
    for (var = variable_alist; var != NULL; var = var->next)
      if (var->key[0] != '_')
	printf ("%s\t%s\n", var->key, var->value);
    return;

  case 2:
    /*
     * Show value of variable
     */
    if ((s = variable_ref (argv[1])) != NULL)
      puts (s);
    else
      printf ("Unbounded variable: %s\n", argv[1]);
    return;

  default:
    printf ("%s: too many arguments\n", argv[0]);
  }
}

void
command_set (argc, argv)
     int argc;
     char *argv[];
{
  switch (argc) {
  case 1:
    printf ("%s: too few arguments\n", argv[0]);
    return;

  case 2:
    argv[2] = "";
  case 3:
    variable_set (argv[1], argv[2]);
    return;

  default:
    printf ("%s: too many arguments\n", argv[0]);
  }
}

void
command_unset (argc, argv)
     int argc;
     char *argv[];
{
  int i;
  if (argc == 1) {
    printf ("%s: too few arguments\n", argv[0]);
  } else {
    for (i = 1; i < argc; i++)
      variable_set (argv[i], NULL);
    return;
  }
}

void
command_help (argc, argv)
     int argc;
     char *argv[];
{
  int i;
  char buff[256];
  const char *p;
  switch (argc) {
  case 1:
    /*
     * List up all command helps
     */
    for (i = 0; command_table[i].name != NULL; i++) {
      sprintf (buff, "%s %s",
	      command_table[i].name, command_table[i].option_string);
      printf (" %-22s - ", buff);
      for (p = command_table[i].help;
	   *p != '\0' && *p != '.' && *p != '\n';
	   p++)
	putchar (*p);
      putchar ('\n');
    }
    return;

  case 2:
    /*
     * Show command help
     */
    for (i = 0; command_table[i].name != NULL; i++) {
      if (strcmp (command_table[i].name, argv[1]) == 0)
	break;
    }
    if (command_table[i].name == NULL) {
      printf ("No such command: %s\n", argv[1]);
    } else {
      printf ("Usage: %s %s\n\n%s",
	      command_table[i].name, command_table[i].option_string,
	      command_table[i].help);
    }
    return;

  default:
    printf ("%s: too many arguments\n", argv[0]);
  }
}


int
check_book ()
{
  if (eb_is_bound (&current_book)) {
    return 1;
  } else {
    puts ("You should specify a book directory first");
    return 0;
  }
}

int
check_subbook ()
{
  if (check_book ()) {
    if (eb_subbook (&current_book) >= 0)
      return 1;
    else
      puts ("You should select a subbook first");
  }
  return 0;
}

int
internal_set_font (book, height)
     EB_Book *book;
     char *height;
{
  EB_Font_Code font;

  if (height == NULL)
    if ((height = variable_ref ("font")) == NULL)
      height = "16";

  font = atoi (height);
  if (font != EB_FONT_16 && font != EB_FONT_24 &&
      font != EB_FONT_30 && font != EB_FONT_48) {
    printf ("Illegal font height: %s\n", height);
    return 0;
  }

  if (!eb_have_font (book, font) ) {
    printf ("Invalid font for %s: %s\n", eb_subbook_directory (book), height);
    set_error_message ();
    return 0;
  }

  if (eb_set_font (book, font) < 0) {
    printf ("An error occurred in internal_set_font: %s\n",
	   eb_error_message ());
    set_error_message ();
    return 0;
  }
  return 1;
}

int
parse_dict_id (name, book)
     char *name;
     EB_Book *book;
{
  int i, num;
  EB_Subbook_Code sublist[EB_MAX_SUBBOOKS];

  if ((num = eb_subbook_list (book, sublist)) < 0)
    goto error;

  if ((i = atoi (name)) > 0) {
    /*
     * Numbered dictionry
     */
    if (--i < num) {
      if (eb_set_subbook (book, sublist[i]) < 0)
	goto error;
      return 1;
    } else {
      printf ("No such numberd dictionary : %s\n", name);
      return 0;
    }
  } else {
    /*
     * Named dictionry
     */
    const char *dir;
    for (i = 0; i < num; i++) {
      if ((dir = eb_subbook_directory2 (book, sublist[i])) == NULL)
	goto error;

      if (strcmp (name, dir) == 0) {
	if (eb_set_subbook (book, sublist[i]) < 0)
	  goto error;
	return 1;
      }
    }
    printf ("No such dictionry: %s\n", name);
    return 0;
  }

 error:
  printf ("An error occurred in parse_dict_id: %s\n", eb_error_message ());
  set_error_message ();
  return 0;
}

int
parse_entry_id (code, pos)
     char *code;
     EB_Position *pos;
{
  if (strchr (code, ':') != NULL) {
    /*
     * Encoded position
     */
    char *endp;
    pos->page = strtol (code, &endp, 16);
    if (*endp != ':')
      goto illegal;

    pos->offset = strtol (endp + 1, &endp, 16);
    if (*endp != '\0')
      goto illegal;

    return 1;

  illegal:
    printf ("Illegal position: %s\n", code);
    return 0;

  } else {
    /*
     * Numbered entry
     */
    int num, count;
    const char *pattern = variable_ref ("_last_search_pattern");
    EB_Hit list[MAX_HIT_SIZE];

    if (!pattern) {
      puts ("No search has been executed yet.");
      return 0;
    }
    if ((count = atoi (code) - 1) < 0) {
      printf ("Invalid entry number: %s\n", code);
      return 0;
    }
    if (check_subbook ()) {
      if (last_search_function (&current_book, pattern) < 0) {
	printf ("An error occured in parse_entry_id: %s\n",
		eb_error_message ());
	set_error_message ();
	return 0;
      }
      while ((num = eb_hit_list (&current_book, list, MAX_HIT_SIZE)) > 0) {
	if (count < num) {
	  pos->page = list[count].text.page;
	  pos->offset = list[count].text.offset;
	  return 1;
	}
	count -= num;
	pattern = NULL;
      }
      if (num == 0)
	printf ("Too big: %s\n", code);
    }
    return 0;
  }
}


int
search_pattern (book, appendix, pattern, begin, length)
     EB_Book *book;
     EB_Appendix *appendix;
     char *pattern;
     int begin;
     int length;
{
  int i, num, point;
  char headbuf1[BUFSIZ];
  char headbuf2[BUFSIZ];
  char *head;
  const char *s;
  int (*search) EB_P ((EB_Book *, const char *));
  EB_Hit hitlist[MAX_HIT_SIZE];

  char* prevhead;
  int prevpage;
  int prevoffset;

  if (pattern == NULL) {
    /* check last search */
    begin = last_search_begin;
    length = last_search_length;
    search = last_search_function;
    pattern = variable_ref ("_last_search_pattern");
    if (pattern == NULL) {
      puts ("No search has been executed yet.");
      return 0;
    }
    if (last_search_begin == 0) {
      printf ("Last search had finished\n");
      return 0;
    }
  } else {
    /* get search method */
    if ((s = variable_ref ("search-method")) == NULL)
      s = default_method;

    if (strcmp (s, "exact") == 0)
      search = eb_search_exactword;
    else if (strcmp (s, "word") == 0)
      search = eb_search_word;
    else if (strcmp (s, "endword") == 0)
      search = eb_search_endword;
    else if (strcmp (s, "glob") == 0) {
      search = eb_search_exactword;

      /* check for word search */
      i = strlen (pattern) - 1;
      if (pattern[i] == '*') {
	pattern[i] = '\0';
	search = eb_search_word;
      }

      /* check for endword search */
      if (pattern[0] == '*') {
	pattern++;
	search = eb_search_endword;
      }
    } else {
      printf ("Invalid search method: %s\n", s);
      return 0;
    }
  }

  /* reserve search information */
  variable_set ("_last_search_book", eb_path (book));
  variable_set ("_last_search_dict", eb_subbook_directory (book));
  variable_set ("_last_search_pattern", pattern);
  last_search_begin = 0;
  last_search_length = length;
  last_search_function = search;

  /* search */
  point = 0;
  if (search (book, pattern) < 0) {
    printf ("An error occured in search_pattern: %s\n", eb_error_message ());
    set_error_message ();
    return 0;
  }


  head = headbuf1;
  prevhead = headbuf2;
  *prevhead = '\0';
  prevpage = 0;
  prevoffset = 0;

  while ((num = eb_hit_list (book, hitlist, MAX_HIT_SIZE)) > 0) {
    for (i = 0; i < num; i++) {
      point++;
      if (point >= begin + length && length > 0) {
	printf ("<more point=%d>\n", point);
	last_search_begin = point;
	goto exit;
      }

      if (point >= begin) {
	eb_seek (book, &hitlist[i].heading);
	eb_heading (book, appendix, &text_hookset, head, BUFSIZ - 1);

	if (prevpage == hitlist[i].text.page &&
	    prevoffset == hitlist[i].text.offset &&
	    strcmp(head, prevhead) == 0)
	  continue;

	printf ("%2d. %x:%x\t", point,
	       hitlist[i].text.page, hitlist[i].text.offset);
	eblook_write (head, stdout);
	fputc ('\n', stdout);
      }

      if (head == headbuf1) {
	head = headbuf2;
	prevhead = headbuf1;
      } else {
	head = headbuf1;
	prevhead = headbuf2;
      }
      prevpage = hitlist[i].text.page;
      prevoffset = hitlist[i].text.offset;
    }
  }

 exit:
  return 1;
}

int
insert_content (book, appendix, pos, begin, length)
     EB_Book *book;
     EB_Appendix *appendix;
     EB_Position *pos;
     int begin;
     int length;
{
  int point, len;
  char last = '\n';
  char buff[EB_SIZE_PAGE];

  /* insert */
  point = 0;
  eb_seek (book, pos);
  while ((len = eb_text (book, appendix, &text_hookset, buff, EB_SIZE_PAGE - 1)) > 0) {
    /* count up */
    point++;
    if (point >= begin + length && length > 0) {
      printf ("<more point=%d>\n", point);
      goto exit;
    }

    /* insert */
    if (point >= begin) {
      eblook_write (buff, stdout);
      last = buff[len - 1];
    }
  }

  /* insert a newline securely */
  if (last != '\n')
    putchar ('\n');

 exit:
  return 1;
}

int
insert_font (book, id)
     EB_Book *book;
     const char *id;
{
  int ch, size, width, height;
  unsigned char bitmap[EB_SIZE_WIDE_FONT_48];
  unsigned char xbm[EB_SIZE_WIDE_FONT_48_XBM];

  switch (*id) {
  case 'h':
    ch = strtol (id + 1, NULL, 16);
    if (eb_narrow_font_start (book) <= ch && ch <= eb_narrow_font_end (book)) {
      width = eb_narrow_font_width (book);
      if (eb_narrow_font_character_bitmap (book, ch, bitmap) < 0)
	goto error;
    } else {
      printf ("No such character font: %s\n", id);
      return 0;
    }
    break;

  case 'z':
    ch = strtol (id + 1, NULL, 16);
    if (eb_wide_font_start (book) <= ch && ch <= eb_wide_font_end (book)) {
      width = eb_wide_font_width (book);
      if (eb_wide_font_character_bitmap (book, ch, bitmap) < 0)
	goto error;
    } else {
      printf ("No such character font: %s\n", id);
      return 0;
    }
    break;

  default:
    printf ("Invalid font id: %s\n", id);
    return 0;
  }

  height = eb_font (book);
  size = eb_bitmap_to_xbm (xbm, bitmap, width, height);
  xbm[size] = '\0';
  fputs (xbm, stdout);
  return 1;

 error:
  printf ("An error occured in insert_font: %s\n", eb_error_message ());
  set_error_message ();
  return 0;
}

int
insert_font_list (book)
     EB_Book *book;
{
  int ch, width, height, size, start, end;
  unsigned char bitmap[EB_SIZE_WIDE_FONT_48];
  unsigned char xbm[EB_SIZE_WIDE_FONT_48_XBM];

  height = eb_font (book);

  width = eb_narrow_font_width (book);
  start = eb_narrow_font_start (book);
  end = eb_narrow_font_end (book);
  for (ch = start; ch < end; ch++)
    if (eb_narrow_font_character_bitmap (book, ch, bitmap) >= 0) {
      size = eb_bitmap_to_xbm (xbm, bitmap, width, height);
      xbm[size] = '\0';
      fprintf (stdout, "\nid = h%04x\n", ch);
      fputs (xbm, stdout);
    }

  width = eb_wide_font_width (book);
  start = eb_wide_font_start (book);
  end = eb_wide_font_end (book);
  for (ch = start; ch < end; ch++)
    if (eb_wide_font_character_bitmap (book, ch, bitmap) >= 0) {
      size = eb_bitmap_to_xbm (xbm, bitmap, width, height);
      xbm[size] = '\0';
      fprintf (stdout, "\nid = z%04x\n", ch);
      fputs (xbm, stdout);
    }
  return 1;
}


int
hook_font (book, appendix, buff, code, argc, argv)
     EB_Book *book;
     EB_Appendix *appendix;
     char *buff;
     EB_Hook_Code code;
     int argc;
     const int *argv;
{
  switch (code) {
  case EB_HOOK_NARROW_FONT:
    if (eb_narrow_alt_character_text (appendix, argv[0], buff) < 0)
      sprintf (buff, "<gaiji=h%04x>", argv[0]);
    break;

  case EB_HOOK_WIDE_FONT:
    if (eb_wide_alt_character_text (appendix, argv[0], buff) < 0)
      sprintf (buff, "<gaiji=z%04x>", argv[0]);
    break;
  }
  return 0;
}

int
hook_newline (book, appendix, buff, code, argc, argv)
     EB_Book *book;
     EB_Appendix *appendix;
     char *buff;
     EB_Hook_Code code;
     int argc;
     const int *argv;
{
  strcpy (buff, "\n");
  return 0;
}

int
hook_stopcode (book, appendix, buff, code, argc, argv)
     EB_Book *book;
     EB_Appendix *appendix;
     char *buff;
     EB_Hook_Code code;
     int argc;
     const int *argv;
{
  const char *stop = variable_ref ("stop-code");

  if (stop) {
    int c;
    if (strncmp (stop, "0x", 2) == 0)
      c = strtol (stop + 2, NULL, 16);
    else
      c = atoi (stop);

    if (c == (argv[0] << 16) + argv[1])
      return -1;
    else
      return 0;
  }

  return eb_hook_stopcode_mixed (book, appendix, buff, code, argc, argv);
}

int
hook_tags (book, appendix, buff, code, argc, argv)
     EB_Book *book;
     EB_Appendix *appendix;
     char *buff;
     EB_Hook_Code code;
     int argc;
     const int *argv;
{
  switch (code) {
  case EB_HOOK_BEGIN_PICTURE:
    strcpy (buff, "<picture>"); break;
  case EB_HOOK_END_PICTURE:
    sprintf (buff, "</picture=%x:%x>", argv[1], argv[2]); break;

  case EB_HOOK_BEGIN_SOUND:
    strcpy (buff, "<sound>"); break;
  case EB_HOOK_END_SOUND:
    sprintf (buff, "</sound=%x:%x>", argv[1], argv[2]); break;

  case EB_HOOK_BEGIN_MENU:
  case EB_HOOK_BEGIN_REFERENCE:
    strcpy (buff, "<reference>"); break;
  case EB_HOOK_END_MENU:
  case EB_HOOK_END_REFERENCE:
    sprintf (buff, "</reference=%x:%x>", argv[1], argv[2]); break;
  }
  return 0;
}


void
show_version ()
{
  printf ("%s %s (with EB %d.%d)\n", program_name, program_version,
	  EB_VERSION_MAJOR, EB_VERSION_MINOR);
  puts ("Copyright (C) 1997,1998,1999 NISHIDA Keisuke");
  puts ("eblook may be distributed under the terms of the GNU General Public Licence;");
  puts ("certain other uses are permitted as well.  For details, see the file");
  puts ("`COPYING', which is included in the Guile distribution.");
  puts ("There is no warranty, to the extent permitted by law.");
}

void
show_help ()
{
  fprintf (stderr, "Usage: %s [option...] [book-directory [appendix-directory]]\n", program_name);
  fprintf (stderr, "Options:\n");
  fprintf (stderr, "  -q, --no-init-file    ignore user init file\n");
  fprintf (stderr, "  -h, --help            show this message\n");
  fprintf (stderr, "  -v, --version         show version number\n");
  fflush (stderr);
}

void
show_try_help ()
{
  fprintf (stderr, "Try `%s --help' for more information.\n", invoked_name);
  fflush (stderr);
}

void
set_error_message ()
{
  variable_set ("_error", strerror (errno));
  variable_set ("_eb_error", eb_error_message ());
}

void
unset_error_message ()
{
  variable_set ("_error", NULL);
  variable_set ("_eb_error", NULL);
}


StringAlist *
salist_set (alist, key, value)
     StringAlist *alist;
     const char *key;
     const char *value;
{
  StringAlist *var;

  if (value) {
    /*
     * Set KEY to VALUE
     */
    StringAlist *prev = NULL;
    for (var = alist; var != NULL; var = var->next) {
      if (strcmp (key, var->key) == 0) {
	/* update original value */
	char *p = strdup (value);
	if (p != NULL) {
	  free (var->value);
	  var->value = p;
	} else {
	  puts ("memory full");
	  if (prev)
	    prev->next = var->next;
	  else
	    alist = var->next;

	  free (var->key);
	  free (var->value);
	  free (var);
	}
	break;
      }
      prev = var;
    }
    if (var == NULL) {
      /* add new element */
      if ((var = malloc (sizeof (StringAlist))) == NULL) {
	puts ("memory full");
      } else if ((var->key = strdup (key)) == NULL ||
	       (var->value = strdup (value)) == NULL) {
	puts ("memory full");
	free (var->key);
	free (var);
      } else {
	var->next = alist;
	alist = var;
      }
    }
  } else {
    /*
     * Delete element
     */
    StringAlist *prev = NULL;
    for (var = alist; var != NULL; var = var->next) {
      if (strcmp (key, var->key) == 0) {
	/* delete from alist */
	if (prev)
	  prev->next = var->next;
	else
	  alist = var->next;

	/* free */
	free (var->key);
	free (var->value);
	free (var);
	break;
      }
      prev = var;
    }
  }

  return alist;
}

char *
salist_ref (alist, key)
     StringAlist *alist;
     const char *key;
{
  StringAlist *var;
  for (var = alist; var != NULL; var = var->next)
    if (strcmp (key, var->key) == 0)
      return var->value;

  return NULL;
}


int
eblook_read (string)
     char *string;
{
  int ret = 0;
  switch (kanji_code) {
  case JIS:
    ret = jis2euc (string);
    break;
  case SJIS:
    ret = sjis2euc (string);
    break;
  case EUC:
    break;
  }
  return ret;
}

void
eblook_write (string, fp)
     const char *string;
     FILE *fp;
{
  switch (kanji_code) {
  case JIS:
    write_euc2jis (string, fp);
    break;
  case SJIS:
    write_euc2sjis (string, fp);
    break;
  default:
    fputs (string, fp);
  }
}

int
jis2euc (string)
     unsigned char *string;
{
  int kanji = 0;
  unsigned char *i;
  unsigned char *o;

  for (i = o = string; *i; i++, o++) {
    if (!kanji && *i == 033 && i[1] == '$' && i[2] == 'B') {
      kanji = 1;
      i += 3;
    } else if (kanji && *i == 033 && i[1] == '(' && i[2] == 'B') {
      kanji = 0;
      i += 3;
    }
    if (kanji)
      *o = *i | 0x80;
    else
      *o = *i;
  }
  *o = '\0';
  return 0;
}

int
sjis2euc (string)
     unsigned char *string;
{
  unsigned char *p;

  for (p = string; *p; p++) {
    if ((0xa1 <= *p) && (*p <= 0xdf)) {
      /* we don't handle jisx0201-kana yet (ever?) */
      return -1;
    } else if (*p >= 0x80) {
      if (p[1] == '\0') return -1;
      if (p[1] >= 0x9f) {
	p[0] += p[0] - ((p[0] >= 0xe0) ? 0xe0 : 0x60);
	p[1] += 2;
      } else {
	p[0] += p[0] - ((p[0] >= 0xe0) ? 0xe1 : 0x61);
	p[1] += 0x60 + ((p[1] < 0x7f) ? 1 : 0);
      }
      p++;
    }
  }
  return 0;
}

void
write_euc2jis (string, fp)
     const unsigned char *string;
     FILE *fp;
{
  int kanji = 0;
  const unsigned char *p;

  for (p = string; *p; p++) {
    if (*p > 0x80) {
      if (!kanji) {
	fputs ("\033$B", fp);
	kanji = 1;
      }
      fputc (*p & 0x7f, fp);
    } else {
      if (kanji) {
	fputs ("\033(B", fp);
	kanji = 0;
      }
      fputc (*p, fp);
    }
  }
  if (kanji)
    fputs ("\033(B", fp);
}

void
write_euc2sjis (string, fp)
     const unsigned char *string;
     FILE *fp;
{
  const unsigned char *p;
  unsigned char c1;
  unsigned char c2;

  for (p = string; *p; p++) {
    if (*p < 0x80) {
      fputc(*p, fp);
    } else {
      c1 = p[0];
      c2 = p[1];
      if (c1 % 2) {
	c1 = (c1 >> 1) + ((c1 < 0xdf) ? 0x31 : 0x71);
	c2 -= 0x60 + ((c2 < 0xe0) ? 1 : 0);
      } else {
	c1 = (c1 >> 1) + ((c1 < 0xdf) ? 0x30 : 0x70);
	c2 -= 2;
      }
      fputc(c1, fp);
      fputc(c2, fp);
      p++;
    }
  }
}

