
#ifndef __autil_string_map__
#define __autil_string_map__

#include <string>
#include <iostream>
#include <utility>
#include "copy_ptr.hh"
#include "emulation.hh"
#include "container.hh"

// A string map is a simple hash table where the key and values
// are strings.  It also has the ability to write and read data
// files of a standard format.
// It is perfect for storing word pairs for "Replace All".

namespace autil {

  class StringMap : public MutableContainer {
  private:
    class  Data;
    class  String;
    struct Parms;
    CopyPtr<Data> data;
  protected:
    string & find(const string & key);
  public:
    typedef pair<const string,string> Value;
    typedef const Value &             Ref;
    typedef VirEmulation<Ref>         VirEmul;
    typedef Emulation<Ref>            Emul;

    StringMap();
    StringMap(const StringMap & other);
    virtual ~StringMap();

    VirEmul * elements() const;
    
    virtual void insert(const string & key, const string & value);
    // note: insert will NOT overwrite an existing entry
    virtual void replace(const string & key, const string & value);

    virtual void remove(const string & key);
    const string & operator[] (const string & key) const {return lookup(key);}
    virtual const string & lookup (const string & key) const;
    virtual bool have(const string & key) const;

    void insert(const char * key) {insert(key,"true");}
    void remove(const char * key) {remove(static_cast<string>(key));}
    
    void clear();

    void merge(const StringMap & other);

    enum AddAction {Insert, Replace};
    virtual void read_in_stream(istream &, char delim = '\n', AddAction a = Replace);
    virtual void read_in_file(const string &, AddAction a = Replace);
    virtual void read_in_string(const string &, AddAction a = Replace);

    virtual void write_to_stream(ostream &) const;
    virtual void write_to_file(const string &) const;
  };
  
}

#endif
