# mktex.pm -- shared stuff for Perl mktex* versions
# 
# Based on the various /bin/sh versions:
#  mktex.opt:
#   te@informatik.uni-hannover.de and kb@mail.tug.org. Public domain.
#   RCS Id: mktex.opt,v 1.28 1999/02/18 19:57:20 olaf Exp
#  mktexnam:
#   te@informatik.uni-hannover.de, kb@mail.tug.org, and infovore@xs4all.nl.
#   Public domain.
#   RCS Id: mktexnam,v 1.22 1999/05/30 18:57:40 olaf Exp
#  mktexnam.opt:
#   te@informatik.uni-hannover.de and kb@mail.tug.org. Public domain.
#   RCS Id: mktexnam.opt,v 1.10 1999/02/14 16:04:57 olaf Exp
#  mktexdir:
#   te@informatik.uni-hannover.de and kb@mail.tug.org. Public domain.
#   RCS Id: mktexdir,v 1.13 1999/05/30 18:36:24 olaf Exp
#  mktexupd:
#   te@informatik.uni-hannover.de and kb@mail.tug.org. Public domain.
#   RCS Id: mktexupd,v 1.15 1999/02/14 16:08:59 olaf Exp
# 
# 
# Perl version:
# $Id: Mktex.pm,v 1.11 1999/11/18 00:33:01 jdg Exp $
# Copyright Julian Gilbey <jdg@debian.org> 1999
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

package TeX::Mktex;

# Modules we need within this file
use POSIX ();
use Cwd;
use Getopt::Long;
use TeX::Kpsewhich;
use File::Basename;

require Exporter;
@ISA = qw(Exporter);

@EXPORT=qw(strip_quotes mktex_opt mktex_names mktex_dir mktex_upd
		$kpse_plain @cleanfiles);
@EXPORT_OK=qw(mknam_nomfdrivers $MT_FEATURES $TEMPDIR $DPI $BDPI $MODE
		$MAG $ps_to_pk);
%EXPORT_TAGS=qw();

# And we import progname
*progname = \$main::progname;

# Some useful routines
sub setdefault {
	my ($var,$val) = @_;
	$$var = $ENV{$var} || $val;
}

sub setordefault {
	my ($var,$val) = @_;
	$$var ||= $ENV{$var} || $val;
}

sub setkpsedefault {
	my ($var,$name,$fallback) = @_;
	$$var = $ENV{VAR} || $kpse_w2c->find($name) || $fallback;
}

sub strip_quotes {
	my $lines=$_[0];
	$lines =~ s/^:[ \t]+//gm;
	return $lines;
}

################   Stuff from mktex.opt   ####################

# This deals with command-line options, setting up variables, temp dirs
# and the like.

# Start with the BEGIN and END functions, which deal with cleaning up
# on normal or abnormal exit.
# The "cd /" is for MS-DOS and MS-Windows, where there is a separate
# current directory on each drive, and therefore "cd $KPSE_DOT" might
# still leave $TMPDIR current directory on its drive, in which case it
# cannot be removed.
# We start by setting signal handlers to call die.  In the END routine,
# we ensure that we only go further if we have at least initialised $TEMPDIR.
BEGIN {
	# We use the same signal list as used in the shell-script version
	$SIG{'HUP'} = $SIG{'INT'} = $SIG{'QUIT'} = $SIG{'PIPE'} =
		$SIG{'TERM'} = $SIG{'BUS'} = \&CORE::die;
	# We define this now so that its use later won't cause
	# problems due to malloc being non-reentrant
	@cleanfiles=();
}
END {
	# Have we actually done anything yet?
	return unless defined($TEMPDIR) && -d $TEMPDIR;
	unlink @cleanfiles;
	chdir '/';
	chdir $ENV{'KPSE_DOT'};
	system ("rm -rf $TEMPDIR") >> 8 == 0
		or warn "$progname: Couldn't remove temporary directory on exit!\n";
}


# Handle the command-line options; the GetOptions parameters are
# passed as the function parameters.
sub parse_options {
	Getopt::Long::config(qw(require_order));
	if (! GetOptions('help' => \$help, 'version' => \$opt_version, @_)) {
		die "$progname: error parsing command-line options!\n" .
			"Try $progname --help for more information.\n";
	}

	if ($help) {
		print $main::usage;
		exit 0;
	}
	if ($opt_version) {
		print "$progname $main::version\n";
		print TeX::Kpsewhich::version();
		exit 0;
	}
}

# The init function handles printing help or version info if requested,
# checking the remaining arguments, setting up the temporary directory
# and setting up a few necessary variables,

sub init {
	# Defaults...
	$main::mt_min_args=1 if ! defined $main::mt_min_args;
	$main::mt_max_args=1 if ! defined $main::mt_max_args;

	if (@ARGV < $main::mt_min_args) {
		die "$progname: Missing argument(s).\n" .
			"Try `$progname --help' for more information.\n";
	}
	if (@ARGV > $main::mt_max_args) {
		die "$progname: Extra argument(s): @ARGV[$main::mt_max_args..$#ARGV]\n" .
			"Try `$progname --help' for more information.\n";
	}

	# MS-DOS and MS-Windows define $COMSPEC or $ComSpec and use `;' to
	# separate directories in path lists whereas Unix uses `:'.
	# Create a variable that holds the right character to be used by
	# the scripts.
	$SEP=(exists $ENV{'COMSPEC'} or exists $ENV{'ComSpec'}) ? ';' : ':';

	# We will change search paths to include $KPSE_DOT. This is necessary
	# since we cd to $TEMPDIR.
	setdefault('KPSE_DOT',cwd());
	$ENV{'KPSE_DOT'}=$KPSE_DOT;

	# TEMPDIR needs to be unique to each process because of the possibility
	# of two people running dvips (or whatever) simultaneously.
	$TEMPDIR=($ENV{'TMPDIR'} || '/tmp') . "/mt$$.tmp";

	mkdir "$TEMPDIR", 0700 or die "$progname: Couldn't make $TEMPDIR!\n";
	chdir "$TEMPDIR"  or die "$progname: Couldn't chdir $TEMPDIR!\n";

	# The alternative location for fonts.  The directory need not exist,
	# so we use --expand-var here.  We also perform a sanity check.
	$MT_VARTEXFONTS = $ENV{'MT_VARTEXFONTS'} ||
		$kpse_plain->expand_var('$VARTEXFONTS');
	$MT_VARTEXFONTS =~ s/^!!//;
	if ($MT_VARTEXFONTS eq '') {
		warn "$progname: VARTEXFONTS is not defined.  Defaulting to \`$KPSE_DOT'.\n";
		$MT_VARTEXFONTS=$KPSE_DOT;
	}
	elsif ($MT_VARTEXFONTS =~ /(.*?)$SEP/) {
		warn "$progname: VARTEXFONTS has multiple paths; using only the first.";
		$MT_VARTEXFONTS=$1;
	}

	# The supporting scripts: "perl" means call the internal Perl version
	setdefault('MT_MKTEXNAM', 'perl');
	setdefault('MT_MKTEXNAM_OPT', 'perl');
	if (($MT_MKTEXNAM eq 'perl' and $MT_MKTEXNAM_OPT ne 'perl') or
		 ($MT_MKTEXNAM ne 'perl' and $MT_MKTEXNAM_OPT eq 'perl')) {
		die "$progname: Either neither of or both of MT_MKTEXNAM and MT_MKTEXNAM_OPT\nmay be set to 'perl'!\n";
	}
	setdefault('MT_MKTEXDIR', 'perl');
	setdefault('MT_MKTEXDIR_OPT', 'perl');
	if (($MT_MKTEXDIR eq 'perl' and $MT_MKTEXDIR_OPT ne 'perl') or
		 ($MT_MKTEXDIR ne 'perl' and $MT_MKTEXDIR_OPT eq 'perl')) {
		die "$progname: Either neither of or both of MT_MKTEXDIR and MT_MKTEXDIR_OPT\nmay be set to 'perl'!\n";
	}
	setdefault('MT_MKTEXUPD', 'perl');

	# mktexupd and mktexlsr make sure they're coordinated via this.  A copy of
	# the first string is found in mktexlsr.
	$ls_R_magic=
		'% ls-R -- filename database for kpathsea; do not change this line.';
	# Old ls-R files should continue to work.
	$old_ls_R_magic=
		'% ls-R -- maintained by MakeTeXls-R; do not change this line.';
}

# Read defaults from mktex.cnf if that file exists. Can be used to
# overwrite anything defined below.  In that file, please use the syntax
# ``: ${MODE=ljfour}'', not just ``MODE=ljfour'', as that is what this
# script expects, and that is what is necessary for the shell version of
# mktex.opt.  We put the results into the appropriate envvars, as the cnf
# file values have that sort of status within the suite.

sub read_cnf {
	setkpsedefault('MT_MKTEX_CNF', 'mktex.cnf', '');
	if ($MT_MKTEX_CNF && -r $MT_MKTEX_CNF) {
		open CNF, "<$MT_MKTEX_CNF"
			or die "$progname: Couldn't open $MT_MKTEX_CNF for reading: $!\n";
		while (<CNF>) {
			next if /^\s*#/;
			next if /^\s*$/;
			# Careful not to use \w because of locale problems; "no locale"
			# would be awkward here if these scripts want to be localised
			if (/^\s*:\s*\${([A-Za-z][A-Za-z0-9_]*)=([^{}]*)}\s*$/) {
				$ENV{$1} ||= $2;
				next;
			}
			warn "$progname: skipping unparseable line in $MT_MKTEX_CNF:\n$_";
		}
		close CNF
			or die "$progname: Problem reading $MT_MKTEX_CNF: $!\n";
	}
}

sub setdefaults {
	# We have to be careful, as MT_FEATURES might be set to be empty
	if (! defined $MT_FEATURES) {
		if (exists $ENV{'MT_FEATURES'}) { $MT_FEATURES=$ENV{'MT_FEATURES'} }
		else {
			$MT_FEATURES = exists($ENV{'COMSPEC'}) ?
				'appendonlydir:dosnames' : 'appendonlydir' }
	}
	setdefault('MODE', 'ljfour');
	setdefault('BDPI', 600);
	setdefault('DPI', 600);
	setdefault('MAG', 1.0);
	setdefault('ps_to_pk', 'gsftopk');

	# Care about the umask; we will change it if necessary before
	# performing a mkdir
	umask 022;

	# Export the values we have found ???
}

# Tie them all together:
sub mktex_opt {
	# These will be needed before we go much further!
	$kpse_plain = TeX::Kpsewhich->new();
	$kpse_w2c = TeX::Kpsewhich->new('-format' => 'web2c files');

	parse_options(@_);
	init();
	read_cnf();
	setdefaults();
}


################   Stuff from mktexnam   ####################

# We will call the mktexnam program through this subroutine instead of
# through an external program.  We check first, though, whether an
# external mktexnam has been selected instead.  In that case, we presume
# that the program returns a string of the form "PKNAME:TFMNAME:MFNAME".

sub mktex_names {
	if ($MT_MKTEXNAM eq 'perl') { return mktex_names_perl(@_); }
	else {
		$ENV{'MT_MKTEXNAM_OPT'} = $MT_MKTEXNAM_OPT;
		my $cmd_args = join ' ', map { "'$_'" } @_;
		my $ret = `$MT_MKTEXNAM $cmd_args`;
		die "$MT_MKTEXNAM $cmd_args failed!\n" if ($? >> 8);
		return wantarray ? split /$SEP/o, $ret : $ret;
	}
}

sub mktex_names_perl {
	$NAME=$_[0];
	local($MODE,$DPI,$DEST)=($MODE,$DPI);
	if (@_ == 2) { $DEST=$_[1] }
	elsif (@_ > 2) { $DPI=$_[1]; $MODE=$_[2]; $DEST=$_[3]; }

	setdefault('MT_PKDESTDIR', '$MT_DESTROOT/$MT_PKDESTREL');
	setdefault('MT_TFMDESTDIR', '$MT_DESTROOT/$MT_TFMDESTREL');
	setdefault('MT_MFDESTDIR', '$MT_DESTROOT/$MT_MFDESTREL');
	setdefault('MT_PKBASE', '$NAME.${DPI}pk');
	setdefault('MT_TFMBASE', '$NAME.tfm');
	setdefault('MT_MFBASE', '$NAME.mf');
	setdefault('MT_DEFAULT_SUPPLIER', 'unknown');
	setdefault('MT_DEFAULT_TYPEFACE', 'unknown');
	setdefault('MT_DEFAULT_NAMEPART', '$MT_SUPPLIER/$MT_TYPEFACE');
	setdefault('MT_DEFAULT_PKDESTREL', 'pk/modeless/$MT_NAMEPART');
	setdefault('MT_DEFAULT_TFMDESTREL', 'tfm/$MT_NAMEPART');
	setdefault('MT_DEFAULT_MFDESTREL', 'source/$MT_NAMEPART');
	setdefault('MT_DEFAULT_DESTROOT', $KPSE_DOT);
	# Also have to set $USE_VARTEXFONTS if it's set in the environment
	$USE_VARTEXFONTS=$ENV{'USE_VARTEXFONTS'};

	# Take care to clear certain variables we will assume are undefined
	# before we begin.
	undef $MT_SUPPLIER;
	undef $MT_TYPEFACE;
	undef $MT_NAMEPART;
	undef $MT_PKDESTREL;
	undef $MT_TFMDESTREL;
	undef $MT_MFDESTREL;

	# Find the font: test tfm first, then mf, then possible sauterized mf.
	$FULLNAME = $kpse_plain->find("$NAME.tfm") ||
		$kpse_plain->find("$NAME.mf");
	if (! $FULLNAME) {
		($ROOTNAME=$NAME) =~ s/\d*$//;
		$FULLNAME = $kpse_plain->find("b-$ROOTNAME.mf");
		# Some fonts get special treatment:
		if (! $FULLNAME) {
			# Czech/Slovak fonts get special treatment:
			if ($ROOTNAME =~ /^(cs|i?lcsss|ics(csc|tt))/) {
				$FULLNAME = $kpse_plain->find("cscode.mf");
			}
			# LH fonts get special treatment:
			elsif ($ROOTNAME =~ m/^( # This is a horrendous regexp
						wn[bcdfirstuv]|
						rx[bcdfiorstuvx][bcfhilmostx]|
						l[abchl][bcdfiorstuvx]
					)/x) {
				$FULLNAME = $kpse_plain->find(substr($ROOTNAME,0,2) . 'codes.mf');
			}
			else {
				$FULLNAME = $kpse_plain->find("$ROOTNAME.mf");
			}
		}
	}

	# After all this, do we _have_ a font?
	if (! $FULLNAME) {
		setdefault('MT_DESTROOT', $MT_VARTEXFONTS);
	}
	else {
		# Normalize $FULLNAME.
		$FULLNAME =~ s,/+,/,g;
		# See if $FULLNAME comes from a standard location.
		@font_locs = split /$SEP/o,
			$kpse_plain->expand_path('$TEXMF/fonts' . $SEP . $MT_VARTEXFONTS);
		foreach $loc (@font_locs) {
			next if ! $loc;
			next unless $FULLNAME =~ m,^$loc/,;
			# We now have a preliminary value for the destination root.
			setdefault('MT_DESTROOT', $loc);

			# When we're done, $relfmt contains one of these:
			# "/source/$MT_NAMEPART/"
			# "/$MT_NAMEPART/source/"
			# while namepart contains the actual namepart.
			# We perform this substitution in multiple steps for clarity and
			# to parallel the shell version.
			$relfmt = $FULLNAME;
			$relfmt =~ s,^$loc(/.*/)[^/]*$,$1,;
			$relfmt =~ s,^/tfm/,/source/,;
			$relfmt =~ s,/tfm/$,/source/,;
			($namepart=$relfmt) =~ s,/source/,/,;

			# See if namepart is empty.
			if ($namepart ne '/') {
				$relfmt =~ s,$namepart,/\$MT_NAMEPART/,;
			} else {
				# Assume TDS.
				$relfmt='/source/$MT_NAMEPART/';
				$namepart='';
			}

			# Determine supplier and typeface from namepart.  If there is
			# only one part in the namepart, we take it to be the typeface.
			($MT_SUPPLIER, $MT_TYPEFACE) = ($namepart =~ m,^/([^/]*)/(.*)$,);
			if (! $MT_TYPEFACE) {
				$MT_TYPEFACE=$MT_SUPPLIER;
				$MT_SUPPLIER='';
			}

			# Phew.  Now we create the relative paths for pk, tfm and source.
			setdefault('MT_NAMEPART', '$MT_SUPPLIER/$MT_TYPEFACE');
			($tmp=$relfmt) =~ s,/source/,/pk/\$MT_MODE/,;
			setdefault('MT_PKDESTREL', $tmp);
			($tmp=$relfmt) =~ s,/source/,/tfm/,;
			setdefault('MT_TFMDESTREL', $tmp);
			setdefault('MT_MFDESTREL', $relfmt);

			# And we're done with the loop now.
			last;
		}
	}

	# In case some variables remain unset.
	setordefault('MT_SUPPLIER', $MT_DEFAULT_SUPPLIER);
	setordefault('MT_TYPEFACE', $MT_DEFAULT_TYPEFACE);
	setordefault('MT_NAMEPART', $MT_DEFAULT_NAMEPART);
	setordefault('MT_PKDESTREL', $MT_DEFAULT_PKDESTREL);
	setordefault('MT_TFMDESTREL', $MT_DEFAULT_TFMDESTREL);
	setordefault('MT_MFDESTREL', $MT_DEFAULT_MFDESTREL);

	# We have found nothing, so force using the fontmaps as a last resort.
	# This also means mktexnam can be queried for advice on where to
	# place fonts.
	if ($MT_SUPPLIER eq 'unknown' or $MT_TYPEFACE eq 'unknown') {
		$MT_FEATURES .= ':fontmap';
	}

	# Handle the options; this is now performed by calling the appropriate
	# Perl sub
	mktexnames_opt();

	# The following is modified to perform a more detailed check of the
	# directory permissions than in the shell-script version.

	if (defined $DEST) {
		if ($DEST =~ m,^([A-Za-z]:)?/,) {
			# Absolute, explicit destdir => use it.
			$MT_PKDESTDIR = $MT_TFMDESTDIR = $MT_MFDESTDIR = $DEST;
			$MT_NAMEPART = '';
			$MT_DESTROOT = $DEST;
			$USE_ALTERNATE=0;
		} else {
			# Relative destdir => append to the default.
			$MT_NAMEPART = $DEST;
		}
	}

	$SAVE_MODE = $MT_MODE;
	$SAVE_NAMEPART = $MT_NAMEPART;
	$SAVE_DESTROOT = $MT_DESTROOT;
	$SAVE_PKDESTREL = $MT_PKDESTREL;
	$SAVE_TFMDESTREL = $MT_TFMDESTREL;
	$SAVE_MFDESTREL = $MT_MFDESTREL;
	$SAVE_PKDESTDIR = $MT_PKDESTDIR;
	$SAVE_TFMDESTDIR = $MT_TFMDESTDIR;
	$SAVE_MFDESTDIR = $MT_MFDESTDIR;
	$SAVE_PKNAME = $MT_PKNAME;
	$SAVE_TFMNAME = $MT_TFMNAME;
	$SAVE_MFNAME = $MT_MFNAME;

	$MT_MODE = eval qq("$MODE");
	$MT_NAMEPART = eval qq("$MT_NAMEPART");
	$MT_DESTROOT = eval qq("$MT_DESTROOT");
	$MT_PKDESTREL = eval qq("$MT_PKDESTREL");
	$MT_TFMDESTREL = eval qq("$MT_TFMDESTREL");
	$MT_MFDESTREL = eval qq("$MT_MFDESTREL");
	$MT_PKDESTDIR = eval qq("$MT_PKDESTDIR");
	$MT_TFMDESTDIR = eval qq("$MT_TFMDESTDIR");
	$MT_MFDESTDIR = eval qq("$MT_MFDESTDIR");
	$MT_PKNAME = eval qq("$MT_PKDESTDIR/$MT_PKBASE");
	$MT_TFMNAME = eval qq("$MT_TFMDESTDIR/$MT_TFMBASE");
	$MT_MFNAME = eval qq("$MT_MFDESTDIR/$MT_MFBASE");

	# Adjust MT_DESTROOT, if necessary.
	if ($MT_DESTROOT) {
		# Do we have write access and permission?
		my $d;
		foreach $d ($MT_MFDESTDIR, $MT_TFMDESTDIR, $MT_PKDESTDIR) {
			while (! -d $d) { $d=dirname($d); }
			unless (POSIX::access($MT_DESTROOT, POSIX::W_OK()) &&
					-w $MT_DESTROOT) {
				$USE_ALTERNATE=1;
				last;
			}
		}
		# We distinguish system trees from the rest.
		$systexmf = $SEP . $kpse_plain->expand_path('{$SYSTEXMF}/fonts') .
			$SEP . $MT_VARTEXFONTS . $SEP;
		if ($systexmf =~ /$SEP$MT_DESTROOT$SEP/) {
			# A system tree, check for varfonts.
			if ($USE_VARTEXFONTS || $USE_ALTERNATE) {
				$SAVE_DESTROOT = $MT_VARTEXFONTS;  # SAVE_, not MT_
			}
		} else {
			# A non-system tree.
			$USE_ALTERNATE and $SAVE_DESTROOT = '';  # Ditto: SAVE_, not MT_
		}
	}

	$MT_MODE = $SAVE_MODE;
	$MT_NAMEPART = $SAVE_NAMEPART;
	$MT_DESTROOT = $SAVE_DESTROOT;
	$MT_PKDESTREL = $SAVE_PKDESTREL;
	$MT_TFMDESTREL = $SAVE_TFMDESTREL;
	$MT_MFDESTREL = $SAVE_MFDESTREL;
	$MT_PKDESTDIR = $SAVE_PKDESTDIR;
	$MT_TFMDESTDIR = $SAVE_TFMDESTDIR;
	$MT_MFDESTDIR = $SAVE_MFDESTDIR;
	$MT_PKNAME = $SAVE_PKNAME;
	$MT_TFMNAME = $SAVE_TFMNAME;
	$MT_MFNAME = $SAVE_MFNAME;

	if (! $MT_DESTROOT) {
		$MT_DESTROOT = $MT_PKDESTDIR = $MT_TFMDESTDIR = $MT_MFDESTDIR =
			$MT_DEFAULT_DESTROOT;
	}

	$MT_MODE = eval qq("$MODE");
	$MT_NAMEPART = eval qq("$MT_NAMEPART");
	$MT_DESTROOT = eval qq("$MT_DESTROOT");
	$MT_PKDESTREL = eval qq("$MT_PKDESTREL");
	$MT_TFMDESTREL = eval qq("$MT_TFMDESTREL");
	$MT_MFDESTREL = eval qq("$MT_MFDESTREL");
	$MT_PKDESTDIR = eval qq("$MT_PKDESTDIR");
	$MT_TFMDESTDIR = eval qq("$MT_TFMDESTDIR");
	$MT_MFDESTDIR = eval qq("$MT_MFDESTDIR");
	$MT_PKNAME = eval qq("$MT_PKDESTDIR/$MT_PKBASE");
	$MT_TFMNAME = eval qq("$MT_TFMDESTDIR/$MT_TFMBASE");
	$MT_MFNAME = eval qq("$MT_MFDESTDIR/$MT_MFBASE");

	foreach $name ($MT_PKNAME, $MT_TFMNAME, $MT_MFNAME) {
		$name =~ s,/+,/,g;
	}

	return wantarray ? ($MT_PKNAME, $MT_TFMNAME, $MT_MFNAME) :
		join($SEP, $MT_PKNAME, $MT_TFMNAME, $MT_MFNAME);
}


################   Stuff from mktexnam.opt   ####################

# Get 8.3 filenames like dpiNNN/NAME.pk
sub mknam_dosnames {
	$MT_PKBASE='dpi$DPI/$NAME.pk';
}

# Omit the mode directory (e.g., ljfour)
sub mknam_nomode {
	$MODE='';
}

################################################################
# Use this feature if you wish to use the alias files from the 
# ftp://ftp.tug.org/tex/fontname.tar.gz distribution.
################################################################
sub mknam_fontmap {
	$MT_NAMEPART='$MT_SUPPLIER/$MT_TYPEFACE';
	$MT_PKDESTREL='pk/$MT_MODE/$MT_NAMEPART';
	$MT_TFMDESTREL='tfm/$MT_NAMEPART';
	$MT_MFDESTREL='source/$MT_NAMEPART';
	$SPECIALMAP=$kpse_plain->find('special.map');
	$TYPEFACEMAP=$kpse_plain->find('typeface.map');
	$SUPPLIERMAP=$kpse_plain->find('supplier.map');
	if (-r $SPECIALMAP) {
		open MAP, "<$SPECIALMAP"
			or die "$progname: Couldn't open $SPECIALMAP for reading: $!\n";
		while (<MAP>) {
			@fields = split;
			if ($fields[0] eq $NAME or
				$NAME =~ /^$fields[0].*\d/ && $fields[0] !~ /\d$/) {
				($SUPPLIER,$TYPEFACE)=@fields[1,2];
				last;
			}
		}
		close MAP
			or die "$progname: Problem reading $SPECIALMAP: $!\n";

		# Did we succeed in setting the supplier from the special map?
		# If not, let's try the supplier map.
		if (! $SUPPLIER && -r $SUPPLIERMAP) {
			$s_abbrev=substr($NAME,0,1);
			open MAP, "<$SUPPLIERMAP"
				or die "$progname: Couldn't open $SUPPLIERMAP for reading: $!\n";
			while (<MAP>) {
				@fields = split;
				if ($fields[0] eq $s_abbrev) {
					$SUPPLIER=$fields[1];
					last;
				}
			}
			close MAP
				or die "$progname: Problem reading $SUPPLIERMAP: $!\n";

			# If we have now found the supplier from the supplier map,
			# we search the typeface map for the typeface name. 
			if ($SUPPLIER && -r $TYPEFACEMAP) {
				$t_abbrev=substr($NAME,1,2);
				open MAP, "<$TYPEFACEMAP"
					or die "$progname: Couldn't open $TYPEFACEMAP for reading: $!\n";
				while (<MAP>) {
					@fields = split;
					if ($fields[0] eq $t_abbrev) {
						$TYPEFACE=$fields[1];
						last;
					}
				}
				close MAP
					or die "$progname: Problem reading $TYPEFACEMAP: $!\n";
			} # if ($SUPPLIER && -r $TYPEFACEMAP)
		} # if (! $SUPPLIER && -r $SUPPLIERMAP)
	} # if (-r $SPECIALMAP)

	# How well have we managed?
	if (! $SUPPLIER) {
		warn "$progname: " .
			"Could not map source abbreviation $s_abbrev for $NAME.\n" .
			"   Need to update $SPECIALMAP?\n";
		$MT_SUPPLIER=$MT_DEFAULT_SUPPLIER;
		$MT_TYPEFACE=$MT_DEFAULT_TYPEFACE;
	} else {
		$MT_SUPPLIER=$SUPPLIER;
		if (! $TYPEFACE) {
			warn "$progname: " .
				"Could not map typeface abbreviation $t_abbrev for $NAME.\n" .
				"   Need to update $SPECIALMAP?\n";
			$MT_TYPEFACE=$MT_DEFAULT_TYPEFACE;
		} else {
			$MT_TYPEFACE=$TYPEFACE;
		}
	}
}

#############################################################
# Use this feature to strip the "supplier" part (e.g. ams)
# of the target name:
#############################################################

sub mknam_stripsupplier {
	$MT_SUPPLIER='';
}

#############################################################
# Use this feature to strip the "typeface" part (e.g. euler)
# of the target name:
#############################################################

sub mknam_striptypeface {
	$MT_TYPEFACE='';
}

# Put new fonts into the directory named by the VARTEXFONTS environment
# variable or config file value.  (A default value for VARTEXFONTS is
# already set in the default texmf.cnf, q.v.)
# 
# A user can override this setting in either direction by setting
# USE_VARTEXFONTS to 1 or 0.
sub mknam_varfonts {
	$USE_VARTEXFONTS=1 unless defined $USE_VARTEXFONTS;
}

sub mktexnames_opt {
	":$MT_FEATURES:" =~ /:dosnames:/ && mknam_dosnames();
	":$MT_FEATURES:" =~ /:nomode:/ && mknam_nomode();
	":$MT_FEATURES:" =~ /:fontmap:/ && mknam_fontmap();
	":$MT_FEATURES:" =~ /:stripsupplier:/ && mknam_stripsupplier();
	":$MT_FEATURES:" =~ /:striptypeface:/ && mknam_striptypeface();
	":$MT_FEATURES:" =~ /:varfonts:/ && mknam_varfonts();
}

# One other one from mktex{tfm,pk}, which doesn't fit in nicely with
# the rest (its effect is on any subsequent METAFONT invocation)
sub mknam_nomfdrivers {
     $ENV{'MT_MFDESTDIR'} ||= cwd();
}

################   Stuff from mktexdir   ####################

# The original mktexdir is designed to handle multiple directories
# being given on the command line.  But the reality is that in mktex*,
# we never use this facility.  So we assume here that only one
# directory is being asked for.

sub mktex_dir {
	if ($MT_MKTEXDIR eq 'perl') { return mktex_dir_perl(@_); }
	else {
		$ENV{'MT_MKTEXDIR_OPT'} = $MT_MKTEXDIR_OPT;
		my $ret = `$MT_MKTEXDIR "$_[0]"`;
		die "$MT_MKTEXDIR @_ failed!\n" if ($? >> 8);
		return $ret;
	}
}

sub mktex_dir_perl {
	if (":$MT_FEATURES:" =~ /:appendonlydir:/) {
		$mt_append_mask = 01000; # sitcky bit
	} else {
		$mt_append_mask = 0;
	}

	$save_umask=umask 0;
	$save_cwd=cwd();

	@dircomps=split '/', $_[0];
	$dir_so_far=$dircomps[0];

	unless ($dir_so_far eq '' or $dir_so_far =~ /^[A-Za-z]:$/) {
		$KPSE_DOT or
			die "$progname: mktex_dir: Couldn't handle relative dir: $_[0]\n".
				"   (KPSE_DOT has not been set to a sensible value -- why not?)\n";
		chdir $KPSE_DOT
			or die "$progname: couldn't chdir to $KPSE_DOT: $!\n";
	}
	
	foreach $comp (@dircomps) {
		unless (-d "$dir_so_far/$comp") {
			$mode=(stat $dir_so_far)[2] & 07777;
			mkdir "$dir_so_far/$comp", $mode|$mt_append_mask or
				die "$progname: couldn't mkdir $dir_so_far/$comp: $!\n";
		}
		$dir_so_far .= '/' . $comp;
	}

	umask $save_umask;
	chdir $save_cwd
		or die "$progname: couldn't chdir back to $save_cwd: $!\n";
}
	

################   Stuff from mktexupd   ####################

sub mktex_upd {
	if ($MT_MKTEXUPD eq 'perl') { mktex_upd_perl(@_); }
	else {
		system($MT_MKTEXUPD, @_);
		if ($? >> 8) { die "$MT_MKTEXUPD @_ failed!\n"; }
	}
}

sub mktex_upd_perl {
	($dir,$file)=@_;

	return if ! $dir;
	-d $dir or die "$progname(mktex_upd): $dir not a directory!\n";
	-f "$dir/$file" or die "$progname(mktex_upd): $dir/$file not a file!\n";

	# We ensure that $dir does not end with a '/'
	$dir =~ s,/$,,;

	@db_dirs = split /$SEP/o, $kpse_plain->show_path('ls-R');
	foreach $db (@db_dirs) {
		# Ensure that the $db path does *not* ends with a '/'.  This
		# codes ensures that we don't get confused between things like
		# /usr/lib/texmf and /usr/lib/texmf.local (as I have seen used)
		$db =~ s,/$,,;
		if ("$dir/" =~ m,^$db/,) {
			$texmfls_r = $db;
			last;
		}
	}
	return unless $texmfls_r;

	$db_file = "$texmfls_r/ls-R";
	$dir =~ s,^$texmfls_r/,./,;
	if (! -f $db_file and system("mktexlsr", $texmfls_r) >> 8 != 0) {
		warn "$progname warning: mktexlsr $texmfls_r failed; ignoring.\n";
		return;
	}

	if (! -w $db_file) {
		warn "$progname warning: $db_file unwritable; ignoring.\n";
		return;
	}

	unless (open DB, ">>+$db_file") {
		warn "$progname warning: Couldn't open $db_file for reading; ignoring: $!\n";
		return;
	}

	chomp($first_line=<DB>);
	if ($first_line ne $ls_R_magic and $first_line ne $old_ls_R_magic) {
		if (close DB) {
			warn "$progname warning: $db_file lacks magic string `$ls_R_magic'\n";
		} else {
			warn "$progname warning: Problem reading $db_file; ignoring: $!\n";
		}
		return;
	}

	print DB "$dir:\n$file\n";
	close DB or
		warn "$progname warning: Problem reading/writing $db_file; ignoring: $!\n";
}

1;
