
/*
 * The Real SoundTracker - GUI configuration dialog
 *
 * Copyright (C) 1999 Michael Krause
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>

#include <gtk/gtk.h>
#ifdef USE_GNOME
#include <gnome.h>
#endif

#include "i18n.h"
#include "gui-settings.h"
#include "gui-subs.h"
#include "preferences.h"
#include "scope-group.h"
#include "track-editor.h"
#include "extspinbutton.h"

gui_prefs gui_settings = {
    1,
    0,
    1,

    1,
    0,
    0,
    1,
    16,

    1,

    50,
    40,
    500000,

    0,

    "~/",
    "~/",
    "~/",
    "~/",
    "~/",
    "~/",
    "~/",
};

static GtkWidget *configwindow = NULL;

static void           prefs_scopesfreq_changed                  (int value);
static void           prefs_trackerfreq_changed                 (int value);

static void
gui_settings_close_requested (void)
{
    gtk_widget_destroy(configwindow);
    configwindow = NULL;
}

static gui_subs_slider prefs_scopesfreq_slider = {
    N_("Scopes Frequency"), 1, 80, prefs_scopesfreq_changed
};
static gui_subs_slider prefs_trackerfreq_slider = {
    N_("Tracker Frequency"), 1, 80, prefs_trackerfreq_changed
};

static void
prefs_scopesfreq_changed (int value)
{
    extern ScopeGroup *scopegroup;
    gui_settings.scopes_update_freq = value;
    scope_group_set_update_freq(scopegroup, value);
}

static void
prefs_trackerfreq_changed (int value)
{
    gui_settings.tracker_update_freq = value;
    tracker_set_update_freq(value);
}

static void
gui_settings_hexmode_toggled (GtkWidget *widget)
{
    int o = gui_settings.tracker_hexmode;
    if(o != (gui_settings.tracker_hexmode = GTK_TOGGLE_BUTTON(widget)->active)) {
	gtk_widget_queue_resize(GTK_WIDGET(tracker));
    }
}

static void
gui_settings_upcase_toggled (GtkWidget *widget)
{
    int o = gui_settings.tracker_upcase;
    if(o != (gui_settings.tracker_upcase = GTK_TOGGLE_BUTTON(widget)->active)) {
	gtk_widget_queue_resize(GTK_WIDGET(tracker));
    }
}

static void
gui_settings_advance_cursor_toggled (GtkWidget *widget)
{
    gui_settings.advance_cursor_in_fx_columns = GTK_TOGGLE_BUTTON(widget)->active;
}

static void
gui_settings_use_aa_canvas_toggled (GtkWidget *widget)
{
    gui_settings.gui_use_aa_canvas = GTK_TOGGLE_BUTTON(widget)->active;
}

static void
gui_settings_non_empty_patterns_toggled (GtkWidget *widget)
{
    gui_settings.save_all_patterns = GTK_TOGGLE_BUTTON(widget)->active;
}

static void
gui_settings_scopebufsize_changed (GtkSpinButton *spin)
{
    double n = gtk_spin_button_get_value_as_float(spin);

    gui_settings.scopes_buffer_size = n * 1000000;
}

static void
gui_settings_highlight_rows_toggled (GtkWidget *widget)
{
    gui_settings.highlight_rows = GTK_TOGGLE_BUTTON(widget)->active;

    tracker_redraw(tracker);
}

static void
gui_settings_highlight_rows_changed (GtkSpinButton *spin)
{
    int n = gtk_spin_button_get_value_as_int(spin);

    gui_settings.highlight_rows_n = n;
    if(gui_settings.highlight_rows)
	tracker_redraw(tracker);
}

void
gui_settings_dialog (void)
{
    GtkWidget *mainbox, *thing, *box1;
    GtkTooltips *tooltips;

    if(configwindow != NULL) {
	gdk_window_raise(configwindow->window);
	return;
    }
    
#ifdef USE_GNOME
    configwindow = gnome_app_new("SoundTracker", _("GUI Configuration"));
#else
    configwindow = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(configwindow), _("GUI Configuration"));
#endif
    gtk_signal_connect (GTK_OBJECT (configwindow), "delete_event",
			GTK_SIGNAL_FUNC (gui_settings_close_requested), NULL);

    mainbox = gtk_vbox_new(FALSE, 2);
    gtk_container_border_width(GTK_CONTAINER(mainbox), 4);
#ifdef USE_GNOME
    gnome_app_set_contents(GNOME_APP(configwindow), mainbox);
#else
    gtk_container_add(GTK_CONTAINER(configwindow), mainbox);
#endif
    gtk_widget_show(mainbox);

    tooltips = gtk_tooltips_new();
    gtk_object_set_data(GTK_OBJECT(configwindow), "tooltips", tooltips);

    thing = gui_subs_create_slider(&prefs_scopesfreq_slider);
    gtk_box_pack_start(GTK_BOX(mainbox), thing, FALSE, TRUE, 0);
    gtk_widget_show(thing);

    thing = gui_subs_create_slider(&prefs_trackerfreq_slider);
    gtk_box_pack_start(GTK_BOX(mainbox), thing, FALSE, TRUE, 0);
    gtk_widget_show(thing);

    thing = gtk_check_button_new_with_label(_("Use Hexadecimal Numbers"));
    gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(thing), gui_settings.tracker_hexmode);
    gtk_box_pack_start(GTK_BOX(mainbox), thing, FALSE, TRUE, 0);
    gtk_widget_show(thing);
    gtk_signal_connect(GTK_OBJECT(thing), "toggled",
		       GTK_SIGNAL_FUNC(gui_settings_hexmode_toggled), NULL);

    thing = gtk_check_button_new_with_label(_("Use upper case letters for hex numbers"));
    gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(thing), gui_settings.tracker_upcase);
    gtk_box_pack_start(GTK_BOX(mainbox), thing, FALSE, TRUE, 0);
    gtk_widget_show(thing);
    gtk_signal_connect(GTK_OBJECT(thing), "toggled",
		       GTK_SIGNAL_FUNC(gui_settings_upcase_toggled), NULL);

    thing = gtk_check_button_new_with_label(_("Advance Cursor in FX Columns"));
    gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(thing), gui_settings.advance_cursor_in_fx_columns);
    gtk_box_pack_start(GTK_BOX(mainbox), thing, FALSE, TRUE, 0);
    gtk_widget_show(thing);
    gtk_signal_connect(GTK_OBJECT(thing), "toggled",
		       GTK_SIGNAL_FUNC(gui_settings_advance_cursor_toggled), NULL);

    thing = gtk_check_button_new_with_label(_("Use anti-aliased envelope editor"));
    gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(thing), gui_settings.gui_use_aa_canvas);
    gtk_box_pack_start(GTK_BOX(mainbox), thing, FALSE, TRUE, 0);
    gtk_widget_show(thing);
    gtk_signal_connect(GTK_OBJECT(thing), "toggled",
		       GTK_SIGNAL_FUNC(gui_settings_use_aa_canvas_toggled), NULL);
    gtk_tooltips_set_tip(tooltips, thing, _("You need to restart SoundTracker for this change to come into effect."), NULL);

    gui_subs_set_slider_value(&prefs_scopesfreq_slider, gui_settings.scopes_update_freq);
    gui_subs_set_slider_value(&prefs_trackerfreq_slider, gui_settings.tracker_update_freq);

    box1 = gtk_hbox_new(FALSE, 4);
    gtk_widget_show(box1);
    gtk_box_pack_start(GTK_BOX(mainbox), box1, FALSE, TRUE, 0);

    thing = gtk_label_new(_("Scopes buffer size [MB]"));
    gtk_box_pack_start(GTK_BOX(box1), thing, FALSE, TRUE, 0);
    gtk_widget_show(thing);
    add_empty_hbox(box1);
    thing = extspinbutton_new(GTK_ADJUSTMENT(gtk_adjustment_new((double)gui_settings.scopes_buffer_size / 1000000, 0.5, 5.0, 0.1, 1.0, 0.0)), 0, 0);
    gtk_box_pack_start(GTK_BOX(box1), thing, FALSE, TRUE, 0);
    gtk_widget_show(thing);
    gtk_spin_button_set_digits(GTK_SPIN_BUTTON(thing), 1);
    gtk_signal_connect(GTK_OBJECT(thing), "changed",
		       GTK_SIGNAL_FUNC(gui_settings_scopebufsize_changed), NULL);

    thing = gtk_hseparator_new();
    gtk_widget_show(thing);
    gtk_box_pack_start(GTK_BOX(mainbox), thing, FALSE, TRUE, 0);

    box1 = gtk_hbox_new(FALSE, 4);
    gtk_widget_show(box1);
    gtk_box_pack_start(GTK_BOX(mainbox), box1, FALSE, TRUE, 0);

    thing = gtk_check_button_new_with_label(_("Highlight rows:"));
    gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(thing), gui_settings.highlight_rows);
    gtk_signal_connect(GTK_OBJECT(thing), "toggled",
		       GTK_SIGNAL_FUNC(gui_settings_highlight_rows_toggled), NULL);
    gtk_box_pack_start(GTK_BOX(box1), thing, FALSE, TRUE, 0);
    gtk_widget_show(thing);
    add_empty_hbox(box1);
    thing = extspinbutton_new(GTK_ADJUSTMENT(gtk_adjustment_new((double)gui_settings.highlight_rows_n, 1, 16, 1, 2, 0.0)), 0, 0);
    gtk_box_pack_start(GTK_BOX(box1), thing, FALSE, TRUE, 0);
    gtk_widget_show(thing);
    gtk_spin_button_set_digits(GTK_SPIN_BUTTON(thing), 1);
    gtk_signal_connect(GTK_OBJECT(thing), "changed",
		       GTK_SIGNAL_FUNC(gui_settings_highlight_rows_changed), NULL);

    thing = gtk_hseparator_new();
    gtk_widget_show(thing);
    gtk_box_pack_start(GTK_BOX(mainbox), thing, FALSE, TRUE, 0);

    thing = gtk_check_button_new_with_label(_("`Save XM' saves all non-empty patterns"));
    gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(thing), gui_settings.save_all_patterns);
    gtk_box_pack_start(GTK_BOX(mainbox), thing, FALSE, TRUE, 0);
    gtk_widget_show(thing);
    gtk_signal_connect(GTK_OBJECT(thing), "toggled",
		       GTK_SIGNAL_FUNC(gui_settings_non_empty_patterns_toggled), NULL);

    gtk_widget_show (configwindow);
}

void
gui_settings_load_config (void)
{
    char buf[256];
    FILE *f;

    sprintf(buf, "%s/settings", prefs_get_prefsdir());

    f = fopen(buf, "rb");
    if(f) {
	prefs_get_int(f, "gui-use-hexadecimal-numbers", &gui_settings.tracker_hexmode);
	prefs_get_int(f, "gui-use-upper-case", &gui_settings.tracker_upcase);
	prefs_get_int(f, "gui-advance-cursor-in-fx-columns", &gui_settings.advance_cursor_in_fx_columns);
	prefs_get_int(f, "gui-display-scopes", &gui_settings.gui_display_scopes);
	prefs_get_int(f, "gui-use-backing-store", &gui_settings.gui_use_backing_store);
	prefs_get_int(f, "gui-use-aa-canvas", &gui_settings.gui_use_aa_canvas);
	prefs_get_int(f, "tracker-highlight-rows", &gui_settings.highlight_rows);
	prefs_get_int(f, "tracker-highlight-rows-n", &gui_settings.highlight_rows_n);
	prefs_get_int(f, "save-settings-on-exit", &gui_settings.save_settings_on_exit);
	prefs_get_int(f, "tracker-update-frequency", &gui_settings.tracker_update_freq);
	prefs_get_int(f, "scopes-update-frequency", &gui_settings.scopes_update_freq);
	prefs_get_int(f, "scopes-buffer-size", &gui_settings.scopes_buffer_size);
	prefs_get_int(f, "save-all-non-empty-patterns", &gui_settings.save_all_patterns);
	fclose(f);
    }

    sprintf(buf, "%s/settings-always", prefs_get_prefsdir());

    f = fopen(buf, "rb");
    if(f) {
	prefs_get_string(f, "loadmod-path", gui_settings.loadmod_path);
	prefs_get_string(f, "savemod-path", gui_settings.savemod_path);
	prefs_get_string(f, "savemodaswav-path", gui_settings.savemodaswav_path);
	prefs_get_string(f, "loadsmpl-path", gui_settings.loadsmpl_path);
	prefs_get_string(f, "savesmpl-path", gui_settings.savesmpl_path);
	prefs_get_string(f, "loadinstr-path", gui_settings.loadinstr_path);
	prefs_get_string(f, "saveinstr-path", gui_settings.saveinstr_path);
	fclose(f);
    }
}

void
gui_settings_save_config (void)
{
    char buf[256];
    FILE *f;

    prefs_check_prefs_dir();
    sprintf(buf, "%s/settings", prefs_get_prefsdir());

    f = fopen(buf, "wb");
    if(!f)
	return;

    prefs_put_int(f, "gui-use-hexadecimal-numbers", gui_settings.tracker_hexmode);
    prefs_put_int(f, "gui-use-upper-case", gui_settings.tracker_upcase);
    prefs_put_int(f, "gui-advance-cursor-in-fx-columns", gui_settings.advance_cursor_in_fx_columns);
    prefs_put_int(f, "gui-display-scopes", gui_settings.gui_display_scopes);
    prefs_put_int(f, "gui-use-backing-store", gui_settings.gui_use_backing_store);
    prefs_put_int(f, "gui-use-aa-canvas", gui_settings.gui_use_aa_canvas);
    prefs_put_int(f, "tracker-highlight-rows", gui_settings.highlight_rows);
    prefs_put_int(f, "tracker-highlight-rows-n", gui_settings.highlight_rows_n);
    prefs_put_int(f, "save-settings-on-exit", gui_settings.save_settings_on_exit);
    prefs_put_int(f, "tracker-update-frequency", gui_settings.tracker_update_freq);
    prefs_put_int(f, "scopes-update-frequency", gui_settings.scopes_update_freq);
    prefs_put_int(f, "save-all-non-empty-patterns", gui_settings.save_all_patterns);

    fclose(f);
}

void
gui_settings_save_config_always (void)
{
    char buf[256];
    FILE *f;

    prefs_check_prefs_dir();
    sprintf(buf, "%s/settings-always", prefs_get_prefsdir());

    f = fopen(buf, "wb");
    if(!f)
	return;

    prefs_put_string(f, "loadmod-path", gui_settings.loadmod_path);
    prefs_put_string(f, "savemod-path", gui_settings.savemod_path);
    prefs_put_string(f, "savemodaswav-path", gui_settings.savemodaswav_path);
    prefs_put_string(f, "loadsmpl-path", gui_settings.loadsmpl_path);
    prefs_put_string(f, "savesmpl-path", gui_settings.savesmpl_path);
    prefs_put_string(f, "loadinstr-path", gui_settings.loadinstr_path);
    prefs_put_string(f, "saveinstr-path", gui_settings.saveinstr_path);

    fclose(f);
}
