#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <signal.h>
#include <string.h>
#include <linux/cdrom.h>

#include "cdindex.h"
extern "C"
{
   #include "sha.h"
   #include "base64.h"
}

void TestGenerateId(void)
{
   SHA_INFO       sha;
   unsigned char  digest[20], *base64;
   unsigned long  size;

   sha_init(&sha);
   sha_update(&sha, (unsigned char *)"0123456789", 10);
   sha_final(digest, &sha);

   base64 = rfc822_binary(digest, 20, &size);
   if (strncmp((char *)base64, "h6zsF82dzSCnFsws9nQXtxyKcBY-", size))
   {
       free(base64);
       printf("The SHA1 hash function failed to properly generate the\n");
       printf("test key. Please review endian.h.\n");
       exit(0);
   }
   free(base64);
}

void GenerateId(PCDINDEX_CDINFO pCDInfo, char DiscId[33])
{
   SHA_INFO       sha;
   unsigned char  digest[20], *base64;
   unsigned long  size;
   char           temp[9];
   int            i;

   sha_init(&sha);

   // Before we ran the hash on the binary data, but now to make
   // sure people on both types of endian systems create the same
   // keys, we convert the data to hex-ASCII first. :-)
   sprintf(temp, "%02X", pCDInfo->FirstTrack);
   sha_update(&sha, (unsigned char *)temp, strlen(temp));

   sprintf(temp, "%02X", pCDInfo->LastTrack);
   sha_update(&sha, (unsigned char *)temp, strlen(temp));

   for(i = 0; i < 100; i++)
   {
       sprintf(temp, "%08X", pCDInfo->FrameOffset[i]);
       sha_update(&sha, (unsigned char *)temp, strlen(temp));
   }
   sha_final(digest, &sha);

   base64 = rfc822_binary(digest, 20, &size);
   memcpy(DiscId, base64, size);
   DiscId[size] = 0;
   free(base64);

}

int ReadTOCHeader(int fd, int &first, int &last)
{
   struct cdrom_tochdr tochdr;

   int ret = ioctl(fd, CDROMREADTOCHDR, &tochdr);

   if (!ret)
   {
      first = tochdr.cdth_trk0;
      last  = tochdr.cdth_trk1;
   }

   return ret;
}

int ReadTOCEntry(int fd, int track, int &lba, int &control, int &adr)
{
   struct cdrom_tocentry tocentry;
   tocentry.cdte_track = track;
   tocentry.cdte_format = CDROM_LBA;

   int ret = ioctl(fd, CDROMREADTOCENTRY, &tocentry);
   if (!ret)
   {
      assert (tocentry.cdte_format == CDROM_LBA);
      lba = tocentry.cdte_addr.lba;
      control = tocentry.cdte_ctrl;
      adr = tocentry.cdte_adr;
   }

   return ret;
}

void Usage(void)
{
   printf("cdindex version 1.0.0\n\n");
   printf("usage: cdindex <options> [cdrom device]\n\n");
   printf("Options:\n  -s  Submit the given CD\n");
   printf("  -g  Get the information for this CD\n");
   printf("  -l  Print the url used, but don't launch browser\n\n");
   printf("If no device is given, /dev/cdrom will be used.\n");

   exit(0);
}

void main(int argc, char *argv[])
{
   unsigned char   buf[CD_FRAMESIZE_RAW];
   int             ret, first, last, i, lba, control, adr, opt_s=0, opt_g=0,
opt_l=0;
   char           *device=DEFAULT_DEVICE;
   char *home;
   CDINDEX_CDINFO  cdinfo;
   char            id[33], url[URL_SIZE], url2[URL_SIZE], toc_string[URL_SIZE];
   char            lockfile[255];
   int lockfile_fd;
   struct stat sb;

   TestGenerateId();

   if (argc == 1 || (argc > 1 && strcasecmp(argv[1], "--help") == 0))
        Usage();

   // command-line argument processing loop
   for (i=1; i<argc; i++)
   {
       if(strcmp(argv[i], "-s") == 0)
       {
           opt_s = 1;
           continue;
       }
       if(strcmp(argv[i], "-g") == 0)
       {
           opt_g = 1;
           continue;
       }
       if(strcmp(argv[i], "-l") == 0)
       {
           opt_l = 1;
           continue;
       }

       // special case the standalone optional final argument (device)
       if(i != argc-1) Usage();
       device = argv[i];
   }

   int fd = open(device, O_RDONLY);
   if (fd < 0)
   {
       printf("Cannot open '%s'\n", device);
       exit(0);
   }

   memset(&cdinfo, 0, sizeof(CDINDEX_CDINFO));
   if (ReadTOCHeader(fd, first, last))
   {
      printf("Cannot read table of contents.\n");
      close(fd);
      exit(0);
   }

   ReadTOCEntry(fd, CDROM_LEADOUT, lba, control, adr);
   cdinfo.FrameOffset[0] = lba + 150;
   sprintf(toc_string, "%d+%d+%d", first, last, cdinfo.FrameOffset[0]);
   for (i = first; i <= last; i++)
   {
      ReadTOCEntry(fd, i, lba, control, adr);
      cdinfo.FrameOffset[i] = lba + 150;
      sprintf(toc_string + strlen(toc_string), "+%d", cdinfo.FrameOffset[i]);
   }

   cdinfo.FirstTrack = first;
   cdinfo.LastTrack = last;

   GenerateId(&cdinfo, id);
   close(fd);

   if (opt_s)
      sprintf(url, "http://www.freeamp.org/cgi-bin/cdi/submit.pl?"
               "id=%s&tracks=%d&toc=%s", id, (last - first) + 1, toc_string);
   else
      sprintf(url, "http://www.freeamp.org/cgi-bin/cdi/hget.pl?"
               "id=%s", id);

   if (opt_l)
   {   // just print the url
       printf("%s\n", url);
       exit(0);
   }
   sprintf(url2, "openURL(%s)", url);

   printf("Opening netscape to: '%s'\n(Please open it by hand if the "
         "browser doesn't start automatically)\n", url);

   home=getenv("HOME");
   if(!home) home="/";
   sprintf(lockfile,"%.200s/.netscape/lock",home);
   if(fork()>0) _exit(0);
   if((lockfile_fd=lstat(lockfile, &sb))!=-1) {
      execlp("netscape", "netscape", "-remote", url2, NULL);
   } else {
      execlp("netscape", "netscape", url, NULL);
   }

   perror("Could not launch netscape");
}

