## This is the appletalk module for tkchooser2
## This file defines and provides a higher-level
## API to the underlying software layer (Netatalk, in this case)
## and handles non service-specific stuff
## Ideally a plugin relying on appletalk doesn't need to know
## what the underlying appletalk layer is built with. A
## CAP appletalk module could drop in and replace this one
## without the plugins caring at all.
##
## Major protocol modules must provide a start function,
## a setcurrzone function, and an unload function
##
## Ethan Gold <etgold@cs.vassar.edu> 2/24/98
##
#### appletalk module plugin API spec (function followed by description):
##
## proc appletalk.nbplkup {entity}
#### Function to lookup a given appletalk entity.
#### name:type@zone or =:=@* for wildcards.
#### Returns an unordered list of raw output from
#### the nbplkup program. Not very platform independant.
#### Should be used in conjunction with namesfromlkup.
##
## proc appletalk.getcurrzone {}
#### function to return the current zone
#### returns a "*" if none is set or there are no zones.
##
## proc appletalk.namesfromlkup {results}
#### function to parse names out of entity lists
#### takes results from nbplkup and returns just the names
#### in dictionary sorted order.
##
## proc appletalk.print {printer filename}
#### function to send a file to the printer using
#### whatever the printer-talking utility is called.
#### This MIGHT have to move into the plugin if all
#### sorts of happy printer options are to be supported.


if {[debug]} {
    puts "Loading netatalk appletalk module..."
}

#### Appletalk global variables ####
## global "appletalkdir" defined in the config file
set atalk_currzone ""
set atalk_defzone ""
set atalk_zonelist ""
## delay between zonelist refreshed in milliseconds
## this doesn't need to happen very often
set atalk_zonedelay 60000
set atalk_afterID ""

## external program names
set getzones "getzones"
set nbplkup "nbplkup"
set print "pap"
set appletalk_dependancies "$getzones $nbplkup $print"
#### End Appletalk global variables ####

############ Required functions - API to main ##############

## startup procedure
proc appletalk.start {} {
    if {[debug]} {
	puts "appletalk: starting appletalk module"
    }
    appletalk.delzones
    appletalk.delentities
    appletalk.setzones
    ## call once to get the default zone
    appletalk.setdefzone
    appletalk.setcurrzone [appletalk.getdefzone]
    appletalk.refresh
    ## call again to make sure it's active in the listbox
    appletalk.setdefzone
}

## shutdown procedure
proc appletalk.stop {} {
    global atalk_afterID
    if {[debug]} {
	puts "appletalk: stopping appletalk module"
    }
    after cancel $atalk_afterID
    appletalk.delentities
}

## procedure to set the current zone
proc appletalk.setcurrzone {newzone} {
    global atalk_currzone
    set atalk_currzone $newzone
}

############ End Required functions - API to main ##############
############ Function API for plugins ##############

## Function to lookup a given entity.
## Returns an unordered list of raw output from
## the nbplkup program. Not very platform independant.
## Should be used in conjunction with namesfromlkup.
proc appletalk.nbplkup {entity} {
    global nbplkup
    set results [list]

    if {[debug]} { puts "appletalk: looking up up entity $entity" }

    ## put quotes around the name so the caller doesn't have to
    set lkupfd [open "|$nbplkup \"$entity\"" r]
    while { [gets $lkupfd line] != -1 } {
	#set results [lappend $results, $line]
	lappend results $line
    }
    if {[debug]} { puts "nbplkup results: $results" }
    #return [lsort $results]
    return $results
}

## function to return the current zone
## returns a "*" if none is set or there are no zones.
proc appletalk.getcurrzone {} {
    global atalk_currzone
    if {[string compare $atalk_currzone ""] == 0} { return "*" }
    return $atalk_currzone
}

## function to parse names out of entity lists
## takes results from nbplkup and returns just the names
## in dictionary sorted order.
proc appletalk.namesfromlkup {results} {
    set names [list]
    foreach result $results {
	set name [lindex [split $result :] 0]
	set name [string trim $name \ ]
	lappend names $name
    }
    ## kludge in case our tcl version is too old to
    ## support the -dictionary sorting option
    if { [catch {set newnames [lsort -dictionary $names]}] !=0 } {
	set newnames [lsort -ascii $names]
    }
    return $newnames
}

## function to send a file to the printer using
## whatever the printer-talking utility is called.
## This MIGHT have to move into the plugin if all
## sorts of happy printer options are to be supported.
proc appletalk.print {printer filename filterflag} {
    global print printfilterfile
    set plugframe [plug_frame]

    if {!$filterflag} {
	if {[debug]} {puts "appletalk: executing $print \
		-p \"$printer\" $filename"}
	set pipefd [open "|$print -p \"$printer\" \
		$filename" r]
    } else {
	if {[file readable $printfilterfile]} {
	    ## read in filter command
	    set filtfd [open "$printfilterfile" r]
	    gets $filtfd filtcmd
	    close $filtfd
	    if {[debug]} {puts "appletalk: executing: $filtcmd $filename | \
		    $print -p \"$printer\""}
	    set pipefd [open "|$filtcmd $filename | $print \
		    -p \"$printer\"" r]
	} else {
	    error "appletalk" "Could not find the printfilter file \
		    $printfilterfile. Please uncheck the filter \
		    option and try again."
	    return ""
	}
    }
    set result ""
    while {[gets $pipefd line] != -1} {
	set result "$result\n$line"
	$plugframe.status configure -text $line
    }
    catch {close $pipefd}
    return $result
}

#########  End Function API for plugins  ############

######### begin general utility functions ###########
## function to return zones
proc appletalk.getzones {} {
    global getzones
    set atalk_zonelist [list]
    set zonefd [open "|$getzones" r]
    while { [gets $zonefd line] != -1 } {
	lappend atalk_zonelist $line
    }
    catch {close $zonefd}
    return $atalk_zonelist
}

proc appletalk.getdefzone {} {
    global getzones
    set fd [open "|$getzones -m" r]
    while { [gets $fd line] != -1 } {
	return $line
    }
}

## procedure to show current zonelist
proc appletalk.setzones {} {
    foreach name [appletalk.getzones] {
	.leftside.bot.zones insert end $name
    }
}

## procedure to set the default zone (at startup)
proc appletalk.setdefzone {} {
    set thezone [appletalk.getdefzone]
    set size [.leftside.bot.zones size]

    for {set i 0} {$i<$size} {incr i} {
	set name [.leftside.bot.zones get $i]
	if {[string compare $name $thezone] == 0} {
	    .leftside.bot.zones selection set $i
	    update
	    return
	}
    }
    
}

## procedure to delete zone listings
proc appletalk.delzones {} {
    .leftside.bot.zones del 0 end
}

proc appletalk.delentities {} {
    set pluglist [plug_list]
    $pluglist delete 0 end
}

## looping procedure to refresh zone listings
proc appletalk.refresh {} {
    global atalk_zonedelay atalk_afterID

    if {[debug]} { puts "appletalk: refreshing zonelist" }
    appletalk.delzones
    appletalk.setzones
    set atalk_afterID [after $atalk_zonedelay {appletalk.refresh}]
}

############# End general utility functions ###############

## make sure we can find everything we need. if not,
## unset our global flag! we only need one of the dependancies
## in this case, so check separately
## this is a really ugly attempt at streamlining
#set atalk_ok 1
#if {[info exists appletalkdir]} {
#    foreach item $appletalk_dependancies {
#	set loc $appletalkdir/bin/$item
#	if {[file executable $loc]} {
#	    lappend locs $loc
#	} else {set atalk_ok 0}
#    }
#}
## assume they're all in the same place - if not
## who knows what will happen anyway
#if {!$atalk_ok} {
#    puts "appletalk: could not find $appletalk_dependancies in \
#	    $appletalkdir/bin. Check chooser.cfg and netatalk installation."
#    set locs [check_deps $appletalk_dependancies]
#    set atalk_ok 1
#}

set locs [check_deps $appletalk_dependancies]
set atalk_ok 1

## make sure we found them all
foreach item $locs {
    if {![file executable $item]} {
	set atalk_ok 0
	break
    }
}
if {!$atalk_ok} {
    puts "appletalk: Can't find required programs: \
	    $appletalk_dependancies."
    puts "appletalk: Netatalk may not be installed or your \
	    PATH may be incorrect. Disabling appletalk."
    set appletalkflag 0
}

if {$appletalkflag} {
    if {[debug]} {puts "appletalk: found $locs"}
    set getzones [lindex $locs 0]
    set nbplkup [lindex $locs 1]
    set print [lindex $locs 2]
}
catch {unset item}
catch {unset locs}
catch {unset loc}

if {[debug]} { puts "Finished loading netatalk appletalk module." }


