/*
** Copyright (C) 10 Feb 1999 Jonas Munsin <jmunsin@iki.fi>
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
** 
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
** 
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software 
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <gtk/gtk.h>
#include <string.h>
#include <sys/types.h>
#include <sys/time.h>
#include <unistd.h>
#include <signal.h>
#include <stdlib.h>
#include <ctype.h>

#include "linebuffer.h"
#include "readers.h"
#include "progressbars_gui.h"
#include "status_text.h"
#include "common_gtk.h"
#include "cdrecord_options.h"
#include "globals.h"

int countdown_counter;

static linebuffer cdrecord_stderr, cdrecord_stdout, mkisofs_stderr, mkisofs_stdout;
static GtkWidget *done_win;

static int frozen, total_track_size, total_track_written, current_track_size;

static void do_countdown(int starting) {
	char warn[] = "Last chance to quit, starting in\n 9 seconds.";

	estimate_finnish_time_start();

	if (starting) {
		gtk_label_set(GTK_LABEL(info_label),
				_("Burn in progress, please wait."));
		countdown_counter = 0;
	} else {
		g_snprintf(warn, sizeof(warn),
				"Last chance to quit, starting in\n %d seconds.",
				countdown_counter--);
		gtk_label_set(GTK_LABEL(info_label), warn);
	}
}

static void fixating_starting(void) {
	gtk_label_set(GTK_LABEL(info_label), _("Fixating..."));
}

static void kill_ack(GtkWidget *widget, gpointer data) {
	gtk_widget_destroy(data);
}

static void all_done(void) {
	GtkWidget *done, *label;

	done_win = gtk_dialog_new();

	done = gtk_button_new_with_label("Ok");
	gtk_box_pack_start (GTK_BOX(GTK_DIALOG(done_win)->action_area), done,
			TRUE, TRUE, 0);
	gtk_widget_show(done);

	gtk_signal_connect(GTK_OBJECT(done), "clicked",
			GTK_SIGNAL_FUNC(kill_ack), done_win);

	label = gtk_label_new(_("Operation finished successfully!"));
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(done_win)->vbox), label, TRUE,
			TRUE, 0);
	gtk_widget_show(label);

	gtk_widget_show(done_win);
}

static void fixating_done(void) {
	all_done();
}

void init_buffers(int which) {
	total_track_size = 0;
	current_track_size = 0;

	if (which & BUF_MKISOFS_STDERR)
		init_buffer(&mkisofs_stderr);
	else
		mkisofs_stderr.buffer = NULL;

	if (which & BUF_MKISOFS_STDOUT)
		init_buffer(&mkisofs_stdout);
	else
		mkisofs_stdout.buffer = NULL;

	if (which & BUF_CDRECORD_STDERR)
		init_buffer(&cdrecord_stderr);
	else
		cdrecord_stderr.buffer = NULL;

	if (which & BUF_CDRECORD_STDOUT)
		init_buffer(&cdrecord_stdout);
	else
		cdrecord_stdout.buffer = NULL;

	/* should actually only be called just before mkisofs starts, but there's
	 * nowhere obvious place to put it, so here is just as good as any place */
	estimate_finnish_time_start();
}

void destroy_buffers(void) {
/* before the destroy buffer calls it would be possible to a while (extract_line) add_text, but
 * passing them to destroy_buffer and let it print out warnings is more useful for catching potential
 * bugs (even if it causes a false alarm for strings staring with \r and with no \n or \r at the end
 * (like "Track XX:   YY of ZZZ MB written." */
	if (mkisofs_stderr.buffer != NULL) {
		destroy_buffer(&mkisofs_stderr);
	}
	if (mkisofs_stdout.buffer != NULL) {
		destroy_buffer(&mkisofs_stdout);
	}
	if (cdrecord_stderr.buffer != NULL) {
		destroy_buffer(&cdrecord_stderr);
	}
	if (cdrecord_stdout.buffer != NULL) {
		destroy_buffer(&cdrecord_stdout);
	}
}

/*
 * pre:  read_into() will only get called when select() reported there was
 *       something to be read from fd, so if read() returns 0, EOF was read
 *       and the process attached
 *       to the pipe has exited.
 * post: returns FALSE on error or if EOF (read returns 0) was read, otherwise
 *       whatever was read goes into lbuf
 */

int read_into(linebuffer *lbuf, int fd) {
	char in_buf[BUF_S];
	int result;

	if ((result = read(fd, in_buf, BUF_S-1)) < 0) {
		g_warning("read returned negative!");
		return FALSE;
	} else if (result == 0)
		return FALSE;
	else 
		in_buf[result] = 0;

	add_to_buffer(lbuf, in_buf);

	if (result == BUF_S -1) /* there might be more available, recurse baby! */
		return read_into(lbuf, fd);
	else
		return TRUE;
}


int ni_read_mkisofs_stdout(int fd) {

	if (!read_into(&mkisofs_stdout, fd))
		return FALSE;

	if (extract_line(&mkisofs_stdout, "\n")) {
		gtk_text_freeze(GTK_TEXT(mkisofs_text));
		add_text_mkisofs_stdout(mkisofs_stdout.newline);
		while (extract_line(&mkisofs_stdout, "\n"))
			add_text_mkisofs_stdout(mkisofs_stdout.newline);
		gtk_text_thaw(GTK_TEXT(mkisofs_text));
	}
	if (GTK_TOGGLE_BUTTON(autoscroll_text)->active) {
		gtk_text_freeze(GTK_TEXT(mkisofs_text));
		scroll_down_text(mkisofs_text, mkisofs_text_scrollbar);
		gtk_text_thaw(GTK_TEXT(mkisofs_text));
	}
	return TRUE;
}

static void add_mkisofs_stderr_output(void) {
	if ((cdrecord_stdout.buffer == NULL) &&
			strstr(mkisofs_stderr.newline, " extents written (")) {
		all_done();
	} else if (strstr(mkisofs_stderr.newline, "% done, estimate finish")) {
		mkisofs_stderr.newline[6]=0;
		if ((atof(mkisofs_stderr.newline)/100.0 < 0) ||
				(atof(mkisofs_stderr.newline)/100.0 > 1))
			g_warning("%s %i: add_mkisofs_stderr_output estimated finish < 0 or > 1!",
					__FILE__, __LINE__);
		else {
			gtk_progress_bar_update(GTK_PROGRESS_BAR(progress_bar),
					atof(mkisofs_stderr.newline)/100.0);
			estimate_finnish_time_update(atof(mkisofs_stderr.newline)/100.0);
		}
	} else if (strstr(mkisofs_stderr.newline, "Invalid node - ")) /* TODO: open an error window */
		g_warning("%s", mkisofs_stderr.newline);
	else if (strstr(mkisofs_stderr.newline, "cannot fwrite ")) {
		g_warning("%s", mkisofs_stderr.newline);
		g_warning("Maybe you ran out of hd space for image?");
	} else {
		if (!frozen) {
			gtk_text_freeze(GTK_TEXT(mkisofs_text));
			frozen = TRUE;
		}
		add_text_mkisofs_stderr(mkisofs_stderr.newline);
	}
}
/* returns false when it detects mkisofs has exited, otherwise true.
 * Tests for cdrecord_stdout.buffer == NULL to determine if it should
 * call all_done() when mksisofs is done
 */
int ni_read_mkisofs_stderr(int fd) {
	frozen = FALSE;
	if (!read_into(&mkisofs_stderr, fd))
		return FALSE;

	while (extract_line(&mkisofs_stderr, "\n"))
		add_mkisofs_stderr_output();
	if (frozen) {
		if (GTK_TOGGLE_BUTTON(autoscroll_text)->active)
			scroll_down_text(mkisofs_text, mkisofs_text_scrollbar);
		gtk_text_thaw(GTK_TEXT(mkisofs_text));
	}
/*	if (frozen)
		gtk_text_thaw(GTK_TEXT(mkisofs_text));
	if (GTK_TOGGLE_BUTTON(autoscroll_text)->active)
		scroll_down_text(mkisofs_text, mkisofs_text_scrollbar); blu */
	return TRUE;
}

static int add_cdda2wav_stderr_output(void) {
	char *tmp;
	if (strstr(mkisofs_stderr.newline, " successfully recorded")) {
		if (!frozen) {
			gtk_text_freeze(GTK_TEXT(mkisofs_text));
			frozen = TRUE;
		}
		add_text_mkisofs_stderr(mkisofs_stderr.newline);
		return FALSE;
	} else if (NULL != (tmp = strstr(mkisofs_stderr.newline, "%"))) {
			float p;
			tmp--;
			if (isdigit(tmp[0]))
				tmp--;
			p = ((float)atoi(tmp))/100.0;
			if (p <= 1 && p >= 0) {
				gtk_progress_bar_update(GTK_PROGRESS_BAR(progress_bar), p);
				estimate_finnish_time_update(p);
			} else {
				g_warning("%s:%i percentage done is wrong", __FILE__, __LINE__);
			}
	} else {
		if (!frozen) {
			gtk_text_freeze(GTK_TEXT(mkisofs_text));
			frozen = TRUE;
		}
		add_text_mkisofs_stderr(mkisofs_stderr.newline);
	}
	return TRUE;
}

int ni_read_cdda2wav_stderr(int fd) {
/*	int r = TRUE; */
	frozen = FALSE;
	if (!read_into(&mkisofs_stderr, fd))
		return FALSE;

	while (extract_line(&mkisofs_stderr, "\n") || extract_line(&mkisofs_stderr, "\r")) {
/*		r = r && add_cdda2wav_stderr_output(); */
		add_cdda2wav_stderr_output();
	}
	if (frozen) {
		if (GTK_TOGGLE_BUTTON(autoscroll_text)->active)
			scroll_down_text(mkisofs_text, mkisofs_text_scrollbar);
		gtk_text_thaw(GTK_TEXT(mkisofs_text));
	}

/*	return r; */
	return TRUE;
}

static void add_cdrecord_stderr_output(int *status, int check_for_end) {
	if (strstr(cdrecord_stderr.newline, 
				"cdrecord: Operation not permitted. Cannot do mlockall(2)")
			|| strstr(cdrecord_stderr.newline, "Operation not permitted. WARNING: Cannot do mlockall(2).")) {
		g_warning("You are probably not running cdrecord as root or you have the wrong permissions on the burner device");
		*status = FALSE;
	} else if (strstr(cdrecord_stderr.newline, "cdrecord: No disk / Wrong disk!")) {
		alert_user_of_error(_("You need to insert an empty disk"));
	} else if (strstr(cdrecord_stderr.newline, "cdrecord: Premature EOF on stdin.")
			|| strstr(cdrecord_stderr.newline, "cdrecord: Input buffer error, aborting.")) {
		g_warning("Something went wrong with mkisofs/the image");
	} else if (check_for_end && strstr(cdrecord_stderr.newline, "fifo was")) {
		*status = FALSE;
	} else if (strstr(cdrecord_stderr.newline, "cdrecord: Cannot send CUE sheet.")) {
		*status = FALSE;
	}
	if (!frozen) {
		gtk_text_freeze(GTK_TEXT(cdrecord_text));
		frozen = TRUE;
	}
	add_text_cdrecord_stderr(cdrecord_stderr.newline);
}

/* returns false when it detects cdrecord has exited, otherwise true
 */
int ni_read_cdrecord_stderr(int fd, int check_for_end) {
	int status = TRUE;
	frozen = FALSE;
	if (!read_into(&cdrecord_stderr, fd))
		return FALSE;

	while (extract_line(&cdrecord_stderr, "\n"))
		add_cdrecord_stderr_output(&status, check_for_end);
	if (frozen) {
		if (GTK_TOGGLE_BUTTON(autoscroll_text)->active)
			scroll_down_text(cdrecord_text, cdrecord_text_scrollbar);
		gtk_text_thaw(GTK_TEXT(cdrecord_text));
	}

	return status;
}

static int add_cdrecord_stdout_output(void) {
	/*		if (strstr(cdrecord_stdout.newline, "Writing  time:"))
			{}
			else */
	if (strstr(cdrecord_stdout.newline, "Fixating..."))
		fixating_starting();
	else if (strstr(cdrecord_stdout.newline, "Fixating time:")) {
		fixating_done();
		if (frozen)
			gtk_text_thaw(GTK_TEXT(cdrecord_text));
		frozen = FALSE;
		return FALSE;
	} else if (strstr(cdrecord_stdout.newline,
				"Waiting for reader process to fill input-buffer ... "
				"input-buffer ready.")) {
		do_countdown(1);
		estimate_finnish_time_start();
	} else if (strstr(cdrecord_stdout.newline, "Track ")
			&& strstr(cdrecord_stdout.newline, " MB written (fifo ")) {
		if (mkisofs_stderr.buffer == NULL) {
			/* mkisofs isn't running, update progressbar */
			float written, left;
			cdrecord_stdout.newline[13] = 0;
			cdrecord_stdout.newline[20] = 0;
			if (GTK_TOGGLE_BUTTON(audio_tracks)->active) {
				written = atof(&cdrecord_stdout.newline[9]);
				written += total_track_written;
				current_track_size = atof(&cdrecord_stdout.newline[16]);
				left = (float)(total_track_size + current_track_size);
			} else {
				written = atof(&cdrecord_stdout.newline[9]);
				left = atof(&cdrecord_stdout.newline[16]);
			}
			if ((0 == left) || (written / left < 0) || (written / left > 1))
				g_warning("%s %i: add_cdrecord_stdout_output estimated finish "
						"< 0 or > 1 or left == 0!", __FILE__, __LINE__);
			else {
				gtk_progress_bar_update(GTK_PROGRESS_BAR(progress_bar),
						written/left);
				estimate_finnish_time_update(written/left);
			}
			cdrecord_stdout.newline[8] = 0;
			set_track_burning_now(atoi(&cdrecord_stdout.newline[6]));
			cdrecord_stdout.newline[8] = ' ';
			cdrecord_stdout.newline[13] = ' ';
			cdrecord_stdout.newline[20] = ' ';
		}
		set_mb_fifo_info(&cdrecord_stdout.newline[9]);
	} else if (strstr(cdrecord_stdout.newline, "Last chance to quit, starting "))
		do_countdown(0);
	else if (strstr(cdrecord_stdout.newline, " seconds.")
			&& !(strstr(cdrecord_stdout.newline, "ATAPI early return: sleeping")))
		do_countdown(0);
	else if (strstr(cdrecord_stdout.newline, "Blanking "))
		gtk_label_set(GTK_LABEL(info_label), cdrecord_stdout.newline);
	else if (strstr(cdrecord_stdout.newline, "Starting new track at sector"))
		total_track_written += current_track_size;
	else if (strstr(cdrecord_stdout.newline, "Total size:"))
		total_track_size = atoi(&cdrecord_stdout.newline[strlen("Total size: ")]);
	else {
		if (!frozen) {
			gtk_text_freeze(GTK_TEXT(cdrecord_text));
			frozen = TRUE;
		}
		add_text_cdrecord_stdout(cdrecord_stdout.newline);
	}
	return TRUE;
}


/* returns false when it detects cdrecord has exited, otherwise true.
 * Tests for mkisofs_stderr.buffer == NULL to determine if it should
 * update the progressbar or let ni_read_mksisofs_stderr do it.
 */
int ni_read_cdrecord_stdout(int fd) {
	frozen = FALSE;
	if (!read_into(&cdrecord_stdout, fd))
		return FALSE;

	/* FIXME: after "(fifo xx%)" messages there can be error messages without
	 *        a \n or \r inbetweeen - should probably split on %), too, but that requieres
	 *        changing add_cdrecord_stdout_output as well */
	while (extract_line(&cdrecord_stdout, "\n")
			|| extract_line(&cdrecord_stdout, "\r")
			|| extract_line(&cdrecord_stdout, " seconds.")) /* countdown */ {
		if (FALSE == add_cdrecord_stdout_output())
			return FALSE;
	}
	if (frozen) {
		if (GTK_TOGGLE_BUTTON(autoscroll_text)->active)
			scroll_down_text(cdrecord_text, cdrecord_text_scrollbar);
		gtk_text_thaw(GTK_TEXT(cdrecord_text));
	}

	return TRUE;
}

