/*  Protocol compatible masqdialer server written in C
    Copyright (C) 1998 Charles P. Wright 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

    Parameter Database Functions by Shane Kerr <kerr@wizard.net>
*/

#include <syslog.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include "mserver.h"

/* initial number of entries in the parameter database */
#define INIT_SPACE_FOR_PARAM 32

/* boolean data type */
typedef enum { FALSE, TRUE } boolean_t;

#ifndef NDEBUG
/* invariant for a single parameter */
boolean_t param_invariant(const struct param_t* param)
{
    if (param == NULL)
    {
        return FALSE;
    }
    if (param->key == NULL)
    {
        return FALSE;
    }
    if (param->val == NULL)
    {
        return FALSE;
    }
    return TRUE;
}
#endif /* NDEBUG */

/* initialize a single parameter */
static void init_param(struct param_t* param, const char* key, const char* val)
{
    param->key = strdup(key);
    if (param->key == NULL)
    {
        fprintf(stderr, "Out of memory storing key\n");
#ifndef UTIL
	mserver_cleanup();
#endif
	exit(1);
    }
    param->val = strdup(val);
    if (param->val == NULL)
    {
        fprintf(stderr, "Out of memory storing key\n");
#ifndef UTIL
	mserver_cleanup();
#endif
	exit(1);
    }

    if(!param_invariant(param))
    {
	syslog(LOG_WARNING, "init_param created null entry");	
    }
}

/* set the value of a parameter to a new value */
static void set_param_val(struct param_t* param, const char *val)
{
	if (!param_invariant(param))
	{
		syslog(LOG_WARNING, "set_param_val was passed null entry");
	}
	if (val == NULL)
	{
		syslog(LOG_WARNING, "set_param_val was passed a null value");
	}

    /* release the old value */
    free(param->val);

    /* initialize the new value */
    param->val = strdup(val);
    if (param->val == NULL)
    {
        fprintf(stderr, "Out of memory changing key\n");
#ifndef UTIL
	mserver_cleanup();
#endif
	exit(1);
    }

    if(!param_invariant(param))
    {
	syslog(LOG_WARNING, "set_param_val created null entry");	
    }
}

/* release memory used by a parameter */
static void dispose_param(struct param_t* param)
{
    if(!param_invariant(param))
    {
	syslog(LOG_WARNING, "dispose_param passed null entry");
    }

    free(param->key);
    param->key = NULL;
    free(param->val);
    param->val = NULL;
}


#ifndef NDEBUG
/* parameter database invariant */
boolean_t param_dbase_invariant(const struct param_dbase_t* dbase)
{
    if (dbase == NULL)
    {
        return FALSE;
    }
    if (dbase->param == NULL)
    {
        return FALSE;
    }
    if (dbase->space_for_param < INIT_SPACE_FOR_PARAM)
    {
        return FALSE;
    }
    if (dbase->num_param > dbase->space_for_param)
    {
        return FALSE;
    }
    return TRUE;
}
#endif /* NDEBUG */

/* initialize a parameter database */
void init_param_dbase(struct param_dbase_t* dbase)
{
	if (dbase == NULL)
	{
		syslog(LOG_WARNING, "Trying to initialize NULL database!");
	}

    dbase->num_param = 0;
    dbase->space_for_param = INIT_SPACE_FOR_PARAM;
    dbase->param = (struct param_t *) calloc(INIT_SPACE_FOR_PARAM, sizeof(struct param_t));
    if (dbase->param == NULL)
    {
        fprintf(stderr, "Out of memory initializing parameter database\n");
#ifndef UTIL
	mserver_cleanup();
#endif
	exit(1);
    }

    if(!param_dbase_invariant(dbase))
    {
	syslog(LOG_WARNING, "init_param_dbase created null entry");
    }
}

/* add a value with the given key to the database */
void param_dbase_setval(struct param_dbase_t* dbase, const char* key, const char *val)
{
    boolean_t found_key;
    int i;
    int mem_size;

	if(!param_dbase_invariant(dbase))
	{
		syslog(LOG_WARNING, "param_dbase_setval passed null entry");
	}
	if(key == NULL)
	{
		syslog(LOG_WARNING, "param_dbase_setval passed null key");
	}	
	if(val == NULL)
	{
		syslog(LOG_WARNING, "param_dbase_setval passed null value");
	}	

    found_key = FALSE;
    for (i=0; (i<dbase->num_param) && !found_key; i++)
    {
        if (strcmp(dbase->param[i].key, key) == 0)
	{
	    set_param_val(dbase->param + i, val);
	    found_key = TRUE;
	}
    }

    if (!found_key)
    {
	/* expand database if necessary */
	if (dbase->num_param >= dbase->space_for_param)
	{
	    dbase->space_for_param *= 2;
	    mem_size = dbase->space_for_param * sizeof (struct param_t);
	    dbase->param = (struct param_t *)realloc(dbase->param, mem_size);
	    if (dbase->param == NULL)
	    {
                fprintf(stderr, "Out of memory adding to parameter database\n");
#ifndef UTIL
		mserver_cleanup();
#endif
	        exit(1);
	    }
	}

        init_param(dbase->param + dbase->num_param, key, val);
	dbase->num_param += 1;
    }

	if(!param_dbase_invariant(dbase))
	{
		syslog(LOG_WARNING, "param_dbase_setval passed null entry");
	}
	if(strcmp(param_dbase_getval(dbase, key), val))
	{
		syslog(LOG_WARNING, "param_dbase_setval incorrectly set value");
	}	
	if(param_dbase_getval(dbase, key) == NULL)
	{
		syslog(LOG_WARNING, "param_dbase_setval did not set value");
	}	
}

/* find a key with the given value */
char* param_dbase_getval(const struct param_dbase_t* dbase, const char* key)
{
    char* ret_val;
    int i;

	if(!param_dbase_invariant(dbase))
	{
		syslog(LOG_WARNING, "param_dbase_getval passed null entry");
	}
	if(key == NULL)
	{
		syslog(LOG_WARNING, "param_dbase_getval passed null key");
	}	

    ret_val = NULL;
    for (i=0; (i<dbase->num_param) && (ret_val == NULL); i++)
    {
        if (strcmp(dbase->param[i].key, key) == 0)
	{
	    ret_val = dbase->param[i].val;
	}
    }

    if(!param_dbase_invariant(dbase))
    {
	syslog(LOG_WARNING, "param_dbase_getval returned null entry");
    }
    return ret_val;
}

/* clear the database */
void clear_param_dbase(struct param_dbase_t* dbase)
{
    int i;

	if(!param_dbase_invariant(dbase))
	{
		syslog(LOG_WARNING, "clear_param_dbase passed null dbase");
	}

    for (i=0; i<dbase->num_param; i++)
    {
        dispose_param(dbase->param + i);
    }
    dbase->num_param = 0;

    if(!param_dbase_invariant(dbase))
    {
	syslog(LOG_WARNING, "clear_param_dbase didn't clear dbase");
    }
}

/* for stub testing */
#ifdef UNIT_TEST
int main()
{
    param_dbase_t dbase;

    char inbuf[256];
    char key[256];
    char val[256];

    const char* query_result;

    printf("Starting parameter database tester.\n");
    init_param_dbase(&dbase);
    printf("Database initialized.\n");
    printf("\n");

    printf("Commands are:\n");
    printf("  (S)et key value\n");
    printf("  (G)et key\n");
    printf("  (C)lear the database\n");
    printf("  (Q)uit\n");

    for (;;)
    {
        if (fgets(inbuf, sizeof(inbuf), stdin) == NULL)
	{
            return 0;
	}
	if (strcmp(inbuf, "q\n") == 0)
	{
	    printf("Test ended\n");
	    return 0;
	}
	else if (strcmp(inbuf, "s\n") == 0)
	{
	    printf("key:"); fflush(stdout);
	    if (fgets(key, sizeof(key), stdin) == NULL)
	    {
	        return 0;
	    }
	    printf("val:"); fflush(stdout);
	    if (fgets(val, sizeof(val), stdin) == NULL)
	    {
	        return 0;
	    }
	    param_dbase_setval(&dbase, key, val);
	}
	else if (strcmp(inbuf, "g\n") == 0)
	{
	    printf("key:"); fflush(stdout);
	    if (fgets(key, sizeof(key), stdin) == NULL)
	    {
	        return 0;
	    }
	    query_result = param_dbase_getval(&dbase, key);
	    if (query_result == NULL)
	    {
	        printf("key not found\n");
	    }
	    else
	    {
	        printf("val:%s", query_result);
	    }
	}
	else if (strcmp(inbuf, "c\n") == 0)
	{
	    clear_param_dbase(&dbase);
	    printf("database cleared\n");
	}
	else 
	{
	    printf("unrecognized command\n");
	}
    }
}
#endif /* UNIT_TEST */
