/*
 gui-nicklist.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"
#include "gui-dcc.h"
#include "gui-nicklist-popup.h"

#include "pixmaps/op.xpm"
#include "pixmaps/voice.xpm"
#include "pixmaps/ircop.xpm"
#include "pixmaps/ircop_op.xpm"
#include "pixmaps/ircop_voice.xpm"

static GdkPixmap *op_pixmap, *voice_pixmap, *ircop_pixmap, *ircop_op_pixmap, *ircop_voice_pixmap;
static GdkBitmap *op_mask, *voice_mask, *ircop_mask, *ircop_op_mask, *ircop_voice_mask;

static gint gonecheck_tag;

static GtkTargetEntry dnd_nicklist_target[] =
{
    { "text/uri-list", 0, 0 }
};
#define ELEMENTS(x) (sizeof (x) / sizeof (x[0]))

static void nickdnd_drop(GtkWidget *widget,
                         GdkDragContext *context,
                         gint x,
                         gint y,
                         GtkSelectionData *selection_data,
                         guint info,
                         guint32 time,
                         WINDOW_REC *window)
{
#ifdef HAVE_GNOME
    GList *names;
    gchar *fname;
    NICK_REC *nick;
    gint row, col;

    if (gtk_clist_get_selection_info(GTK_CLIST(widget), x, y, &row, &col) < 0)
        return;

    nick = gtk_clist_get_row_data(GTK_CLIST(widget), row);
    names = gnome_uri_list_extract_uris(selection_data->data);
    if (names != NULL)
    {
        fname = names->data;

        if (strncmp(fname, "file:", 5) == 0)
            gui_dcc_send(window->active->server, nick->nick, fname+5);
    }
    gnome_uri_list_free_strings(names);
#endif
}

/* callback : data moved in the window */
static gboolean nickdnd_motion(GtkWidget *widget,
                               GdkDragContext *context,
                               gint x,
                               gint y,
                               guint time)
{
    gint row, col;

    if (gtk_clist_get_selection_info(GTK_CLIST(widget), x, y, &row, &col) >= 0)
        gtk_clist_select_row(GTK_CLIST(widget), row, col);
    return TRUE;
}

/* callback : drag leave the window */
static gboolean nickdnd_leave (GtkWidget *widget,
                               GdkDragContext *context,
                               guint time)
{
    gtk_clist_unselect_all(GTK_CLIST(widget));
    return TRUE;
}

static void gui_nicklist_draw_label(CHANNEL_REC *channel)
{
    GUI_WINDOW_REC *gui;
    GList *nicklist, *tmp;
    gint nicks, normal, voices, ops;
    gchar *str;

    /* redraw number of ops/nicks in channel */
    gui = WINDOW_GUI(CHANNEL_PARENT(channel));

    nicks = normal = voices = ops = 0;
    nicklist = nicklist_getnicks(channel);
    for (tmp = nicklist; tmp != NULL; tmp = tmp->next)
    {
	NICK_REC *rec = tmp->data;

	if (rec->op)
	    ops++;
	else if (rec->voice)
	    voices++;
	else
	    normal++;
	nicks++;
    }
    g_list_free(nicklist);

    str = g_strdup_printf(_("%d ops, %d total"), ops, nicks);
    gtk_label_set(GTK_LABEL(gui->nlist_label), str);
    g_free(str);
}

static void gui_nicklist_add(CHANNEL_REC *channel, NICK_REC *nick)
{
    GUI_WINDOW_REC *gui;
    GdkPixmap *pix;
    GdkBitmap *mask;
    GList *list;
    gchar *text[1];
    gint row;

    g_return_if_fail(channel != NULL);
    g_return_if_fail(nick != NULL);

    gui = WINDOW_GUI(CHANNEL_PARENT(channel));
    if (!gui_channel_active(channel)) return;

    list = GTK_CLIST(gui->nicklist)->row_list;
    for (row = 0; list != NULL; row++, list = list->next)
        if (nicklist_compare(GTK_CLIST_ROW(list)->data, nick) > 0)
            break;

    text[0] = nick->nick;
    row = gtk_clist_insert(GTK_CLIST(gui->nicklist), row, text);

    if (nick->op || nick->voice || nick->ircop)
    {
	/* op / voice / ircop - needs a pixmap anyway */
	if (nick->ircop)
	{
	    pix = nick->op ? ircop_op_pixmap :
		(nick->voice ? ircop_voice_pixmap : ircop_pixmap);
	    mask = nick->op ? ircop_op_mask :
		(nick->voice ? ircop_voice_mask : ircop_mask);
	}
	else
	{
	    pix = nick->op ? op_pixmap : voice_pixmap;
	    mask = nick->op ? op_mask : voice_mask;
	}
	gtk_clist_set_pixtext(GTK_CLIST(gui->nicklist), row, 0,
                              nick->nick, 3, pix, mask);
    }

    if (!setup_get_bool("toggle_buggy_gtkthemes"))
    {
	gtk_clist_set_foreground(GTK_CLIST(gui->nicklist), row,
				 &THEME_GUI(current_theme)->colors[nick->gone ? COLOR_NICKLIST_GONE : COLOR_NICKLIST_DEFAULT]);
    }

    gtk_clist_set_row_data(GTK_CLIST(gui->nicklist), row, nick);
}

static void gui_nicklist_remove(CHANNEL_REC *channel, NICK_REC *nick)
{
    WINDOW_REC *window;
    gint row;
 
    g_return_if_fail(channel != NULL);
    g_return_if_fail(nick != NULL);

    window = CHANNEL_PARENT(channel);
    if (!gui_channel_active(channel)) return;

    row = gtk_clist_find_row_from_data(GTK_CLIST(WINDOW_GUI(window)->nicklist), nick);
    if (row != -1)
        gtk_clist_remove(GTK_CLIST(WINDOW_GUI(window)->nicklist), row);
}

static void gui_nicklist_set_color(CHANNEL_REC *channel, NICK_REC *nick)
{
    WINDOW_REC *window;
    gint row;
 
    g_return_if_fail(channel != NULL);
    g_return_if_fail(nick != NULL);

    window = CHANNEL_PARENT(channel);
    if (!gui_channel_active(channel)) return;

    row = gtk_clist_find_row_from_data(GTK_CLIST(WINDOW_GUI(window)->nicklist), nick);
    if (row != -1)
    {
	gtk_clist_set_foreground(GTK_CLIST(WINDOW_GUI(window)->nicklist), row,
				 &THEME_GUI(current_theme)->colors[nick->gone ? COLOR_NICKLIST_GONE : COLOR_NICKLIST_DEFAULT]);
    }
}

static void gui_nicklist_redraw_nick(gpointer key, NICK_REC *nick, CHANNEL_REC *channel)
{
    gui_nicklist_add(channel, nick);
}

void gui_nicklist_redraw(CHANNEL_REC *channel)
{
    GUI_WINDOW_REC *gui;

    gui = WINDOW_GUI(CHANNEL_PARENT(channel));
    if (!gui_channel_active(channel)) return;

    gtk_clist_freeze(GTK_CLIST(gui->nicklist));
    gtk_clist_clear(GTK_CLIST(gui->nicklist));
    g_hash_table_foreach(channel->nicks, (GHFunc) gui_nicklist_redraw_nick, channel);
    gtk_clist_thaw(GTK_CLIST(gui->nicklist));
    gui_nicklist_draw_label(channel);
}

static gboolean sig_massjoin(CHANNEL_REC *channel, GList *nicks)
{
    GUI_CHANNEL_REC *gui;
    gboolean who_all;
    gchar *str;
    GList *tmp, *next;
    gint tag, len;

    g_return_val_if_fail(channel != NULL, FALSE);
    g_return_val_if_fail(nicks != NULL, FALSE);

    if (!gui_channel_active(channel)) return TRUE;
    gui = CHANNEL_GUI(channel);

    /* Get the number of WHO requests in idle queue for channel */
    len = 0;
    for (tmp = g_list_first(gui->who_tags); tmp != NULL; tmp = tmp->next)
    {
	next = tmp->next;
	if (server_idle_find(channel->server, GPOINTER_TO_INT(tmp->data)))
	    len++;
	else
	{
	    /* Doesn't exist anymore, remove.. */
	    gui->who_tags = g_list_remove(gui->who_tags, tmp->data);
	}
    }

    /* If there's more than 5 requests to be sent, just forget it and send
       WHO #channel instead */
    who_all = len+g_list_length(nicks) > 5;
    if (len > 0 && who_all)
    {
	/* We decided to send WHO to channel, remove the existing WHO requests
	   from idle queue */
	for (tmp = g_list_first(gui->who_tags); tmp != NULL; tmp = tmp->next)
	    server_idle_remove(channel->server, GPOINTER_TO_INT(tmp->data));
	g_list_free(gui->who_tags);
	gui->who_tags = NULL;
    }

    gtk_clist_freeze(GTK_CLIST(WINDOW_GUI(CHANNEL_PARENT(channel))->nicklist));
    for (tmp = g_list_first(nicks); tmp != NULL; tmp = tmp->next)
    {
	NICK_REC *rec = tmp->data;

	/* add to nicklist widget */
	gui_nicklist_add(channel, rec);

	if (!who_all && rec->realname == NULL)
	{
	    /* if there's only few nicks joined, get their names separately
	       with WHO requests */
	    str = g_strdup_printf("WHOIS %s", rec->nick);
	    tag = server_idle_add(channel->server, str, rec->nick, 1,
				  "event 318", "event empty", 1,
				  "event 401", "event empty", 1,
				  "event 311", "silent event whois", 1,
				  "event 301", "event empty", 1,
				  "event 312", "event empty", 1,
				  "event 313", "event empty", 1,
				  "event 317", "event empty", 1,
				  "event 319", "event empty", 1, NULL);
	    gui->who_tags = g_list_append(gui->who_tags, GINT_TO_POINTER(tag));
	    g_free(str);
	}
    }

    if (who_all)
    {
	/* More than a few new nicks, list the entire channel */
	/* first check if there's any WHO requests in the same channel,
	   remove all of them */
	for (tmp = g_list_first(gui->who_tags); tmp != NULL; tmp = tmp->next)
	    server_idle_remove(channel->server, GPOINTER_TO_INT(tmp->data));
	g_list_free(gui->who_tags);

	/* Add the WHO request to idle queue */
	str = g_strdup_printf("WHO %s", channel->name);
	tag = server_idle_add(channel->server, str, channel->name, 1,
			      "event 315", "event empty", 1,
			      "event 352", "silent event who", 1, NULL);
	gui->who_tags = g_list_append(NULL, GINT_TO_POINTER(tag));
	g_free(str);
    }

    gtk_clist_thaw(GTK_CLIST(WINDOW_GUI(CHANNEL_PARENT(channel))->nicklist));
    gui_nicklist_draw_label(channel);
    return TRUE;
}

static gboolean sig_nicklist_new(CHANNEL_REC *channel, NICK_REC *nick)
{
    g_return_val_if_fail(channel != NULL, FALSE);

    if (channel->gui_data != NULL && !nick->send_massjoin)
    {
	gui_nicklist_add(channel, nick);
        gui_nicklist_draw_label(channel);
    }
    return TRUE;
}

static gboolean sig_nicklist_remove(CHANNEL_REC *channel, NICK_REC *nick)
{
    g_return_val_if_fail(channel != NULL, FALSE);

    if (channel->gui_data != NULL)
    {
	gui_nicklist_remove(channel, nick);
        gui_nicklist_draw_label(channel);
    }
    return TRUE;
}

static gboolean sig_nicklist_changed(CHANNEL_REC *channel, NICK_REC *nick)
{
    g_return_val_if_fail(channel != NULL, FALSE);

    if (!nick->send_massjoin)
    {
	gtk_clist_freeze(GTK_CLIST(WINDOW_GUI(CHANNEL_PARENT(channel))->nicklist));
	gui_nicklist_remove(channel, nick);
	gui_nicklist_add(channel, nick);
	gtk_clist_thaw(GTK_CLIST(WINDOW_GUI(CHANNEL_PARENT(channel))->nicklist));
        gui_nicklist_draw_label(channel);
    }
    return TRUE;
}

static gboolean sig_nick_gone_changed(CHANNEL_REC *channel, NICK_REC *nick)
{
    g_return_val_if_fail(channel != NULL, FALSE);

    if (channel->wholist)
	gui_nicklist_set_color(channel, nick);
    return TRUE;
}

static gboolean sig_nick_ircop_changed(CHANNEL_REC *channel, NICK_REC *nick)
{
    g_return_val_if_fail(channel != NULL, FALSE);

    if (!nick->send_massjoin)
    {
	gtk_clist_freeze(GTK_CLIST(WINDOW_GUI(CHANNEL_PARENT(channel))->nicklist));
	gui_nicklist_remove(channel, nick);
	gui_nicklist_add(channel, nick);
	gtk_clist_thaw(GTK_CLIST(WINDOW_GUI(CHANNEL_PARENT(channel))->nicklist));
	gui_nicklist_draw_label(channel);
    }
    return TRUE;
}

static gboolean sig_channel_wholist(CHANNEL_REC *channel)
{
    GList *tmp, *nicks;

    if (setup_get_bool("toggle_buggy_gtkthemes"))
	return TRUE;

    /* Entire WHO list received, redraw all gone nicks */
    nicks = nicklist_getnicks(channel);
    gtk_clist_freeze(GTK_CLIST(WINDOW_GUI(CHANNEL_PARENT(channel))->nicklist));
    for (tmp = nicks; tmp != NULL; tmp = tmp->next)
    {
	NICK_REC *rec = tmp->data;

	if (rec->gone)
	    gui_nicklist_set_color(channel, rec);
    }
    gtk_clist_thaw(GTK_CLIST(WINDOW_GUI(CHANNEL_PARENT(channel))->nicklist));
    g_list_free(nicks);
    return TRUE;
}

/* signal: button pressed in nicklist */
static gint sig_nicklist_button_pressed(GtkWidget *widget, GdkEventButton *event, WINDOW_REC *window)
{
    NICK_REC *nick;
    gint row, col;

    g_return_val_if_fail(widget != NULL, 0);
    g_return_val_if_fail(event != NULL, 0);

    if (gtk_clist_get_selection_info(GTK_CLIST(widget), event->x, event->y, &row, &col) < 0)
        return 0;

    if (event->button == 3 && g_list_find(GTK_CLIST(widget)->selection, GINT_TO_POINTER(row)) == NULL)
    {
        gtk_clist_unselect_all(GTK_CLIST(widget));
        gtk_clist_select_row(GTK_CLIST(widget), row, col);
    }

    nick = gtk_clist_get_row_data(GTK_CLIST(widget), row);
    if (nick == NULL) return 0;

    if (event->button == 1 && event->type == GDK_2BUTTON_PRESS)
    {
        /* create query with doubleclick */
        signal_emit("command query", 3, nick->nick, window->active->server, window->active);
    }

    if (event->button == 3)
    {
        /* show popup menu */
        gtk_clist_select_row(GTK_CLIST(widget), row, col);
        gui_nicklist_popup(NULL, window->active, NULL, event);
    }

    return 1;
}

/* display realname in statusbar when mouse is over nick in nicklist */
static gint sig_nicklist_motion(GtkCList *clist, GdkEventMotion *event, WINDOW_REC *window)
{
    static gint lastrow = -1;
    MAIN_WINDOW_REC *mainwin;
    NICK_REC *nick;
    gint ret, row, column, x, y;
    GdkModifierType mask;

    mainwin = WINDOW_GUI(window)->parent;
    gdk_window_get_pointer(GTK_WIDGET(clist)->window, &x, &y, &mask);

    ret = gtk_clist_get_selection_info(clist, event->x, event->y, &row, &column);
    if (ret < 1) row = -1;
    if (lastrow == row && (lastrow == -1 || mainwin->nicklist_id != 0))
    {
	/* same row as last time, no update */
	return 1;
    }
    lastrow = row;

    nick = row < 0 ? NULL : gtk_clist_get_row_data(clist, row);
    if (nick == NULL || nick->realname == NULL)
    {
        /* moved off the nick or no realname got yet, remove statusbar */
        if (mainwin->nicklist_id > 0)
        {
            gui_statusbar_pop(mainwin, mainwin->nicklist_id);
            mainwin->nicklist_id = 0;
        }
        return 1;
    }

    /* nick over nicklist, display realname in statusbar.. */
    if (mainwin->nicklist_id > 0)
	gui_statusbar_pop(mainwin, mainwin->nicklist_id);
    mainwin->nicklist_id = gui_statusbar_push(mainwin, nick->realname);

    return 1;
}

static gint sig_nicklist_leave(GtkCList *clist, GdkEventCrossing *event, WINDOW_REC *window)
{
    MAIN_WINDOW_REC *mainwin;

    mainwin = WINDOW_GUI(window)->parent;

    if (mainwin->nicklist_id > 0)
    {
        gui_statusbar_pop(mainwin, mainwin->nicklist_id);
        mainwin->nicklist_id = 0;
    }
    return 1;
}

static gboolean sig_window_created(WINDOW_REC *window)
{
    GtkWidget *clist;
    GUI_WINDOW_REC *gui;

    g_return_val_if_fail(window != NULL, FALSE);

    /* create op and voice pixmaps */
    gui = WINDOW_GUI(window);
    gtk_widget_realize(gui->window);

    /* initialize drag'n'drop */
    clist = gui->nicklist;
    gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 0, TRUE);
    gtk_clist_set_selection_mode(GTK_CLIST(clist), GTK_SELECTION_EXTENDED);

    gtk_drag_dest_set(clist, GTK_DEST_DEFAULT_ALL,
                      dnd_nicklist_target, ELEMENTS(dnd_nicklist_target),
                      GDK_ACTION_COPY | GDK_ACTION_MOVE | GDK_ACTION_LINK);
    gtk_signal_connect(GTK_OBJECT(clist), "drag_data_received",
                       GTK_SIGNAL_FUNC(nickdnd_drop), window);
    gtk_signal_connect(GTK_OBJECT(clist), "drag_motion",
                       GTK_SIGNAL_FUNC(nickdnd_motion), window);
    gtk_signal_connect(GTK_OBJECT(clist), "drag_leave",
                       GTK_SIGNAL_FUNC(nickdnd_leave), window);

    gtk_widget_set_events(gui->nicklist,
			  GDK_POINTER_MOTION_MASK |
			  GDK_POINTER_MOTION_HINT_MASK |
			  GDK_LEAVE_NOTIFY_MASK);
    gtk_signal_connect(GTK_OBJECT(gui->nicklist), "motion_notify_event",
                       GTK_SIGNAL_FUNC(sig_nicklist_motion), window);
    gtk_signal_connect(GTK_OBJECT(gui->nicklist), "leave_notify_event",
                       GTK_SIGNAL_FUNC(sig_nicklist_leave), window);
    gtk_signal_connect(GTK_OBJECT(clist), "button_press_event",
                       GTK_SIGNAL_FUNC(sig_nicklist_button_pressed), window);
    return TRUE;
}

static gboolean sig_mainwindow_created(MAIN_WINDOW_REC *window)
{
    if (op_pixmap != NULL)
	return TRUE;

    op_pixmap = gdk_pixmap_create_from_xpm_d(window->window->window, &op_mask, NULL, op_xpm);
    voice_pixmap = gdk_pixmap_create_from_xpm_d(window->window->window, &voice_mask, NULL, voice_xpm);
    ircop_pixmap = gdk_pixmap_create_from_xpm_d(window->window->window, &ircop_mask, NULL, ircop_xpm);

    /* ircop + op, ircop + voice */
    ircop_op_pixmap = gdk_pixmap_create_from_xpm_d(window->window->window, &ircop_op_mask, NULL, ircop_op_xpm);
    ircop_voice_pixmap = gdk_pixmap_create_from_xpm_d(window->window->window, &ircop_voice_mask, NULL, ircop_voice_xpm);

    return TRUE;
}

static gboolean event_names_list(gchar *data, SERVER_REC *server)
{
    CHANNEL_REC *chanrec;
    gchar *params, *type, *channel;

    params = event_get_params(data, 3, NULL, &type, &channel);

    chanrec = channel_find(server, channel);
    if (chanrec != NULL && !server->names_coming)
        gtk_clist_freeze(GTK_CLIST(WINDOW_GUI(CHANNEL_PARENT(chanrec))->nicklist));

    g_free(params);
    return TRUE;
}

static gboolean event_end_of_names(gchar *data, SERVER_REC *server)
{
    CHANNEL_REC *chanrec;
    gchar *params, *channel;

    g_return_val_if_fail(server != NULL, FALSE);

    params = event_get_params(data, 2, NULL, &channel);

    chanrec = channel_find(server, channel);
    if (chanrec != NULL)
        gtk_clist_thaw(GTK_CLIST(WINDOW_GUI(CHANNEL_PARENT(chanrec))->nicklist));

    g_free(params);
    return TRUE;
}

static void nicklist_gonecheck(SERVER_REC *server)
{
    GList *chan, *nick, *querylist;
    GUI_SERVER_REC *gui;
    CHANNEL_REC *chanrec;
    NICK_REC *nickrec;
    GString *query;
    gint nicks;
    time_t now;

    if (server->usermode_away || server_idle_find(server, SERVER_GUI(server)->userhost_tag))
    {
	/* USERHOST command is still in idle queue, just wait for a while and
           try again.. also, don't check others' gone status if you are gone.. */
        return;
    }

    gui = SERVER_GUI(server);
    now = time(NULL);

    /* get the first channel where we're supposed to find next gone people.. */
    chan = g_list_find(channels, gui->gone_channel);
    if (chan == NULL)
    {
	chan = g_list_first(channels);
	if (gui->gone_nicks != NULL)
	{
	    g_list_free(gui->gone_nicks);
	    gui->gone_nicks = NULL;
	}
    }
    gui->gone_channel = NULL;

    nicks = 0; querylist = NULL;
    for (;; chan = chan->next)
    {
        if (chan == NULL)
        {
            chan = g_list_first(channels);
            if (chan == NULL) break; /* ?? */
        }

        if (gui->gone_channel == NULL)
            gui->gone_channel = chan;
        else if (gui->gone_channel == chan)
        {
            /* went through the list. */
            break;
        }

        chanrec = chan->data;

        if (chanrec->server != server || chanrec->type != CHANNEL_TYPE_CHANNEL ||
            !chanrec->synced) /* not synced yet.. */
            continue;

	if (gui->gone_nicks == NULL)
	{
	    if (CHANNEL_GUI(chanrec)->gone_oldest != 0 &&
		now-CHANNEL_GUI(chanrec)->gone_oldest < MAX_GONE_REFRESH_TIME)
	    {
		/* not old enough nicks in this channel */
		continue;
	    }

	    /* Create the gone nicks list */
            gui->gone_nicks = nicklist_getnicks(chanrec);
	    CHANNEL_GUI(chanrec)->gone_oldest = now;
	}

        while (gui->gone_nicks != NULL)
        {
            nickrec = gui->gone_nicks->data;

	    gui->gone_nicks = g_list_remove(gui->gone_nicks, gui->gone_nicks->data);

	    /*
  	      1. not our own nick
	      2. haven't been checked for MAX_GONE_REFRESH_TIME seconds
	      3. nick isn't being sent in massjoin event - WHOIS/WHO request
	         will take care of it
              4. nick isn't already in list
	     */
	    if (g_strcasecmp(nickrec->nick, server->nick) != 0 &&
		!nickrec->send_massjoin &&
                glist_find_icase_string(querylist, nickrec->nick) == NULL)
	    {
		if (now-nickrec->last_check < MAX_GONE_REFRESH_TIME)
		{
                    /* update the oldest nickrec in channel */
		    if (CHANNEL_GUI(chanrec)->gone_oldest > nickrec->last_check)
			CHANNEL_GUI(chanrec)->gone_oldest = nickrec->last_check;
		}
		else
		{
		    /* got a nick */
		    querylist = g_list_append(querylist, nickrec->nick);
		    nickrec->last_check = now;
		    nicks++;
		    if (nicks == 4) break; /* max. 4 nick queries / line */
		}
            }
        }
        if (nicks == 4)
        {
            if (gui->gone_nicks == NULL) chan = chan->next;
            break;
        }
    }
    gui->gone_channel = chan;

    if (querylist == NULL)
    {
        /* i already know everyone, great! :) */
        return;
    }

    /* send the query to server */
    query = g_string_new("USERHOST :");
    for (nick = g_list_first(querylist); nick != NULL; nick = nick->next)
    {
        if (nick->next == NULL)
            g_string_append(query, nick->data);
        else
            g_string_sprintfa(query, "%s ", (gchar *) nick->data);
    }

    /* need to redirect it so ui-common doesn't print anything.. */
    SERVER_GUI(server)->userhost_tag =
	server_idle_add(server, query->str, query->str+10, 1,
			"event 302", "userhost event", -1,
			"event 401", "event empty", 1, NULL);

    g_list_free(querylist);
    g_string_free(query, TRUE);
}

static gint sig_gonecheck(void)
{
    g_list_foreach(servers, (GFunc) nicklist_gonecheck, NULL);
    return 1;
}

static gboolean sig_connected(SERVER_REC *server)
{
    g_return_val_if_fail(server != NULL, FALSE);

    server_redirect_init(server, "command userhost", 1, "event 302", "event 401", NULL);
    SERVER_GUI(server)->gone_channel = NULL;
    SERVER_GUI(server)->gone_nicks = NULL;
    return TRUE;
}

void gui_nicklist_init(void)
{
    gonecheck_tag = gui_timeout_add(5000, (GUITimeoutFunction) sig_gonecheck, NULL);
    signal_add("gui window created", (SIGNAL_FUNC) sig_window_created);
    signal_add("gui mainwindow created", (SIGNAL_FUNC) sig_mainwindow_created);
    signal_add("massjoin", (SIGNAL_FUNC) sig_massjoin);
    signal_add("nicklist new", (SIGNAL_FUNC) sig_nicklist_new);
    signal_add("nicklist remove", (SIGNAL_FUNC) sig_nicklist_remove);
    signal_add("nicklist changed", (SIGNAL_FUNC) sig_nicklist_changed);
    signal_add("nick gone changed", (SIGNAL_FUNC) sig_nick_gone_changed);
    signal_add("nick ircop changed", (SIGNAL_FUNC) sig_nick_ircop_changed);
    signal_add("channel wholist", (SIGNAL_FUNC) sig_channel_wholist);
    signal_add("nick mode changed", (SIGNAL_FUNC) sig_nicklist_changed);
    signal_add("event 353", (SIGNAL_FUNC) event_names_list);
    signal_add("event 366", (SIGNAL_FUNC) event_end_of_names);
    signal_add("server connected", (SIGNAL_FUNC) sig_connected);
}

void gui_nicklist_deinit(void)
{
    gui_timeout_remove(gonecheck_tag);
    signal_remove("gui window created", (SIGNAL_FUNC) sig_window_created);
    signal_remove("gui mainwindow created", (SIGNAL_FUNC) sig_mainwindow_created);
    signal_remove("massjoin", (SIGNAL_FUNC) sig_massjoin);
    signal_remove("nicklist new", (SIGNAL_FUNC) sig_nicklist_new);
    signal_remove("nicklist remove", (SIGNAL_FUNC) sig_nicklist_remove);
    signal_remove("nicklist changed", (SIGNAL_FUNC) sig_nicklist_changed);
    signal_remove("nick gone changed", (SIGNAL_FUNC) sig_nick_gone_changed);
    signal_remove("nick ircop changed", (SIGNAL_FUNC) sig_nick_ircop_changed);
    signal_remove("channel wholist", (SIGNAL_FUNC) sig_channel_wholist);
    signal_remove("nick mode changed", (SIGNAL_FUNC) sig_nicklist_changed);
    signal_remove("event 353", (SIGNAL_FUNC) event_names_list);
    signal_remove("event 366", (SIGNAL_FUNC) event_end_of_names);
    signal_remove("server connected", (SIGNAL_FUNC) sig_connected);
}
