/*
 dialog-links.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

typedef struct
{
    gchar *name;
    gchar *hub;
    gchar *desc;

    gboolean mark;
}
LINK_REC;

typedef struct
{
    gchar *name;
    gchar *hub;
    GNode *hublink;
    LINK_REC *link;

    gboolean found;
    gint best;
}
LINK_FIND_REC;

typedef struct
{
    SERVER_REC *server;
    GtkWidget *ctree;
}
LINK_DIALOG_REC;

static GList *link_dialogs, *resend_stats;
static gint resend_tag;

static gboolean link_find_node(GNode *node, LINK_FIND_REC *findrec)
{
    LINK_REC *rec;

    rec = node->data;
    if (rec == NULL || strcmp(rec->name, findrec->name) != 0)
	return FALSE;

    findrec->link = rec;
    return TRUE;
}

static LINK_REC *link_find(SERVER_REC *server, gchar *link)
{
    LINK_FIND_REC rec;

    rec.name = link;
    rec.link = NULL;
    g_node_traverse(SERVER_GUI(server)->links, G_IN_ORDER, G_TRAVERSE_ALL, -1,
		    (GNodeTraverseFunc) link_find_node, &rec);

    return rec.link;
}

static gboolean gnode2link(GtkCTree *ctree, gint depth, GNode *node, GtkCTreeNode *cnode, gpointer data)
{
    LINK_REC *rec;

    rec = node->data;
    gtk_ctree_set_node_info(ctree, cnode, rec == NULL ? _("Links") : rec->name,
			    0, NULL, NULL, NULL, NULL,
			    node->children == NULL, TRUE);
    if (rec != NULL)
    {
	gtk_ctree_node_set_text(ctree, cnode, 1, rec->desc);
	gtk_ctree_node_set_row_data(ctree, cnode, rec);

	if (rec->mark && !setup_get_bool("toggle_buggy_gtkthemes"))
	    gtk_ctree_node_set_foreground(ctree, cnode, &THEME_GUI(current_theme)->colors[RED]);
    }
    return TRUE;
}

static void sig_destoy(GtkWidget *dialog, GtkWidget *ctree)
{
    GList *tmp;

    /* remove dialog from link dialogs list */
    for (tmp = link_dialogs; tmp != NULL; tmp = tmp->next)
    {
	LINK_DIALOG_REC *rec = tmp->data;

	if (rec->ctree == ctree)
	{
	    g_free(rec);
	    link_dialogs = g_list_remove(link_dialogs, rec);
	    break;
	}
    }
}

static void redraw_links(GtkCTree *ctree, SERVER_REC *server)
{
    if (SERVER_GUI(server)->links == NULL)
	return;

    gtk_clist_clear(GTK_CLIST(ctree));
    gtk_ctree_insert_gnode(GTK_CTREE(ctree), NULL, NULL, g_node_get_root(SERVER_GUI(server)->links),
			   (GtkCTreeGNodeFunc) gnode2link, server);
    gtk_clist_columns_autosize(GTK_CLIST(ctree));
}

static void links_get(SERVER_REC *server, gchar *link, gboolean add_list)
{
    GUI_SERVER_REC *gui;
    gchar *str;

    g_return_if_fail(server != NULL);

    gui = SERVER_GUI(server);
    str = link == NULL ? g_strdup("LINKS") : g_strdup_printf("LINKS %s %s", link, link);
    if (!gui->stats_waiting)
    {
	/* queue empty, send the /LINKS command */
	server_redirect_event(server, "LINKS", 2,
			      "event 365", "gui event 365", -1,
			      "event 263", "gui event 263", 1,
			      "event 364", "gui event 364", -1, NULL);

	irc_send_cmd(server, str);
	gui->stats_waiting = TRUE;
    }

    if (!add_list)
	g_free(str);
    else
	gui->last_stats = g_list_append(gui->last_stats, str);
}

static void linklist_mark(GtkCTree *ctree, SERVER_REC *server, gchar *link)
{
    LINK_REC *rec;
    GtkCTreeNode *cnode;

    if (setup_get_bool("toggle_buggy_gtkthemes"))
	return;

    /* mark the row with red */
    rec = link_find(server, link);
    if (rec != NULL)
    {
	rec->mark = TRUE;
	cnode = gtk_ctree_find_by_row_data(GTK_CTREE(ctree), NULL, rec);
	gtk_ctree_node_set_foreground(ctree, cnode, &THEME_GUI(current_theme)->colors[RED]);
    }
}

static void stats_get(SERVER_REC *server, gchar type, gchar *statserver, gboolean add_list)
{
    GUI_SERVER_REC *gui;
    gchar *str;

    g_return_if_fail(server != NULL);

    gui = SERVER_GUI(server);
    str = g_strdup_printf(statserver == NULL ? "STATS %c" : "STATS %c %s", type, statserver);
    if (!gui->stats_waiting)
    {
	/* queue empty, send the /LINKS command */
	switch (type)
	{
	    case 'i':
	    case 'I':
		server_redirect_event(server, "STATS i", 2,
				      "event 219", "gui event 219", 1,
				      "event 263", "gui event 263", 1,
				      "event 215", "gui event 215", -1, NULL);
		break;
	    case 'k':
	    case 'K':
		server_redirect_event(server, "STATS k", 2,
				      "event 219", "gui event 219", 1,
				      "event 263", "gui event 263", 1,
				      "event 216", "gui event 215", -1, NULL);
		break;
	}
	irc_send_cmd(server, str);
	gui->stats_waiting = TRUE;
    }

    if (!add_list)
	g_free(str);
    else
	gui->last_stats = g_list_append(gui->last_stats, str);
}

/* Get /LINKS for selected servers */
static void links_popup_get_links(GtkWidget *widget, GList *links)
{
    GtkWidget *menu;
    GtkCTree *ctree;
    SERVER_REC *server;
    GList *tmp;

    /* get the server, check that it hasn't been disconnected yet.. */
    menu = gtk_object_get_data(GTK_OBJECT(widget), "menu");
    server = gtk_object_get_data(GTK_OBJECT(menu), "server");
    ctree = gtk_object_get_data(GTK_OBJECT(menu), "ctree");
    if (g_list_find(servers, server) == NULL)
	return;

    /* send all the /LINKS queries */
    for (tmp = links; tmp != NULL; tmp = tmp->next)
    {
        linklist_mark(ctree, server, tmp->data);
	links_get(server, tmp->data, TRUE);
    }
}

#if 0
/* Get the I/K/B-lines for selected servers */
static void links_popup_get_lines(GtkWidget *widget, GList *links)
{
    GtkWidget *menu;
    SERVER_REC *server;
    GList *tmp;

    /* get the server, check that it hasn't been disconnected yet.. */
    menu = gtk_object_get_data(GTK_OBJECT(widget), "menu");
    server = gtk_object_get_data(GTK_OBJECT(menu), "server");
    if (g_list_find(servers, server) == NULL)
	return;

    /* send all the /STATS queries */
    for (tmp = links; tmp != NULL; tmp = tmp->next)
	stats_get(server, 'i', tmp->data, TRUE);
}
#endif

static void linkspopup_free(GList *list)
{
    g_list_foreach(list, (GFunc) g_free, NULL);
    g_list_free(list);
}

static void links_popup_menu(GtkCTree *ctree, GdkEventButton *event, SERVER_REC *server)
{
    GtkWidget *menu;
    GtkObject *listobject;
    LINK_REC *link;
    GList *list, *tmp;

    /* get the selected servers */
    list = NULL;
    for (tmp = GTK_CLIST(ctree)->selection; tmp != NULL; tmp = tmp->next)
    {
        link = gtk_ctree_node_get_row_data(ctree, tmp->data);
	if (link != NULL)
	    list = g_list_append(list, g_strdup(link->name));
    }

    menu = gtk_menu_new();
    gtk_object_set_data(GTK_OBJECT(menu), "server", server);
    gtk_object_set_data(GTK_OBJECT(menu), "ctree", ctree);

    /* create GtkObject for list so we can use reference counting.. */
    listobject = gtk_object_new(GTK_TYPE_OBJECT, NULL);
    gtk_signal_connect_object(listobject, "destroy",
                              GTK_SIGNAL_FUNC(linkspopup_free), (GtkObject *) list);

    /* destroy list after menu is closed */
    gtk_signal_connect_object(GTK_OBJECT(menu), "selection_done",
                              GTK_SIGNAL_FUNC(gtk_object_unref), listobject);

    popup_add(menu, _("Get the /LINKS for hubs"), links_popup_get_links, list);
    /*popup_add(menu, _("Get the I/K/B-lines"), links_popup_get_lines, list);*/
    gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL, event->button, event->time);
}

static gint sig_button_press(GtkCTree *ctree, GdkEventButton *event, SERVER_REC *server)
{
    GtkCTreeNode *cnode;
    GtkCTreeRow *crow;
    LINK_REC *link;
    gint row, col;

    g_return_val_if_fail(event != NULL, 0);

    if (g_list_find(servers, server) == NULL)
	return 0;

    if (gtk_clist_get_selection_info(GTK_CLIST(ctree), event->x, event->y, &row, &col) < 0)
        return 0;

    if (event->button == 3)
    {
	/* create popup menu */
	links_popup_menu(ctree, event, server);
	return 0;
    }

    if (event->type != GDK_2BUTTON_PRESS || event->button != 1)
	return 0;

    /* doubleclicked */
    link = gtk_clist_get_row_data(GTK_CLIST(ctree), row);
    if (link == NULL) return 0;

    cnode = gtk_ctree_node_nth(ctree, row);
    if (cnode == NULL) return 0;

    crow = GTK_CTREE_ROW(cnode);

    if (strchr(link->name, '*') != NULL && crow->is_leaf)
    {
	/* Ask the servers behind the hub */
	linklist_mark(ctree, server, link->name);
	links_get(server, link->name, TRUE);
    }
    else if (strchr(link->name, '*') == NULL && crow->is_leaf)
    {
	/* open server query dialog */
    }

    return 0;
}

void dialog_links(SERVER_REC *server)
{
    GtkWidget *dialog, *scrollbox, *ctree;
    LINK_DIALOG_REC *rec;
    gchar *titles[2];

    g_return_if_fail(server != NULL);

    titles[0] = _("Server");
    titles[1] = _("Description");

    dialog = gnome_dialog_new(PACKAGE, GNOME_STOCK_BUTTON_CLOSE, NULL);
    gtk_widget_set_usize(dialog, 700, 500);
    gtk_window_set_policy(GTK_WINDOW(dialog), TRUE, TRUE, FALSE);
    gtk_signal_connect(GTK_OBJECT(dialog), "delete_event",
                       GTK_SIGNAL_FUNC(gtk_widget_destroy), NULL);

    /* Create server ctree widget */
    scrollbox = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollbox),
                                   GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), scrollbox, TRUE, TRUE, 0);

    ctree = gtk_ctree_new_with_titles(2, 0, titles);
    gtk_signal_connect(GTK_OBJECT(dialog), "destroy",
                       GTK_SIGNAL_FUNC(sig_destoy), ctree);
    gtk_signal_connect(GTK_OBJECT(ctree), "button_press_event",
                       GTK_SIGNAL_FUNC(sig_button_press), server);
    gtk_clist_set_column_resizeable(GTK_CLIST(ctree), 0, TRUE);
    gtk_clist_set_column_resizeable(GTK_CLIST(ctree), 1, TRUE);
    gtk_container_add(GTK_CONTAINER(scrollbox), ctree);

    gtk_clist_set_selection_mode(GTK_CLIST(ctree), GTK_SELECTION_EXTENDED);
    if (SERVER_GUI(server)->links != NULL)
	redraw_links(GTK_CTREE(ctree), server);
    else
    {
	/* get the links list */
	links_get(server, NULL, TRUE);
    }

    rec = g_new0(LINK_DIALOG_REC, 1);
    rec->ctree = ctree;
    rec->server = server;
    link_dialogs = g_list_append(link_dialogs, rec);

    gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 0, GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(dialog));
    gtk_widget_show_all(dialog);
}

static void link_free(LINK_REC *link)
{
    g_free(link->name);
    if (link->hub != NULL) g_free(link->hub);
    g_free(link->desc);
    g_free(link);
}

static gboolean link_add(GNode *node, LINK_FIND_REC *findrec)
{
    LINK_REC *rec;
    gint len;

    rec = node->data;
    if (rec == NULL) return FALSE;

    if (g_strcasecmp(rec->name, findrec->name) == 0)
    {
	/* link found, replace it */
	link_free(rec);
	node->data = findrec->link;
	findrec->found = TRUE;
        findrec->hublink = node;
	return TRUE;
    }

    if (findrec->hublink == NULL && findrec->hub != NULL &&
	g_strcasecmp(rec->name, findrec->hub) == 0)
    {
	/* exact hub match found */
        findrec->hublink = node;
    }

    len = strlen(rec->name);
    if (findrec->hub == NULL && findrec->best < len &&
	match_wildcards(rec->name, findrec->name))
    {
	/* wildcarded hub found (*.irchub.net) */
	findrec->hublink = node;
	findrec->best = len;
    }

    return FALSE;
}

static gboolean link_fix(GNode *node, LINK_FIND_REC *findrec)
{
    LINK_REC *rec;

    rec = node->data;
    if (rec != NULL && rec->hub != NULL &&
	node->parent != findrec->hublink != 0 &&
	match_wildcards(findrec->name, rec->hub))
    {
	/* hub found for record */
	g_node_unlink(node);
	g_node_append(findrec->hublink, node);
    }

    return FALSE;
}

static gboolean event_links(gchar *data, SERVER_REC *server)
{
    GUI_SERVER_REC *gui;
    LINK_REC *rec;
    LINK_FIND_REC *findrec;
    gchar *params, *name, *hub, *desc;
    GNode *node;

    g_return_val_if_fail(data != NULL, FALSE);

    params = event_get_params(data, 4, NULL, &name, &hub, &desc);
    /* skip hop count */
    while (*desc != '\0' && *desc != ' ') desc++;
    while (*desc == ' ') desc++;

    gui = SERVER_GUI(server);

    /* create link record */
    rec = g_new0(LINK_REC, 1);
    rec->name = g_strdup(name);
    rec->desc = g_strdup(desc);
    rec->hub = g_strcasecmp(name, hub) == 0 ? NULL : g_strdup(hub);

    if (gui->links == NULL)
    {
	/* no links yet, just add the first */
	gui->links = g_node_new(NULL);
	g_node_append_data(gui->links, rec);
    }
    else
    {
	/* add/replace link */
	findrec = g_new0(LINK_FIND_REC, 1);
	findrec->name = g_strdup(name);
	findrec->hub = g_strcasecmp(name, hub) == 0 ? NULL : g_strdup(hub);
	findrec->link = rec;

	g_node_traverse(gui->links, G_IN_ORDER, G_TRAVERSE_ALL, -1,
			(GNodeTraverseFunc) link_add, findrec);

	if (!findrec->found)
	{
	    /* add link to list */
	    node = g_node_new(rec);
	    g_node_append(findrec->hublink != NULL ?
			  findrec->hublink : gui->links, node);
            findrec->hublink = node;
	}

	/* check if there's childs that belong under this node */
	g_node_traverse(gui->links, G_IN_ORDER, G_TRAVERSE_ALL, -1,
			(GNodeTraverseFunc) link_fix, findrec);

	g_free(findrec->name);
	g_free(findrec->hub);
	g_free(findrec);
    }

    g_free(params);
    return TRUE;
}

static gboolean event_end_of_links(gchar *data, SERVER_REC *server)
{
    GUI_SERVER_REC *gui;
    GList *tmp;

    gui = SERVER_GUI(server);
    if (gui->last_stats != NULL)
    {
	LINK_REC *rec;
	gchar *p;

	/* unmark the link .. LINKS *.blah *.blah */
	p = strchr(gui->last_stats->data+6, ' ');
	if (p != NULL) *p = '\0';
	rec = link_find(server, gui->last_stats->data+6);
	if (rec != NULL) rec->mark = FALSE;

	/* remove the expected LINKS reply from queue */
	g_free(gui->last_stats->data);
	gui->last_stats = g_list_remove(gui->last_stats, gui->last_stats->data);
    }

    /* redraw dialogs */
    for (tmp = link_dialogs; tmp != NULL; tmp = tmp->next)
    {
	LINK_DIALOG_REC *rec = tmp->data;

	if (rec->server == server)
	{
	    redraw_links(GTK_CTREE(rec->ctree), server);
	    break;
	}
    }

    gui->stats_waiting = FALSE;
    if (gui->last_stats != NULL)
    {
        /* send the next /LINKS query */
	links_get(server, gui->last_stats->data, FALSE);
    }
    return TRUE;
}

static gboolean event_stats_i(gchar *data, SERVER_REC *server)
{
    return TRUE;
}

static gboolean event_end_of_stats(gchar *data, SERVER_REC *server)
{
    return TRUE;
}

static gboolean event_try_again(gchar *data, SERVER_REC *server)
{
    /* need to send the STATS/LINKS again. */
    SERVER_GUI(server)->stats_waiting = FALSE;
    if (g_list_find(resend_stats, server) == NULL)
	resend_stats = g_list_append(resend_stats, server);
    return TRUE;
}

static gboolean sig_connected(SERVER_REC *server)
{
    g_return_val_if_fail(server != NULL, FALSE);

    server_redirect_init(server, "command links", 2, "event 365", "event 263", "event 364", NULL);
    return TRUE;
}

static gboolean linknode_free(GNode *node)
{
    if (node->data != NULL) link_free(node->data);
    return FALSE;
}

static gboolean sig_disconnected(SERVER_REC *server)
{
    GUI_SERVER_REC *gui;

    g_return_val_if_fail(server != NULL, FALSE);

    gui = SERVER_GUI(server);

    /* remove all the server links */
    if (gui->links != NULL)
    {
	g_node_traverse(gui->links, G_IN_ORDER, G_TRAVERSE_ALL, -1,
			(GNodeTraverseFunc) linknode_free, NULL);
	g_node_destroy(gui->links);
    }

    /* destroy the /LINKS queue */
    while (gui->last_stats != NULL)
    {
	g_free(gui->last_stats->data);
        gui->last_stats = g_list_remove(gui->last_stats, gui->last_stats->data);
    }

    /* remove from resend /LINKS list */
    resend_stats = g_list_remove(resend_stats, server);
    return TRUE;
}

static gint sig_resend_stats(void)
{
    GUI_SERVER_REC *gui;
    SERVER_REC *server;
    GList *tmp, *next;
    gchar *params, *cmd, *type, *args;

    /* try to send STATS/LINKS again */
    for (tmp = resend_stats; tmp != NULL; tmp = next)
    {
	next = tmp->next;

	server = resend_stats->data;
	gui = SERVER_GUI(server);
	if (gui->stats_waiting)
	    continue; /* already waiting for STATS/LINKS reply, skip */

        params = cmd_get_params(gui->last_stats->data, 3, &cmd, &type, &args);
	if (gui->last_stats != NULL)
	{
	    if (strcmp(cmd, "LINKS") == 0)
		links_get(server, args, FALSE);
	    else if (strcmp(cmd, "STATS") == 0)
		stats_get(server, *type, args, FALSE);

	}
	g_free(params);

	if (gui->last_stats == NULL || gui->last_stats->next == NULL)
	    resend_stats = g_list_remove(resend_stats, server);
    }
    return 1;
}

void dialog_links_init(void)
{
    resend_stats = NULL;
    resend_tag = gui_timeout_add(10000, (GUITimeoutFunction) sig_resend_stats, NULL);

    signal_add("server connected", (SIGNAL_FUNC) sig_connected);
    signal_add("server disconnected", (SIGNAL_FUNC) sig_disconnected);

    signal_add("event 364", (SIGNAL_FUNC) event_links);
    signal_add("event 365", (SIGNAL_FUNC) event_end_of_links);
    signal_add("gui event 364", (SIGNAL_FUNC) event_links);
    signal_add("gui event 365", (SIGNAL_FUNC) event_end_of_links);
    signal_add("gui event 263", (SIGNAL_FUNC) event_try_again);
    signal_add("gui event 215", (SIGNAL_FUNC) event_stats_i);
    signal_add("gui event 219", (SIGNAL_FUNC) event_end_of_stats);
}

void dialog_links_deinit(void)
{
    g_list_foreach(link_dialogs, (GFunc) g_free, NULL);
    g_list_free(link_dialogs);

    gui_timeout_remove(resend_tag);
    g_list_free(resend_stats);

    signal_remove("server connected", (SIGNAL_FUNC) sig_connected);
    signal_remove("server disconnected", (SIGNAL_FUNC) sig_disconnected);

    signal_remove("event 364", (SIGNAL_FUNC) event_links);
    signal_remove("event 365", (SIGNAL_FUNC) event_end_of_links);
    signal_remove("gui event 364", (SIGNAL_FUNC) event_links);
    signal_remove("gui event 365", (SIGNAL_FUNC) event_end_of_links);
    signal_remove("gui event 263", (SIGNAL_FUNC) event_try_again);
    signal_remove("gui event 215", (SIGNAL_FUNC) event_stats_i);
    signal_remove("gui event 219", (SIGNAL_FUNC) event_end_of_stats);
}
