/*
 gnome-stuff.c : functions to get gmasqdialer build without GNOME libraries.
                 these functions are pretty slow and dum, if you want better,
                 use GNOME libraries!

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
    */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include <glib.h>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "gnome-stuff.h"

#define CONFIG_FILE PACKAGE".conf"

/* configuration file handling */
static GSList *prefixes;

void gnome_config_push_prefix(gchar *path)
{
    prefixes = g_slist_append(prefixes, g_strdup(path));
}

void gnome_config_pop_prefix(void)
{
    GSList *last;

    if (prefixes != NULL)
    {
        last = g_slist_last(prefixes);
        g_free(last->data);
        prefixes = g_slist_remove_link(prefixes, last);
    }
}

static gchar *get_prefix(gchar *path)
{
    if (prefixes == NULL)
        return g_strdup(path);

    return g_strconcat((gchar *) g_slist_last(prefixes)->data, path, NULL);
}

gchar *gnome_config_get_string(gchar *path)
{
    static gchar buf[1024];
    gchar *origpath, *def, *def2, *ptr;
    gint catlen, taglen;
    gboolean rightcat;
    GString *str;
    FILE *f;

    g_return_val_if_fail(path != NULL, NULL);

    def = strrchr(path, '=');
    if (def != NULL) def++;

    str = g_string_new(NULL);
    g_string_sprintf(str, "%s/."PACKAGE".conf", g_get_home_dir());

    f = fopen(str->str, "r");
    g_string_free(str, TRUE);
    if (f == NULL) return def;

    origpath = path = get_prefix(path);

    path = strchr(path+1, '/');
    if (path == NULL)
    {
        g_warning("gnome_config_get_string() : Invalid path");
        fclose(f);
        g_free(origpath);
        return NULL;
    }
    path++;

    ptr = strrchr(path, '/');
    if (ptr != NULL) ptr++; else ptr = path;

    catlen = (gint) (ptr-path-1);
    def2 = strrchr(ptr, '=');
    taglen = def2 == NULL ? strlen(ptr) : (gint) (def2-ptr);

    rightcat = FALSE;
    while (fgets(buf, sizeof(buf), f) != NULL)
    {
        if (*buf == '[')
        {
            if (rightcat) break;

            if (g_strncasecmp(buf+1, path, catlen) == 0)
            {
                /* section found */
                rightcat = TRUE;
            }
        }

        if (rightcat && g_strncasecmp(buf, ptr, taglen) == 0 && buf[taglen] == '=')
        {
            /* got it! */
            buf[strlen(buf)-1] = '\0';
            fclose(f);
            g_free(origpath);
            return buf+taglen+1;
        }
    }

    fclose(f);
    g_free(origpath);
    return def;
}

gint gnome_config_get_int(gchar *path)
{
    gchar *str;
    gint num;

    g_return_val_if_fail(path != NULL, 0);

    str = gnome_config_get_string(path);
    sscanf(str, "%d", &num);
    return num;
}

gboolean gnome_config_get_bool(gchar *path)
{
    g_return_val_if_fail(path != NULL, FALSE);

    return g_strcasecmp(gnome_config_get_string(path), "TRUE") == 0;
}

gpointer gnome_config_init_iterator(gchar *path)
{
    FILE *f;
    gchar buf[1024], *ptr;
    GString *str;

    g_return_val_if_fail(path != NULL, NULL);

    path = get_prefix(path);

    str = g_string_new(NULL);
    g_string_sprintf(str, "%s/."PACKAGE".conf", g_get_home_dir());

    f = fopen(str->str, "r");
    g_string_free(str, TRUE);
    if (f == NULL)
    {
        g_free(path);
        return NULL;
    }

    ptr = strrchr(path, '/');
    if (ptr != NULL) ptr++; else ptr = path;

    while (fgets(buf, sizeof(buf), f) != NULL)
    {
        if (buf[0] == '[' && g_strncasecmp(buf+1, ptr, strlen(ptr)) == 0 && buf[strlen(ptr)+1] == ']')
        {
            /* got it! */
            g_free(path);
            return f;
        }
    }

    fclose(f);
    g_free(path);
    return NULL;
}

gpointer gnome_config_iterator_next(gpointer handle, gchar **key, gchar **value)
{
    FILE *f = handle;
    static gchar buf[1024];

    if (handle == NULL) return NULL;

    g_return_val_if_fail(key != NULL, NULL);
    g_return_val_if_fail(value != NULL, NULL);

    while (fgets(buf, sizeof(buf), f) != NULL)
    {
        if (buf[0] == '\0') continue;
        if (buf[0] == '[') break;

        buf[strlen(buf)-1] = '\0';
        *value = strchr(buf, '=');
        if (*value == NULL) continue;
        *(*value)++ = '\0';
        *key = buf;
        return handle;
    }

    fclose(f);
    return NULL;
}

static void gnome_config_write(gchar *path, gchar *value, gboolean cleansect)
{
    FILE *f, *fout;
    gchar buf[1024], *ptr, *tempname, *origpath;
    gint catlen, taglen;
    gboolean rightcat, configwrote;
    GString *str;

    g_return_if_fail(path != NULL);

    str = g_string_new(NULL);
    g_string_sprintf(str, "%s/."PACKAGE".conf", g_get_home_dir());

    f = fopen(str->str, "r");

    tempname = tempnam(g_get_home_dir(), PACKAGE);
    fout = fopen(tempname, "w+");
    if (fout == NULL)
    {
        g_warning("Can't create temporary configuration file %s\n", tempname);
        g_string_free(str, TRUE);
        return;
    }

    origpath = path = get_prefix(path);

    ptr = strchr(path+1, '/');
    if (ptr == NULL)
    {
        g_warning("gnome_config_write() : invalid path");
        if (f != NULL) fclose(f);
        fclose(fout);
        remove(tempname);
        g_string_free(str, TRUE);
        g_free(origpath);
        return;
    }

    path = g_strdup(ptr+1);
    ptr = strchr(path, '/');
    if (ptr != NULL) *ptr++ = '\0';
    if (ptr[0] == '\0') ptr = NULL;

    catlen = strlen(path);
    taglen = ptr == NULL ? 0 : strlen(ptr);

    /* path = [section], ptr = tag */

    configwrote = FALSE;
    if (f != NULL)
    {
        rightcat = FALSE;
        while (fgets(buf, sizeof(buf), f) != NULL)
        {
            if (*buf == '[')
            {
                if (rightcat)
                {
                    /* tag not found, append it. */
                    if (value != NULL) fprintf(fout, "%s=%s\n", ptr, value);
                    fprintf(fout, "%s", buf);
                    configwrote = TRUE;
                    break;
                }

                if (g_strncasecmp(buf+1, path, catlen) == 0)
                {
                    /* section found */
                    rightcat = TRUE;
                }
            }

            if (rightcat && ptr != NULL && g_strncasecmp(buf, ptr, taglen) == 0)
            {
                /* tag found! */
                if (value != NULL) fprintf(fout, "%s=%s\n", ptr, value);
                configwrote = TRUE;
                break;
            }

            if (!cleansect || !rightcat) fprintf(fout, "%s", buf);
        }

        if (rightcat && !configwrote)
        {
            /* tag not found, append it. */
            if (value != NULL) fprintf(fout, "%s=%s\n", ptr, value);
            configwrote = TRUE;
        }
    }

    if (!configwrote && value != NULL)
    {
        fprintf(fout, "\n[%s]\n", path);
        fprintf(fout, "%s=%s\n", ptr, value);
    }

    if (f != NULL)
    {
        while (fgets(buf, sizeof(buf), f) != NULL)
            fprintf(fout, "%s", buf);

        fclose(f);
    }

    fclose(fout);
    g_free(path);

    g_string_sprintf(str, "%s/."PACKAGE".conf", g_get_home_dir());
    remove(str->str);

    if (rename(tempname, str->str) != 0/* && !copyfile(tempname, str->str)*/)
    {
        g_warning("gnome_config_write() : can't copy %s to %s", tempname, str->str);
        remove(tempname);
    }

    g_string_free(str, TRUE);
    g_free(origpath);
}

void gnome_config_set_string(gchar *path, gchar *value)
{
    g_return_if_fail(path != NULL);
    g_return_if_fail(value != NULL);

    gnome_config_write(path, value, FALSE);
}

void gnome_config_set_int(gchar *path, gint value)
{
    GString *str;

    g_return_if_fail(path != NULL);

    str = g_string_new(NULL);
    g_string_sprintf(str, "%d", value);
    gnome_config_set_string(path, str->str);
    g_string_free(str, TRUE);
}

void gnome_config_set_bool(gchar *path, gboolean value)
{
    g_return_if_fail(path != NULL);

    gnome_config_set_string(path, value ? "TRUE" : "FALSE");
}

void gnome_config_clean_key(gchar *path)
{
    g_return_if_fail(path != NULL);

    gnome_config_write(path, NULL, FALSE);
}

void gnome_config_clean_section(gchar *path)
{
    g_return_if_fail(path != NULL);

    gnome_config_write(path, NULL, TRUE);
}

void gnome_config_sync(void)
{
}

/* property box dialog */
static void sig_property_apply(GtkWidget *widget, GnomePropertyBox *box)
{
    g_return_if_fail(box != NULL);

    gtk_widget_set_sensitive(box->ok_button, FALSE);
    gtk_widget_set_sensitive(box->apply_button, FALSE);

    box->applyfunc(box->applyfuncdata);
}

static void sig_property_ok(GtkWidget *widget, GnomePropertyBox *box)
{
    g_return_if_fail(box != NULL);

    sig_property_apply(widget, box);
    gtk_widget_destroy(GTK_WIDGET(box->dialog));
}

GtkWidget *gnome_property_box_new(void)
{
    GnomePropertyBox *box;

    box = g_new(GnomePropertyBox, 1);

    box->dialog = GTK_DIALOG(gtk_dialog_new());
    gtk_signal_connect_object(GTK_OBJECT(box->dialog), "destroy",
                              GTK_SIGNAL_FUNC(g_free), (GtkObject *) box);

    box->notebook = gtk_notebook_new();
    gtk_box_pack_start(GTK_BOX(box->dialog->vbox), box->notebook, TRUE, TRUE, 0);

    box->ok_button = gtk_button_new_with_label(_("Ok"));
    gtk_signal_connect(GTK_OBJECT(box->ok_button), "clicked",
                       GTK_SIGNAL_FUNC(sig_property_ok), box);
    gtk_box_pack_start(GTK_BOX(box->dialog->action_area), box->ok_button, TRUE, TRUE, 10);

    box->apply_button = gtk_button_new_with_label(_("Apply"));
    gtk_signal_connect(GTK_OBJECT(box->apply_button), "clicked",
                       GTK_SIGNAL_FUNC(sig_property_apply), box);
    gtk_box_pack_start(GTK_BOX(box->dialog->action_area), box->apply_button, TRUE, TRUE, 10);

    box->close_button = gtk_button_new_with_label(_("Cancel"));
    gtk_signal_connect_object(GTK_OBJECT(box->close_button), "clicked",
                              GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(box->dialog));
    gtk_box_pack_start(GTK_BOX(box->dialog->action_area), box->close_button, TRUE, TRUE, 10);

    gtk_widget_set_sensitive(box->apply_button, FALSE);
    gtk_widget_set_sensitive(box->ok_button, FALSE);

    gtk_object_set_data(GTK_OBJECT(box->dialog), "propertybox", box);
    return GTK_WIDGET(box->dialog);
}

void gnome_property_box_set_apply_func(GtkWidget *box, GtkFunction func, gpointer data)
{
    GnomePropertyBox *rec;

    g_return_if_fail(box != NULL);

    rec = gtk_object_get_data(GTK_OBJECT(box), "propertybox");
    rec->applyfunc = func;
    rec->applyfuncdata = data;
}

void gnome_property_box_changed(GtkWidget *box)
{
    GnomePropertyBox *rec;

    g_return_if_fail(box != NULL);

    rec = gtk_object_get_data(GTK_OBJECT(box), "propertybox");
    gtk_widget_set_sensitive(rec->ok_button, TRUE);
    gtk_widget_set_sensitive(rec->apply_button, TRUE);
}

gint gnome_property_box_append_page(GtkWidget *box, GtkWidget *child, GtkWidget *tab_label)
{
    GnomePropertyBox *rec;

    g_return_val_if_fail(box != NULL, 0);

    rec = gtk_object_get_data(GTK_OBJECT(box), "propertybox");
    gtk_notebook_append_page(GTK_NOTEBOOK(rec->notebook), child, tab_label);
    return g_list_length(GTK_NOTEBOOK(rec->notebook)->children) - 1;
}

void popupfunc(GtkWidget *widget, gpointer data)
{
    gpointer data2;
    void (*func) (GtkWidget *, gpointer);

    func = gtk_object_get_data(GTK_OBJECT(widget), "menu_activate_func");
    if (func == NULL) return;

    data2 = gtk_object_get_data(GTK_OBJECT(widget->parent), "menu_activate_data");
    func(widget, data2 != NULL ? data2 : data);
}

void gnome_app_fill_menu_with_data(GtkMenuShell *menu, GnomeUIInfo *uiinfo, GtkAccelGroup *accel_group, gboolean insert_shortcuts, gint pos, gpointer data)
{
    gchar *str, *ptr;
    gint n;

    g_return_if_fail(menu != NULL);
    g_return_if_fail(uiinfo != NULL);

    for (n = 0; uiinfo[n].type != GNOME_APP_UI_ENDOFINFO; n++)
    {
        if (uiinfo[n].label == NULL)
            str = NULL;
        else
        {
            str = g_strdup(uiinfo[n].label);
            ptr = strchr(str, '_');
            if (ptr != NULL)
                memmove(ptr, ptr+1, strlen(ptr));
        }

        switch (uiinfo[n].type)
        {
            case GNOME_APP_UI_SEPARATOR:
                uiinfo[n].widget = gtk_menu_item_new();
                break;
            case GNOME_APP_UI_TOGGLEITEM:
                uiinfo[n].widget = gtk_check_menu_item_new_with_label(dgettext(PACKAGE, str));
                gtk_check_menu_item_set_show_toggle(GTK_CHECK_MENU_ITEM(uiinfo[n].widget), TRUE);
                break;
            case GNOME_APP_UI_ITEM:
            case GNOME_APP_UI_SUBTREE:
                uiinfo[n].widget = gtk_menu_item_new_with_label(dgettext(PACKAGE, str));
                break;
        }
        gtk_menu_shell_insert(menu, uiinfo[n].widget, pos++);
        if (str != NULL) g_free(str);

        if (uiinfo[n].type == GNOME_APP_UI_SUBTREE)
        {
            GtkWidget *submenu;

            submenu = gtk_menu_new();
            gtk_menu_item_set_submenu(GTK_MENU_ITEM(uiinfo[n].widget), submenu);
            gnome_app_fill_menu_with_data(GTK_MENU_SHELL(submenu), (GnomeUIInfo *) uiinfo[n].moreinfo,
                                          NULL, TRUE, pos, data);
        }
        else
        {
            gtk_object_set_data(GTK_OBJECT(uiinfo[n].widget), "menu_activate_func", uiinfo[n].moreinfo);
            gtk_signal_connect(GTK_OBJECT(uiinfo[n].widget), "activate",
                               GTK_SIGNAL_FUNC(popupfunc), data != NULL ? data : uiinfo[n].user_data);
        }
    }

    gtk_widget_show_all(GTK_WIDGET(menu));
}

void gnome_app_create_menus_with_data(GtkWidget *window, GnomeUIInfo *uiinfo, gpointer data)
{
    GtkWidget *menubar;

    g_return_if_fail(window != NULL);
    g_return_if_fail(uiinfo != NULL);

    menubar = gtk_menu_bar_new();
    gtk_object_set_data(GTK_OBJECT(window), "menubar", menubar);
    gtk_widget_show(menubar);
    gtk_box_pack_start(GTK_BOX(gtk_container_children(GTK_CONTAINER(window))->data),
                       menubar, FALSE, FALSE, 0);

    gnome_app_fill_menu_with_data(GTK_MENU_SHELL(menubar), uiinfo, NULL, TRUE, 0, data);
}

GtkWidget *gnome_popup_menu_new(GnomeUIInfo *uiinfo)
{
    GtkWidget *menu;

    menu = gtk_menu_new();
    gtk_widget_show(menu);

    gtk_object_set_data(GTK_OBJECT(menu), "parent", menu);
    gnome_app_fill_menu_with_data(GTK_MENU_SHELL(menu), uiinfo, NULL, TRUE, 0, NULL);
    return menu;
}

static gint popup_button_pressed(GtkWidget *widget, GdkEventButton *event, gpointer data)
{
    gpointer user_data;

    if (event->button != 3)
        return FALSE;

    user_data = gtk_object_get_data(GTK_OBJECT(widget), "menu_activate_data");
    gtk_signal_emit_stop_by_name (GTK_OBJECT (widget), "button_press_event");

    gnome_popup_menu_do_popup(data, NULL, NULL, event, user_data);
    return TRUE;
}

static void popup_attach_widget_destroyed(GtkWidget *widget, gpointer data)
{
    gtk_object_unref(GTK_OBJECT(data));
}

void gnome_popup_menu_attach(GtkWidget *menu, GtkWidget *widget, gpointer data)
{
    gtk_object_ref(GTK_OBJECT(menu));
    gtk_object_sink(GTK_OBJECT(menu));

    gtk_object_set_data(GTK_OBJECT(menu), "menu_activate_data", data);
    gtk_widget_add_events(widget, GDK_BUTTON_PRESS_MASK);

    gtk_signal_connect(GTK_OBJECT(widget), "button_press_event",
                       (GtkSignalFunc) popup_button_pressed, menu);

    gtk_signal_connect(GTK_OBJECT(widget), "destroy",
                       (GtkSignalFunc) popup_attach_widget_destroyed, menu);
}

void gnome_popup_menu_do_popup(GtkWidget *menu, GtkMenuPositionFunc posfunc, gpointer posdata, GdkEventButton *event, gpointer data)
{
    if (data != NULL)
	gtk_object_set_data(GTK_OBJECT(menu), "menu_activate_data", data);

    gtk_menu_popup(GTK_MENU(menu), NULL, NULL, posfunc, posdata,
                   event == NULL ? 0 : event->button, event == NULL ? GDK_CURRENT_TIME : event->time);
}

GtkWidget *gnome_app_new(gchar *id, gchar *title)
{
    GtkWidget *window, *vbox;

    g_return_val_if_fail(title != NULL, NULL);

    window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(window), title);

    vbox = gtk_vbox_new(FALSE, 0);
    gtk_widget_show(vbox);
    gtk_container_add(GTK_CONTAINER(window), vbox);

    return window;
}

void gnome_app_set_contents(GtkWidget *window, GtkWidget *child)
{
    g_return_if_fail(window != NULL);
    g_return_if_fail(child != NULL);

    gtk_box_pack_end(GTK_BOX(gtk_container_children(GTK_CONTAINER(window))->data),
                     child, TRUE, TRUE, 0);
}
