// File containing helper functions that do parsing

#include "global.h"

static int ActivityCount = 13;

char *DX_ActivityList[][2] = {
    { "Unknown", "??" },
    { "Sleeping", "ZZ" },
    { "Paused", "PS" },
    { "Complete", "CP" },
    { "Despooling file...", "DS" },
    { "Preparing for data transfer...", "PR" },
    { "Transferring file", "TR" },
    { "Getting file size...", "SZ" },
    { "Looking up host", "LK" },
    { "Connecting to host...", "CN" },
    { "Connected; waiting for reply...", "RP" },
    { "Log in as", "LG" },
    { "Sending password...", "PW" }
};

int DX_ParseFileLogLine(const char *log_line, char *date, int date_max,
                        char *time, int time_max, char *protocol,
                        char *server, int server_max, char *path,
                        int path_max, char *local_path, int local_path_max,
                        int *current_size, int *total_size, int *total_time,
                        int *rate_overall, char *reason, int reason_max,
                        char *file_id)
{
    char buffer[1024], *field_ptr;

    memset(buffer, 0, sizeof(buffer));
    memset(date, 0, date_max);
    memset(time, 0, time_max);
    strcpy(buffer, log_line);
    
    field_ptr = DX_GetNextField(buffer);
    if (*field_ptr)
    {
        if (*field_ptr == '"')
            field_ptr++;
        if (date_max > 0)
        {
            strncpy(date, field_ptr, date_max - 1);
            if (date[strlen(date) - 1] == '"')
                date[strlen(date) - 1] = '\0';
        }
    }
    field_ptr += strlen(field_ptr) + 1;
    
    field_ptr = DX_GetNextField(field_ptr);
    if (*field_ptr)
    {
        if (*field_ptr == '"')
            field_ptr++;
        if (time_max > 0)
        {
            strncpy(time, field_ptr, time_max - 1);
            if (time[strlen(time) - 1] == '"')
                time[strlen(time) - 1] = '\0';
        }
    }
    field_ptr += strlen(field_ptr) + 1;

    dl_error(E_TRACE, "Parsed log line: date %s, time %s", date, time);
    return DX_ParseFileStatus(field_ptr, protocol, server, server_max,
                              path, path_max, local_path, local_path_max,
                              current_size, total_size, total_time,
                              rate_overall, reason, reason_max, file_id);
}

int DX_ParseFileStatus(const char *status_line, char *protocol,
                       char *server, int server_max, char *path,
                       int path_max, char *local_path, int local_path_max,
                       int *current_size, int *total_size, int *total_time,
                       int *rate_overall, char *activity, int activity_max,
                       char *file_id)
{
    char buffer[1024], url[256], *field_ptr;

    // The format of a file status line is:
    // [proto]://[server]/[path] [local path] [current size] [total size]
    // [total time] [rx rate] [activity]
    // All quantities are measured in bytes, and the rx rate is per-session.
    
    memset(buffer, 0, sizeof(buffer));
    memset(server, 0, server_max);
    memset(path, 0, path_max);
    memset(local_path, 0, local_path_max);
    memset(activity, 0, activity_max);
    memset(file_id, 0, 4);
    *current_size = *total_size = *total_time = *rate_overall = -1;
    strcpy(buffer, status_line);
    field_ptr = DX_GetNextField(buffer);
    if (*field_ptr)
    {
        // because the GetNextField call does not remove
        // quotes, we have to do it ourselves
        if (*field_ptr == '"')
            field_ptr++;
        strcpy(url, field_ptr);
        if (url[strlen(url) - 1] == '"')
            url[strlen(url) - 1] = '\0';
    }
    DX_StripControlChars(url);
    DX_ParseUrl(url, protocol, server, server_max - 1, path, path_max - 1);
    field_ptr += strlen(field_ptr) + 1;
    
    field_ptr = DX_GetNextField(field_ptr);
    if (*field_ptr)
    {
        if (*field_ptr == '"')
            field_ptr++;
        if (local_path_max > 0)
        {
            strncpy(local_path, field_ptr, local_path_max - 1);
            if (local_path[strlen(local_path) - 1] == '"')
                local_path[strlen(local_path) - 1] = '\0';
        }
    }
    field_ptr += strlen(field_ptr) + 1;
    
    field_ptr = DX_GetNextField(field_ptr);
    if (*field_ptr)
    {
        if (!strcmp(field_ptr, "?"))
            *current_size = -1;
        else
            *current_size = atoi(field_ptr);
    }
    field_ptr += strlen(field_ptr) + 1;
    
    field_ptr = DX_GetNextField(field_ptr);
    if (*field_ptr)
    {
        if (!strcmp(field_ptr, "?"))
            *total_size = -1;
        else
            *total_size = atoi(field_ptr);
    }
    field_ptr += strlen(field_ptr) + 1;
    
    field_ptr = DX_GetNextField(field_ptr);
    if (*field_ptr)
    {
        if (!strcmp(field_ptr, "?"))
            *total_time = -1;
        else
            *total_time = atoi(field_ptr);
    }
    field_ptr += strlen(field_ptr) + 1;
    
    field_ptr = DX_GetNextField(field_ptr);
    if (*field_ptr)
    {
        if (!strcmp(field_ptr, "?"))
            *rate_overall = -1;
        else
            *rate_overall = atoi(field_ptr);
    }
    field_ptr += strlen(field_ptr) + 1;
    
    field_ptr = DX_GetNextField(field_ptr);
    if (*field_ptr)
    {
        if (*field_ptr == '"')
            field_ptr++;
        if (activity_max > 0)
        {
            strncpy(activity, field_ptr, activity_max - 1);
            if (activity[strlen(activity) - 1] == '"')
                activity[strlen(activity) - 1] = '\0';
        }
    }
    //Added by Manuel Clos. Was missing and not parsing correctly ;)
    field_ptr += strlen(field_ptr) + 1;

    field_ptr = DX_GetNextField(field_ptr);
    if (*field_ptr)
        strncpy(file_id, field_ptr, 3);
    
    dl_error(E_TRACE, "Parsed status line: protocol %s, server %s, path %s, "
             "local path %s, current size %d, total size %d, total time %d, "
             "overall rx rate %d, activity %s", protocol, server, path,
             local_path, *current_size, *total_size, *total_time,
             *rate_overall, activity);
    
    return DX_LIB_OK;
}

// Parses a URL... should be fault-tolerant
int DX_ParseUrl(const char *url, char *protocol, char *server,
                 int server_max, char *path, int path_max)
{
    char buffer[1024], buffer2[1024], *buf_ptr;

    if (!url)
        return DX_LIB_ERROR;
    
    // The format of a URL is [protocol://]server[path]
    // Items in brackets are optional.
    memset(buffer, 0, sizeof(buffer));
    memset(buffer2, 0, sizeof(buffer2));
    strcpy(buffer, url);
    if (server)
        memset(server, 0, server_max);
    if (path)
        memset(path, 0, path_max);
    buf_ptr = strstr(buffer, "://");
    if (buf_ptr)
    {
        *buf_ptr = '\0';
        buf_ptr += 3;
        strcpy(buffer2, buf_ptr);
        strcpy(protocol, buffer);
        strcpy(buffer, buffer2);
    }
    else if (!strncmp(buffer, "file:/", 6))
    {
        // allow for Netscape-ish case
        strcpy(protocol, "file");
        strcpy(buffer2, buffer + 5);
        strcpy(buffer, buffer2);
        buf_ptr = buffer;
    }
    else
    {
        strcpy(protocol, "ftp");
        buf_ptr = buffer;
    }
    
    // NB: you *can't* do "buf_ptr = strchr(buf_ptr, x)", at least in glibc2.1
    buf_ptr = strchr(buffer, '/');
    // check to see if there's a path in the URL
    if (buf_ptr)
    {
        dl_error(E_TRACE, "buffer: %d, buf_ptr: %d", buffer, buf_ptr);
        *buf_ptr++ = '\0';
        dl_error(E_TRACE, "buffer: %s", buffer);
        if (server_max > 0)
            strncpy(server, buffer, server_max - 1);
        strcpy(path, "/");
        if (path_max > 1)
            strncat(path, buf_ptr, path_max - 2);
    }
    else
    {
        if (server_max > 0)
            strncpy(server, buffer, server_max - 1);
        if (path_max > 0)
            strncpy(path, "/", path_max - 1);
    }
    dl_error(E_TRACE, "Protocol: %s, Server: %s, Path: %s",
             protocol, server, path);
    return DX_LIB_OK;
}

// parses a file count string
int DX_ParseFileCount(const char *count_line, int *transfer_size,
                      int *total_size, int *transfer_count, int *total_count)
{
    char buffer[256], *field_ptr;

    // format: current size | total size | current count | total_count
    
    memset(buffer, 0, sizeof(buffer));
    *transfer_size = *total_size = *transfer_count = *total_count = 0;
    strncpy(buffer, count_line, sizeof(buffer) - 1);
    
    field_ptr = DX_GetNextField(buffer);
    if (*field_ptr)
        *transfer_size = atoi(field_ptr);
    field_ptr += strlen(field_ptr) + 1;
    
    field_ptr = DX_GetNextField(field_ptr);
    if (*field_ptr)
        *total_size = atoi(field_ptr);
    field_ptr += strlen(field_ptr) + 1;
    
    field_ptr = DX_GetNextField(field_ptr);
    if (*field_ptr)
        *transfer_count = atoi(field_ptr);
    field_ptr += strlen(field_ptr) + 1;
    
    field_ptr = DX_GetNextField(field_ptr);
    if (*field_ptr)
        *total_count = atoi(field_ptr);
    
    dl_error(E_TRACE, "Parsed count line: current size %d, total size %d, "
             "current count %d, total count %d", *transfer_size, *total_size,
             *transfer_count, *total_count);
    
    return DX_LIB_OK;
}

// Creates an activity code from an activity
char *DX_CodeFromActivity(char *activity)
{
    int i;

    for (i = 0; i < ActivityCount; i++)
    {
        if (!strcasecmp(DX_ActivityList[i][0], activity))
            return DX_ActivityList[i][1];
    }
    return DX_ActivityList[0][1];
}

// Creates an activity from an activity code
char *DX_ActivityFromCode(char *code)
{
    int i;

    for (i = 0; i < ActivityCount; i++)
    {
        if (!strcasecmp(DX_ActivityList[i][1], code))
            return DX_ActivityList[i][0];
    }
    return DX_ActivityList[0][0];
}

// parses a "quick file status" string
// format: ID | current size | time | rx rate | activity
int DX_ParseFastFileStatus(const char *status_line, char *file_id,
                            int *current_size, int *total_size,
                            int *total_time, int *rate_overall,
                            char *activity)
{
    char buffer[256], *field_ptr;

    memset(buffer, 0, sizeof(buffer));
    *current_size = *total_size = *total_time = *rate_overall = 0;
    strncpy(buffer, status_line, sizeof(buffer) - 1);
    
    field_ptr = DX_GetNextField(buffer);
    if (*field_ptr)
        strcpy(file_id, field_ptr);
    field_ptr += strlen(field_ptr) + 1;

    field_ptr = DX_GetNextField(field_ptr);
    if (*field_ptr)
        *current_size = atoi(field_ptr);
    field_ptr += strlen(field_ptr) + 1;
    
    field_ptr = DX_GetNextField(field_ptr);
    if (*field_ptr)
        *total_size = atoi(field_ptr);
    field_ptr += strlen(field_ptr) + 1;
    
    field_ptr = DX_GetNextField(field_ptr);
    if (*field_ptr)
        *total_time = atoi(field_ptr);
    field_ptr += strlen(field_ptr) + 1;
    
    field_ptr = DX_GetNextField(field_ptr);
    if (*field_ptr)
        *rate_overall = atoi(field_ptr);
    field_ptr += strlen(field_ptr) + 1;
    
    field_ptr = DX_GetNextField(field_ptr);
    if (*field_ptr)
        strcpy(activity, field_ptr);
    
    dl_error(E_TRACE, "Parsed fast status line: current size %d, total size "
             "%d, total time %d, rate overall %d, activity %s",
             *current_size, *total_size, *total_time, *rate_overall, activity);
    
    return DX_LIB_OK;
}

// parses a "redirection event" line
int DX_ParseRedirectionEvent(const char *redirect_line, char *old_url,
                             int old_max, char *new_url, int new_max)
{
    char buffer[1024], *field_ptr;

    memset(buffer, 0, sizeof(buffer));
    strcpy(buffer, redirect_line);
    
    field_ptr = DX_GetNextField(buffer);
    if (*field_ptr)
    {
        if (*field_ptr == '"')
            field_ptr++;
        if (old_max > 0)
        {
            strncpy(old_url, field_ptr, old_max - 1);
            if (old_url[strlen(old_url) - 1] == '"')
                old_url[strlen(old_url) - 1] = '\0';
        }
    }
    field_ptr += strlen(field_ptr) + 1;
    
    field_ptr = DX_GetNextField(buffer);
    if (*field_ptr)
    {
        if (*field_ptr == '"')
            field_ptr++;
        if (new_max > 0)
        {
            strncpy(new_url, field_ptr, new_max - 1);
            if (new_url[strlen(new_url) - 1] == '"')
                new_url[strlen(new_url) - 1] = '\0';
        }
    }
    dl_error(E_TRACE, "Parsed redirection line: old %s, new %s", old_url,
             new_url);
    
    return DX_LIB_OK;
}

// Strips '%20' etc. characters from a url
char *DX_StripControlChars(char *url)
{
    char *buffer, *src, *dst, hex[3];
    int char_code, size;

    dl_error(E_TRACE, "Stripping \"%s\"...", url);
    if (url == NULL)
        return NULL;
    size = strlen(url) + 1;
    buffer = malloc(size);
    memset(buffer, 0, size);
    src = url;
    dst = buffer;
    while (*src)
    {
        if ((*src == '%') && (*(src+1)) && (*(src+2)))
        {
            hex[0] = *(src + 1);
            hex[1] = *(src + 2);
            hex[2] = '\0';
            if (sscanf(hex, "%x", &char_code) == 1)
                *dst++ = char_code;
            src += 3;
        }
        else
        {
            *dst++ = *src++;
        }
    }
    memset(url, 0, size);
    strcpy(url, buffer);
    dl_error(E_TRACE, "Stripped to \"%s\"", url);
    free(buffer);
    return url;
}
