
/*********************************************************************
 *                
 * Filename:      rtcmosram.c
 * Description:   rtcmosram is a kernel module that serves to interface
 *                with the RT/CMOS RAM in IBM ThinkPads
 * Status:        beta
 * Author:        Thomas Hood <jdthood@mail.com>
 * Created:       24 July 1999 
 *
 * Please report bugs to the author ASAP.
 * 
 *     Copyright (c) 1999 J.D. Thomas Hood, All rights reserved
 *     
 *     This program is free software; you can redistribute it and/or 
 *     modify it under the terms of the GNU General Public License as 
 *     published by the Free Software Foundation; either version 2 of 
 *     the License, or (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *     GNU General Public License for more details.
 * 
 *     To receive a copy of the GNU General Public License, please write
 *     to the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 *     Boston, MA 02111-1307 USA
 *     
 ********************************************************************/

#include "thinkpad_mod_defines.h"

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/ioport.h>
#ifdef USE_PROC_FS
#include <linux/proc_fs.h>
#endif
#include <asm/io.h>
#include <asm/uaccess.h>
#include "thinkpad_common.h"
#include "rtcmosram.h"


/****** defines ******/

#define SZ_RTCMOSRAM_MODULE_VERSION "0.8.0"
#define SZ_RTCMOSRAM_MODULE_NAME "rtcmosram"


/****** declarations ******/

#ifdef USE_PROC_FS
#ifndef NEW_PROC
extern struct proc_dir_entry _proc_dir_entryThinkpadDir;
static int rtcmosram_read_proc(
	char *pchBuf,
	char **ppchStart,
	off_t offThe,
	int intLen,
	int intUnused
);
#endif
#endif


/****** variables ******/

static char _szRtcmosramName[] = SZ_RTCMOSRAM_MODULE_NAME;
static char _szRtcmosramVersion[] = SZ_RTCMOSRAM_MODULE_VERSION;
static flag_t _fRtcmosramReady = 0;

#ifdef USE_PROC_FS
#ifndef NEW_PROC
static flag_t _fCreatedProcRtcmosram = 0;
static struct proc_dir_entry _proc_dir_entryRtcmosram = {
	0,                  /* low_ino: the inode--dynamic */
	9, "rtcmosram",     /* len of name, name */
	S_IFREG | S_IRUGO,  /* mode */
	1, 0, 0,            /* nlinks, owner, group */
	0,                  /* size--unused */
	NULL,               /* point to operations--use default */
	&rtcmosram_read_proc
};
#endif
#endif


/****** functions *******/

/*
 * We use _p variants just for safety.
 */
static byte cmos_getb( byte bWhere )
{
	byte bGot;
	unsigned long flags;

	save_flags( flags ); cli();
	outb_p( bWhere, 0x70 );
	bGot = inb_p( 0x71 );
	outb_p( 0x0f, 0x70 ); /* required, according to Tech Ref */
	inb( 0x71 );
	restore_flags( flags );

	return bGot;
}


static void cmos_putb( byte bWhat, byte bWhere )
{
	unsigned long flags;

	save_flags( flags ); cli();
	outb_p( bWhere, 0x70 );
	outb_p( bWhat, 0x71 );
	outb_p( 0x0F, 0x70 ); /* required, according to Tech Ref */
	inb( 0x71 );
	restore_flags( flags );

	return;
}


#ifdef USE_PROC_FS

static int rtcmosram_read_proc(
#ifdef NEW_PROC
	char *pchBuf, 
	char **ppchStart, 
	off_t offThe,
	int intLen, 
	int *eof, 
	void *data
#else
	char *pchBuf,
	char **ppchStart,
	off_t offThe,
	int intLen,
	int intUnused
#endif
) {

#ifdef NEW_PROC
	if ( _fRtcmosramReady ) {
#else
	if ( _fRtcmosramReady && _fCreatedProcRtcmosram ) {
#endif
		return sprintf(
			pchBuf,
			"%s version %s accessing RT CMOS RAM at ioports 0x%x, 0x%x\n",
			_szRtcmosramName, _szRtcmosramVersion,
			0x70, 0x71
		);
	}

	return sprintf(
		pchBuf,
		"%s version %s not ready\n",
		_szRtcmosramName, _szRtcmosramVersion
	);

}

#endif


int rtcmosram_do( 
	unsigned long ulongIoctlArg,
	flag_t fCallerHasWritePerm
) {
	rtcmosram_ioparm_t ioparmMy;
	unsigned long ulRtnCopy;

	if ( !_fRtcmosramReady ) return -ETHINKPAD_EXECUTION;

	ulRtnCopy = copy_from_user(
		(byte *)&ioparmMy,
		(byte *)(rtcmosram_ioparm_t *)ulongIoctlArg,
		sizeof( ioparmMy )
	);
	if ( ulRtnCopy ) return -EFAULT;

	switch ( ioparmMy.in.wFunc ) {

		case RTCMOSRAM_FUNC_GETDATA: {
			rtcmosram_data_t dataThe;
			int i;

			for ( i=0; i<sizeof(dataThe); i++ )
				*(((byte *)&dataThe)+i) = cmos_getb( (byte)i );

			ulRtnCopy =  copy_to_user(
				(byte *)&(((rtcmosram_ioparm_t *)ulongIoctlArg)->data),
				(byte *)&dataThe,
				sizeof(dataThe)
			);
			if ( ulRtnCopy ) return -EFAULT;

			return 0;
		}

		case RTCMOSRAM_FUNC_DAYLIGHTSAVINGTIME_ABLIFY:
			if ( ! fCallerHasWritePerm )
				return -EACCES;

			if ( (flag_t)ioparmMy.in.dwParm1 ) {
				cmos_putb( cmos_getb(0xB) | 1, 0xB );
			} else {
				cmos_putb( cmos_getb(0xB) & ~1, 0xB );
			}
			return 0;

		case RTCMOSRAM_FUNC_DAYLIGHTSAVINGTIME_GET: {
			byte bGot;
			bGot = cmos_getb( 0xB );
			ioparmMy.out.wRtn = 0;
			ioparmMy.out.dwParm1 = (bGot & 1) ? (dword)1 : (dword)0;
			ulRtnCopy = copy_to_user(
				(byte *)(rtcmosram_ioparm_t *)ulongIoctlArg,
				(byte *)&ioparmMy,
				sizeof(ioparmMy)
			);
			if ( ulRtnCopy ) return -EFAULT;
			return 0;
		}

		default:
			printk(
				KERN_ERR
				"%s: function %d not recognized\n",
				_szRtcmosramName, ioparmMy.in.wFunc
			);
			return -EINVAL;
	} /* switch */

	return -ETHINKPAD_PROGRAMMING; /* we should not reach here */
}


int init_module( void )
{
	int intRtn;

	intRtn = check_region( 0x70, 2 );
	if ( intRtn < 0 ) {
#if 0
/*
 * In some configs this region
 * is already claimed by "rtc"
 */
		printk(
			KERN_ERR
			"%s: io ports not available\n",
			_szRtcmosramName
		);
		return -EBUSY;
#endif
	} else {
		request_region( 0x70, 2, _szRtcmosramName );
	}

	_fRtcmosramReady = 1;

#ifdef USE_PROC_FS
#ifdef NEW_PROC
	create_proc_read_entry(
		"thinkpad/rtcmosram",
		S_IFREG | S_IRUGO,
		NULL,
		rtcmosram_read_proc,
		NULL
	);
#else
	/*** Add /proc entry ***/
	_fCreatedProcRtcmosram = 0;
	intRtn = proc_register( &_proc_dir_entryThinkpadDir, &_proc_dir_entryRtcmosram );
	if ( intRtn ) {
		printk( KERN_ERR "%s: error returned by proc_register()\n", _szRtcmosramName );
	} else {
		_fCreatedProcRtcmosram = 1;
	}
#endif
#endif

	return 0;
}


void cleanup_module( void )
{

	if ( !_fRtcmosramReady ) return;

	_fRtcmosramReady = 0;

	release_region( 0x70, 2 );

#ifdef USE_PROC_FS
#ifdef NEW_PROC
 	remove_proc_entry( "thinkpad/rtcmosram", NULL );
#else
	/*** Remove /proc entry ***/
	if ( _fCreatedProcRtcmosram ) {
		int intRtn;
		_fCreatedProcRtcmosram = 0;
		intRtn = proc_unregister( &_proc_dir_entryThinkpadDir, _proc_dir_entryRtcmosram.low_ino );
		if ( intRtn ) printk( KERN_ERR "%s: error returned by proc_unregister()\n", _szRtcmosramName );
	}
#endif
#endif

	return;
}
