/* This file is part of Malaga, a system for Natural Language Analysis.
 * Copyright (C) 1995-1999 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/* description ==============================================================*/

/* Tools for a command line interpreter. */

/* includes =================================================================*/

#include <stdio.h>
#include <stdlib.h>
#include <setjmp.h>
#include <signal.h>
#include "basic.h"
#include "input.h"
#include "files.h"

#undef GLOBAL
#define GLOBAL
#include "commands.h"

/* types ====================================================================*/

/* an alias */
typedef struct ALIAS_T
{
  struct ALIAS_T *next;
  string_t name;
  string_t line; /* command line which this alias stands for */
} alias_t;

/* variables ================================================================*/

LOCAL bool_t user_break_requested = FALSE;
/* flag that indicates that user has send a break signal */

LOCAL command_t **local_commands; 
/* local copy of command table for <help_command> */

LOCAL alias_t *aliases;

/* functions for user interrupts ============================================*/

LOCAL void set_user_break (int signal_number)
/* Signal handler for notification that user wants to interrupt a command. */
{
  user_break_requested = TRUE;
  signal (SIGINT, set_user_break);
}

/*---------------------------------------------------------------------------*/

GLOBAL void check_user_break (void)
/* Abort command execution if user has sent an interrupt signal. */
{
  if (user_break_requested)
    error ("user break");
}

/* functions for reading and executing commands =============================*/

LOCAL command_t *find_command (string_t command_name,
			       command_t *commands[])
/* Find the entry for <command_name> in <commands> and return it.
 * Return NULL if <command_name> can't be found. */
{
  int_t i;

  for (i = 0; commands[i] != NULL; i++) 
    {
      string_t names;

      names = commands[i]->names;
      while (*names != EOS)
	{
	  string_t name = parse_word (&names);
	  bool_t found_command = (strcmp_no_case (command_name, name) == 0);

	  free_mem (&name);
	  if (found_command) /* Execute the command. */
	    return commands[i];
	}
    }
  
  return NULL;
}

/*---------------------------------------------------------------------------*/

LOCAL void execute_command (string_t command_line, 
			    string_t command_type,
			    command_t *commands[],
			    bool_t use_aliases)
/* Execute <command_line> using <commands>.
 * <command_type> may be "command" or "option", used for error messages.
 * If there is no such command, print an error.
 * The function returns FALSE iff a quit-command has been executed. */
{
  string_t command_name, alias_line;
  command_t *command;
  bool_t command_found;

  /* Divide command line into strings <command> and <arguments>. */
  parse_whitespace (&command_line);
  if (*command_line == EOS)
    return;
  command_name = parse_word (&command_line);

  command_found = FALSE;
  user_break_requested = FALSE;
  local_commands = commands;
  command = find_command (command_name, commands);
  if (command != NULL)
  { /* Execute the command. */
    command_found = TRUE;
    command->command (command_line);
  }
  else if (use_aliases)
  {
    alias_t *alias;
    
    for (alias = aliases; alias != NULL; alias = alias->next)
    {
      if (strcmp_no_case (alias->name, command_name) == 0)
      {
	command_found = TRUE;
	alias_line = replace_arguments (alias->line, "a", command_line);
	execute_command (alias_line, command_type, commands, FALSE);
	free_mem (&alias_line);
	break;
      }
    }
  }
  
  if (! command_found)
    error ("unknown %s \"%s\", try \"help\"", command_type, command_name);

  free_mem (&command_name);
}

/*---------------------------------------------------------------------------*/

GLOBAL void execute_command_file (string_t command_file, string_t line_header, 
				  command_t *commands[])
/* Execute commands in <command_file> using <commands>.
 * If <line_header> != NULL, command lines must start with <line_header>. */
{
  FILE *command_stream = open_stream (command_file, "r");

  while (! feof (command_stream)) 
  {
    string_t command_line, command_line_ptr;
    bool_t is_command_line;
    
    command_line = read_line (command_stream);
    command_line_ptr = command_line;
    
    if (*command_line_ptr == EOS)
      is_command_line = FALSE;
    else if (line_header == NULL)
      is_command_line = TRUE;
    else
    { /* Check if current line begins with <line_header>. */
      
      string_t header = parse_word (&command_line_ptr);
      
      is_command_line = FALSE;
      if (strcmp_no_case (header, line_header) == 0)
	    is_command_line = TRUE;
      else if (strcmp_no_case (header, "include:") == 0)
      {
	string_t include_file = parse_word (&command_line_ptr);
	string_t path = absolute_path (include_file, command_file);
	
	parse_end (command_line_ptr);
	execute_command_file (path, line_header, commands);
	free_mem (&path);
	free_mem (&include_file);
      }
      free_mem (&header);
    }
    if (is_command_line)
      execute_command (command_line_ptr, "command", commands, TRUE);

    free_mem (&command_line);
  }
  close_stream (&command_stream, command_file);
}

/*---------------------------------------------------------------------------*/

GLOBAL void command_loop (string_t prompt, command_t *commands[])
/* Read user commands, look for them in <commands> and execute them
 * until a command function returns FALSE. */
{
  string_t command_line;
  jmp_buf local_error_jump_point; /* where to jump after an error */
  jmp_buf *prev_error_jump_point; /* jump point active on entry */ 

  /* Install a new return point in case of an error. */
  prev_error_jump_point = error_jump_point;
  setjmp (local_error_jump_point);
  error_jump_point = &local_error_jump_point;
    
  signal (SIGINT, set_user_break);

  while (! quit_requested)
  { /* Read command line. */
    printf ("%s> ", prompt);
    command_line = read_line (stdin);
    if (feof (stdin))
      break;
    execute_command (command_line, "command", commands, TRUE);
    free_mem (&command_line);
    if (leave_command_loop)
    {
      leave_command_loop = FALSE;
      break;
    }
  }
  
  /* Reestablish old error jump point. */
  error_jump_point = prev_error_jump_point;
}

/* general commands =========================================================*/

LOCAL void print_help_summary (command_t *commands[])
/* Print names of all commands in <commands>. */
{
  int i;

  for (i = 0; commands[i] != NULL; i++)
  {
    int_t j;
    string_t names = commands[i]->names;
    string_t name = parse_word (&names);
    
    if (i > 0 && i % 5 == 0)
      printf ("\n");
    printf ("%s", name);
    for (j = strlen (name); j < 15; j++)
      printf (" ");
    
    free_mem (&name);
  }
  printf ("\n");
}

/*---------------------------------------------------------------------------*/

LOCAL void print_help (string_t command_type, command_t *command)
/* Print help about <command>, which is of <command_type> (either "command"
 * or "option"). */
{ 
  string_t names, name;

  names = command->names;
  
  /* Print full name. */
  name = parse_word (&names);
  printf ("%s \"%s\"", command_type, name);
  free_mem (&name);
  
  /* Print shortcuts. */
  while (*names != EOS)
  {
    name = parse_word (&names);
    printf (", \"%s\"", name);
    free_mem (&name);
  }
  
  /* Print help text. */
  printf (":\n%s", command->help);
}

/*---------------------------------------------------------------------------*/

LOCAL void do_help (string_t arguments)
/* Give help on commands. */
{
  if (*arguments == EOS)
  {
    printf ("commands available "
	    "(use \"help command <command>\" to get help about <command>):\n");
    print_help_summary (local_commands);
    printf ("\noptions available "
	    "(use \"help option <option>\" to get help about <option>):\n");
    print_help_summary (options);
  }
  else
  {
    string_t command_name;
    command_t *command, *option;

    command_name = parse_word (&arguments);
    if (strcmp_no_case (command_name, "command") == 0)
    {
      free_mem (&command_name);
      command_name = parse_word (&arguments);
      parse_end (arguments);
      command = find_command (command_name, local_commands);
      option = NULL;
      if (command == NULL)
	error ("\"%s\" is no command", command_name);
    }
    else if (strcmp_no_case (command_name, "option") == 0)
    {
      free_mem (&command_name);
      command_name = parse_word (&arguments);
      command = NULL;
      option = find_command (command_name, options);
      if (option == NULL)
	error ("\"%s\" is no option", command_name);
    }
    else
    {
      command = find_command (command_name, local_commands);
      option = find_command (command_name, options);
      if (command != NULL && option != NULL)
	error ("\"%s\" is a command as well as an option", command_name);
      else if (command == NULL && option == NULL)
	error ("\"%s\" is neither a command nor an option", command_name);
    }

    parse_end (arguments);

    if (command != NULL)
      print_help ("command", command);
    if (option != NULL)
      print_help ("option", option);

    free_mem (&command);
  }
}

GLOBAL command_t help_command = 
{
  "help h ?", do_help,
  "Usage:\n"
  "  help command <command> -- print help about <command>\n"
  "  help option <option> -- print help about <option>\n"
  "  help <command_or_option> -- "
  "print help about <command_or_option> if unique\n"
  "  help -- get a list of all available commands and options\n"
};

/*---------------------------------------------------------------------------*/

LOCAL void do_quit (string_t arguments)
/* Quit the program. */
{
  parse_end (arguments);
  quit_requested = TRUE;
}

GLOBAL command_t quit_command =
{
  "quit q", do_quit,
  "Leave the program.\n"
  "Usage: quit\n"
};

/*---------------------------------------------------------------------------*/

LOCAL void do_get (string_t arguments)
/* Get setting for <arguments>. */
{
  if (*arguments == EOS)
  {
    int_t i;
    
    for (i = 0; options[i] != NULL; i++) 
      options[i]->command ("");
  }
  else
  {
    string_t option;
    
    option = parse_word (&arguments);
    parse_end (arguments);
    execute_command (option, "option", options, FALSE);
    free_mem (&option);
  }
}

GLOBAL command_t get_command =
{
  "get", do_get,
  "Query program settings.\n"
  "Usage:\n"
  "  get <option> -- print the setting of <option>\n"
  "  get -- print all settings\n"
};

/*---------------------------------------------------------------------------*/

LOCAL void do_set (string_t arguments)
/* Set an option. */
{
  string_t option_arguments, option;

  option_arguments = arguments;
  option = parse_word (&arguments);
    
  if (*arguments == EOS)
    error ("missing arguments to set \"%s\"", option);

  execute_command (option_arguments, "option", options, FALSE);
  free_mem (&option);
}

GLOBAL command_t set_command =
{
  "set", do_set,
  "Change program settings.\n"
  "Usage:\n"
  "  set <option> <argument> -- change value of <option> to <argument>\n"
};

/*---------------------------------------------------------------------------*/

LOCAL command_t *set_commands[] = {&set_command, NULL};
/* the commands that can be called during initialisation */

GLOBAL void execute_set_commands (string_t file_name, string_t prefix)
/* Execute set commands in file <file_name> that are prefixed with <prefix>. */
{
  execute_command_file (file_name, prefix, set_commands);
}

/*---------------------------------------------------------------------------*/

LOCAL void do_alias_option (string_t arguments)
{
  alias_t *alias;
  
  if (*arguments == EOS)
  {
    if (aliases == NULL)
      printf ("alias: none defined\n");
    else 
    {
      for (alias = aliases; alias != NULL; alias = alias->next)
      {
	string_t line = new_string_readable (alias->line, NULL);
	printf ("alias \"%s\": %s\n", alias->name, line);
	free_mem (&line);
      }
    }
  }
  else
  {
    string_t name;
    alias_t **alias_ptr;

    name = parse_word (&arguments);
    
    /* Find the alias <name>. */
    alias_ptr = &aliases;
    while (*alias_ptr != NULL 
	   && strcmp_no_case ((*alias_ptr)->name, name) != 0)
      alias_ptr = &(*alias_ptr)->next;

    if (*arguments == EOS) /* Delete an alias. */
    {
      if (*alias_ptr == NULL)
	error ("alias \"%s\" not defined");

      /* remove the alias from the list and delete it. */
      alias = *alias_ptr;
      *alias_ptr = alias->next;
      free_mem (&alias->name);
      free_mem (&alias->line);
      free_mem (&alias);
    } 
    else /* Define an alias. */
    {
      string_t line;

      if (*alias_ptr == NULL) /* Create alias entry if alias doesn't exist. */
      {
	alias = new_mem (sizeof (alias_t));
	alias->name = new_string (name, NULL);
	*alias_ptr = alias;
      }
      else
	alias = *alias_ptr;

      line = parse_word (&arguments);
      parse_end (arguments);

      free_mem (&alias->line);
      alias->line = line;
    }
    
    free_mem (&name);
  }
}

GLOBAL command_t alias_option =
{
  "alias", do_alias_option,
  "Define command line abbreviations.\n"
  "Usage:\n"
  "  alias <name> <line> -- define abbreviation <name> for <line>\n"
  "  alias <name> -- undefine abbreviation <name>\n"
};

/* end of file ==============================================================*/
