/*

    Emulation of BSD usleep()

    Note that this is not a complete emulation in that
    it destroys any pre-existing setitimer(), but it's
    good enough for programs which don't have interval
    timers running across calls on usleep().  It does
    save and restore an existing SIGALRM handler.

*/

#ifdef USLEEP
#include <stdio.h>
#include <signal.h>
#include <unistd.h>
#include <sys/signal.h>
#include <sys/time.h>

#ifdef Solaris
typedef void (*signalFUNC)(int);
#define signal(a, b)	sigset(a, (signalFUNC) b)
#define signalFUNCreturn (signalFUNC)
#endif

#ifndef signalFUNCreturn
#define signalFUNCreturn
#endif

#ifdef FLUSH
#include <termios.h>
int flush_fd = -1;		      /* File descriptor for auto-flush */
#endif

volatile static int waiting;

static void getalrm(int i)
{
    waiting = 0;
}

void sf_usleep(unsigned t)
{
    static struct itimerval it;
    void (*oldsig)();

    oldsig = signalFUNCreturn signal(SIGALRM, getalrm);
#ifdef FLUSH
#define FLUSHTIME 100000	      /* Flush interval in microseconds */
    if (flush_fd != -1) {
	while (t > FLUSHTIME) {
	    it.it_value.tv_sec = 0;
	    it.it_value.tv_usec = FLUSHTIME;
	    t -= FLUSHTIME;
	    tcflush(flush_fd, TCIFLUSH);
	    waiting = 1;
	    if (setitimer(ITIMER_REAL, &it, NULL))
		return /*error*/;
	    while (waiting) {
		pause();
	    }
	    (void) signal(SIGALRM, getalrm);
	}
    }
#endif

    it.it_value.tv_sec = t / 1000000;
    it.it_value.tv_usec = t % 1000000;
    waiting = 1;    
    if (setitimer(ITIMER_REAL, &it, NULL))
	return /*error*/;
    while (waiting) {
	pause();
    }
    signal(SIGALRM, oldsig);
}
#endif
