
#include "mathuserfunc.h"
#include "lispobject.h"
#include "lispeval.h"
#include "standard.h"

#define InternalEval aEnvironment.iEvaluator->Eval

BranchingUserFunction::BranchingUserFunction(LispPtr& aParameters)
    
{
    iParamList.Set(aParameters.Get());

    LispIterator iter(aParameters);

    while (iter())
    {
        Check(iter()->String() != NULL,KLispErrCreatingUserFunction);
        BranchParameter param(iter()->String());
        iParameters.Append(param);
        iter.GoNext();
    }
}

BranchingUserFunction::~BranchingUserFunction()
{
    LispInt i;
    LispInt nrc=iRules.NrItems();
    for (i=0;i<nrc;i++)
    {
        delete iRules[i];
        iRules[i] = NULL;
    }
}

void BranchingUserFunction::Evaluate(LispPtr& aResult,LispEnvironment& aEnvironment,
                                     LispPtr& aArguments)
{
    LispInt arity = Arity();
    LispInt i;

    //hier
    if (Traced())
    {
        LispPtr tr;
        tr.Set(LispSubList::New(aArguments.Get()));
        TraceShowEnter(aEnvironment,tr);
        tr.Set(NULL);
    }
    
    LispIterator iter(aArguments);
    iter.GoNext();

    // unrollable arguments 
    LispPtr* arguments = new LispPtr[arity];
    LocalArgs args(arguments);

    // Walk over all arguments, evaluating them as necessary
    for (i=0;i<arity;i++)
    {
        Check(iter() != NULL, KLispErrWrongNumberOfArgs);
        if (iParameters[i].iHold)
        {
            arguments[i].Set(iter()->Copy(LispFalse));
        }
        else
        {
            Check(iter.Ptr() != NULL, KLispErrWrongNumberOfArgs);
            InternalEval(aEnvironment, arguments[i], *iter.Ptr());
        }
        iter.GoNext();
    }

    if (Traced())
    {
        LispIterator iter(aArguments);
        iter.GoNext();
        for (i=0;i<arity;i++)
        {
            TraceShowArg(aEnvironment,*iter.Ptr(),
                  arguments[i]);

            iter.GoNext();
        }
    }
    
    // declare a new local stack.
    LispLocalFrame frame(aEnvironment,Fenced());

    // define the local variables.
    for (i=0;i<arity;i++)
    {
        LispStringPtr variable = iParameters[i].iParameter;
        // set the variable to the new value
        aEnvironment.NewLocal(variable,arguments[i].Get());
    }

    // walk the rules database, returning the evaluated result if the
    // predicate is LispTrue.
    LispInt nrRules = iRules.NrItems();
    for (i=0;i<nrRules;i++)
    {
        LISPASSERT(iRules[i] != NULL);
        LispPtr pred;
        InternalEval(aEnvironment, pred, iRules[i]->iPredicate);
        if (IsTrue(aEnvironment, pred))
        {
            InternalEval(aEnvironment, aResult, iRules[i]->iBody);
            goto FINISH;
        }
    }
    
    // No predicate was LispTrue: return a new expression with the evaluated
    // arguments.

    {
        LispPtr full;
        full.Set(aArguments.Get()->Copy(LispFalse));
        full.Get()->Next().Set(arguments[0].Get());
        for (i=0;i<arity-1;i++)
        {
            arguments[i].Get()->Next().Set(arguments[i+1].Get());
        }
        aResult.Set(LispSubList::New(full.Get()));
    }

FINISH:
    if (Traced())
    {
        LispPtr tr;
        tr.Set(LispSubList::New(aArguments.Get()));
        TraceShowLeave(aEnvironment, aResult,tr);
        tr.Set(NULL);
    }
}

void BranchingUserFunction::HoldArgument(LispStringPtr aVariable)
{
    LispInt i;
    LispInt nrc=iParameters.NrItems();
    for (i=0;i<nrc;i++)
    {
        if (iParameters[i].iParameter == aVariable)
            iParameters[i].iHold = LispTrue;
    }
}

LispInt BranchingUserFunction::Arity() const
{
    return iParameters.NrItems();
}

void BranchingUserFunction::DeclareRule(LispInt aPrecedence, LispPtr& aPredicate,
                         LispPtr& aBody)
{
    // New branching rule.
    BranchRule* newRule = new BranchRule(aPrecedence,aPredicate,aBody);
    Check(newRule != NULL,KLispErrCreatingRule);

    // Find place to insert
    LispInt low,high,mid;
    low=0;
    high=iRules.NrItems();

    // Constant time: find out if the precedence is before any of the
    // currently defined rules or past them.
    if (high>0)
    {
        if (iRules[0]->iPrecedence > aPrecedence)
        {
            mid=0;
            goto CONTINUE;
        }
        if (iRules[high-1]->iPrecedence < aPrecedence)
        {
            mid=high;
            goto CONTINUE;
        }
    }

    // Otherwise, O(log n) search algorithm for place to insert
    for(;;)
    {
        if (low>=high)
        {
            mid=low;
            goto CONTINUE;
        }
        mid = (low+high)>>1;

        if (iRules[mid]->iPrecedence > aPrecedence)
        {
            high = mid;
        }
        else if (iRules[mid]->iPrecedence < aPrecedence)
        {
            low = (++mid);
        }
        else
        {
            goto CONTINUE;
        }
    }
    CONTINUE:
    // Insert it
    iRules.Insert(mid,newRule);
}


LispPtr& BranchingUserFunction::ArgList()
{
    return iParamList;
}





