/*
   Copyright (C) 1997-1999  Ulric Eriksson <ulric@edu.stockholm.se>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include <X11/Xlib.h>
#include <X11/keysym.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>

#include <X11/Shell.h>
#include <X11/xpm.h>

#include <X11/Xaw/Box.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Viewport.h>
#include <X11/Xaw/List.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Dialog.h>

#include "../common/common.h"
#include "../common/bitmaps/unknown.xpm"

#include "dialogs.h"
#include "xcommon.h"
#include "TextField.h"
#include "Rudegrid.h"
#include "Frame.h"

static Widget pshell, dialog, dialogIcon, dialogLabel, dialogText,
	dialogCancel, dialogDone;

static Widget alert, listshell;
static long status;

static Widget add_button(Widget pw, char *name, char *label,
        void (*cb)(Widget, XtPointer, XtPointer), XtPointer data)
{
        Widget w = XtVaCreateManagedWidget(name,
                commandWidgetClass, pw,
                (char *)0);
        label_set(w, label);
        XtVaSetValues(w, XtNwidth, 80, (char *)0);
        if (cb) XtAddCallback(w, XtNcallback, cb, data);
        return w;
}

static void
alert_clicked(Widget w, XtPointer client_data, XtPointer call_data)
{
	XtPopdown(alert);
	status = (long)client_data;
}

/* ---
Pops up a box with a text and a bunch of buttons.
*/

int alertbox(Widget pw, char *text, char *buttons[], int nbuttons)
{
	Widget topbox, buttonbox, label, command;
	int i;
	Position x, y;
	Dimension width, height, w1, h1;
	XtAppContext app_context = XtWidgetToApplicationContext(pw);

	alert = XtVaCreatePopupShell("alert",
		transientShellWidgetClass, pw,
		XtNtitle, translate("Alert"),
		(char *)0);
	topbox = XtVaCreateManagedWidget("topbox",
		boxWidgetClass, alert, (char *)0);
	label = XtVaCreateManagedWidget("label",
		labelWidgetClass, topbox,
		(char *)0);
	label_set(label, text);
	buttonbox = XtVaCreateManagedWidget("buttonbox",
		boxWidgetClass, topbox, (char *)0);
	for (i = 0; i < nbuttons; i++) {
		command = add_button(buttonbox, "command", buttons[i],
			alert_clicked, (XtPointer)i);
	}
	status = -1;
	XtRealizeWidget(alert);
	XtUnrealizeWidget(alert);
	XtVaGetValues(pw,
		XtNwidth, &width,
		XtNheight, &height, (char *)0);
	XtVaGetValues(alert,
		XtNwidth, &w1,
		XtNheight, &h1, (char *)0);
	XtTranslateCoords(pw,
		(Position) (width-w1)/2, (Position) (height-h1)/2,
		&x, &y);
	XtVaSetValues(alert,
		XtNx, x,
		XtNy, y, (char *)0);
	XtPopup(alert, XtGrabNonexclusive);
	while (status < 0) {
		XEvent event_return;
		XtAppNextEvent(app_context, &event_return);
		XtDispatchEvent(&event_return);
	}
	XtDestroyWidget(alert);
	return status;
}

/* ---
Same as an alertbox with only an "OK" button.
*/

int errorbox(Widget pw, char *message)
{
	char *buttons[] = {"OK"};

	return alertbox(pw, message, buttons, 1);
}

static void
listsel_clicked(Widget w, XtPointer client_data, XtPointer call_data)
{
	XtPopdown(listshell);
	status = (long)client_data;
}

/* ---
Place the child centered relative to the parent.
*/

void center(Widget parent, Widget child)
{
	Position x, y;
	Dimension width, height, w1, h1;

	XtRealizeWidget(child);
	XtUnrealizeWidget(child);
	XtVaGetValues(parent,
		XtNwidth, &width,
		XtNheight, &height, (char *)0);
	XtVaGetValues(child,
		XtNwidth, &w1,
		XtNheight, &h1, (char *)0);
	XtTranslateCoords(parent,
		(Position)(width-w1)/2, (Position)(height-h1)/2, &x, &y);
	XtVaSetValues(child,
		XtNx, x,
		XtNy, y, (char *)0);
}

/* ---
Pops up a list box. Returns the index of the selected item, or
-1 for none.
*/

int listsel(Widget pw, char *text, char *choices[], int nchoices)
{
	Widget form, buttonframe,
     		buttonbox, label, viewport, listframe, list,
		okbutton, cancelbutton;
	int n;
	XtAppContext app_context = XtWidgetToApplicationContext(pw);

	listshell = XtVaCreatePopupShell("lsel_shell",
		transientShellWidgetClass, pw,
		XtNtitle, translate("List"),
		(char *)0);
	form = XtVaCreateManagedWidget("lsel_form",
		rudegridWidgetClass, listshell, (char *)0);
	buttonframe = XtVaCreateManagedWidget("lsel_buttonframe",
		frameWidgetClass, form, (char *)0);
	buttonbox = XtVaCreateManagedWidget("lsel_buttonbox",
		boxWidgetClass, buttonframe, (char *)0);
	listframe = XtVaCreateManagedWidget("lsel_listframe",
		frameWidgetClass, form, (char *)0);
	label = XtVaCreateManagedWidget("lsel_label",
		labelWidgetClass, listframe,
		(char *)0);
	label_set(label, text);
	XtVaSetValues(listframe,
		XtNtitle, label,
		(char *)0);
	viewport = XtVaCreateManagedWidget("lsel_viewport",
		viewportWidgetClass, listframe, (char *)0);
	list = XtVaCreateManagedWidget("lsel_list",
		listWidgetClass, viewport, (char *)0);
	XawListChange(list, choices, nchoices, 0, True);
	XtVaSetValues(list, XtNwidth, width_get(list)+20, (char *)0);
	okbutton = add_button(buttonbox, "lsel_okbutton",
		"OK", listsel_clicked, (XtPointer)1);
	cancelbutton = add_button(buttonbox, "lsel_cancelbutton",
		"Cancel", listsel_clicked, (XtPointer)0);
	status = -1;
	center(pw, listshell);
	XtPopup(listshell, XtGrabNonexclusive);
	XawListChange(list, choices, nchoices, 0, True);

	while (status < 0) {
		XEvent event_return;
		XtAppNextEvent(app_context, &event_return);
		XtDispatchEvent(&event_return);
	}
	n = XawListShowCurrent(list)->list_index;
	XtDestroyWidget(listshell);
	if (status)	/* clicked OK */
		return n;
	else		/* clicked Cancel */
		return -1;
}

static void DialogDone(Widget w, XtPointer client_data, XtPointer call_data)
{
	XtPopdown(pshell);
	status = DONE;
}

static void DialogAbort(Widget w, XtPointer client_data, XtPointer call_data)
{
	XtPopdown(pshell);
	status = ABORT;
}

static void DialogCancelAction(Widget w, XEvent * event,
		String * params, Cardinal * n)
{
	XtPopdown(pshell);
	status = ABORT;
}

static void DialogDoneAction(Widget w, XEvent * event,
		String * params, Cardinal * n)
{
	XtPopdown(pshell);
	status = DONE;
}

static XtActionsRec actions[] = {
	{"dialog-done", DialogDoneAction},
	{"dialog-cancel", DialogCancelAction},
};

static Atom wm_delete_window;
static Pixmap default_icon = None;

/* ---
Pops up a dialog box with an optional icon, a title, a prompt, a text field
and an OK and a Cancel button.
The text buffer must contain a valid initial value, i.e. it
must at least be cleared before calling this function.
*/

int dialog_input_icon(Widget pw, char *title,
		char *prompt, char *buffr, Pixmap icon)
{
	XtAppContext app_context = XtWidgetToApplicationContext(pw);
	String string;
	static int init_done = 0;

	if (!init_done) {
		XpmAttributes xpm_attr;
		static XpmColorSymbol none_color = { NULL, "None", (Pixel)0 };

		xpm_attr.valuemask = XpmReturnPixels|XpmColorSymbols;
		xpm_attr.colorsymbols = &none_color;
		xpm_attr.numsymbols = 1;
		XtVaGetValues(pw,
			XtNbackground, &none_color.pixel,
			(char *)0);
		XpmCreatePixmapFromData(XtDisplay(pw),
			DefaultRootWindow(XtDisplay(pw)),
			unknown_xpm, &default_icon, NULL, &xpm_attr);

		XtAppAddActions(app_context, actions, XtNumber(actions));
		init_done = 1;
	}

	pshell = XtVaCreatePopupShell("pshell",
		transientShellWidgetClass, pw,
		XtNtitle, translate("Dialog"),
		(char *)0);
	dialog = XtVaCreateManagedWidget("dialog",
		formWidgetClass, pshell,
		(char *)0);
	dialogIcon = XtVaCreateManagedWidget("dialogIcon",
		labelWidgetClass, dialog,
		XtNpixmap, default_icon,
		(char *)0);
	dialogLabel = XtVaCreateManagedWidget("dialogLabel",
		labelWidgetClass, dialog,
		(char *)0);
	dialogText = XtVaCreateManagedWidget("dialogText",
		textfieldWidgetClass, dialog,
		(char *)0);
	wm_delete_window = XInternAtom(XtDisplay(pshell),
				"WM_DELETE_WINDOW", False);
	XtOverrideTranslations(pshell,
		XtParseTranslationTable(
			"<Message>WM_PROTOCOLS:	dialog-cancel()"));

	XtOverrideTranslations(dialogText,
		XtParseTranslationTable(
			"<Key>Return:	dialog-done()\n"
			"<Key>Escape:	dialog-cancel()"));

	dialogDone = add_button(dialog, "dialogDone", "OK",
		DialogDone, NULL);
	dialogCancel = add_button(dialog, "dialogCancel", "Cancel",
		DialogAbort, NULL);

	status = WAITING;

	if (icon == None) icon = default_icon;

	XtVaSetValues(dialogIcon,
		XtNbitmap, icon,
		(char *)0);
	XtVaSetValues(pshell,
		XtNtitle, title,
		(char *)0);
	label_set(dialogLabel, prompt);
	XtVaSetValues(dialogText,
		XtNstring, buffr,
		XtNinsertPosition, strlen(buffr),
		(char *)0);
	center(pw, pshell);
	XtPopup(pshell, XtGrabNonexclusive);
	XSetWMProtocols(XtDisplay(pshell), XtWindow(pshell),
			&wm_delete_window, 1);
	if (input_warp_pointer)
		XWarpPointer(XtDisplay(pshell), None, XtWindow(pshell),
				0, 0, 0, 0, 0, 0);
	XtSetKeyboardFocus(pshell, dialogText);

	while (status == WAITING) {
		XEvent event_return;

		XtAppNextEvent(app_context, &event_return);
		XtDispatchEvent(&event_return);
	}

	XtVaGetValues(dialogText,
		XtNstring, &string,
		(char *)0);
	strcpy(buffr, string);
	XtDestroyWidget(pshell);
	return status;
}

/* ---
Pops up a dialog box with a prompt, a text field and an OK and a Cancel
button. The text buffer must contain a valid initial value, i.e. it
must at least be cleared before calling this function.
*/

int dialog_input(Widget pw, char *prompt, char *buffr, int (*comp)(char *))
{
	return dialog_input_icon(pw, "Dialog", prompt, buffr, None);
}

