/*
    Copyright (C) 1996-1998  Ulric Eriksson <ulric@edu.stockholm.se>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston,
    MA 02111-1307, USA.
*/

/* ---
cmalloc.c

A set of functions to allocate and deallocate memory with a little
record keeping and error checking.
--- */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "cmalloc.h"

/* ---
Allocate memory and check if anything was actually allocated.
If not, print an error message or do something else
*/

static void std_alloc_fail(void)
{
	fprintf(stderr, "Out of memory\n");
	exit(EXIT_FAILURE);
}

static void (*alloc_fail)(void) = std_alloc_fail;

/* Also does a few checks that the library functions should already do */

static int paranoia = 1;

typedef struct malloc_node {
	void *p;			/* allocated memory */
	struct malloc_node *next;	/* next in chain */
} malloc_node;

malloc_node *nodes = NULL;

static void remove_node(void *p)
{
	malloc_node *m, *n;

	if (!p) return;		/* deallocating NULL is fine */

	n = nodes;
	if (!n) {
		if (paranoia == 1) return;
		fprintf(stderr,
			"Deallocating %p when nothing allocated\n", p);
		if (paranoia == 2) return;
		alloc_fail();
	}

	if (n->p == p) {
		nodes = n->next;
		free(n);
		return;
	}

	for (m = n->next; m; m = m->next) {
		if (m->p == p) {
			n->next = m->next;
			free(m);
			return;
		}
		n = m;
	}

	if (paranoia == 1) return;
	fprintf(stderr, "Deallocating %p which was not allocated\n", p);
	if (paranoia == 2) return;
	alloc_fail();
}

static void insert_node(void *p)
{
	malloc_node *n;

	if (!p) return;		/* we don't need to remember NULL */

	n = malloc(sizeof(malloc_node));
	if (n == NULL) alloc_fail();
	n->p = p;
	n->next = nodes;
	nodes = n;
}

/* ---
Set up a new failure handler and paranoia level.

	0 = don't check at all
	1 = try to fix errors without telling
	2 = warn about errors and try to fix them
	3 = warn about errors and fail
*/

void cmalloc_init(void (*new_fail)(void), int level)
{
	if (new_fail) alloc_fail = new_fail;
	else alloc_fail = std_alloc_fail;
	paranoia = level;
}

/* ---
Check for memory leaks.
*/

void cmalloc_exit(void)
{
	malloc_node *n;

	if (paranoia == 0) return;

	for (n = nodes; n; n = n->next) {
		if (paranoia >= 2)
			fprintf(stderr, "Didn't deallocate %p\n", n->p);
		if (paranoia == 3) alloc_fail();
		cfree(n->p);
	}
}

/* ---
Allocate memory with a bit of record keeping and error checking.
*/

void *cmalloc(size_t size)
{
	void *p;

	p = malloc(size);
	if (p == NULL) alloc_fail();
	if (paranoia) insert_node(p);
	return p;
}

/* ---
Reallocate memory.
*/

void *crealloc(void *ptr, size_t size)
{
	void *p;

	if (paranoia) remove_node(ptr);
	p = realloc(ptr, size);
	if (p == NULL) alloc_fail();
	if (paranoia) insert_node(p);
	return p;
}

/* ---
Allocate memory and initialize it to all zero.
*/

void *ccalloc(size_t nmemb, size_t size)
{
	void *p;

	p = calloc(nmemb, size);
	if (p == NULL) alloc_fail();
	if (paranoia) insert_node(p);
	return p;
}

/* ---
Allocate a duplicate of a string.
*/

char *cstrdup(const char *s)
{
	char *p;

	if (s) p = malloc(strlen(s)+1);
	else p = NULL;
	if (p == NULL) alloc_fail();
	else strcpy(p, s);
	if (paranoia) insert_node(p);
	return p;
}

/* ---
Free a block of memory.
*/

void cfree(void *ptr)
{
	if (paranoia) remove_node(ptr);
	free(ptr);
}

