/* Copyright (C) 1999 Aaron Lehmann
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */


#include <stdio.h>
#include <stdlib.h>
#include <glib.h>
#include <string.h>
#include <ctype.h>
#include <math.h>
#include <limits.h>

#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <gdk/gdkkeysyms.h>

#include "gtksheet.h"
#include "gtkitementry.h"
#include "gtkplot.h"

#include "app.h"
#include "prefs.h"
#include "data.h"
#include "dialogs.h"
#include "version.h"
#include "curvefit.h"

#define DEFAULT_PRECISION 3
#define DEFAULT_SPACE 8
#define SHEET_LEN 1000

GtkWidget *window;
GtkWidget *main_vbox;
GtkWidget *vbox1;
GtkWidget *notebook;
GtkWidget **sheets;
GtkWidget **scrolled_windows;
GtkWidget *show_hide_box;
GtkWidget *status_box;
GtkWidget *location;
GtkWidget *entry;
GtkWidget *bordercombo;
GtkWidget *notebook;

short strisnum (char *str);

extern GtkWidget *layout, *active_plot;

gint nbrdatasets_ever=0;

extern gint nbrdatasets, nbrrealdatasets;
extern thing **datasets;

extern void set_error (void);

thing *currdata=0;

static gdouble function_reg (GtkPlot *plot, GtkPlotData *dataset, gdouble, gboolean *);
static void dataset_switch_page (GtkNotebook *notebook,
				 GtkNotebookPage *page,
				 gint page_num,
				 gpointer user_data);


void
clipboard_handler(GtkWidget *widget, GdkEventKey *key)
{
    GtkSheet *sheet;
    
    sheet = GTK_SHEET(widget);
    
    if(key->state & GDK_CONTROL_MASK || key->keyval==GDK_Control_L ||
       key->keyval==GDK_Control_R){
	if((key->keyval=='c' || key->keyval == 'C') && sheet->state != GTK_STATE_NORMAL){
            if(GTK_SHEET_IN_CLIP(sheet)) gtk_sheet_unclip_range(sheet);
            gtk_sheet_clip_range(sheet, sheet->range);
	}
	if(key->keyval=='x' || key->keyval == 'X')
            gtk_sheet_unclip_range(sheet);    
    }
}


void
build_example1(GtkWidget *widget)
{
    GtkSheet *sheet;
    
    gchar name[10];
    gint i;
    
    sheet=GTK_SHEET(widget);
    
    for(i=0; i<=1; i++){
	name[0]='X'+i;
	name[1]='\0';
	gtk_sheet_column_button_add_label(sheet, i, name);
	gtk_sheet_set_column_title(sheet, i, name);
    }
    
    
    
    gtk_signal_connect(GTK_OBJECT(sheet),
		       "key_press_event",
		       (GtkSignalFunc) clipboard_handler, 
		       NULL);
}



void
set_cell (GtkSheet *sheet,
	  gint row, gint column)
{
    int i, j=0;
    gchar *c, *d;
    gdouble lowx=0, highx=0, lowy=0, highy=0;
    extern double rangexmi, rangeymi, rangexma, rangeyma, majorticks_h, majorticks_v, minorticks_h, minorticks_v;
    thing *my_thing=NULL;
    
    for (i=0; i != nbrdatasets; ++i)
    {
	if (datasets[i]->sheet == sheet)
	{
	    /* woohoo found a match jolly good */
	    my_thing = datasets[i];
	}
    }

    if (!my_thing) return;

    /*if (my_thing->x) free (my_thing->x);
    if (my_thing->y) free (my_thing->y);
    if (my_thing->bogus) free (my_thing->bogus);*/

    for (i=0; i != SHEET_LEN; ++i)
    {
         c = gtk_sheet_cell_get_text (sheet, i, 0);
	 d = gtk_sheet_cell_get_text (sheet, i, 1);
	 if (c && d && strisnum (c) && strisnum (d))
	 {
	   ++j;
	   my_thing->x = realloc (my_thing->x, j*sizeof(gdouble));
	   my_thing->y = realloc (my_thing->y, j*sizeof(gdouble));
	   my_thing->bogus = realloc (my_thing->bogus, j*sizeof(gdouble));
	   my_thing->x[j-1] = g_strtod(c, (char **)NULL);
	   my_thing->y[j-1] = g_strtod(d, (char **)NULL);
	   my_thing->bogus[j-1] = 0;
	 }
    }
    
    
    gtk_plot_dataset_set_points(my_thing->real_dataset, my_thing->x, my_thing->y, my_thing->bogus, my_thing->bogus, j);
    
    if (j)
    {
	gtk_plot_show_dataset(my_thing->real_dataset);
    }
    else
    {
	gtk_plot_hide_dataset(my_thing->real_dataset);
	if (my_thing->error)
	{
	    free (my_thing->error);
	    my_thing->error = 0;
	}
    }
    

    if (my_thing->regrtype == REGRESSION_NONE && my_thing->error)
    {
	free (my_thing->error);
	my_thing->error = 0;
    }

    if ((j > 1))
	{
	    if (curve_fit (my_thing->x, my_thing->y, j, my_thing))
	    {
		char info[200];

		if (my_thing->error)
		{
		    free (my_thing->error);
		    my_thing->error = 0;
		}
		
		gtk_plot_show_dataset(my_thing->reg_dataset);
		
		sprintf (info, "  %s = %g\n  %s = %g\n  %s = %Lg\n", "M", my_thing->m, "B", my_thing->b, _("Mean squared error"), my_thing->mse);
		/*gtk_misc_set_alignment (GTK_MISC(datalabel), 0, 0);*/
		if (my_thing->datalabel)
		    gtk_label_set_text (GTK_LABEL(my_thing->datalabel), info);
		else
		{
		    my_thing->datalabel = gtk_label_new (info);
		    gtk_box_pack_start (GTK_BOX(my_thing->hbox1), my_thing->datalabel, FALSE, FALSE, 0);
		    gtk_label_set_justify (GTK_LABEL(my_thing->datalabel), GTK_JUSTIFY_LEFT);
		    gtk_widget_show (my_thing->datalabel);
		}
	    }
	    else
	    {
		if (my_thing->datalabel)
		{
		    gtk_widget_destroy (my_thing->datalabel);
		    my_thing->datalabel = 0;
		    gtk_widget_queue_draw (GTK_WIDGET(my_thing->sheet));
		}
	    }
	}
    else
    {
	gtk_plot_hide_dataset(my_thing->reg_dataset);
	if (my_thing->datalabel)
	{
	    gtk_widget_destroy (my_thing->datalabel);
	    my_thing->datalabel = 0;
            gtk_widget_queue_draw (GTK_WIDGET(my_thing->sheet));
	}
	if (my_thing->error)
	{
	    free (my_thing->error);
	    my_thing->error = 0;
	}
	my_thing->datalabel = 0;
	my_thing->m = 0;
        my_thing->b = 0;
	my_thing->mse = 0;
    }

    if (my_thing->autoresize)
    {
	/* If lowest [xy] is less than minimum [xy]
	   or highest [xy] is greater than maximum [xy]
	   Resize it to fit */
	
	for (i=0; i != j; ++i)
	{
	    if      (my_thing->x[i] > highx )     highx = my_thing->x[i];
	    else if (my_thing->x[i] < lowx  )     lowx  = my_thing->x[i];
	    if      (my_thing->y[i] > highy )     highy = my_thing->y[i];
	    else if (my_thing->y[i] < lowy  )     lowy  = my_thing->y[i];
	}
	
	if (lowx < rangexmi)     rangexmi = lowx;
	if (lowy < rangeymi)     rangeymi = lowy;
	if (highx > rangexma)    rangexma = highx;
	if (highy > rangeyma)    rangeyma = highy;
	
	if (rangexma > rangeyma) rangeyma = rangexma;
	if (rangeyma > rangexma) rangexma = rangeyma;
	if (rangexmi < rangeymi) rangeymi = rangexmi;
	if (rangeymi < rangexmi) rangexmi = rangeymi;
	
	majorticks_v = (rangeyma - rangeymi) / 10;
	minorticks_v = (rangeyma - rangeymi) / 20;
	majorticks_h = (rangexma - rangexmi) / 10;
	minorticks_h = (rangexma - rangexmi) / 20;
	
	gtk_plot_set_range (GTK_PLOT(active_plot), rangexmi, rangexma, rangeymi, rangeyma);
	
	gtk_plot_axis_set_ticks(GTK_PLOT(active_plot), GTK_ORIENTATION_HORIZONTAL, majorticks_h, minorticks_h);
	gtk_plot_axis_set_ticks(GTK_PLOT(active_plot), GTK_ORIENTATION_VERTICAL, majorticks_v, minorticks_v);
    }
    gtk_widget_queue_draw (layout);

    set_error ();
}


void data_new (void) 
{
    /* Menu layout struct */
    GtkItemFactoryEntry menu_items[] =
    {
	{ _("/_File"),                           NULL,         NULL,                           0, "<Branch>"     },
	{ _("/_File/New dataset"),               NULL,         (GtkSignalFunc)data_new_entry,  0, NULL           },
	{ _("/_File/Delete dataset"),            NULL,         data_del_entry,                 0, NULL           },
	{ _("/_File/Change dataset name"),       NULL,         data_chname_entry,              0, NULL           },
	{ _("/_File/sep0"),                      NULL,         NULL,                           0, "<Separator>"  },
	{ _("/_File/L_oad dataset..."),          "<control>O", data_load,                      0, NULL           },
	{ _("/_File/S_ave dataset..."),          "<control>A", data_save,                      0, NULL           },
	{ _("/_File/sep0"),                      NULL,         NULL,                           0, "<Separator>"  },
	{ _("/_File/_Close"),                    "<control>C", (GtkSignalFunc)data_hide,       0, NULL           }
    };
    
    GtkItemFactory *item_factory;
    GtkAccelGroup *accel_group;
    GtkWidget *menubar;
    gint nmenu_items = sizeof (menu_items) / sizeof (menu_items[0]);
    
    window=gtk_window_new (GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title (GTK_WINDOW(window), _("Data Entry"));
    gtk_widget_set_usize (GTK_WIDGET(window), 265, 400);
    gtk_window_set_policy (GTK_WINDOW(window), TRUE, TRUE, TRUE);
    
    gtk_signal_connect (GTK_OBJECT(window), "delete-event", GTK_SIGNAL_FUNC(data_hide), GTK_OBJECT(window));
    
    main_vbox=gtk_vbox_new(FALSE,1);
    gtk_container_set_border_width(GTK_CONTAINER(main_vbox),0); 
    gtk_container_add(GTK_CONTAINER(window), main_vbox);
    gtk_widget_show(main_vbox);
    
    
    accel_group = gtk_accel_group_new ();
    
    item_factory = gtk_item_factory_new (GTK_TYPE_MENU_BAR, "<main>", accel_group);
    
    gtk_item_factory_create_items (item_factory, nmenu_items, menu_items, NULL);
    
    gtk_window_add_accel_group (GTK_WINDOW(window), item_factory->accel_group);
    
    menubar = gtk_item_factory_get_widget (item_factory, "<main>");
    
    gtk_box_pack_start (GTK_BOX(main_vbox), menubar, FALSE, TRUE, 0);
    gtk_widget_show (menubar);
    
    
    notebook=gtk_notebook_new();
    gtk_notebook_set_tab_pos(GTK_NOTEBOOK(notebook), GTK_POS_TOP);
    gtk_signal_connect(GTK_OBJECT(notebook), "switch-page", GTK_SIGNAL_FUNC(dataset_switch_page), NULL);
    gtk_box_pack_start(GTK_BOX(main_vbox), notebook, TRUE, TRUE, 0);
    gtk_widget_show(notebook);
    
    
    data_new_entry ();
    currdata = datasets[0];
}


void data_edit (GtkWidget *widget, gpointer data)
{
	gtk_widget_show (window);
}


gboolean data_hide (GtkObject *obj, gpointer data)
{
	gtk_widget_hide (window);
	return (TRUE);
}


short strisnum (char *str)
{
    int i;
    
    for (i=0; str[i] != '\0'; ++i)
    {
	if (!isdigit(str[i]) && str[i] != '.' && str[i] != '+' && str[i] != '-')
	    return 0;
    }
    
    if (i > 0) return (1);
    return (0);
}


void data_load (GtkWidget *w, gpointer data)
{
    GtkWidget *filew = gtk_file_selection_new (_("Load dataset"));
    gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (filew)-> ok_button), "clicked", (GtkSignalFunc) data_load_ok, filew);
    gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION (filew)->cancel_button), "clicked", (GtkSignalFunc) gtk_widget_destroy, GTK_OBJECT (filew));
    gtk_widget_show (filew);
}


void data_save (GtkWidget *w, gpointer data)
{
    GtkWidget *filew = gtk_file_selection_new (_("Save dataset"));
    gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (filew)-> ok_button), "clicked", (GtkSignalFunc) data_save_ok, filew);
    gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION (filew)->cancel_button), "clicked", (GtkSignalFunc) gtk_widget_destroy, GTK_OBJECT (filew));
    gtk_widget_show (filew);
}


void      data_load_ok       (GtkWidget *w, GtkFileSelection *fs)
{
    thing *a_thing = data_new_entry ();
    char d[32], e[32];
    int i;

    FILE *file = fopen(gtk_file_selection_get_filename (GTK_FILE_SELECTION (fs)), "r");

    if (!file)
    {
	perror (gtk_file_selection_get_filename (GTK_FILE_SELECTION (fs)));
	return;
    }

   
    gtk_sheet_freeze (a_thing->sheet);

    for (i=0; !feof(file) && i < SHEET_LEN; ++i)
    {
	fscanf (file, "%31s %31s\n", d, e);
	gtk_sheet_set_cell_text (a_thing->sheet, i, 0, d);
	gtk_sheet_set_cell_text (a_thing->sheet, i, 1, e);
    }

    fclose (file);

    gtk_notebook_set_page (GTK_NOTEBOOK(notebook), -1); /* Switch to the new dataset */
    currdata = a_thing;

    /* Plot points */
    set_cell (a_thing->sheet, 0, 0);

    gtk_sheet_thaw (a_thing->sheet);

    gtk_widget_destroy (GTK_WIDGET(fs));
}


void      data_save_ok       (GtkWidget *w, GtkFileSelection *fs)
{
    int i;
    gchar *c, *d;
    FILE * dataout;

    if (!currdata) /* no datasets, goto end */
	goto end;

    dataout = fopen(gtk_file_selection_get_filename (GTK_FILE_SELECTION (fs)), "w");

    if (!dataout)
    {
	fprintf (stderr, "%s\n", _("Can't open the file you selected. Sorry."));
	goto end;
    }

    for (i=0; i != SHEET_LEN; ++i)
    {
         c = gtk_sheet_cell_get_text (currdata->sheet, i, 0);
	 d = gtk_sheet_cell_get_text (currdata->sheet, i, 1);
	 if (c && d && strisnum (c) && strisnum (d))
	 {
	     fprintf (dataout, "%s\t%s\n", c, d);
	 }
    }

    fclose (dataout);


end:
    gtk_widget_destroy (GTK_WIDGET(fs));
}


static char * default_colors [] = { "red",
				    "slate grey",
				    "blue",
				    "brown",
				    "turquoise",
				    "green",
				    "HotPink1",
				    "magenta",
				    "purple",
				    "orange" };


thing * data_new_entry (void)
{
    char s[30], *regname;
    int i;
    GtkWidget *label, *chkbutton;
    thing *dataset=0;

    /* Try to find an unused dataset */
    for (i=0; i != nbrdatasets; ++i)
    {
	if (!datasets[i]->usable)
	{
	    dataset = datasets[i];
	}
    }
   
    if (!dataset)
    {
	datasets = (thing **) realloc (datasets, (++nbrdatasets)*sizeof(thing *));
	datasets[nbrdatasets-1] = malloc (sizeof(thing));
	if (!datasets[nbrdatasets-1]) return NULL;
	dataset = datasets[nbrdatasets-1];
    }

    ++nbrrealdatasets;
    ++nbrdatasets_ever;

    /* Allocated, now null it out woohoo */
    dataset->real_dataset = gtk_plot_dataset_new (GTK_PLOT(active_plot));
    gtk_plot_add_dataset (GTK_PLOT(active_plot), dataset->real_dataset);
    gdk_color_parse (default_colors[(nbrdatasets_ever-1)%10], &(dataset->symbol_color));
    gdk_color_alloc (gdk_colormap_get_system(), &(dataset->symbol_color));
    
    gdk_color_parse (default_colors[(nbrdatasets_ever-1)%10], &(dataset->line_color));
    gdk_color_alloc (gdk_colormap_get_system(), &(dataset->line_color));

    gtk_plot_dataset_set_symbol (dataset->real_dataset,
				 GTK_PLOT_SYMBOL_DIAMOND,
				 GTK_PLOT_SYMBOL_OPAQUE,
				 10, 2, dataset->symbol_color);
    gtk_plot_dataset_set_line_attributes (dataset->real_dataset,
					  GTK_PLOT_LINE_SOLID,
					  1,
					  dataset->line_color);
    gtk_plot_dataset_set_connector (dataset->real_dataset,
				    GTK_PLOT_CONNECT_NONE);
    gtk_plot_dataset_hide_yerrbars (dataset->real_dataset);
    gtk_plot_hide_dataset (dataset->real_dataset);
    
    strcpy (s, _("Untitled dataset"));
    
    gtk_plot_dataset_set_legend (dataset->real_dataset, s);
    
    dataset->sheet = GTK_SHEET(gtk_sheet_new(SHEET_LEN,2,""));


    /* Datalabel */
    dataset->datalabel = 0;
    /* Settings dialog (used to determine if it's open) */
    dataset->dlg_data = 0;
    /* data points for GtkPlot */
    dataset->x = 0;
    dataset->y = 0;
    dataset->bogus = 0;
    /* Display least-squares approximation? */
    dataset->regrtype = REGRESSION_LINEAR;
    /* Number of significant figures */
    dataset->sigfigs = 0;
    /* Dynamically resize plot? */
    dataset->autoresize = 1;
    /* Regression stuff */
    dataset->m = 0;
    dataset->b = 0;
    dataset->corr = 0;
    dataset->mse = 0;
    /* Connect line (GtkPlotConnector) */
    dataset->connect = GTK_PLOT_CONNECT_NONE;
    /* Dataset name */
    dataset->name = malloc ((strlen(s)+1)*sizeof(char));
    strcpy (dataset->name, s);
    /* Error string */
    dataset->error = 0;
    /* Usable */
    dataset->usable = TRUE;
    /* Tab # */
    dataset->index = nbrrealdatasets-1;


    dataset->reg_dataset = gtk_plot_add_function (GTK_PLOT(active_plot), (GtkPlotFunc)function_reg);
    dataset->reg_dataset->data = (gpointer) dataset;
    gtk_plot_dataset_set_line_attributes (dataset->reg_dataset,
					  GTK_PLOT_LINE_SOLID,
					  0, dataset->line_color);


    for (i=0; i != SHEET_LEN; ++i)
    {
	char name[5];
	sprintf (name, "%i", i+i);
	gtk_sheet_set_row_title (dataset->sheet,
				 i,
				 name);
    }
    
    vbox1 = gtk_vbox_new (FALSE, 0);
    gtk_widget_show (vbox1);
    
    scrolled_windows=(GtkWidget **)realloc(scrolled_windows, sizeof(GtkWidget *));
    scrolled_windows[0]=gtk_scrolled_window_new(NULL, NULL);
    gtk_container_add(GTK_CONTAINER(scrolled_windows[0]), GTK_WIDGET(dataset->sheet));
    
    gtk_box_pack_start(GTK_BOX(vbox1), scrolled_windows[0], TRUE, TRUE, 0);
    
    chkbutton = gtk_button_new_with_label (_("Settings"));
    gtk_box_pack_start (GTK_BOX(vbox1), chkbutton, FALSE, FALSE, 0);
    gtk_signal_connect (GTK_OBJECT(chkbutton), "clicked", (GtkSignalFunc)dialog_data, (gpointer)dataset);
    gtk_widget_show (chkbutton);
    
    /*chkbutton = gtk_check_button_new_with_label (_("Display line of best fit"));
      gtk_box_pack_start (GTK_BOX(main_vbox), chkbutton, FALSE, FALSE, 0);
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(chkbutton), dispbest);
      gtk_signal_connect (GTK_OBJECT(chkbutton), "clicked", (GtkSignalFunc) data_toggle_line, NULL);
      gtk_widget_show (chkbutton);*/
    
    gtk_widget_show(GTK_WIDGET(dataset->sheet));
    gtk_widget_show(scrolled_windows[0]);
    
    dataset->hbox1 = gtk_hbox_new (FALSE, 5);
    gtk_box_pack_start (GTK_BOX(vbox1), dataset->hbox1, FALSE, FALSE, 0);
    gtk_widget_show (dataset->hbox1);
    

    /*sprintf (s, "%s %i", _("Dataset"), nbrdatasets);*/

    label=gtk_label_new(s);
    /* Bookmark label */
    dataset->label = label;
    gtk_notebook_append_page (GTK_NOTEBOOK(notebook), GTK_WIDGET(vbox1), label);
    gtk_notebook_set_page (GTK_NOTEBOOK(notebook), -1); /* Go to last page */
    currdata = dataset;

    /* Notebook item */
    dataset->notebook_item = vbox1;

    
    regname = (char *) malloc (strlen(s)+strlen(_("curve fit"))+4);
    sprintf (regname, "%s (%s)", s, _("curve fit"));
    gtk_plot_dataset_set_legend (dataset->reg_dataset, regname);
    free (regname);

    gtk_plot_hide_dataset (dataset->reg_dataset);
    
    gtk_signal_connect(GTK_OBJECT(dataset->sheet),
		       "set_cell",
		       set_cell,
		       NULL);
    gtk_signal_connect(GTK_OBJECT(dataset->sheet),
		       "clear_cell",
		       set_cell,
		       NULL);
    
    build_example1(GTK_WIDGET(dataset->sheet));

    gtk_widget_queue_draw (GTK_WIDGET(dataset->sheet));

    return (dataset);
}


static void dataset_switch_page (GtkNotebook *notebook,
				 GtkNotebookPage *page,
				 gint page_num,
				 gpointer user_data)
{
    int i;
    for (i=0; i != nbrdatasets; ++i)
    {
	if (datasets[i]->index == page_num)
	{
	    currdata = datasets[i];
	}
    }
}


static gdouble function_reg (GtkPlot *plot, GtkPlotData *dataset, gdouble x, gboolean *err)
{
    thing *funky_thing = (thing *) dataset->data;
    gdouble m = funky_thing->m;
    gdouble b = funky_thing->b;

    *err = FALSE;

    switch (funky_thing->regrtype)
    {
    case REGRESSION_LINEAR:
    {
	return m * x + b;
    }
    case REGRESSION_LOGARITHMIC:
    {
	if (x > 0.0 && !funky_thing->error)
	    return b + m * log(x);
	else
	{
	    goto default_out;
	}
    }
    case REGRESSION_EXPONENTIAL:
    {
	if (!funky_thing->error)
	{
	    gdouble lny = log(b) + m * x;
	    return exp (lny);
	}
	else
	{
	    goto default_out;
	}
    }
    case REGRESSION_POWER:
    {
	if (x > 0.0 && !funky_thing->error)
	{
	    gdouble lny = log(b) + m * log(x);
	    return exp (lny);
	}
	else
	{
	    goto default_out;
	}
    }
    default:
	goto default_out;
    }
default_out:
    *err = TRUE;
    gtk_plot_hide_dataset (funky_thing->reg_dataset);
    return 0;
}

void data_del_entry (GtkWidget *w, gpointer data)
{
    int i;
    signed long dist=LONG_MAX;
    thing *dataset=currdata, *new_sel=0;

    gtk_notebook_remove_page (GTK_NOTEBOOK(notebook), dataset->index);
    /*gtk_widget_destroy (dataset->notebook_item);*/

    for (i=0; i != nbrdatasets; ++i)
    {
	/* Adjust indexes to be correct now that this dataset has been removed */
	if (datasets[i]->index > dataset->index)
	{
	    /* This came after the dataset we are deleting, it will need correction */
	    ++(datasets[i]->index);
	}
    }

    /* Usable */
    dataset->usable = FALSE;

    for (i=0; i != nbrdatasets; ++i)
    {
	/* Find the closest dataset */
	if (datasets[i]->usable && abs(datasets[i]->index - dataset->index) < dist)
	{
	    new_sel = datasets[i];
	    dist = abs(datasets[i]->index - dataset->index);
	}
    }
    
    if (!new_sel)
    {
	currdata = 0;
    }
    else
    {
	gtk_notebook_set_page (GTK_NOTEBOOK(notebook), new_sel->index); /* Go to new page */
	currdata = new_sel;
    }

    --nbrrealdatasets;

    /* data points for GtkPlot */
    dataset->x = 0;
    dataset->y = 0;
    dataset->bogus = 0;
    /* Display least-squares approximation? */
    dataset->regrtype = REGRESSION_LINEAR;
    /* Number of significant figures */
    dataset->sigfigs = 0;
    /* Dynamically resize plot? */
    dataset->autoresize = 1;
    /* Regression stuff */
    dataset->m = 0;
    dataset->b = 0;
    dataset->corr = 0;
    dataset->mse = 0;
    /* Connect line (GtkPlotConnector) */
    dataset->connect = GTK_PLOT_CONNECT_NONE;
    /* Dataset name */
    if (dataset->name)
	free (dataset->name);
    dataset->name = 0;
    /* Error string */
    if (dataset->error)
	free (dataset->error);
    dataset->error = 0;
    /* Notebook item */
    dataset->notebook_item = 0;
    /* Main hbox */
    dataset->hbox1 = 0;
    /* Datalabel */
    dataset->datalabel = 0;
    /* Settings dialog (used to determine if it's open) */
    if (dataset->dlg_data)
    {
	gtk_widget_destroy (dataset->dlg_data);
    }
    dataset->dlg_data = 0;
    /* Tab # */
    dataset->index = -1;
    /* Notebook item */
    dataset->notebook_item = 0;
    /* Bookmark label */
    dataset->label = 0;
}

void data_chname_entry (GtkWidget *w, gpointer data)
{
    if (currdata == 0) return;
    dialog_rename (currdata, 1 /* 1 == dataset */);
}
