/*
The DsTool program is the property of:
 
                             Cornell University 
                        Center of Applied Mathematics 
                              Ithaca, NY 14853
                      dstool_bugs@macomb.tn.cornell.edu
 
and may be used, modified and distributed freely, subject to the following
restrictions:
 
       Any product which incorporates source code from the DsTool
       program or utilities, in whole or in part, is distributed
       with a copy of that source code, including this notice. You
       must give the recipients all the rights that you have with
       respect to the use of this software. Modifications of the
       software must carry prominent notices stating who changed
       the files and the date of any change.
 
DsTool is distributed in the hope that it will be useful, but WITHOUT ANY 
WARRANTY; without even the implied warranty of FITNESS FOR A PARTICULAR PURPOSE.
The software is provided as is without any obligation on the part of Cornell 
faculty, staff or students to assist in its use, correction, modification or
enhancement.
*/

/*
 * twod.c
 */

#include <stdio.h>
#include <math.h>

#include <tk.h>
#include <pm.h>
#include <memory.h>
#include <constants.h>
#include <symbols.h>
#include <view.h>
#include <utilities.h>
#include "dscolor.h"
#include <math_utils.h>
#include <memory.h>
#include "../memory/memory_local.h"

static void twod_xplot(VIEW_ITEM *view, int x, int y, int color, int symbol);
static void twod_plot(VIEW_ITEM *view, int x, int y, int color, int symbol);

void
twod_dismiss(int n)
{
#ifdef DEBUG
  fprintf(stdout,"Deregistering TWOD %d\n",n);
#endif
  dismiss_view(TWOD, n);
}

void
twod_range(int n, int hortype, int horindex, double hormin, double hormax,
	   int vertype, int verindex, double vermin, double vermax)
{
  VIEW_ITEM *view;
  TWOD_VIEW_ITEM *vi;

#ifdef DEBUG
  fprintf(stdout, "Setting range for TWOD %d\n",n);
#endif
  if ( NULL == (view = get_view(TWOD, n)) )
    {
      fprintf(stdout, " Attempt to set range for invalid TWOD view %d\n",n);
      return;
    }
  vi = &(view->data.twod);
  vi->hortype = hortype;
  vi->horindex = horindex;
  vi->hormin = hormin;
  vi->hormax = hormax;
  vi->vertype = vertype;
  vi->verindex = verindex;
  vi->vermin = vermin;
  vi->vermax = vermax;
  
}

void
twod_colormode( int n, int mode )
{
  VIEW_ITEM *view;
  TWOD_VIEW_ITEM *vi;

  if( NULL == (view = get_view(TWOD, n)) ) {
    fprintf(stdout, "Attempt to set pick color mode for invalid TWOD view %d\n",n);
    return;
  }
  vi = &(view->data.twod);
  vi->colormode = mode;
}

void
twod_config(int n, int width, int height)
{
  VIEW_ITEM *view;
  TWOD_VIEW_ITEM *vi;

#ifdef DEBUG
  fprintf(stdout, "Configuring TWOD %d\n",n);
#endif
  if ( NULL == (view = get_view(TWOD, n)) )
    {
      fprintf(stdout, "Attempt to config for invalid TWOD view %d\n",n);
      return;
    }
  vi = &(view->data.twod);
  vi->width = width;
  vi->height = height;
}

void
twod_activate(int n)
{
  VIEW_ITEM *v;
  char buf[20];

#ifdef DEBUG
  fprintf(stdout, "Activating TWOD %d\n",n);
#endif
  v = add_view(TWOD, n);

  /* get tk info about window */
  sprintf(buf,".twoD%d.c",n);
  v->data.twod.clientData = get_ClientData(buf);
}

void
twod_size(int n)
{
#ifdef DEBUG
  fprintf(stdout, "Setting size for TWOD %d\n",n);
#endif
}


void
twod_dump(VIEW_ITEM *v)
{
  TWOD_VIEW_ITEM *vi = &(v->data.twod);

  fprintf(stdout, "  %s %d  %dx%d\n", win_keyindex(v->type), v->number,
	  vi->width, vi->height);
  fprintf(stdout, "    Hor: %s %d in (%g,%g) \n",
	  win_keyindex(vi->hortype), vi->horindex, vi->hormin, vi->hormax);
  fprintf(stdout, "    Ver: %s %d in (%g,%g) \n",
	  win_keyindex(vi->vertype), vi->verindex, vi->vermin, vi->vermax);
}

static void
twod_xplot(VIEW_ITEM *view, int x, int y, int color, int symbol)
{
  int sym = symbol / NUM_SYM_SIZES;
  int size = symbol % NUM_SYM_SIZES;

  Tk_PlotSymbol((TkPlot *) (view->data.twod.clientData), 
		x, y, color, sym, size);
}

static void
twod_plot(VIEW_ITEM *view, int x, int y, int color, int symbol)
{
  int sym = symbol / NUM_SYM_SIZES;
  int size = symbol % NUM_SYM_SIZES;

  Tk_PlotSymbol((TkPlot *) (view->data.twod.clientData), 
		x, y, color, sym, size);
}


void twod_point(int n, int x, int y, int color, int symbol)
{
  VIEW_ITEM *view = get_view(TWOD,n);

  twod_plot(view, x, y, color, symbol);
}

/*
 * plots all the data in a specified memory object to the specified twod
 * window
 * 
 * NOTE: this procedure may create spruious data points in the presence of  
 * parameter data (loaded from DsTool 2).  In particular, if a twoD view
 * panel has a function as either the horizontal or vertical variable, then 
 * that function will be evaluated for points in the parameter memory at the
 * specified parameters and all variables equal to zero.  If the twoD view 
 * panel has a variable as horizontal or vertical variable, then no data will 
 * be plotted on the twoD view from the parameter memory. 
 * This discrepancy is due to the fact that we cannot tell in advance whether
 * a function will depend upon variables or parameters or both. 
 */
void
twod_mem(VIEW_ITEM *v, memory m)
{
  TWOD_VIEW_ITEM *twod = &(v->data.twod);
  int fn = FALSE;
  int x, y, *color;
  double *fpoints, *points, *params, dx, dy;
  int pcolor;
  int n_varb = *((int *)pm(GET, "Model.Varb_Dim", NULL));
  int i, ignore=0;
  
  /* will need function values */
  if ( (twod->hortype == FUNCT) || (twod->vertype == FUNCT) ) fn = TRUE;

  if (where_are_varbs(((Memory) m)->mem_type) == NOWHERE) {
      fpoints = dvector(0,n_varb-1);    
      for (i=0;i<n_varb;i++)
          fpoints[i] = 0.;
      if ((twod->hortype == VARB) || (twod->vertype == VARB) ) ignore = 1;
      }

  if (ignore == 0)
    if (memory_reset_read(m) == 0) 
      {
      while (memory_read_next_flow(m, NULL, NULL, NULL, NULL, NULL) == 0)  
	{
	  while (memory_read_next_traj(m, NULL, NULL, NULL) == 0)
	    {
	      while (memory_read_next_point(m, &points, &params, &color, 
					    NULL, NULL) == 0)
		{
		  if (fn == TRUE)
  		      if (where_are_varbs(((Memory) m)->mem_type) == NOWHERE) 
		          view_auxf(view_f, fpoints, params);
                      else 
                          view_auxf(view_f, points, params);
		  switch ( twod->hortype )
		    {
		    case VARB:
			dx = points[twod->horindex];
		        break;
		    case PARAM:
		        dx = params[twod->horindex];
		        break;
		    case FUNCT:
		        dx = view_f[twod->horindex];
		        break;
		    }
		  switch ( twod->vertype )
		    {
		    case VARB:
			dy = points[twod->verindex];
                        break;
		    case PARAM:
		        dy = params[twod->verindex];
		        break;
		    case FUNCT:
		        dy = view_f[twod->verindex];
		        break;
		    }
		  dx = (dx - twod->hormin) / (twod->hormax - twod->hormin);
		  dy = (twod->vermax - dy) / (twod->vermax - twod->vermin);

		  if ((dx >= 0.0) && (dx < 1.0) && (dy > 0.0) && (dy <= 1.0))
		    {
		      x = (int) floor(dx*twod->width);
		      y = (int) floor(dy*twod->height);

                      pcolor = DsColor_get_index( twod->colormode, color[0], color[1] );
		      twod_plot( v, x, y, pcolor, color[2]);

		    }
		}
	    }
	}
    }
    if (where_are_varbs(((Memory) m)->mem_type) == NOWHERE)
        free_dvector(fpoints, 0, n_varb-1);

}


/*
 * plots all the data in all the memory objects to a specified twod window
 */
void
twod_all_mem(VIEW_ITEM *v)
{
  char *name;
  memory m;
  int i=0;

  while ( (name = (char *) pm(QUERY, "Memory", MEMRY, i, NULL)) )
    {
      i++;
      m = (memory) pm(GET, name, NULL);
      twod_mem(v,m);
    }
}
