;;; wl-user-agent.el --- compose-mail support for wanderlust

;;;  Copyright (C) 1999 Sen Nagata

;; Author: Sen Nagata <sen@eccosys.com>
;; Version: 0.4
;; License: GPL 2
;; Warning: only partial compose-mail support at this point (see docstring)

;; This file is not a part of GNU Emacs.

;;; Commentary:
;;
;; required elisp packages:
;;
;;   -wl
;;   -sendmail.el
;;
;; installation:
;;
;;   -put this file in an appropriate directory (so emacs can find it)
;;
;;   <necessary>
;;   -put:
;;
;;     (add-hook 'wl-init-hook (lambda () (require 'wl-user-agent)))
;;
;;    in .emacs or .wl
;;
;;   <optional>
;;   -put:
;;
;;     (autoload 'wl-user-agent-compose "wl" nil t)
;;     (if (boundp 'wl-user-agent)
;;         (setq mail-user-agent 'wl-user-agent))
;;     (if (fboundp 'define-mail-user-agent)
;;         (define-mail-user-agent
;;           'wl-user-agent
;;           'wl-user-agent-compose
;;           'wl-draft-send
;;           'wl-draft-kill
;;           'mail-send-hook))   ; no wl-send-hook?
;;
;;    in .emacs -- putting the above in .wl seems pointless
;;
;;    (looked at mew.el for ideas)
;;
;; todo:
;;
;;   -add support for switch-function in wl-user-agent-compose
;;
;;   -consider whether values of nil for TO and SUBJECT for
;;    wl-user-agent-compose ought to override any to and subject entries
;;    in OTHER-HEADERS
;;
;; notes and details:
;;
;; i thought of rewriting wl-draft, but it's non-trivial.  soooo, i
;; thought of the following hack...since wl-draft calls run-hooks on
;; wl-mail-setup-hook, add-hook'ing an appropriate hook function to
;; manipulate headers and body text is an option.  it isn't pretty but
;; for the moment, it seems better than rewriting wl-draft...note that
;; run-hooks is only called if wl-draft thinks it is called
;; interactively...good thing there is a function called
;; call-interactively :-)
;;
;; hopefully, there will be a native implementation of wl-user-agent-compose
;; at some point in the near future...
;;
;; goal:
;;
;;   a way of preparing a new message w/ specific headers and body text
;;
;; ingredients:
;;
;;   wl-user-agent-compose
;;     a wrapper function around wl-draft -- attempts to simulate
;;     the compose-mail interface from simple.el
;;
;;   wl-user-agent-compose-hook-function
;;     a hook function to be add-hook'ed to wl-mail-setup-hook -- this
;;     function performs the actual work of inserting headers and
;;     body text.

;;; History:
;;
;; 0.4:
;;
;;  modified comments a bit
;;  started using checkdoc -- checkdoc-current-buffer
;;  tested w/ xemacs -- found a bug -- member-ignore-case used
;;   compare-strings, which doesn't exist in xemacs :-(
;;  removed member-ignore-case dependency -- so, this should now work w/ 
;;   xemacs.
;;
;; 0.3:
;;
;;  fixed bug in wl-user-agent-compose-hook-function
;;
;; 0.2:
;;
;;  brushing up code
;;
;; 0.1:
;;
;;  initial implementation

;;; Code:
(defconst wl-user-agent-version "wl-user-agent.el 0.4")

;; how should we handle the dependecy on wl?
;; will this work?
(eval-when-compile
  (require 'wl))

;; use mail-mode support: mail-position-on-field, mail-header-separator
(require 'sendmail)

;; this appears to be necessarily global...
(defvar wl-user-agent-compose-p nil)

;; this should be a generic function for mail-mode -- i wish there was
;; something like it in sendmail.el
(defun wl-user-agent-insert-header (header-name header-value)
  "Insert HEADER-NAME w/ value HEADER-VALUE into a message."
  ;; it seems like overriding existing headers is acceptable -- should
  ;; we provide an option?
  
  ;; plan was: unfold header (might be folded), remove existing value, insert
  ;;           new value
  ;; wl doesn't seem to fold header lines yet anyway :-)
  
  (let ((kill-whole-line t)
	end-of-line)
    (mail-position-on-field (capitalize header-name))
    (setq end-of-line (point))
    (beginning-of-line)
    (re-search-forward ":" end-of-line)
    (insert (concat " " header-value "\n"))
    (kill-line)))

;; this should be a generic function for mail-mode -- i wish there was
;; something like it in sendmail.el
;;
;; ** haven't dealt w/ case where the body is already set **
(defun wl-user-agent-insert-body (body-text)
  "Insert a body of text, BODY-TEXT, into a message."
  ;; code defensively... :-P
  (goto-char (point-min))
  (search-forward mail-header-separator)
  (forward-line 1)
  (insert body-text))

;;;###autoload
(defun wl-user-agent-compose (&optional to subject other-headers continue
					switch-function yank-action
					send-actions)
  "Support the `compose-mail' interface for wl.
Only support for TO, SUBJECT, and OTHER-HEADERS has been implemented.
Support for CONTINUE, SWITCH-FUNCTION, YANK-ACTION, and SEND-ACTIONS has not
been implemented yet.  Support for SWITCH-FUNCTION can probably be
implemented by performing duplicative surgery on `wl-draft'."

  ;; protect these -- to and subject get bound at some point, so it looks
  ;; to be necessary to protect the values used w/in
  (let ((wl-user-agent-headers-and-body-alist other-headers))

    (if to
	(if (assoc-ignore-case "to" wl-user-agent-headers-and-body-alist)
	    (setcdr
	     (assoc-ignore-case "to" wl-user-agent-headers-and-body-alist)
	     to)
	  (setq wl-user-agent-headers-and-body-alist
		(cons (cons "to" to)
		      wl-user-agent-headers-and-body-alist))))

    (if subject
	(if (assoc-ignore-case "subject" wl-user-agent-headers-and-body-alist)
	    (setcdr
	     (assoc-ignore-case "subject" wl-user-agent-headers-and-body-alist)
	     subject)
	  (setq wl-user-agent-headers-and-body-alist
		(cons (cons "subject" subject)
		      wl-user-agent-headers-and-body-alist))))

    ;; i think this is what we want to use...
    (unwind-protect
	(progn
	  
	  ;; tell the hook-function to do its stuff
	  (setq wl-user-agent-compose-p t)
	  
	  ;; because to get the hooks working, wl-draft has to think it has
	  ;; been called interactively
	  (call-interactively 'wl-draft)))
    
    (setq wl-user-agent-compose-p nil)))

;; note that this function assumes the existence of the following variables:
;;
;;   wl-user-agent-compose-p (global),
;;   wl-user-agent-headers-and-body-alist (defined in wl-user-agent-compose)
;;
;; see:
;;
;;   ftp://ftp.dl.ac.uk/pub/fx/emacs/emacs-AI-memo/AI-519a.html#SEC18
;;
;; for some reasoning.
(defun wl-user-agent-compose-hook-function ()
  "Manipulate headers and/or a body of a draft message.
The author's intention was for this function to be add-hook'ed to
`wl-mail-setup-hook'."

  ;; being called from wl-user-agent-compose?
  (if wl-user-agent-compose-p
      (progn

	;; insert headers
	(let ((case-fold-search t))
	  (mapcar
	   (lambda (x)
	     (let ((header-name (car x))
		   (header-value (cdr x)))
	       ;; skip body
	       (if (not (string-match "^body$" header-name))
		   (wl-user-agent-insert-header header-name header-value)
		 t)))
	   wl-user-agent-headers-and-body-alist))

	;; highlight headers (from wl-draft in wl-draft.el)
	(let (wl-highlight-x-face-func)
	  (wl-highlight-headers))

	;; insert body
	(if (assoc-ignore-case "body" wl-user-agent-headers-and-body-alist)
	    (wl-user-agent-insert-body
	     (cdr (assoc-ignore-case
		   "body"
		   wl-user-agent-headers-and-body-alist)))))

    ;; not being called from wl-user-agent-compose, so don't do anything --
    ;; don't remember whether returning nil is a bad idea...the info pages
    ;; on Hooks in the elisp reference don't say.  pity.  let's return t --
    ;; i have this weird feeling...
    t))

;; this is very important!
(add-hook 'wl-mail-setup-hook 'wl-user-agent-compose-hook-function)

;; since this will be used via 'require'...
(provide 'wl-user-agent)

;;; wl-user-agent.el ends here
