/*  Spruce
 *  Copyright (C) 1999 Susixware
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include "pgp.h"

static gchar pgp_stdin[1024];
static gchar pgp_stdout[1024];
static gchar pgp_stderr[1024];

static gchar tmp_path[256];

static gchar pgp_file[1024];
static gchar pgp_file_asc[1024];
static gchar chars[] = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789";

gint set_tmp_path ()
{
   DIR *tmpdir;

   /* find out temp directory path */
   memset(tmp_path, 0, sizeof(tmp_path));
   g_snprintf(tmp_path, sizeof(tmp_path)-1, "%s/.tmp", getenv("HOME"));

   if (strlen(tmp_path) >= 200)
   {
      /* looks like some is trying to play with our heads...
       * a home directory shouldn't have a path longer than 200 chars */
      memset(tmp_path, 0, sizeof(tmp_path));
      strncpy(tmp_path, "/tmp/spruce", sizeof(tmp_path)-1);
      return 1;
   }

   /* since we aren't using /tmp, we have to make sure it exists */
   tmpdir = opendir(tmp_path);
   if (tmpdir == NULL)
	{
      /* directory doesn't exist, lets create it */
      return (!mkdir(tmp_path, 0700));
	}
   closedir(tmpdir);
   return 1;
}

void pgp_bzero_stdio()
{
   memset(pgp_stdin, 0, sizeof(pgp_stdin));
   memset(pgp_stdout, 0, sizeof(pgp_stdout));
	memset(pgp_stderr, 0, sizeof(pgp_stderr));
}

gint pgp_gen_stdio()
{
   gchar tmp[10];
   guint r, i;
   time_t t;
	
	srand((unsigned) time(&t));

   pgp_bzero_stdio();

	if (!set_tmp_path ())
      return 0;

   strncpy(pgp_stdin, tmp_path, sizeof(pgp_stdin)-1);
   strncpy(pgp_stdout, tmp_path, sizeof(pgp_stdout)-1);
   strncpy(pgp_stderr, tmp_path, sizeof(pgp_stderr)-1);

   memset(tmp, 0, sizeof(tmp));
   for (i = 0; i < 6; i++)
   {
      r = rand() % 61;
      tmp[i] = chars[r];
   }
   strncat(pgp_stdin,  tmp, sizeof(pgp_stdin)-1);
   strncat(pgp_stdout, tmp, sizeof(pgp_stdout)-1);
   strncat(pgp_stderr, tmp, sizeof(pgp_stderr)-1);

   strncat(pgp_stdin,  ".stdin", sizeof(pgp_stdin)-1);
   strncat(pgp_stdout, ".stdout", sizeof(pgp_stdout)-1);
   strncat(pgp_stderr, ".stderr", sizeof(pgp_stderr)-1);

   strncat(pgp_stdin,  "XXXXXX", sizeof(pgp_stdin)-1);
   strncat(pgp_stdout, "XXXXXX", sizeof(pgp_stdout)-1);
   strncat(pgp_stderr, "XXXXXX", sizeof(pgp_stderr)-1);

   if (mktemp(pgp_stdin) == NULL)
      return 0;
   if (mktemp(pgp_stdout) == NULL)
      return 0;
   if (mktemp(pgp_stderr) == NULL)
      return 0;

   return 1;
}

gint pgp_gen_filename()
{
   gchar tmp[10];
   guint r, i;
   time_t t;
	
	srand((unsigned) time(&t));

	if (!set_tmp_path ())
      return 0;

   memset(pgp_file, 0, sizeof(pgp_file));
   strncpy(pgp_file, tmp_path, sizeof(pgp_file)-1);

   memset(tmp, 0, sizeof(tmp));
   for (i = 0; i < 6; i++)
   {
      r = rand() % 61;
      tmp[i] = chars[r];
   }
   strncat(pgp_file, tmp, sizeof(pgp_file)-1);
   strncat(pgp_file, "XXXXXX", sizeof(pgp_file)-1);
	
   if (mktemp(pgp_file) == NULL)
      return 0;

   g_snprintf(pgp_file_asc, sizeof(pgp_file)-1, "%s.asc", pgp_file);

   return 1;
}

gint pgp_detect(gchar *text)
{
   if (strstrcase(text, "-----BEGIN PGP MESSAGE-----"))
      return 1;
   return 0;
}

gchar *pgp_encrypt(gchar *text, gchar *recipients, gchar *passphrase, gint use_gnupg, gint pgp_sign)
{
   gchar pgpexec[256], pgpargv0[50];
   gchar line[256];
   gchar *buffer;
   FILE *fp;
   gint pgpin, pgpout, pgperr;   /* stdio file descriptors for the pgp exec */
   pid_t pgppid;                 /* pgp process id */
   gint *status = NULL;          /* process exit status */
   gint retval = 0;
   gboolean got_prompt, pgp_error = FALSE;

   if (passphrase == (gchar*)NULL)
      return NULL;

   if (!pgp_gen_filename())
      return NULL;

   /* lets make a temp file to dump our message in */
   fp = fopen(pgp_file, "wt");
   if (fp == NULL)
      return NULL;

   /* lets make secure permissions */
   chmod(pgp_file, 0600);

   /* lets write our mesg to a temp file */
   fprintf(fp, "%s\n", text);
   fflush(fp);

   fclose(fp);

   /* lets generate the command line */
   memset(pgpexec, 0, sizeof(pgpexec));
   memset(pgpargv0, 0, sizeof(pgpargv0));
   if (!use_gnupg)
   {
      strcpy(pgpexec, "pgpe");
      strcpy(pgpargv0, "-a");
   }
   else
   {
      strcpy(pgpexec, "gpg");
      strcpy(pgpargv0, "-ea");
   }

   if (pgp_sign)
      strcat(pgpargv0, "s");

   strcat(pgpargv0, "r");

   /*g_snprintf(command, 512, "%s %s %s %s", pgpexec, pgpargv0, recipients, pgp_file);*/

   /* time to create file descriptors for the pgp process... */
   if (!pgp_gen_stdio())
   {
      remove (pgp_file);
      return NULL;
   }
   pgpin = open(pgp_stdin, O_CREAT | O_RDWR, 0600);
   pgpout = open(pgp_stdout, O_CREAT | O_RDWR, 0600);
   pgperr = open(pgp_stderr, O_CREAT | O_RDWR, 0600);

   /* if we remove them now, they will be cleaned up even if we crash */
   remove(pgp_stdin);
   remove(pgp_stdout);
   remove(pgp_stderr);

   /* we're now ready to invoke the pgp process */
   pgppid = fork();
   switch (pgppid)
   {
      case -1:
         remove(pgp_file);
         perror("pgp_encrypt()");
         return NULL;
         break;
      case 0:
         /* child */
         dup2(pgpout, 1);
         close(pgpout);
         dup2(pgperr, 2);
         close(pgperr);
         dup2(pgpin, 0);
         close(pgpin);

         retval = execlp(pgpexec, pgpargv0, recipients, pgp_file, NULL);
         if (retval)
            return NULL;
         break;
      default:
         /* parent */
         memset(line, 0, sizeof(line));
         got_prompt = FALSE;
         do
         {
            /* Pgp50i sends: "Enter pass phrase:" but Gpg sends: "Enter passphrase:" */
            if (strstrcase(line, "phrase:") != NULL)
               got_prompt = TRUE;
            memset(line, 0, sizeof(line));
         } while (read(pgpout, line, 255) > 0 || !got_prompt);

         /* now lets send the passphrase */
         write(pgpin, passphrase, strlen(passphrase) + 1);
         write(pgpin, "\n", sizeof(gchar));

         memset(line, 0, sizeof(line));
         do
         {
            /* on error, Pgp sends: "Error:..." but Gpg sends "gpg: Invalid..." */
            if (strstrcase(line, "Error:") != NULL || strstrcase(line, "Invalid") != NULL)
            {
               /* we musta sent an invalid passphrase? */
               kill(pgppid, SIGKILL);
               pgp_error = TRUE;
               break;
            }
            memset(line, 0, sizeof(line));
         } while (read(pgpout, line, 255) > 0);

         if (pgp_error)
            break;

         waitpid(pgppid, status, 0);
         break;
   }

   /* close the stdio file descriptors */
   close(pgpin);
   close(pgpout);
   close(pgperr);
   pgp_bzero_stdio();

   /* lets remove that pgp_file right away! */
   remove(pgp_file);

   /* if we had encountered an error before... */
   if (pgp_error)
      return NULL;

   /* now lets read the encrypted message in */
   fp = fopen(pgp_file_asc, "rt");
   if (fp == NULL)
      return NULL;
   /* if we remove now, it will be cleaned up even if we crash */
   remove(pgp_file_asc);

   buffer = g_malloc0(sizeof(gchar));
   while (!feof(fp))
   {
      memset(line, 0, sizeof(line));
      fgets(line, 255, fp);
      buffer = g_realloc(buffer, strlen(buffer) + strlen(line) + 1);
      strcat(buffer, line);
   }

   fclose(fp);

   return buffer;
}

gchar *pgp_decrypt(gchar *text, gchar *passphrase, gint use_gnupg)
{
   gchar crypt_bin[50];   /* Binary used for PGP. */
   gchar command[512];    /* Command used to execute binary. */
   gchar line[256];
   gchar *buffer;
   FILE *fp;

   pgp_gen_filename();

   /* lets write our mesg to a temp file */
   fp = fopen(pgp_file_asc, "wt");
   if (fp == NULL)
      return NULL;

   /* lets make secure permissions */
   chmod(pgp_file_asc, 0600);

   fprintf(fp, "%s\r\n", text);
   fflush(fp);

   fclose(fp);

   /* Generate the command line. */
   if (!use_gnupg)
      strcpy(crypt_bin, "pgpv");
   else
      strcpy(crypt_bin, "gpg -d");

   g_snprintf(command, 512, "%s %s", crypt_bin, pgp_file_asc);

   if (use_gnupg)
   {
      strcat(command, " > ");
      strcat(command, pgp_file);
   }

   fp = popen(command, "wt");
   if (fp == NULL)
      return NULL;

   fprintf(fp, "%s\n", passphrase);  /* lets send our passphrase... */
   fflush(fp);

   /* pgp should now have decrypted our mesg */
   pclose(fp);

   fp = fopen(pgp_file, "rt");
   if (fp == NULL)
   {
      remove(pgp_file_asc);
      return NULL;
   }

   buffer = g_malloc0(sizeof(gchar));
   while (!feof(fp))
   {
      memset(line, 0, 255);
      fgets(line, 256, fp);
      buffer = g_realloc(buffer, strlen(buffer) + strlen(line) + 1);
      strcat(buffer, line);
   }

   fclose(fp);
   remove(pgp_file);
   remove(pgp_file_asc);

   return buffer;
}
