/*
 * dump-imports.c --
 *
 *      Operations to dump import hierarchies in a human readable format.
 *
 * Copyright (c) 1999 Frank Strauss, Technical University of Braunschweig.
 * Copyright (c) 1999 J. Schoenwaelder, Technical University of Braunschweig.
 *
 * See the file "COPYING" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * @(#) $Id: dump-imports.c,v 1.4 1999/10/05 06:31:01 strauss Exp $
 */

#include <stdio.h>
#include <string.h>

#include "smi.h"
#include "smidump.h"


typedef struct Imports {
    char *module;
    int  count;
} Imports;



static Imports *getImports(char *modulename, int *n)
{
    SmiImport *smiImport;
    Imports   *imports;
    int       i, size;
    
    for(smiImport = smiGetFirstImport(modulename), *n = 0;
	smiImport; smiImport = smiGetNextImport(smiImport)) {
	(*n)++;
    }

    size = (*n + 1) * sizeof(Imports);
    imports = xmalloc(size);
    memset(imports, 0, size);

    for(smiImport = smiGetFirstImport(modulename), *n = 0;
	smiImport; smiImport = smiGetNextImport(smiImport)) {
	
	for (i = 0; i < *n; i++) {
	    if (strcmp(smiImport->importmodule, imports[i].module) == 0) {
		break;
	    }
	}
	
	if (i == *n) {
	    imports[i].module = xstrdup(smiImport->importmodule);
	    if (imports[i].module) {
		imports[i].count = 0;
		(*n)++;
	    }
	}
	imports[i].count++;
    }

    return imports;
}



static void freeImports(Imports *imports, int n)
{
    int i;

    for (i = 0; i < n; i++) {
	xfree(imports[i].module);
    }

    xfree(imports);
}



static int printImports(char *modulename, char *prefix)
{
    Imports *imports;
    int     i, n, recurse = 0, done = 0;

    imports = getImports(modulename, &n);

    for (i = 0; i < n; i++) {
	char *newprefix;
	SmiImport *firstImport;

	firstImport = smiGetFirstImport(imports[i].module);
	recurse = (firstImport == NULL);
	smiFreeImport(firstImport);
	if (recurse) {
	    printf("%s  |\n", prefix);
	}
	printf("%s  +--%s [%d identifier%s]\n", prefix, imports[i].module, 
	       imports[i].count, imports[i].count > 1 ? "s" : "");
	newprefix = xmalloc(strlen(prefix)+10);
	strcpy(newprefix, prefix);
	if (i == n-1) {
	    strcat(newprefix, "   ");
	} else {
	    strcat(newprefix, "  |");
	}
	done = printImports(imports[i].module, newprefix);
	if (! recurse && done) {
	    if (i == n-1) {
		printf("%s   \n", prefix);
	    } else {
		printf("%s  |\n", prefix);
	    }
	}
	xfree(newprefix);
    }

    freeImports(imports, n);

    return recurse;
}



int dumpImports(char *modulename)
{
    SmiModule	 *smiModule;

    smiModule = smiGetModule(modulename);
    if (!smiModule) {
	fprintf(stderr, "smidump: cannot locate module `%s'\n", modulename);
	exit(1);
    }
    
    printf("# %s imports tree (generated by smidump "
	   VERSION ")\n\n", modulename);

    printf("%s\n", smiModule->name);
    printImports(smiModule->name, "");

    smiFreeModule(smiModule);

    return 0;
}
