/* File "hangul.h":
 * Handles conversion of KSC5601 hangul characters
 * and romanised hangul to internal format (and back). */

/* This file is part of Malaga, a system for Left Associative Grammars.
 * Copyright (C) 1995-1998 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/* variables ================================================================*/

extern bool_t convert_to_ksc; 
/* Indicates whether Hangul output is converted to KSC5601
 * (else output is converted to romanised Hangul). */

/* functions ================================================================*/

extern void init_hangul (void);
/* Initialise hangul conversions. */

extern void terminate_hangul (void);
/* Free memory used by hangul conversions. */

extern string_t roman_to_hancode (string_t roman_string);
/* Convert romanised Hangul string <roman_string> to Hancode.
 * The returned string remains valid until this function is called again. */

extern string_t hancode_to_roman (string_t hancode_string);
/* Convert Hancode string <hancode_string> to roman.
 * The returned string remains valid until this function is called again. */

extern string_t hancode_to_ksc (string_t hancode_string);
/* Convert Hancode string <hancode_string> to KSC5601 code.
 * The returned string remains valid until this function is called again. */

extern string_t ksc_to_hancode (string_t ksc_string);
/* Convert KSC5601 string <ksc_string> to Hancode.
 * the returned string remains valid until this function is called again. */

#undef ENCODED_STRING
#define ENCODED_STRING(string) \
  (roman_to_hancode (ksc_to_hancode (string)))
/* Convert text from external representation to internal representation.
 * This overrides the standard ENCODED_STRING definition in "basic.h". */

#undef DECODED_STRING
#define DECODED_STRING(string) \
  ((convert_to_ksc) ? (hancode_to_ksc (string)) : (hancode_to_roman (string)))
/* Convert text from internal representation to external representation.
 * This overrides the standard DECODED_STRING definition in "basic.h" */
