/* File "commands.c":
 * Support for a command line interpreter. */

/* This file is part of Malaga, a system for Left Associative Grammars.
 * Copyright (C) 1995-1998 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/* includes =================================================================*/
#include <stdio.h>
#include <stdlib.h>
#include <setjmp.h>
#include <signal.h>
#include "basic.h"
#include "input.h"
#include "files.h"

#undef GLOBAL
#define GLOBAL
#include "commands.h"

/* constants ================================================================*/

#define COMMAND_LINE_LENGTH 768 /* maximum command line length */

/* types ====================================================================*/

/* an alias */
typedef struct ALIAS_T
{
  struct ALIAS_T *next;
  string_t name;
  string_t line; /* command line which this alias stands for */
} alias_t;

/* variables ================================================================*/

LOCAL bool_t user_break_requested = FALSE;
/* flag that indicates that user has send a break signal */

LOCAL command_t **local_commands; 
/* local copy of command table for <help_command> */

LOCAL alias_t *aliases;

/* functions for user interrupts ============================================*/

LOCAL void set_user_break (int signal_number)
/* Signal handler for notification that user wants to interrupt a command. */
{
  user_break_requested = TRUE;
  signal (SIGINT, set_user_break);
}

/*---------------------------------------------------------------------------*/

GLOBAL void check_user_break (void)
/* Abort command execution if user has sent an interrupt signal. */
{
  if (user_break_requested)
    error ("user break");
}

/* functions for reading and executing commands =============================*/

LOCAL command_t *find_command (string_t command_name,
			       command_t *commands[])
/* Find the entry for <command_name> in <commands> and return it.
 * Return NULL if <command_name> can't be found. */
{
  long_t i;

  for (i = 0; commands[i] != NULL; i++) 
    {
      string_t names;

      names = commands[i]->names;
      while (*names != EOS)
	{
	  string_t name = parse_word (&names);
	  bool_t found_command = (strcmp_no_case (command_name, name) == 0);

	  free (name);
	  if (found_command) /* Execute the command. */
	    return commands[i];
	}
    }
  
  return NULL;
}

/*---------------------------------------------------------------------------*/

LOCAL void execute_command (string_t command_line, 
			    string_t command_type,
			    command_t *commands[],
			    bool_t use_aliases)
/* Execute <command_line> using <commands>.
 * <command_type> may be "command" or "option", used for error messages.
 * If there is no such command, print an error.
 * The function returns FALSE iff a quit-command has been executed. */
{
  string_t command_name;
  command_t *command;

  /* Divide command line into strings <command> and <arguments>. */
  parse_whitespace (&command_line);
  if (*command_line == EOS)
    return;
  command_name = parse_word (&command_line);

  user_break_requested = FALSE;
  local_commands = commands;
  command = find_command (command_name, commands);
  if (command == NULL && use_aliases)
  {
    alias_t *alias;

    for (alias = aliases; alias != NULL; alias = alias->next)
    {
      if (strcmp_no_case (alias->name, command_name) == 0)
      {
	free (command_name);
	parse_end (command_line); /* alias may have no arguments */
	command_line = alias->line;
	command_name = parse_word (&command_line);
	command = find_command (command_name, commands);
	break;
      }
    }
  }
  
  if (command == NULL)
    error ("unknown %s \"%s\", try \"help\"", command_type, command_name);

  free (command_name);

  /* Execute the command. */
  command->command (command_line);
}

/*---------------------------------------------------------------------------*/

GLOBAL void execute_command_file (string_t command_file, string_t line_header, 
				  command_t *commands[])
/* Execute commands in <command_file> using <commands>.
 * If <line_header> != NULL, command lines must start with <line_header>. */
{
  FILE *command_stream = fopen_save (command_file, "r");

  while (! feof (command_stream)) 
  {
    char command_line[COMMAND_LINE_LENGTH];
    string_t command_line_ptr;
    bool_t is_command_line;
    
    read_line (command_stream, command_line, COMMAND_LINE_LENGTH);
    command_line_ptr = command_line;
    
    if (*command_line_ptr == EOS)
      is_command_line = FALSE;
    else if (line_header == NULL)
      is_command_line = TRUE;
    else
    { /* Check if current line begins with <line_header>. */
      
      string_t header = parse_word (&command_line_ptr);
      
      is_command_line = FALSE;
      if (strcmp_no_case (header, line_header) == 0)
	    is_command_line = TRUE;
      else if (strcmp_no_case (header, "include:") == 0)
      {
	string_t include_file = parse_word (&command_line_ptr);
	string_t path = new_string (absolute_path (include_file, 
						   command_file));
	
	parse_end (command_line_ptr);
	execute_command_file (path, line_header, commands);
	free (path);
	free (include_file);
      }
      free (header);
    }
    if (is_command_line)
      execute_command (command_line_ptr, "command", commands, TRUE);
  }
  fclose_save (command_stream, command_file);
}

/*---------------------------------------------------------------------------*/

GLOBAL void command_loop (string_t prompt, command_t *commands[])
/* Read user commands, look for them in <commands> and execute them
 * until a command function returns FALSE. */
{
  char command_line[COMMAND_LINE_LENGTH];
  jmp_buf local_error_jump_point; /* where to jump after an error */
  jmp_buf *prev_error_jump_point; /* jump point active on entry */ 

  /* Install a new return point in case of an error. */
  prev_error_jump_point = error_jump_point;
  setjmp (local_error_jump_point);
  error_jump_point = &local_error_jump_point;
    
  signal (SIGINT, set_user_break);

  while (! quit_requested)
  { /* Read command line. */
    printf ("%s> ", prompt);
    read_line (stdin, command_line, COMMAND_LINE_LENGTH);
    if (feof (stdin))
      break;
    execute_command (command_line, "command", commands, TRUE);
    if (leave_command_loop)
    {
      leave_command_loop = FALSE;
      break;
    }
  }
  
  /* Reestablish old error jump point. */
  error_jump_point = prev_error_jump_point;
}

/* general commands =========================================================*/

LOCAL void print_help_summary (command_t *commands[])
/* Print names of all commands in <commands>. */
{
  int i;

  for (i = 0; commands[i] != NULL; i++)
  {
    long_t j;
    string_t names = commands[i]->names;
    string_t name = parse_word (&names);
    
    if (i > 0 && i % 5 == 0)
      printf ("\n");
    printf ("%s", name);
    for (j = strlen (name); j < 15; j++)
      printf (" ");
    
    free (name);
  }
  printf ("\n");
}

/*---------------------------------------------------------------------------*/

LOCAL void print_help (command_t *commands[], 
		       string_t command_type,
		       string_t arguments)
{ 
  string_t command_name, names, name;
  command_t *command;
  
  command_name = parse_word (&arguments);
  parse_end (arguments);
  
  command = find_command (command_name, commands);
  if (command == NULL)
    error ("unknown %s \"%s\"", command_type, command_name);
  
  names = command->names;
  
  /* Print full name. */
  name = parse_word (&names);
  printf ("%s \"%s\"", command_type, name);
  free (name);
  
  /* Print shortcuts. */
  while (*names != EOS)
  {
    name = parse_word (&names);
    printf (", \"%s\"", name);
    free (name);
  }
  
  printf (":\n%s", command->help);
  
  free (command_name);
}

/*---------------------------------------------------------------------------*/

LOCAL void do_help (string_t arguments)
/* Give help on commands. */
{
  if (*arguments == EOS)
  {
    printf ("commands available "
	    "(use \"help <command>\" to get help about <command>):\n");
    print_help_summary (local_commands);
    printf ("\noptions available "
	    "(use \"help option <option>\" to get help about <option>):\n");
    print_help_summary (options);
  }
  else
  {
    string_t opt_args, command;

    opt_args = arguments;
    command = parse_word (&opt_args);
    if (strcmp_no_case (command, "option") == 0 && *opt_args != EOS)
      print_help (options, "option", opt_args);
    else
      print_help (local_commands, "command", arguments);
    free (command);
  }
}

GLOBAL command_t help_command = 
{
  "help h ?", do_help,
  "Arguments:\n"
  "  <command> -- print help about <command>\n"
  "  option <option> -- print help about <option>\n"
  "  (none) -- get a list of all available commands and options\n"
};

/*---------------------------------------------------------------------------*/

LOCAL void do_quit (string_t arguments)
/* Quit the program. */
{
  parse_end (arguments);
  quit_requested = TRUE;
}

GLOBAL command_t quit_command =
{
  "quit q", do_quit,
  "Leave the program.\n"
  "Arguments: (none)\n"
};

/*---------------------------------------------------------------------------*/

LOCAL void do_get (string_t arguments)
/* Get setting for <arguments>. */
{
  if (*arguments == EOS)
  {
    long_t i;
    
    for (i = 0; options[i] != NULL; i++) 
      options[i]->command ("");
  }
  else
  {
    string_t option;
    
    option = parse_word (&arguments);
    parse_end (arguments);
    execute_command (option, "option", options, FALSE);
    free (option);
  }
}

GLOBAL command_t get_command =
{
  "get", do_get,
  "Query program settings.\n"
  "Arguments:\n"
  "  <option> -- print the setting of <option>\n"
  "  (none) -- print all settings\n"
};

/*---------------------------------------------------------------------------*/

LOCAL void do_set (string_t arguments)
/* Set an option. */
{
  string_t option_arguments, option;

  option_arguments = arguments;
  option = parse_word (&arguments);
    
  if (*arguments == EOS)
    error ("missing arguments to set \"%s\"", option);

  execute_command (option_arguments, "option", options, FALSE);
  free (option);
}

GLOBAL command_t set_command =
{
  "set", do_set,
  "Change program settings.\n"
  "Arguments:\n"
  "  set <option> <argument> -- change value of <option> to <argument>\n"
};

/*---------------------------------------------------------------------------*/

LOCAL command_t *set_commands[] = {&set_command, NULL};
/* the commands that can be called during initialisation */

GLOBAL void execute_set_commands (string_t file_name, string_t prefix)
/* Execute set commands in file <file_name> that are prefixed with <prefix>. */
{
  execute_command_file (file_name, prefix, set_commands);
}

/*---------------------------------------------------------------------------*/

LOCAL void do_alias (string_t arguments)
{
  alias_t *alias;
  
  if (*arguments == EOS)
  {
    if (aliases == NULL)
      printf ("alias: none defined\n");
    else 
    {
      for (alias = aliases; alias != NULL; alias = alias->next)
	printf ("alias \"%s\": %s\n", alias->name, alias->line);
    }
  }
  else
  {
    string_t name;
    alias_t **alias_ptr;

    name = parse_word (&arguments);
    
    /* Find the alias <name>. */
    alias_ptr = &aliases;
    while (*alias_ptr != NULL 
	   && strcmp_no_case ((*alias_ptr)->name, name) != 0)
      alias_ptr = &(*alias_ptr)->next;

    if (*arguments == EOS) /* Delete an alias. */
    {
      if (*alias_ptr == NULL)
	error ("alias \"%s\" not defined");

      /* remove the alias from the list and delete it. */
      alias = *alias_ptr;
      *alias_ptr = alias->next;
      free (alias->name);
      free (alias->line);
      free (alias);
    } 
    else /* Define an alias. */
    {
      string_t line;

      if (*alias_ptr == NULL) /* Create alias entry if alias doesn't exist. */
      {
	alias = new_mem (sizeof (alias_t));
	alias->name = new_string (name);
	*alias_ptr = alias;
      }
      else
	alias = *alias_ptr;

      line = parse_word (&arguments);
      parse_end (arguments);

      free (alias->line);
      alias->line = line;
    }

    free (name);
  }
}

GLOBAL command_t alias_option =
{
  "alias", do_alias,
  "Define command line abbreviations.\n"
  "Arguments:\n"
  "  <name> <line> -- define abbreviation <name> for <line>\n"
  "  <name> -- undefine abbreviation <name>\n"
};

/*---------------------------------------------------------------------------*/
