// ---------------------------------------------------------------------------
// - Strvec.cpp                                                              -
// - standard object library - string vector class implementation            -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Strvec.hpp"
#include "Buffer.hpp"
#include "Exception.hpp"

namespace aleph {

  // check that a character is in the c-string. 
  static bool match_break_sequence (char c, const char* str) {
    long size = String::length (str);
    // loop and compare
    for (long i = 0; i < size; i++)
      if (c == str[i]) return true;
    return false;
  }

  // create an empty string vector

  Strvec::Strvec (void) {
    d_size   = 0;
    d_length = 0;
    p_vector = nilp;
  }

  // create a string vector with an original size

  Strvec::Strvec (const long size) {
    if (size < 0) throw Exception ("size-error","in strvec constructor");
    d_size   = size;
    d_length = 0;
    p_vector = new String[d_size];
  }

  // copy constructor for this string vector

  Strvec::Strvec (const Strvec& that) {
    // copy arguments
    d_size   = that.d_length;
    d_length = that.d_length;
    p_vector = nilp;
    
    // create a new Strvec of strings and copy them
    if ((d_length > 0) && (that.p_vector != nilp)) {
      p_vector = new String[d_length];
      for (long i = 0; i < d_length; i++)
	p_vector[i] = that.p_vector[i];
    }
  }

  // return the class name

  String Strvec::repr (void) const {
    return "Strvec";
  }

  // assign a string vector to this one
  
  Strvec& Strvec::operator = (const Strvec& that) {
    // check againt equal equal
    if (this == &that) return *this;
    
    // delete old value
    delete [] p_vector;
    
    // copy arguments
    d_size   = that.d_length;
    d_length = that.d_length;
    p_vector = nilp;
    
    // create a new string vector of strings and copy them
    if ((d_length > 0) && (that.p_vector != nilp)) {
      p_vector = new String[d_size];
      for (long i = 0; i < d_length; i++)
      p_vector[i] = that.p_vector[i];
    }
    return *this;
  }
  
  // destroy this string vector
  
  Strvec::~Strvec (void) {
    delete [] p_vector;
  }

  // add a new element in this string vector
  
  void Strvec::add (const String& str) {
    // check if we have to resize the Strvec
    if (d_length + 1 >= d_size) {
      long size = (d_size <= 0) ? 1 : d_size * 2;
      String* vector = new String[size];
      for (long i = 0; i < d_length; i++)
	vector[i] = p_vector[i];
      delete [] p_vector;
      d_size   = size;
      p_vector = vector;
    }
  
    // set the string in this Strvec
    p_vector[d_length++] = str;
  }

  // set an string at a certain position in this Strvec
  // the old string is destroyed

  void Strvec::set (const long index, const String& str) {
    // check that we are bounded
    if (index >= d_length) 
      throw Exception ("index-error","in string vector set");
    
    // set the object
    p_vector[index] = str;
  }
  
  // get an string at a certain position

  String Strvec::get (const long index) const {
    // check that we are bounded
    if (index >= d_length) 
      throw Exception ("index-error","in strvec set");
    // get the object
    return p_vector[index];
  }

  // get the number of element in this string vector

  long Strvec::length (void) const {
    return d_length;
  }

  // check that a string exists in this vector

  bool Strvec::exists (const String& name) const {
    if (d_length == 0) return false;
    for (long i = 0; i < d_length; i++)
      if (p_vector[i] == name) return true;
    return false;
  }

  // return the index of a key in this string vector

  long Strvec::index (const String& key) const {
    for (long i = 0; i < d_length; i++)
      if (p_vector[i] == key) return i;
    throw Exception ("key-error","key not found",key);
  }
  
  // return the index of a key or -1

  long Strvec::lookup (const String& key) const {
    for (long i = 0; i < d_length; i++)
      if (p_vector[i] == key) return i;
    return -1;
  }

  // Split this string with a sequence of characters

  Strvec Strvec::split (const String& name, const String& sbrk) {
    Strvec result;
    Buffer buffer;
    
    // first thing first - do we have a nil string
    if (name.length () == 0) return result;
    
    // get a c-string representation
    char* data = name.toChar ();
    char* cptr = data;
    
    // fix the break sequence in case it is nil
    const char* cbrk = (sbrk.length () == 0) ? " \t\n" : sbrk.toChar ();
    
    // loop and accumulate - if a character match the break sequence
    // the buffer is inserted into the vector
    
    char c;
    buffer.reset ();
    while ((c = *data++) != nilp) {
      if (match_break_sequence (c,cbrk) == true) {
	result.add (buffer.toString());
	buffer.reset ();
	continue;
      }
      buffer.add (c);
    }
    
    // check if the buffer is not empty
    if (buffer.length () != 0) result.add (buffer.toString());
    
    // clean the break sequence and return
    if (sbrk.length () != 0) delete [] cbrk;
    delete [] cptr;
    return result;
  }

  // return the maximum string length in this vector

  long Strvec::maxlen (void) const {
    long result = 0;
    for (long i = 0; i < d_length; i++) {
      long len = p_vector[i].length ();
      if (len > result) result = len;
    }
    return result;
  }

  // return the minimum string length in this vector

  long Strvec::minlen (void) const {
    long result = 0;
    for (long i = 0; i < d_length; i++) {
      long len = p_vector[i].length ();
      if (len < result) result = len;
    }
    return result;
  }
}
