// ---------------------------------------------------------------------------
// - Queue.cpp                                                               -
// - standard object library - dynamic queue class implementation            -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Queue.hpp"
#include "Exception.hpp"

namespace aleph {
  
  // create a default queue

  Queue::Queue (void) {
    d_size  = 64;
    p_queue = new Object*[d_size];
    d_qidx  = 0;
    d_didx  = 0;
  }

  // create a queue with a predefined size

  Queue::Queue (const long size) {
    d_size  = size;
    p_queue = new Object*[d_size];
    d_qidx  = 0;
    d_didx  = 0;
  }

  // copy constructor for this queue

  Queue::Queue (const Queue& that) {
    d_size = that.d_size;
    p_queue = new Object*[d_size];
    d_qidx  = 0;
    d_didx  = 0;
    for (long i = that.d_didx; i < that.d_qidx; i++)
      queue (that.p_queue[i]);
  }
    
  // destroy this queue

  Queue::~Queue (void) {
    for (long i = d_didx; i < d_qidx; i++)
      Object::dref (p_queue[i]);
    delete [] p_queue;
  }

  // return the class name

  String Queue::repr (void) const {
    return "Queue";
  }

  // assign a queue to this one

  Queue& Queue::operator = (const Queue& that) {
    if (this == &that) return *this;
    // clean this queue
    for (long i = d_didx; i < d_qidx; i++)
      Object::dref (p_queue[i]);
    delete [] p_queue;
    // create the new one
    d_size  = that.d_size;
    p_queue = new Object*[d_size];
    d_qidx  = 0;
    d_didx  = 0;
    for (long i = that.d_didx; i < that.d_qidx; i++)
      queue (that.p_queue[i]);
    return *this;
  }

  // queue an object into this queue

  void Queue::queue (Object* object) {
    // check first the queue index
    if (d_qidx + 1 >= d_size) resize ();
    // queue the object
    p_queue[d_qidx++] = Object::iref (object);
  }

  // dequeue an object
  Object* Queue::dequeue (void) {
    // any object here
    if (d_didx == d_qidx) return nilp;
    // get the result
    Object* result = p_queue[d_didx++];
    // check if we reset the queue
    if (d_didx == d_qidx) {
      d_qidx = 0;
      d_didx = 0;
    }
    return result;
  }
  
  // return the number of elements in this queue

  long Queue::length (void) const {
    return (d_qidx - d_didx);
  }

  // return true if the queue is empty

  bool Queue::empty (void) const {
    if (d_didx == d_qidx) return true;
    return false;
  }

  // return true if the object is in the queue

  Object* Queue::get (const long index) const {
    long i = index + d_didx;
    if (i >= d_qidx) throw Exception ("bound-error",
				      "out of bound queue get index");
    return p_queue[i];
  }

  // flush this queue

  void Queue::flush (void) {
    while (empty () == false) {
      Object::dref (dequeue ());
    }
  }
      
  // resize this queue - if some objects have been dequeued we shift
  // the object and reset the indexes - if no room is available we resize
  // the array

  void Queue::resize (void) {
    if (d_didx == 0) {
      long size = d_size * 2;
      Object** array = new Object*[size];
      for (long i = 0; i < d_size; i++)
	array[i] = p_queue[i];
      delete [] p_queue;
      d_size  = size;
      p_queue = array;
    } else {
      long shift = d_didx;
      for (long i = d_didx; i < d_qidx; i++)
	p_queue[i - shift] = p_queue[i];
      d_didx = 0;
      d_qidx -= shift;
    }
  }
}
