// ---------------------------------------------------------------------------
// - Directory.cpp                                                           -
// - aleph:sio library - directory class implementation                      -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "cio.hxx"
#include "cdir.hxx"
#include "Interp.hpp"
#include "Method.hpp"
#include "System.hpp"
#include "Directory.hpp"
#include "Exception.hpp"

namespace aleph {

  // open the default directory

  Directory::Directory (void) {
    char* buffer = c_getcwd ();
    d_name       = buffer;
    p_handle = c_opendir (buffer);
    delete [] buffer;
    if (p_handle == nilp)
      throw Exception ("dir-error", "cannot open directory", d_name);
  }

  // open a directory by name

  Directory::Directory (const String& name) {
    d_name = name;
    char* buffer = name.toChar ();
    p_handle = c_opendir (buffer);
    delete [] buffer;
    if (p_handle == nilp) 
      throw Exception ("dir-error", "cannot open directory", name);
  }

  // destroy this directory 

  Directory::~Directory (void) {
    c_closedir (p_handle);
  }

  // return the class name

  String Directory::repr (void) const {
    return "Directory";
  }

  // return a list of this directory names

  List* Directory::getlist (void) {
    List* result = new List;
    while (true) {
      char* name = c_readdir (p_handle);
      if (name == nilp) break;
      result->insert (new String (name));
      delete [] name;
    }
    return result;
  }

  // return a list of this directory files

  List* Directory::getfiles (void) {
    List* result = new List;
    while (true) {
      char* name = c_readdir (p_handle);
      if (name == nilp) break;
      if (c_isfile (name) == true) result->insert (new String (name));
      delete [] name;
    }
    return result;
  }

  // return a list of the sub directories

  List* Directory::getdirs (void) {
    List* result = new List;
    while (true) {
      char* name = c_readdir (p_handle);
      if (name == nilp) break;
      if (c_isdir (name) == true) result->insert (new String (name));
      delete [] name;
    }
    return result;
  }

  // create a new directory by name

  void Directory::mkdir (const String& name) {
    // join the name
    String path = System::join (d_name, name);
    char*  buf  = path.toChar ();
    // try to create the directory
    if (c_mkdir (buf) == false) {
      delete [] buf;
      throw Exception ("directory-error", "cannot create directory", path);
    }
    delete [] buf;
  }

  // remove an empty directory by name
  void Directory::rmdir (const String& name) {
    // join the name
    String path = System::join (d_name, name);
    char*  buf  = path.toChar ();
    // try to create the directory
    
    if (c_rmdir (buf) == false) {
      delete [] buf;
      throw Exception ("directory-error", "cannot remove directory", path);
    }
    delete [] buf;
  }

  // create a new directory in a generic way

  Object* Directory::mknew (Vector* argv) {
    long argc = (argv == nilp) ? 0 : argv->length ();
    // open a default directory
    if (argc == 0) return new Directory;
    // open a directory by name
    if (argc == 1) {
      String val = argv->getstring (0);
      return new Directory (val);
    }
    throw Exception ("argument-error",
		     "too many argument with directory open");
  }

  // evaluate a directory member by name

  Object* Directory::eval (Interp* interp, Nameset* nset, const String& name) {
    return new Method (name, this);
  }

  // apply a directory member by name

  Object* Directory::apply (Interp* interp, Nameset* nset, const String& name,
			    Cons* args) {
    // evaluate the arguments
    Vector* argv = Vector::eval (interp, nset, args);
    long    argc = (argv == nilp) ? 0 : argv->length ();
 
    // check for 0 arguments
    if ((argc == 0) && (name == "get-name")) {
      delete argv;
      return new String (d_name);
    }
    if ((argc == 0) && (name == "get-list")) {
      delete argv;
      return getlist ();
    }
    if ((argc == 0) && (name == "get-files")) {
      delete argv;
      return getfiles ();
    }
    if ((argc == 0) && (name == "get-subdirs")) {
      delete argv;
      return getdirs ();
    }

    // check for one argument
    if ((argc == 1) && (name == "mkdir")) {
      String val = argv->getstring (0);
      delete argv;
      mkdir (val);
      return new Directory (System::join (d_name, val));
    }
    if ((argc == 1) && (name == "rmdir")) {
      String val = argv->getstring (0);
      delete argv;
      rmdir (val);
      return nilp;
    }

    // object method call
    Object* result = nilp;
    try {
      result =  Object::apply (interp, nset, name, argv);
    } catch (...) {
      delete argv;
      throw;
    }
    delete argv;
    return result;
  }
}
