// ct9file.cpp
//
// Copyright (C) 1999  Robert Barron, KA5WSS
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

#include "ct9file.h"
#include <ctype.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#define MAX_CT9_CALL_SIZE        16

/* Used to convert from CT file based band numbers to qso_data type band numbers. */
static unsigned char bands[] = {0,BAND_160M,BAND_80M,BAND_40M,BAND_20M,BAND_15M,
                                   BAND_10M, 0, 0, 0, 0, BAND_NOVICE, BAND_SATELLITE,
                                   BAND_PACKET, 0};
static unsigned char vhfbands[] = {0, BAND_50MHZ, BAND_144MHZ, BAND_222MHZ, BAND_432MHZ,
                                      BAND_902MHZ, BAND_1GHZ, BAND_2GHZ, BAND_3GHZ,
                                      BAND_5GHZ, BAND_10GHZ, BAND_24GHZ, BAND_LIGHT, 0};                                   
static unsigned char dxpedbands[] = {0,BAND_160M,BAND_80M,BAND_40M,BAND_30M,BAND_20M,BAND_18M,
                                       BAND_15M,BAND_12M,BAND_10M,BAND_SATELLITE,0};
static unsigned char modes[] = {0, CW_MODE, SSB_MODE, 0, FM_MODE};
static unsigned char contests[] = {CQ_WW_CONTEST,CQ_160M_CONTEST,CQ_WPX_CONTEST,
                                      ARRL_DX_CONTEST,ARRL_DX_CONTEST_DX,ARRL_10M_CONTEST,
                                      ARRL_160M_CONTEST,ARRL_SS_CONTEST,ARRL_VHF_QSO_PARTY,
                                      FD_CONTEST,WAE_CONTEST,DXPEDITION_MODE,CAL_QP_CONTEST,
                                      IARU_CONTEST,ALL_ASIAN_CONTEST,0};
#if 0
// Not needed until write support is enabled.                                      
static unsigned char CTContests[] = {0, 7, 3, 0, 2, 13, 5, 6, 1, 9, 12, 4, 10, 8, 11, 14, 0};
#endif
static char *radioNames[] = {"NONE", "TS440", "TS850", "TS940", "TS950", "IC725", "IC735", "IC751",
                              "IC761", "IC765", "IC781", "FT890", "FT990", "FT1000", NULL};
#define CT9_MAX_RADIO_NUM 13                              

RCt9File::RCt9File(char *name)
{
    RCt9File::open(name);
}

boolean RCt9File::open(char *name, int mode)
{
    char foo[3*1024];
    
#ifdef LINUX
    if (!RFile::open(name, mode))
        return FALSE;
#else
    // State ios::binary again just in case user does not.
    if (!RFile::open(name, mode | ios::binary))
        return FALSE;
#endif

    if (!(mode & ios::out))
    {        
        RFile::read((void *)&ct9header, sizeof(Ct9Header));
        RFile::read((void *)foo, 3*1024);

        strncpy(header.callsign, ct9header.call, MAX_CT9_CALL_SIZE);
        strncpy(header.name, ct9header.name, 32);
        strncpy(header.address, ct9header.address, 64);
        strncpy(header.city, ct9header.town, 32);
        strncpy(header.state, ct9header.state, 8);
        strncpy(header.zip, ct9header.zip, 16);

        header.stationNumber = (unsigned char)(ct9header.station + 1);
        strcpy(header.lastOperator, ct9header.last_op);
        if (ct9header.radio <= CT9_MAX_RADIO_NUM)
            strcpy(header.radio, radioNames[ct9header.radio]);
        header.tnc = ct9header.tnc;
        header.cwPort = ct9header.cw_port;
        header.dvkPort = ct9header.dvk_port;

        header.zone = (unsigned char)atoi(ct9header.zone);
        strncpy(header.grid, ct9header.my_grid, CT9_GRID_SIZE);
        strncpy(header.club, ct9header.club, CT9_CLUB_SIZE);

        if ((header.contest = contests[ct9header.contest]) == 0)
        {
            /* This contest is not supported. */
            RFile::close();
            return FALSE;
        }
        strncpy(header.contestName, ct9header.contest_title, 32);
        header.category = ct9header.category;

        strncpy(header.CQ_CW_Message, ct9header.cq_msg, 64);
        strncpy(header.Exchange_CW_Message, ct9header.ex_msg, 64);
        strncpy(header.QRZ_CW_Message, ct9header.qrz_msg, 64);
        strncpy(header.F6_CW_Message, ct9header.F6_text, 64);
        strncpy(header.F7_CW_Message, ct9header.F7_text, 64);
        strncpy(header.Dupe_CW_Message, ct9header.dupe_msg, 32);

        // Remember the COM: port information.
        for (int comm_count = 0; comm_count < 4; comm_count++)
        {
            header.comm[comm_count].device = ct9header.comm[comm_count].device;
            header.comm[comm_count].baud = ct9header.comm[comm_count].baud;
        }

        header.dvp.ingain = ct9header.dvp.ingain;
        header.dvp.outgain = ct9header.dvp.outgain;
        header.dvp.onair = ct9header.dvp.onair;
        header.dvp.mon = ct9header.dvp.mon;
        header.dvp.ram_disk = ct9header.dvp.ram_disk;
        header.dvp.ptt = ct9header.dvp.ptt;
        header.dvp.backcopy = ct9header.dvp.backcopy;
        header.dvp.spare = ct9header.dvp.spare;
        header.dvp.clipping_pt = ct9header.dvp.clipping_pt;
        header.dvp.rpt_delay = ct9header.dvp.rpt_delay;
        header.dvp.auto_space = ct9header.dvp.auto_space;
        header.dvp.disable_send_ser = ct9header.dvp.disable_send_ser;
        header.dvp.disable_send_call = ct9header.dvp.disable_send_call;
        
        header.flags.no_work_dupe = ct9header.flags.no_work_dupe;
        header.flags.m_stn = ct9header.flags.m_stn;
        header.flags.post = ct9header.flags.post;
        header.flags.beep = ct9header.flags.beep;
        header.flags.band_rate = ct9header.flags.band_rate;
        header.flags.sound = ct9header.flags.sound;
        header.flags.correct = ct9header.flags.correct;
        header.flags.cw_abbrev = ct9header.flags.cw_abbrev;
        header.flags.autosave = ct9header.flags.autosave;
        header.flags.nocompress = ct9header.flags.nocompress;
        header.flags.see_warc = ct9header.flags.see_warc;
        header.flags.print = ct9header.flags.print;
        header.flags.rpt = ct9header.flags.rpt;
        header.flags.kw_right = ct9header.flags.kw_right;
        
        // Don't bother with these fields unless they are specifically needed.
        if (header.contest == ARRL_SS_CONTEST)
        {
            header.ss.precedence = ct9header.ss.prec[0];
            header.ss.check = (unsigned char)atoi(ct9header.ss.chk);
            strncpy(header.ss.section, ct9header.sec, CT9_SECTION_SIZE);
        }
        else if (header.contest == FD_CONTEST)
        {
            header.fd.category = ct9header.fd.category;
            header.fd.transmitters = ct9header.fd.tx_count;

            header.fd.ep_bonus = ct9header.fd.ep_bonus;
            header.fd.pr_bonus = ct9header.fd.pr_bonus;
            header.fd.loc_bonus = ct9header.fd.loc_bonus;
            header.fd.info_bonus = ct9header.fd.info_bonus;
            header.fd.msg_bonus = ct9header.fd.msg_bonus;
            header.fd.sat_bonus = ct9header.fd.sat_bonus;
            header.fd.nat_bonus = ct9header.fd.nat_bonus;
            header.fd.w1aw_bonus = ct9header.fd.w1aw_bonus;
            header.fd.pkt_bonus = ct9header.fd.pkt_bonus;
            header.fd.tfc_bonus = ct9header.fd.tfc_bonus;
        }
    }
    
    location = 0;

    return TRUE;
}

boolean RCt9File::nextQso(qso_data *qso)
{
    ct9_log_data CTqso;
    int code;
    unsigned short sentRST;
             
    qso->init();
    if (RFile::read((void *)&CTqso, sizeof(ct9_log_data)))
    {
        /* Some CT files somehow got spaces in their info field.  Remove them. */
        /*for (char count = 0; count < strlen(CTqso.info); count++)
        if (!isalnum(CTqso.info[count]))
            CTqso.info[count] = NULL;*/
    
        qso->setCallsign(CTqso.call, strlen(CTqso.call));
        if (ct9header.utc_flag)
            // Is the one day difference still needed???
            qso->setTime(CTqso.time);             // Version 
        else
            /* WARNING!  The following kluge "fixes" a 7 hour time difference between */
            /*  v7 and v8 generated time stamps.  Not 100% sure this is correct! */
            qso->setTime(CTqso.time - (7 * 60 * 60));
            
        if (header.contest == ARRL_VHF_QSO_PARTY)
            qso->setBand(vhfbands[CTqso.band]);
        else if (header.contest == DXPEDITION_MODE)
            qso->setBand(dxpedbands[CTqso.band]);
        else if (header.contest == ARRL_10M_CONTEST)
            qso->setBand(BAND_10M);
        else
            qso->setBand(bands[CTqso.band]);

        qso->setFreq((unsigned long)CTqso.freq[0] / 100);
            
        qso->setMode(modes[CTqso.mode]);
        
        // Error check.  CT can allow an incorrect RST for a mode!
        if ((qso->getMode() != CW_MODE) && (CTqso.rst > 59))
            CTqso.rst = (short)(CTqso.rst / 10);
        else if ((qso->getMode() == CW_MODE) && (CTqso.rst < 100))
            CTqso.rst = (short)((CTqso.rst * 10) + 9);
        if ((qso->getMode() == CW_MODE) || (qso->getMode() == RTTY_MODE)
            || (qso->getMode() == PACKET_MODE))
            sentRST = 599;
        else
            sentRST = 59;
        qso->setQslSent(CTqso.qsl_sent);

        switch (header.contest) {
            case ARRL_SS_CONTEST:
                qso->setSerialNumber(CTqso.serial);
                code = CTqso.status & 0x44;
                if (code == 0)
                    qso->setSSPrecedence('Q');
                else if (code == 0x40)
                    qso->setSSPrecedence('B');
                else
                    qso->setSSPrecedence('A');
                qso->setSSCheck(CTqso.check);
                qso->setSection(CTqso.info, 8);
                break;
            case CQ_WW_CONTEST:
                qso->setReceivedRST(CTqso.rst);
                qso->setSentRST(sentRST);
                qso->setCqZone(CTqso.info);
                break;
            case CQ_WPX_CONTEST:
                qso->setReceivedRST(CTqso.rst);
                qso->setSentRST(sentRST);
                qso->setSerialNumber(CTqso.info);
            case IARU_CONTEST:
                qso->setReceivedRST(CTqso.rst);
                qso->setSentRST(sentRST);
                if (isdigit(CTqso.info[0]))
                    qso->setItuZone(CTqso.info);
                else
                    // The CT9 info field is 8 chars in length.  But CT9 limits the
                    // HQ Station field to 5 characters and sometimes places a string
                    // here without a terminating NULL!  So assume a max of 5 chars.
                    qso->setInfo(CTqso.info, 5);
                break;
            case FD_CONTEST:
                /* Another kluge.  FD information SOMETIMES appears in RST for some reason. */
                /*   If RST is zero get it from where it should be,  else get it from RST.  */
                if (CTqso.rst == 0)
                {
                    qso->setFdTransmitters(CTqso.fd.tx_count);        // Transmitter count.
                    qso->setFdCategory(CTqso.fd.category);            // A-E.
                }
                else
                {
                    qso->setFdTransmitters((char)(CTqso.rst & 0xf));
                    qso->setFdCategory((char)(CTqso.rst >> 8));
                }
                qso->setSection(CTqso.info, 5);         // ARRL section.
                break;
            case WAE_CONTEST:
                qso->setReceivedRST(CTqso.rst);
                qso->setSentRST(sentRST);
                qso->setInfo(CTqso.info, 8);
                qso->setQtcNum(CTqso.qtc_num);
                break;
            case CAL_QP_CONTEST:
                qso->setSerialNumber(CTqso.serial);
                qso->setInfo(CTqso.info, 8);
                break;
            default:
                qso->setReceivedRST(CTqso.rst);
                qso->setSentRST(sentRST);
                qso->setInfo(CTqso.info, 8);
        }
        location++;
        return TRUE;
    }
    else
        return FALSE;
}


boolean RCt9File::writeQso(qso_data *qso)
{
    // This capability is not yet implemented.
    qso = qso;          // Get rid of compiler warning
#if 0    
    ct9_log_data ct_qso;

    memset(&ct_qso, '\0', sizeof(ct_qso));

    strcpy(ct_qso.call, qso->getCallsign());
    return TRUE;
#endif
    return FALSE;    
}

boolean RCt9File::writeHeader(header_data header)
{
    // This capability is not yet implemented.
    header = header;            // Get rid of compiler warning.
#if 0
    Ct9Header ct_header;

    memset(&ct_header, '\0', sizeof(Ct9Header));

    strncpy(ct_header.version_str, "CT 9.15", 16);
    
    strncpy(ct_header.call, header.callsign, CT9_CALL_SIZE);
    strncpy(ct_header.name, header.name, CT9_NAME_SIZE);
    strncpy(ct_header.address, header.address, CT9_ADDRESS_SIZE);
    strncpy(ct_header.town, header.city, CT9_TOWN_SIZE);
    strncpy(ct_header.state, header.state, CT9_STATE_SIZE);
    strncpy(ct_header.zip, header.zip, CT9_ZIP_SIZE);

    if (((ct_header.contest = CTContests[header.contest]) == 0) && (header.contest != CQ_WW_CONTEST))
    {
        /* This contest is not supported. */
        RFile::close();
        return FALSE;
    }
    strncpy(ct_header.contest_title, header.contestName, 32);
    ct_header.category = header.category;

    sprintf(ct_header.zone, "%02d", header.zone);
    strncpy(ct_header.my_grid, header.grid, CT9_GRID_SIZE);
    strncpy(ct_header.club, header.club, CT9_CLUB_SIZE);

    // Remember the COM: port information.
    for (int comm_count = 0; comm_count < 4; comm_count++)
    {
        ct_header.comm[comm_count].device = header.comm[comm_count].device;
        ct_header.comm[comm_count].baud = header.comm[comm_count].baud;
    }
        
    // Don't bother with these fields unless they are specifically needed.
    if (header.contest == ARRL_SS_CONTEST)
    {
        ct_header.ss.prec[0] = header.ss.precedence;
        sprintf(ct_header.ss.chk, "%02d", header.ss.check);
        strncpy(ct_header.sec, header.ss.section, CT9_SECTION_SIZE);
    }
    else if (header.contest == FD_CONTEST)
    {
        ct_header.fd.category = header.fd.category;
        ct_header.fd.tx_count = header.fd.transmitters;
    }
        
    RFile::write((void *)&ct_header, sizeof(Ct9Header));    
    return TRUE;
#endif
    return FALSE;
}
