/* Zgv v3.1 - GIF, JPEG and PBM/PGM/PPM viewer, for VGA PCs running Linux.
 * Copyright (C) 1993-1999 Russell Marks. See README for license details.
 *
 * rcfile.c - config file handling.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>	/* for getopt() */
#include <vga.h>
#include <vgamouse.h>
#include "rc_config.h"
#include "zgv.h"	/* for ZGV_VER */
#include "rcfile.h"


#ifndef GLASTMODE
/* not sure if this is possible, but just in case...
 * (this is fairly conservative, of course)
 */
#define GLASTMODE	G1280x1024x16M
#endif


/* prototypes */
void getconfig(void);
void parseconfig(FILE *in);
int getmodenumber(char *tptr,int *mp,int bpp_force);
void getrgbval(int line,void *arg2,void *arg3);
void getbool(int line,void *arg2,void *arg3);
void getint(int line,void *arg2,void *arg3);
void getdouble(int line,void *arg2,void *arg3);
int strmousetype(char *arg);
void getmousearg(int line,void *arg2,void *arg3);
void getjis(int line,void *arg2,void *arg3);
void do_allmodesbad(int line,void *arg2,void *arg3);
void do_allmodesgood(int line,void *arg2,void *arg3);
void do_badmode(int line,void *arg1,void *arg3);
void do_goodmode(int line,void *arg1,void *arg3);
void do_startmode(int line,void *arg1,void *arg3);
void defaultcfg(void);
void findtoken(char **ptr);
int tokenlength(char *ptr);
int tokencompare(char *tptr, char *txt);
int modematch(int x, int y, int bpp);
int parsecommandline(int argc, char *argv[]);
void usage_help(void);

 

struct zgv_config cfg;

struct cfglookup_tag
  {
  char optname[32];
  void (*routine)(int,void *,void *);
  void *arg3;	/* arg1 is line no., arg2 is the after-token-pointer */
  };

struct cfglookup_tag cfglookup[]=
  {
  {"allmodesbad",	do_allmodesbad,	NULL},
  {"allmodesgood",	do_allmodesgood,NULL},
  {"badmode",		do_badmode,	NULL},
  {"bc_order_rev",	getbool,	&cfg.bc_order_rev},
  {"betterpgm",		getbool,	&cfg.betterpgm},
  {"black",		getrgbval,	&cfg.black},
  {"blockcursor",	getbool,	&cfg.blockcursor},
  {"brightness",	getint,		&cfg.brightness},
  {"center",		getbool,	&cfg.centreflag},
  {"centre",		getbool,	&cfg.centreflag},
  {"cleartext",		getbool,	&cfg.cleartext},
  {"contrast",		getdouble,	&cfg.contrast},
  {"dark",		getrgbval,	&cfg.dark},
  {"fakecols",		getbool,	&cfg.fakecols},
  {"force16fs",		getbool,	&cfg.force16fs},
  {"forgetoldpos",	getbool,	&cfg.forgetoldpos},
  {"fs16col",		getbool,	&cfg.fs16col},
  {"fs_startmode",	do_startmode,	&cfg.fs_startmode},
  {"fullsel",		getbool,	&cfg.fullsel},
  {"gnulitically_correct", getbool,	&cfg.stupid_gnu_verbosity},
  {"goodmode",		do_goodmode,	NULL},
  {"hicolmodes",	getbool,	&cfg.hicolmodes},
  {"hicontrol",		getbool,	&cfg.hicontrol},
  {"jpeg24bit",		getbool,	&cfg.jpeg24bit},
  {"jpegindexstyle",	getjis,		&cfg.jpegindexstyle},
  {"jpegspeed",		getint,		&cfg.jpegspeed},
  {"light",		getrgbval,	&cfg.light},
  {"linetext",		getbool,	&cfg.linetext},
  {"medium",		getrgbval,	&cfg.medium},
  {"mouse",		getbool,	&cfg.svgalib_mouse},
  {"mousekludge",	getbool,	&cfg.mousekludge},
  {"mousescale",	getint,		&cfg.mousescale},
  {"nodelprompt",	getbool,	&cfg.nodelprompt},
  {"onefile_progress",	getbool,	&cfg.onefile_progress},
  {"perfectindex",	getbool,	&cfg.perfectindex},
  {"revert",		getbool,	&cfg.revert},
  {"revert_orient",	getbool,	&cfg.revert_orient},
  {"scrollbar",		getbool,	&cfg.scrollbar},
  {"showxvpicdir",	getbool,	&cfg.showxvpicdir},
  {"shuffleslideshow",	getbool,	&cfg.shuffleslideshow},
  {"slowupdate",	getbool,	&cfg.slowupdate},
  {"smallfstext",	getbool,	&cfg.smallfstext},
  {"startmode",		do_startmode,	&cfg.videomode},
  {"tagged",		getrgbval,	&cfg.marked},
  {"tagtimeout",	getint,		&cfg.tag_timer},
  {"thicktext",		getbool,	&cfg.thicktext},
  {"viewer16col",	getbool,	&cfg.viewer16col},
  {"visual",		getbool,	&cfg.xvpic_index},
  {"vkludge",		getbool,	&cfg.vkludge},
  {"zoom",		getbool,	&cfg.zoom},
  {"",NULL,NULL}
  };



void getconfig()
{
FILE *in;
char cfgfile[1024];
int gotrcfile;

defaultcfg();

gotrcfile=0;
strcpy(cfgfile,getenv("HOME"));
strcat(cfgfile,"/.zgvrc");

if((in=fopen(cfgfile,"r"))!=NULL)
  gotrcfile=1;
else if((in=fopen("/etc/zgv.conf","r"))!=NULL)
  gotrcfile=1;

if(gotrcfile)
  {
  parseconfig(in);
  fclose(in);
  }
}


void fixconfig()
{
/* XXX should probably add many more to this, and perhaps give
 * error messages for some.
 */
 
/* don't allow funny values for this; zero is disallowed because
 * it could make it difficult to abort if all the files load very quickly.
 */
if(cfg.tag_timer<=0) cfg.tag_timer=1;

if(cfg.jpegspeed<1) cfg.jpegspeed=1;
if(cfg.jpegspeed>3) cfg.jpegspeed=3;

if(cfg.mousescale<1) cfg.mousescale=1;
}


void parseconfig(in)
FILE *in;
{
static char inpline[1024];
char *ptr;
int ln,f,c,inpc,found=0;

ln=0;
while(fgets(inpline,sizeof(inpline),in)!=NULL) 
  {
  ln++;
  if(inpline[strlen(inpline)-1]=='\n') inpline[strlen(inpline)-1]=0;
  if((ptr=strchr(inpline,'#'))!=NULL)
    *ptr=0;
  if(inpline[0])
    {
    found=0;
    ptr=inpline;
    findtoken(&ptr);
    inpc=*ptr;
    for(f=0;(c=cfglookup[f].optname[0]);f++)
      if(inpc==c && tokencompare(ptr,cfglookup[f].optname))
        {
        (*cfglookup[f].routine)(ln,ptr,cfglookup[f].arg3);
        found=1;
        break;
        }
    if(!found)
      {
      fprintf(stderr,"zgv: error in line %d of rc file.\n",ln);
      exit(1);
      }
    }
  }
}


/* get mode number from something like `640 480 8' -
 * returns 0 if it can't find one that matches, else 1.
 * (the mode number is put into *mp)
 *
 * If bpp_force is non-zero, it sets the mode's bpp, overriding the
 * value specified in the string.
 */
int getmodenumber(char *tptr,int *mp,int bpp_force)
{
int x,y,bpp,rtn;

/* skip past the current token */
tptr+=tokenlength(tptr);
findtoken(&tptr);
x=atoi(tptr);
tptr+=tokenlength(tptr);

findtoken(&tptr);
y=atoi(tptr);
tptr+=tokenlength(tptr);

if(bpp_force)
  bpp=bpp_force;
else
  {
  findtoken(&tptr);
  bpp=atoi(tptr);
  }

rtn=modematch(x,y,bpp);
if(rtn>=0)
  {
  *mp=rtn;
  return(0);
  }
else
  return(-1);
}


void getrgbval(int line,void *arg2,void *arg3)
{
char *tptr=(char *)arg2;
struct cfgrgb_tag *p=(struct cfgrgb_tag *)arg3;

/* skip past the current token */
tptr+=tokenlength(tptr);
findtoken(&tptr);
p->r=atoi(tptr);
tptr+=tokenlength(tptr);

findtoken(&tptr);
p->g=atoi(tptr);
tptr+=tokenlength(tptr);

findtoken(&tptr);
p->b=atoi(tptr);
}


void getbool(int line,void *arg2,void *arg3)
{
char *tptr=(char *)arg2;
int *bp=(int *)arg3;

/* skip past the current token */
tptr+=tokenlength(tptr);
findtoken(&tptr);
if(tokencompare(tptr,"on") || tptr[0]=='y')
  *bp=1;
else
  if(tokencompare(tptr,"off") || tptr[0]=='n')
    *bp=0;
  else
    fprintf(stderr,"zgv: error in line %d of rc file.\n",line),exit(1);
}


void getint(int line,void *arg2,void *arg3)
{
char *tptr=(char *)arg2;
int *ip=(int *)arg3;

/* skip past the current token */
tptr+=tokenlength(tptr);
findtoken(&tptr);
*ip=atoi(tptr);
}


void getdouble(int line,void *arg2,void *arg3)
{
char *tptr=(char *)arg2;
double *dp=(double *)arg3;

/* skip past the current token */
tptr+=tokenlength(tptr);
findtoken(&tptr);
*dp=atof(tptr);
}


void getjis(int line,void *arg2,void *arg3)
{
int *jisp=(int *)arg3;
getint(line,arg2,arg3);
if(*jisp<1) *jisp=1;
if(*jisp>3) *jisp=3;
}


void do_allmodesbad(int line,void *arg2,void *arg3)
{
int f;

for(f=0;f<256;f++) cfg.mode_allowed[f]=0;
}


void do_allmodesgood(int line,void *arg2,void *arg3)
{
int f;

/* we still disallow ones it hasn't got */
for(f=0;f<=GLASTMODE;f++)
  cfg.mode_allowed[f]=vga_hasmode(f)?1:0;
}


void do_badmode(int line,void *arg1,void *arg3)
{
char *ptr=(char *)arg1;
int f;

if(getmodenumber(ptr,&f,0)==-1)
  {
  fprintf(stderr,"zgv: mode not found on line %d of rc file.\n",line);
  exit(1);
  }
cfg.mode_allowed[f]=0;
}


void do_goodmode(int line,void *arg1,void *arg3)
{
char *ptr=(char *)arg1;
int f;

if(getmodenumber(ptr,&f,0)==-1)
  {
  fprintf(stderr,"zgv: mode not found on line %d of rc file.\n",line);
  exit(1);
  }
cfg.mode_allowed[f]=1;
}


/* used for both startmode and fs_startmode. */
void do_startmode(int line,void *arg1,void *arg3)
{
char *ptr=(char *)arg1;
int *mp=(int *)arg3;
int f;
int bpp_force=0;

/* fairly kludgey... */
if(mp==&(cfg.fs_startmode))
  bpp_force=8;			/* make sure file-sel modes are 8-bit */

if(getmodenumber(ptr,&f,bpp_force)==-1)
  {
  fprintf(stderr,"zgv: mode not found on line %d of rc file.\n",line);
  exit(1);
  }
*mp=f;
}


void defaultcfg()
{
int f;

/* it'll use 360x480x8 if you haven't got 640x480x8,
 * and 320x200x8 if you've locked that out.
 */
cfg.videomode=G640x480x256;
/* for selector, it's 640x480x8 if possible, otherwise 640x480x4. */
cfg.fs_startmode=G640x480x256;
cfg.zoom=0;
cfg.fullsel=0;
cfg.vkludge=0;
cfg.brightness=0;
cfg.jpeg24bit=1;	/* only if possible, of course */
cfg.jpegspeed=2;	/* slow int by default */
cfg.jpegindexstyle=2;	/* jpeg thumbnail speed/quality */
cfg.betterpgm=0;
cfg.centreflag=1;
cfg.blockcursor=0;
cfg.thicktext=0;
cfg.hicolmodes=0;	/* don't force high-colour mode test to true */
cfg.nodelprompt=0;	/* default to prompting on delete */
cfg.perfectindex=0;	/* make selector cols look nice, not index cols */
cfg.xvpic_index=1;	/* visual index */
cfg.onefile_progress=1;	/* progress report while loading file given as arg */
cfg.cleartext=0;	/* clear text screen on startup/exit? */
cfg.repeat_timer=0;	/* don't reread after a timeout */
cfg.tag_timer=4;	/* 4 seconds per tagged file */
cfg.force16fs=0;        /* don't force 16-colour mode for file selector */
cfg.revert=1;		/* auto-reset scaling to off between pics */
cfg.revert_orient=1;	/* auto-reset pic orientation between pics */
cfg.fakecols=1;		/* try to fake more greys/colours in 8-bit modes */
cfg.selecting=0;	/* no selection normally */
cfg.fs16col=0;		/* normally use mono in 16-col file selector */
cfg.viewer16col=0;	/* normally use mono in 16-col viewer mode */
cfg.echotagged=0;	/* echo tagged files on exit */
cfg.forgetoldpos=0;	/* don't return to old position in revisited dir */
cfg.linetext=0;		/* use old line-based text instead of bitmap fonts */
cfg.smallfstext=0;	/* use smaller font for filenames in file selector */
cfg.writefile=0;
cfg.slowupdate=0;
cfg.shuffleslideshow=0;
cfg.scrollbar=0;
cfg.mousekludge=1;
cfg.mousescale=16;	/* gives fair results for trackballs/mice I've tried */
cfg.stupid_gnu_verbosity=0;
cfg.hicontrol=1;	/* allow brightness/contrast in hi-col modes */
cfg.bc_order_rev=0;

cfg.contrast=(double)1;
cfg.black.r =cfg.black.g =cfg.black.b = 0;
cfg.dark.r  =cfg.dark.g  =cfg.dark.b  =20;
cfg.medium.r=cfg.medium.g=cfg.medium.b=30;
cfg.light.r =cfg.light.g =cfg.light.b =40;
cfg.marked.r=cfg.marked.g=cfg.marked.b= 0; cfg.marked.r+=30;
cfg.loop=0;
cfg.svgalib_mouse=0;
for(f=0;f<256;f++) cfg.mode_allowed[f]=0;
cfg.showxvpicdir=0;
for(f=0;f<=GLASTMODE;f++)
  cfg.mode_allowed[f]=vga_hasmode(f)?1:0;
cfg.cmd[0]=0;
cfg.errignore=0;
}


void findtoken(char **ptr)
{
while((*(*ptr)!=0)&&(strchr(" \t",*(*ptr))!=NULL))
  (*ptr)++;
}


int tokenlength(char *ptr)
{
int siz;

siz=0;
while((*ptr!=0)&&(strchr(" \t",*ptr)==NULL))
  {
  ptr++;
  siz++;
  }

return(siz);
}


/* returns 1 if equal, 0 otherwise */
int tokencompare(char *tptr,char *txt)
{
int tlen;

tlen=tokenlength(tptr);
if(tlen!=strlen(txt))
  return(0);
else
  if(strncmp(tptr,txt,tlen))
    return(0);
  else
    return(1);
}


/* returns mode number which matches x,y,bpp or -1 if none did.
 * put `-1' in x,y or bpp to wildcard them.
 *
 * maybe this routine should be somewhere else?
 */
int modematch(int x,int y,int bpp)
{
int numcols,f;
vga_modeinfo *vminfo;

if((bpp>24)||(bpp==2))
  /* they must have used numcols, not bpp. hmm, let 'em get away with it. */
  numcols=bpp;
else
  numcols=(1<<bpp);

/* we check 0 - 255 at the most */
for(f=0;f<256;f++)
  {
  vminfo=vga_getmodeinfo(f);
  if(vminfo!=NULL)
    if(((x==-1)||(x==vminfo->width))&&
       ((y==-1)||(y==vminfo->height))&&
       ((numcols==-1)||(numcols==vminfo->colors)))
      break;
  }

if(f<255)
  return(f);
else
  return(-1);
}


/* all booleans are actually toggles. parsing happens after rc file reading.
 */
int parsecommandline(int argc,char *argv[])
{
int done;
char buf[1024];

if(argc>=2)
  {
  if(strcmp(argv[1],"--help")==0)
    usage_help();	/* doesn't return */
  
  if(strcmp(argv[1],"--version")==0)
    printf("zgv " ZGV_VER "\n"),exit(0);
  }

done=0;
opterr=0;

do
  switch(getopt(argc,argv,"a:bcghijJ:klMm:r:RsS:ptTwz"))
    {
    case 'a':	/* alternative command */
      strcpy(cfg.cmd,optarg);
      if(strstr(cfg.cmd,"%s")==NULL)
        {
        fprintf(stderr,"zgv: "
          "need `%%s' filename placeholder in command string.\n");
        exit(1);
        }
      break;
    case 'b':	/* blockcursor */
      cfg.blockcursor=!cfg.blockcursor; break;
    case 'c':	/* centre */
      cfg.centreflag=!cfg.centreflag; break;
    case 'g':	/* betterpgm */
      cfg.betterpgm=!cfg.betterpgm; break;
    case 'h':	/* help on usage */
      usage_help();	/* doesn't return */
    case 'i':	/* errignore */
      cfg.errignore=!cfg.errignore; break;
    case 'j':	/* jpeg24bit */
      cfg.jpeg24bit=!cfg.jpeg24bit; break;
    case 'J':	/* jpeg speed (takes arg) */
      cfg.jpegspeed=atoi(optarg);
      break;
    case 'k':	/* vkludge */
      cfg.vkludge=!cfg.vkludge; break;
    case 'l':	/* loop in slideshow */
      cfg.loop=1; break;
    case 'M':    /* toggle mouse */
      cfg.svgalib_mouse=!cfg.svgalib_mouse; break;
    case 'm':	/* startup mode (takes arg) */
      {
      int vidnum;
      
      strcpy(buf,"dummytoken ");
      strcat(buf,optarg);
      if(getmodenumber(buf,&vidnum,0)==-1)
        {
        fprintf(stderr,"zgv: mode `%s' not found.\n",optarg);
        exit(1);
        }
      cfg.videomode=vidnum;
      }
      break;
    case 'r':	/* repeat timer (takes arg) */
      cfg.repeat_timer=atoi(optarg); break;
    case 'R':	/* randomise slideshow */
      cfg.shuffleslideshow=1; break;
    case 's':	/* selection - useful with pnmcut */
      cfg.selecting=1;
      break;
    case 'S':	/* slideshow timer */
      cfg.tag_timer=atoi(optarg); break;
    case 'p':   /* onefile_progress */
      cfg.onefile_progress=!cfg.onefile_progress; break;
    case 't':	/* thicktext */
      cfg.thicktext=!cfg.thicktext; break;
    case 'T':	/* echo tagged files on exit */
      cfg.echotagged=1; break;
    case 'w':	/* write file as PPM */
      cfg.writefile=1; break;
    case 'z':	/* zoom */
      cfg.zoom=!cfg.zoom; break;
    case '?':
      switch(optopt)
        {
        case 'm':
          fprintf(stderr,"zgv: "
            "the -m (startup mode) option takes the mode as an argument,\n"
            "  e.g.  zgv -m \"640 480 8\"\n");
          break;
        case 'a':
          fprintf(stderr,"zgv: "
            "the -a (alternative view) option takes the command to run"
            "as an argument,\n"
            "  e.g.  zgv -a \"djpeg -grey %%s | pgmtopbm | "
            "pbmtoascii -2x4 > /dev/tty8\"\n");
          break;
        case 'r':
  	  fprintf(stderr,"zgv: "
            "the -r (repeat timer) option takes an alarm() arg, "
            "i.e. the timeout\n"
            "  in seconds, or -1 for `continuous' update.\n");
          break;
        case 'J':
  	  fprintf(stderr,"zgv: "
            "the -J (jpeg speed) options needs a speed value; "
            "try `zgv -h' for details.\n");
          break;
        case 'S':
  	  fprintf(stderr,"zgv: "
            "the -S (slideshow timer) option takes number of "
            "seconds to wait\n"
            "  as an arg.\n");
          break;
        default:
          fprintf(stderr,"zgv: option `%c' not recognised.\n",optopt);
        }
      exit(1);
    case -1:
      done=1;
    }
while(!done);

return(argc-optind);
}


void usage_help()
{
printf("Zgv v" ZGV_VER
	" - (c) 1993-1999 Russell Marks for improbabledesigns.\n");
printf(
"
usage: zgv [--help] [--version] [-bcghijklpRstTwz] [-a \"command\"]
	[-J jpg_decmp_type] [-M mousetype] [-m modespec] [-r seconds]
	[-S seconds] [dirname | filename [... filenameN]]

	--help	give this usage help
	--version  report version number
	-a	alternative command (see info file or man page for details)
	-b	blockcursor toggle, gives outline cursor not tacky 3d effect
	-c	centre toggle, whether to centre pictures on the screen
		(defaults to on, so using `-c' will turn it off)
	-g	betterpgm toggle, whether to use 8-bit or high-colour modes
		for PGM files (off by default, i.e. 8-bit modes are used)
	-h	give this usage help
	-i	ignore errors with GIF and PNG files and display whatever
		part of the picture could be read anyway, if possible
	-j	jpeg24bit toggle (see documentation - defaults to on, so using
		`-j' will turn it off)
	-J	sets JPEG speed/quality tradeoff:
			1 = floating-point - slow but accurate
			2 = slow integer - faster but not as accurate
			3 = fast integer - fastest but less accurate still
			      (default is 2)
	-k	vkludge toggle, smooths a bit in 320x400 and 360x480 modes,
		and also when `zooming' a big picture down to screen size
	-l	loop forever in slideshow mode
	-M	enable mouse support (or if enabled in config file, disable);
		this uses svgalib's mouse configuration, see the
		libvga.config(5) man page for details
	-m	startup mode; the `modespec' should be in quotes but
		otherwise in the same format as the .zgvrc entry `startmode'
		e.g. \"640 480 8\"
	-p	turn off (or on) progress report when loading a single file
	-r	re-read and redisplay picture every `seconds' seconds
	-R	shuffle (randomise) slideshow order (same as `shuffleslideshow'
		cfg option)
	-s	print dimensions of section of picture being viewed to stdout
		on exit
	-S	set slideshow time-to-wait (same as `tagtimeout' cfg option)
	-t	thicktext toggle, makes the text a little bit clearer, kind of
	-T	echo tagged files on exit
	-w	write file as PPM (on stdout) rather than viewing it. (This
		only works if you run zgv on a single file, specified on the
		command-line.)
	-z	zoom toggle, i.e. whether to scale pictures to the screen size

      dirname			makes zgv start in a certain directory.
     filename			makes zgv view that one file only.
filename ... filenameN		view the N files as a slideshow.
(i.e. more than one file)

All options are processed after any .zgvrc or /etc/zgv.conf file.
");	/* end of printf() */

exit(0);
}
