/* Terraform - (C) 1997-2000 Robert Gasch (r.gasch@chello.nl)
 *  - http://212.187.12.197/RNG/terraform/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

// this file adapted from an -lgtkglmm demo by 
// Karl Nelson <kenelson@ece.ucdavis.edu>


#include "TFOpenGLArea.h" 
#include <iostream.h>
#include "GlobalSanityCheck.h"
#include "MathTrig.h"
#include "TFGui.h"
#include "TFOptions.h"
#include "TFWindowHandler.h"		// circular




/* 
 * TFOpenGLArea::TFOpenGLArea
 * 	This constructor makes the connections necessary to
 * 	handle exposures and to initialize the GL object.
 * 
 *  Note: We can NOT constuct or execute any GL commands
 *  at this point because we have not yet been realized.
 *  Therefore, initialization of the GL odjects must be
 *  done in a seperate realization function.  In this
 *  case we will connect it to initGL().
 *
 *  (the show signal happens before realization and should 
 *   not be used.)
 */ 
TFOpenGLArea::TFOpenGLArea(HeightField *HF) : Gtk_GLArea() 
{
	TFWindowHandler		*TFWH = TFGui::findTFWin (HF);

	SanityCheck::bailout ((!HF), "HF==NULL", "TFOpenGLArea::TFOpenGLArea");
	SanityCheck::bailout ((!TFWH), "TFWH==NULL", "TFOpenGLArea::TFOpenGLArea");

	// assign our real datapointers 
	p_HF = HF;
	p_HFD = TFWH->getHeightFieldDraw ();

	d_terrainList  = d_normalsList = 0;

	d_meshColumnStep = 1;
	d_meshRowStep = 1;
	d_doDrawNormals = GL_FALSE;

	//select events so that we can properly connect to them
	//GDK_POINTER_MOTION_HINT_MASK is used so that we 
	//don't get every point of motion
	this->set_events(GDK_EXPOSURE_MASK | GDK_BUTTON_PRESS_MASK | 
			 GDK_KEY_PRESS_MASK);
}


void TFOpenGLArea::realize_impl()
{
	// we must call the real realize_impl so the object is marked realized
	Gtk_GLArea::realize_impl();

	// then we can call our init
	initGL();
}


gint TFOpenGLArea::expose_event_impl (GdkEventExpose *event)
{
	// we are CPU intensive so we should wait for the last event
	if (event->count>0) 
		return TRUE;

	drawGL();
	return TRUE;
}


gint TFOpenGLArea::configure_event_impl (GdkEventConfigure *event)
{
	if (make_current())
      		setupGL ();
	return TRUE;
}


// Function has not been test (as far as I know)
TFOpenGLArea::~TFOpenGLArea()
{
	if (glIsList(d_terrainList)==GL_TRUE)
		glDeleteLists(d_terrainList,1);
}


/* 
 * TFOpenGLArea::initGL
 * 	This function creates the OpenGL list that will be used 
 * 	to draw the planet.  We cannot do any GL rendering until
 * 	the widget is realized including creating lists.
 */
void TFOpenGLArea::initGL()
{
	int 		i;
	int 		j;
	int 		level;
	int		limi, limj;
	PTYPE		elv;


	// In order to allow the window to be shrunk we will 
	// reset the minumum size to something smaller now 
	// that the inital allocation has been completed.
	//  (this works, but is this what they intended??)
	set_usize(100,100);

	// make_current must be called before performing GL operations
	// and GL code should only be called if make_current returns true
	if (make_current())
		{
		setupGL();
        
                d_terrainList = glGenLists(1);
		if (!d_terrainList)
			return;

		buildMesh ();
                glNewList(d_terrainList, GL_COMPILE);

                //glBegin(GL_TRIANGLES); 
		limi = p_HF->getWidth()-d_meshColumnStep;
		limj = p_HF->getHeight()-d_meshRowStep;
                for(j = 0; j < limj; j += d_meshColumnStep) 
		    {
		    glPolygonMode (GL_BACK, GL_LINE);
                    glBegin(GL_TRIANGLE_STRIP); 
                    for(i = 0; i < limi; i += d_meshRowStep)
			{                
			elv = p_HF->getEl(i,j);
			glColor3f (elv, elv, elv);
			glNormal3fv (p_meshNormals[i+d_meshColumnStep][j]);
			glVertex3fv (p_meshVertices[i+d_meshColumnStep][j]);
			//printf ("%f, %f, %f\n", (p_meshVertices[i+d_meshColumnStep][j][0]), (p_meshVertices[i+d_meshColumnStep][j][1]), (p_meshVertices[i+d_meshColumnStep][j][2])); fflush (stdout);

			elv = p_HF->getEl(i,j);
			glColor3f (elv, elv, elv);
			glColor3f (p_HF->getEl(i,j), p_HF->getEl(i,j), p_HF->getEl(i,j));
			glNormal3fv (p_meshNormals[i][j]);
			glVertex3fv (p_meshVertices[i][j]);
			//printf ("%f, %f, %f\n", (p_meshVertices[i+d_meshColumnStep][j][0]), (p_meshVertices[i+d_meshColumnStep][j][1]), (p_meshVertices[i+d_meshColumnStep][j][2])); fflush (stdout);
                        }
                    glEnd();    
                    }
                glEndList();       

		// draw normals 
		d_normalsList = glGenLists(1);
		glNewList(d_normalsList, GL_COMPILE);
		for (i=0; i<limi; i+=d_meshColumnStep) 
		    {
		    for (j=0; j<limj; j+=d_meshRowStep) 
			{
			glPushMatrix();
			glTranslatef(p_meshVertices[i][j][0], 
				     p_meshVertices[i][j][1], 
				     p_meshVertices[i][j][2]);
			glScalef(0.1, 0.1, 0.1);
			glBegin(GL_LINES);
			glColor3f(1.0, 0.0, 0.0);
			glVertex3f(0.0, 0.0, 0.0);
			glColor3f(0.0, 1.0, 0.0);
			glVertex3fv(p_meshNormals[i][j]);
			glEnd();
			glScalef(1.0, 1.0, 1.0);
			glPopMatrix();
			}
		    }
  		glEndList();
		fflush (stdout);
		}
}


void TFOpenGLArea::buildMesh ()
{
	int 		i, j;
	int		h,w;
	float		max;
	float		ys = TFOptions::s_yscale;
	GLfloat*** 	mesh_facetnormals;
	GLfloat 	u[3], v[3], n[3], l;

	w = p_HF->getWidth();
	h = p_HF->getHeight();
	max = p_HF->getMax ();

	// allocate space for the mesh 
	p_meshVertices	 	= new GLfloat**[w]; 
	p_meshNormals	 	= new GLfloat**[w]; 
	mesh_facetnormals 	= new GLfloat**[w]; 
	for (i = 0; i < w; i++) 
	    {
	    p_meshVertices[i]  	= new GLfloat*[h];
	    p_meshNormals[i]    = new GLfloat*[h];
	    mesh_facetnormals[i] = new GLfloat*[h];
	    for (j = 0; j < h; j++) 
		{
		p_meshVertices[i][j]	= new GLfloat[3];
		p_meshNormals[i][j]	= new GLfloat[3];
		mesh_facetnormals[i][j] = new GLfloat[3];
		}
	    }

	// fill out the vertex array and calculate facet normals for the mesh
	for (i = 0; i < w - 1; i++) 
	    {
	    for (j = 0; j < h - 1; j++) 
		{
		// assign the data to vertices. Some of the vertices will be
		// overwritten in subsequent iterations of the loop, but this is
		// OK, since they will be identical. 
		p_meshVertices[i][j][0] = ((GLfloat)i / (GLfloat)w) - 0.5; 
		p_meshVertices[i][j][1] = (p_HF->getEl(i,j)/max)*ys - (0.5*ys);
		p_meshVertices[i][j][2] = ((GLfloat)j / (GLfloat)h) - 0.5;

		p_meshVertices[i][j+1][0] = p_meshVertices[i][j][0];
		p_meshVertices[i][j+1][1] = (p_HF->getEl(i,j+1)/max)*ys - (0.5*ys);
		p_meshVertices[i][j+1][2] = ((GLfloat)(j+1) / (GLfloat)h) - 0.5;

		p_meshVertices[i+1][j][0] = ((GLfloat)(i+1) / (GLfloat)w)- 0.5;
		p_meshVertices[i+1][j][1] = (p_HF->getEl(i+1,j)/max)*ys - (0.5*ys);
		p_meshVertices[i+1][j][2] = p_meshVertices[i][j][2];

		// get two vectors to cross 
		u[0] = p_meshVertices[i][j+1][0] - p_meshVertices[i][j][0];
		u[1] = p_meshVertices[i][j+1][1] - p_meshVertices[i][j][1];
		u[2] = p_meshVertices[i][j+1][2] - p_meshVertices[i][j][2];

		v[0] = p_meshVertices[i+1][j][0] - p_meshVertices[i][j][0];
		v[1] = p_meshVertices[i+1][j][1] - p_meshVertices[i][j][1];
		v[2] = p_meshVertices[i+1][j][2] - p_meshVertices[i][j][2];

		// get the normalized cross product 
		MathTrig::normalVectorNormalized (u, v, n);
      
		// put the facet normal in the i, j position for later 
		// averaging with other normals. 
		mesh_facetnormals[i][j][0] = n[0];
		mesh_facetnormals[i][j][1] = n[1];
		mesh_facetnormals[i][j][2] = n[2];
		}
	    }

	// fill in the last vertex & it's facet normal 
	p_meshVertices[i][j][0] = ((GLfloat)i / (GLfloat)w) - 0.5; 
	p_meshVertices[i][j][1] = (p_HF->getEl(i,j) / max) - 0.5;
	p_meshVertices[i][j][2] = ((GLfloat)j / (GLfloat)h) - 0.5;

	mesh_facetnormals[i][j][0] = n[0];
	mesh_facetnormals[i][j][1] = n[1];
	mesh_facetnormals[i][j][2] = n[2];

	// calculate normals for the mesh 
	for (i = 1; i < w - 1; i++) 
	    {
	    for (j = 1; j < h - 1; j++) 
		{
		// average all the neighboring normals. 
		n[0] = mesh_facetnormals[i-1][j-1][0];
		n[1] = mesh_facetnormals[i-1][j-1][1];
		n[2] = mesh_facetnormals[i-1][j-1][2];

		n[0] += mesh_facetnormals[i][j-1][0];
		n[1] += mesh_facetnormals[i][j-1][1];
		n[2] += mesh_facetnormals[i][j-1][2];

		n[0] += mesh_facetnormals[i-1][j][0];
		n[1] += mesh_facetnormals[i-1][j][1];
		n[2] += mesh_facetnormals[i-1][j][2];

		n[0] += mesh_facetnormals[i][j][0];
		n[1] += mesh_facetnormals[i][j][1];
		n[2] += mesh_facetnormals[i][j][2];
      
		l = (GLfloat)sqrt(n[0] * n[0] + n[1] * n[1] + n[2] * n[2]);
		p_meshNormals[i][j][0] = n[0] /= l;
		p_meshNormals[i][j][1] = n[1] /= l;
		p_meshNormals[i][j][2] = n[2] /= l;
		}
	    }

	// fill in the normals on the top/bottom edge of the mesh (simply
	// copy the one below/above it). 
	for (i = 0; i < w; i++) 
		{
		p_meshNormals[i][0][0] = p_meshNormals[i][1][0];
		p_meshNormals[i][0][1] = p_meshNormals[i][1][1];
		p_meshNormals[i][0][2] = p_meshNormals[i][1][2];

		p_meshNormals[i][h-1][0] = p_meshNormals[i][h-2][0];
		p_meshNormals[i][h-1][1] = p_meshNormals[i][h-2][1];
		p_meshNormals[i][h-1][2] = p_meshNormals[i][h-2][2];
		}

	// fill in the normals on the left/right edge of the mesh (simply
	// copy the one right/left of it).
	for (j = 0; j < h; j++) 
		{
		p_meshNormals[0][j][0] = p_meshNormals[1][j][0];
		p_meshNormals[0][j][1] = p_meshNormals[1][j][1];
		p_meshNormals[0][j][2] = p_meshNormals[1][j][2];

		p_meshNormals[w-1][j][0] = p_meshNormals[w-2][j][0];
		p_meshNormals[w-1][j][1] = p_meshNormals[w-2][j][1];
		p_meshNormals[w-1][j][2] = p_meshNormals[w-2][j][2];
		}

	// printf("%d x %d x %.0f mesh\n", w, h, max);

	// free the mesh data & facet normals 
	for (i = 0; i < w; i++) 
		{
		for (j = 0; j < h; j++)
			delete mesh_facetnormals[i][j];
		delete [] mesh_facetnormals[i];
		}
	delete [] mesh_facetnormals;
}



/*
 * TFOpenGLArea::draw
 *  These functions draw the objects
 */
void TFOpenGLArea::drawTerrain ()
{
	glPushMatrix();
	glCallList(d_terrainList);

	if (d_doDrawNormals)
		glCallList(d_normalsList);

	glPopMatrix();
}


/*
 * TFOpenGLArea::setupGL
 * This function sets up the projection matrix and
 * configures the OpenGL rendering engine. 
 *
 * This is called once to set up the initial matrix and
 * then again if the window is resized.
 */
void TFOpenGLArea::setupGL()
{
	static GLfloat pos[4] = {5.0, 5.0, 10.0, 0.0 };

	// Setup the rendering engine
	glClearColor (0.0, 0.0, 0.0, 0.0); // frame buffer clears should be to black
	glLightfv (GL_LIGHT0, GL_POSITION, pos);
	glEnable (GL_CULL_FACE);
	glEnable (GL_LIGHTING);
	glEnable (GL_LIGHT0);
	glEnable (GL_DEPTH_TEST);
	glEnable (GL_NORMALIZE);

	glViewport(0, 0, this->width(), this->height());
	//glTranslatef( 0.0, 1, -1.0 );

	// Create the projection matrix
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	if (this->width() > this->height()) 
		{
		GLfloat w = (GLfloat) this->width() / (GLfloat) this->height();
		glFrustum( -w, w, -1.0, 1.0, 2.0, 20.0 );
		}
	else 
		{
		GLfloat h = (GLfloat) this->height() / (GLfloat) this->width();
		glFrustum( -1.0, 1.0, -h, h, 2.0, 20.0 );
		}

	// Go to the place to start the rendering
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
	glTranslatef( 0.0, 0, -2.5 );
}



/*
 * TFOpenGLArea::drawGL
 * This function just calls the OpenGL list to display the
 *  object and then commits the rendering scene.
 */
void TFOpenGLArea::drawGL() 
{
	static GLfloat white[]={1.0,1.0,1.0,1.0};
	//static GLfloat blue[]= {0.3,0.3,1.0,1.0};
	//static GLfloat green[]={0.3,1.0,0.3,1.0};
	//static GLfloat red[]=  {1.0,0.3,0.3,1.0};

	// start opengl rendering, you must put all opengl calls
	// inside make_current
	if (make_current())
		{
		glMatrixMode(GL_MODELVIEW);
		glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

		glPushMatrix();
		glMaterialfv (GL_FRONT, GL_AMBIENT_AND_DIFFUSE, white);
		drawTerrain ();
		glPopMatrix();

		glFlush ();
		}
   
	swap_buffers();
}


/*
 * button_press_event_impl
 */
gint TFOpenGLArea::button_press_event_impl(GdkEventButton *event) 
{
	return TRUE;
}


/*
 * key_press_event_impl
 */
gint TFOpenGLArea::key_press_event_impl(GdkEventKey *event)
{
	char 	key = event->keyval;
	int	i = p_HF->getWidth(),
		j = p_HF->getHeight();

	printf("Keypress was %d\n",key);
	switch (key) 
		{ 
		case 'x':  //GLUT_KEY_UP:
			d_meshRowStep += 1;
			if (d_meshRowStep >= j)
				d_meshRowStep = j - 1;
			printf("d_meshRowStep = %d\n", d_meshRowStep);
			break;

		case 'y': //GLUT_KEY_DOWN:
			d_meshRowStep -= 1;
			if (d_meshRowStep <= 0)
				d_meshRowStep = 1;
			printf("d_meshRowStep = %d\n", d_meshRowStep);
			break;

		case 'z': //GLUT_KEY_RIGHT:
			d_meshColumnStep += 1;
			if (d_meshColumnStep >= i)
				d_meshColumnStep = i - 1;
			printf("d_meshColumnStep = %d\n", d_meshColumnStep);
			break;

		case 'a': //GLUT_KEY_LEFT:
			d_meshColumnStep -= 1;
			if (d_meshColumnStep <= 0)
				d_meshColumnStep = 1;
			printf("d_meshColumnStep = %d\n", d_meshColumnStep);
			break;

		case 'h':
			printf("meshview help\n\n");
			printf("f            -  Filled mesh\n");
			printf("w            -  Wireframe mesh\n");
			printf("l            -  Toggle lighting\n");
			printf("n            -  Toggle normals\n");
			printf("c            -  Toggle colormap\n");
			printf("left arrow   -  Decrease column step\n");
			printf("right arrow  -  Increase column step\n");
			printf("down arrow   -  Decrease row step\n");
			printf("up arrow     -  Increase row step\n");
			printf("escape or q  -  Quit\n\n");
			break;

		case 'f':
			glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
			break;

		case 'w':
			glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
			break;

		case 'l':
		if (glIsEnabled(GL_LIGHTING))
			glDisable(GL_LIGHTING);
		else
			glEnable(GL_LIGHTING);
		break;

		case 'n':
			d_doDrawNormals = !d_doDrawNormals;
			break;

		case 'c':
			if (glIsEnabled(GL_COLOR_MATERIAL))
				glDisable(GL_COLOR_MATERIAL);
			else
				glEnable(GL_COLOR_MATERIAL);
			break;

		case 'q':
		case 27:
			exit(0);
			break;
		}

	initGL ();
	return TRUE;
}

