/* Terraform - (C) 1997-2000 Robert Gasch (r.gasch@chello.nl)
 *  - http://212.187.12.197/RNG/terraform/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "TFDialogRender.h"
#include "TFOptions.h"
#include "GlobalDefs.h"
#include "GlobalTrace.h"
#include "GlobalSanityCheck.h"
#include "HeightFieldDraw.h"
#include "HeightFieldExport.h"
#include "RenderOptions.h"
#include "TFWindowHandler.h"			// Circular dependency
#include "terraform.h"



/*
 *  constructor: initialize all data members. 
 */
TFDialogRender::TFDialogRender (HeightField *HF, HeightFieldDraw *HFD)
	 : GuiDialogOAC (_("Terraform Render Options"), NULL),
	   d_frame1 (_("Camera/Viewpoint")), 
	   d_frame2 (_("Scene Options")),
	   d_frame3 (_("POV Render Options")),
	   d_cbKeepFiles (_("Keep Files")),
	   d_cbAutoRender (_("Auto Render")),
	   d_cbHaveClouds (_("Clouds")),
	   d_cbHaveFog (_("Fog")),
	   d_cbSmooth (_("Smooth")),
	   d_cbhalfYscale (_("Half Y Scale"))
{
	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "+++ TFDialogRender\n");

	SanityCheck::bailout ((!HF), "HF==NULL", "TFDialogRender::TFDialogRender");

	p_da = new GuiBufferedDrawingArea ();

	p_vb2 = new Gtk_VBox (TRUE, 2);
	p_vb3 = new Gtk_VBox (TRUE, 2);
	p_vb1_1 = new Gtk_VBox (TRUE, 2);
	p_vb1_2 = new Gtk_VBox (TRUE, 2);
	p_hb1 = new Gtk_HBox (FALSE, 2);
	p_hb1_0 = new Gtk_HBox (FALSE, 5);
	p_hb1_1 = new Gtk_HBox (FALSE, 5);
	p_hb1_2 = new Gtk_HBox (TRUE, 5);
	p_hb1_3 = new Gtk_HBox (TRUE, 5);
	p_hb1_4 = new Gtk_HBox (TRUE, 5);
	p_hb1_5 = new Gtk_HBox (TRUE, 5);
	p_hb2_1 = new Gtk_HBox (TRUE, 5);
	p_hb2_2 = new Gtk_HBox (TRUE, 5);
	p_hb2_3 = new Gtk_HBox (TRUE, 5);
	p_hb2_4 = new Gtk_HBox (TRUE, 5);
	p_hb3_1 = new Gtk_HBox (TRUE, 5);
	p_hb3_2 = new Gtk_HBox (TRUE, 5);
	p_hb3_3 = new Gtk_HBox (TRUE, 5);
	p_hb3_4 = new Gtk_HBox (TRUE, 5);

	p_lblTitle1 = new Gtk_Label (_("Position"));
	p_lblTitle2 = new Gtk_Label (_("Scene"));
	p_lblTitle3 = new Gtk_Label (_("PovRay"));
	p_lblCamera = new Gtk_Label (_("Camera"));
	p_lblLookAt = new Gtk_Label (_("Look At"));
	p_lblElvSealevel = new Gtk_Label (_("Observe Sealevel"));
	p_lblElvOffset = new Gtk_Label (_("Elevation Offset"));
	p_lblSkySettings = new Gtk_Label (_("Sky Settings"));
	p_lblWaterClarity = new Gtk_Label (_("Water Clarity"));
	p_lblEnvironment = new Gtk_Label (_("Time of Day & North Direction"));
	p_lblRoughness = new Gtk_Label (_("Terrain Roughness"));
	p_lblfileName = new Gtk_Label (_("POV template file"));
	p_lblwidth = new Gtk_Label (_("Render width"));
	p_lbllightning = new Gtk_Label (_("Lighting Level"));
	p_lblkeepFiles = new Gtk_Label (_("Output Options"));

	p_adjWaterClarity = new Gtk_Adjustment (0.5, 0, 1, 0.01, 10, 0);
	p_adjwidth = new Gtk_Adjustment (400, 100, (TFOptions::s_large ? 5000 : 1000), 1, 10, 0);
	p_adjTimeOfDay = new Gtk_Adjustment (12, 0, 24, 0.1, 10, 10);
	p_adjNorthDir = new Gtk_Adjustment (0, 0, 360, 1, 10, 10);
	p_adjVerticalOffset = new Gtk_Adjustment (0.1, -1, 1, 0.01, 10, 10);

	p_spbTimeOfDay = new Gtk_SpinButton (*p_adjTimeOfDay);
	p_spbNorthDir = new Gtk_SpinButton (*p_adjNorthDir);
	p_spbVerticalOffset = new Gtk_SpinButton (*p_adjVerticalOffset);

	p_hsWaterClarity = new Gtk_HScale (*p_adjWaterClarity);
	p_hswidth = new Gtk_HScale (*p_adjwidth);

	p_cmbLighting = new Gtk_Combo ();
	p_cmbTemplateFile = new Gtk_Combo ();
	p_glstLighting = NULL;
	p_glstTemplateFile = NULL;

	buildDialogWindow ();
	p_da->setUsize (100, 100);

	Gtk_Main *gtkMain = Gtk_Main::instance();
	while (gtkMain->events_pending())
		gtkMain->iteration ();

	setHFobjs (HF, HFD);
	sprintf (this->p_windowTitle, _("Terraform Render Options: %s"), HF->getName());
	this->set_title (this->p_windowTitle);
	setValues ();

	//connect_to_method (d_notebook.switch_page, this, 
	//		   &TFDialogRender::handleDAExposeEvent);
	connect_to_method (p_da->expose_event, this, 
			   &TFDialogRender::handleDAExposeEvent);
	connect_to_method (p_da->button_press_event, this,
                        &TFDialogRender::daButtonPressedCallback);

	p_da->setUsize (100, 100);
}


/*
 *  destructor: clean up
 */
TFDialogRender::~TFDialogRender ()
{
	delete p_vb1_1;
	delete p_vb1_2;
	delete p_vb2;
	delete p_vb3;
	delete p_hb1;
	delete p_hb1_0;
	delete p_hb1_1;
	delete p_hb1_2;
	delete p_hb1_3;
	delete p_hb1_4;
	delete p_hb1_5;
	delete p_hb2_1;
	delete p_hb2_2;
	delete p_hb2_3;
	delete p_hb2_4;
	delete p_hb3_1;
	delete p_hb3_2;
	delete p_hb3_3;
	delete p_hb3_4;

	delete p_lblTitle1;
	delete p_lblTitle2;
	delete p_lblTitle3;
	delete p_lblCamera;
	delete p_lblLookAt;
	delete p_lblElvSealevel;
	delete p_lblElvOffset;
	delete p_lblSkySettings;
	delete p_lblWaterClarity;
	delete p_lblEnvironment;
	delete p_lblRoughness;
	delete p_lblfileName;
	delete p_lblwidth;
	delete p_lbllightning;
	delete p_lblkeepFiles;

	delete p_adjWaterClarity;
	delete p_adjwidth;
	delete p_adjTimeOfDay;
	delete p_adjNorthDir;
	delete p_adjVerticalOffset;

	delete p_spbTimeOfDay;
	delete p_spbNorthDir;
	delete p_spbVerticalOffset;

	delete p_hsWaterClarity;
	delete p_hswidth;

	delete p_cmbLighting;
	delete p_cmbTemplateFile;

	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "--- TFDialogRender\n");
}


/*
 *  setHFobjs: set the HF objects we operate on and generate a scaled 
 * 	down version for the preview 
 */
void TFDialogRender::setHFobjs (HeightField *HF, HeightFieldDraw *HFD)
{
	SanityCheck::bailout ((!HF), "HF==NULL", "TFPreviewDialog::setHFobjs");
	SanityCheck::bailout ((!HFD), "HFD==NULL", "TFPreviewDialog::setHFobjs");

	GuiColormap 	*cMap = HFD->getColormap();
	HeightFieldOps	*HFO = NULL;

	if (HF!=p_HF || HFD!=p_HFD)
		{
		p_HF = HF;
		p_HFD = HFD;
		HFO = new HeightFieldOps (p_HF);
		}

	if (p_HFPreview)
		delete p_HFPreview;
	if (p_HFDPreview)
		delete p_HFDPreview;
	p_HFPreview = HFO->hfResize (100, 100);
	p_HFDPreview = new HeightFieldDraw (p_HFPreview, this->p_da);
	p_HFDPreview->setMode ("0");
	p_HFDPreview->setColormap (cMap);
	p_HFDPreview->draw ();
	d_hfTimestamp = p_HF->getUpdateTime();

	if (HFO)
		delete HFO;
}


/* 
 * previewUpToDate(): check if the preview HF is current (ie: if the master 
 *	HF has not been updated)
 */
void TFDialogRender::previewUpToDate ()
{
	SanityCheck::bailout ((p_HF->getUpdateTime()<d_hfTimestamp), "p_HF->getUpdateTime()<d_hfTimestamp", "TFPreviewDialog::previewUpToDate");

	if (p_HF->getUpdateTime() == d_hfTimestamp)
		return;

	setHFobjs (p_HF, p_HFD);
}


/*
 *  exposeDAEventCallback: handle the expose event for the drawing area 
 */
int TFDialogRender::handleDAExposeEvent (GdkEventExpose *e)
{
	//printf ("HERE: %d, %d\n", p1, p2);

	static int	lock = 0;

	Gtk_Main *gtkMain = Gtk_Main::instance();
	while (gtkMain->events_pending())
		gtkMain->iteration ();

	if (lock)
		return 0;

	lock = 1;
	//if (p2 == 0)
	p_HFDPreview->draw ();
	lock = 0;

	return 0;
}


/*
 *  daButtonPressedCallback: update the preview p_da and set camera/viewpoint
 *	coordinates according to which button was pressed
 */
int TFDialogRender::daButtonPressedCallback (GdkEventButton *e)
{
	int	w = this->p_da->width(),
		h = this->p_da->height();
	float	x, y, z;
	char	buf[256];

	x = e->x/w;
	if (d_cbObserveSealevel.get_active())
		y = p_HFPreview->getElSea (e->x, e->y);
	else
		y = p_HFPreview->getEl (e->x, e->y);
	z = e->y/h;

	y += p_spbVerticalOffset->get_value_as_float();

	if (e->button == 1)
		{
		sprintf (buf, "%f", x);
		string s1 (buf);
		d_enPosX.set_text (s1);
		sprintf (buf, "%f", y);
		string s2 (buf);
		d_enPosY.set_text (s2);
		sprintf (buf, "%f", z);
		string s3 (buf);
		d_enPosZ.set_text (s3);
		}
	else
	if (e->button == 3)
		{
		sprintf (buf, "%f", x);
		string s4 (buf);
		d_enLookAtX.set_text (s4);
		sprintf (buf, "%f", y);
		string s5 (buf);
		d_enLookAtY.set_text (s5);
		sprintf (buf, "%f", z);
		string s6 (buf);
		d_enLookAtZ.set_text (s6);
		}

	return 0;
}


/*
 *  fillVBox: fill the dialog's VBox
 */
void TFDialogRender::fillVBox ()
{
	this->get_vbox()->pack_start (d_notebook, TRUE, TRUE, 0);
	d_frame1.set_shadow_type (GTK_SHADOW_ETCHED_IN);
	d_frame2.set_shadow_type (GTK_SHADOW_ETCHED_IN);
	d_frame3.set_shadow_type (GTK_SHADOW_ETCHED_IN);
	d_notebook.append_page (d_frame1, *p_lblTitle1);
	d_notebook.append_page (d_frame2, *p_lblTitle2);
	d_notebook.append_page (d_frame3, *p_lblTitle3);
	d_notebook.show ();

	buildPositionFrame ();
	buildEnvironmentFrame ();
	buildPOVOptionsFrame ();
}


/*
 *  buildDialogWindow:
 */
void TFDialogRender::buildDialogWindow ()
{
	this->set_usize (400, 300);
	this->set_title (this->p_windowTitle);

	this->get_vbox()->set_border_width (2);

	this->setupVBox ();
	fillVBox ();
	fillActionArea ();

	this->show ();
}


/*
 *  buildPositionFrame: frame for the global draw options
 */
void TFDialogRender::buildPositionFrame ()
{
	p_vb1_1->pack_start (*p_da, FALSE, FALSE, 1);
	p_da->show ();
	p_hb1_0->pack_start (*p_vb1_1, FALSE, FALSE, 5);
	p_vb1_1->show ();
	p_hb1->pack_start (*p_hb1_0, FALSE, FALSE, 1);
	p_hb1_0->show ();

	p_hb1_2->pack_start (*p_lblCamera, TRUE, TRUE, s_HBOff);
	p_lblCamera->show ();
	p_hb1_2->pack_start (d_enPosX, TRUE, TRUE, 1);
	d_enPosX.show ();
	p_hb1_2->pack_start (d_enPosY, TRUE, TRUE, 1);
	d_enPosY.show ();
	p_hb1_2->pack_start (d_enPosZ, TRUE, TRUE, 1);
	d_enPosZ.show ();
	p_hb1_2->show ();
	p_vb1_2->pack_start (*p_hb1_2, TRUE, TRUE, s_VBOff);
	p_hb1_2->show();

	p_hb1_3->pack_start (*p_lblLookAt, TRUE, TRUE, s_HBOff);
	p_lblLookAt->show ();
	p_hb1_3->pack_start (d_enLookAtX, TRUE, TRUE, 1);
	d_enLookAtX.show ();
	p_hb1_3->pack_start (d_enLookAtY, TRUE, TRUE, 1);
	d_enLookAtY.show ();
	p_hb1_3->pack_start (d_enLookAtZ, TRUE, TRUE, 1);
	d_enLookAtZ.show ();
	p_vb1_2->pack_start (*p_hb1_3, TRUE, TRUE, s_VBOff);
	p_hb1_3->show();

	p_hb1_4->pack_start (*p_lblElvSealevel, TRUE, TRUE, s_HBOff);
	p_lblElvSealevel->show ();
	d_cbObserveSealevel.set_active (TRUE);
	p_hb1_4->pack_start (d_cbObserveSealevel, TRUE, TRUE, 1);
	d_cbObserveSealevel.show ();
	p_vb1_2->pack_start (*p_hb1_4, TRUE, TRUE, s_VBOff);
	p_hb1_4->show();

	p_hb1_5->pack_start (*p_lblElvOffset, TRUE, TRUE, s_HBOff);
	p_lblElvOffset->show ();
	p_spbVerticalOffset->set_digits (2);
	p_hb1_5->pack_start (*p_spbVerticalOffset, TRUE, TRUE, 1);
	p_spbVerticalOffset->show ();
	p_vb1_2->pack_start (*p_hb1_5, TRUE, TRUE, s_VBOff);
	p_hb1_5->show();

	p_hb1->pack_start (*p_vb1_2, TRUE, TRUE, 1);
	p_vb1_2->show ();
	
	d_frame1.add (*p_hb1);
	p_hb1->show ();
	d_frame1.show ();
}


/*
 *  buildEnvironmentFrame: frame for scene environment 
 */
void TFDialogRender::buildEnvironmentFrame ()
{
	Gtk_HBox *hb0 = new Gtk_HBox (TRUE, 5);

	p_lblRoughness->set_justify (GTK_JUSTIFY_LEFT);
	p_hb2_1->pack_start (*p_lblRoughness, TRUE, TRUE, s_HBOff);
	p_lblRoughness->show ();
	d_cbSmooth.set_active (TRUE);
	hb0->pack_start (d_cbSmooth, TRUE, TRUE, s_HBOff);
        d_cbSmooth.show ();
	d_cbhalfYscale.set_active (TRUE);
	hb0->pack_start (d_cbhalfYscale, TRUE, TRUE, s_HBOff);
        d_cbhalfYscale.show ();
	p_hb2_1->pack_start (*hb0, TRUE, TRUE, s_HBOff);
	hb0->show ();
	p_vb2->pack_start (*p_hb2_1, TRUE, TRUE, s_VBOff);
	p_hb2_1->show ();

	Gtk_HBox *hb1 = new Gtk_HBox (TRUE, 5);

	p_lblSkySettings->set_justify (GTK_JUSTIFY_LEFT);
	p_hb2_2->pack_start (*p_lblSkySettings, TRUE, TRUE, s_HBOff);
	p_lblSkySettings->show ();
	d_cbHaveClouds.set_active (TRUE);
	hb1->pack_start (d_cbHaveClouds, TRUE, TRUE, s_HBOff);
        d_cbHaveClouds.show ();
	d_cbHaveFog.set_active (TRUE);
	hb1->pack_start (d_cbHaveFog, TRUE, TRUE, s_HBOff);
        d_cbHaveFog.show ();
	p_hb2_2->pack_start (*hb1, TRUE, TRUE, s_HBOff);
	hb1->show ();
	p_vb2->pack_start (*p_hb2_2, TRUE, TRUE, s_VBOff);
	p_hb2_2->show ();

	Gtk_HBox *hb2 = new Gtk_HBox (TRUE, 5);

	p_lblEnvironment->set_justify (GTK_JUSTIFY_LEFT);
	p_hb2_4->pack_start (*p_lblEnvironment, TRUE, TRUE, s_HBOff);
	p_lblEnvironment->show ();
	p_spbTimeOfDay->set_digits (1);
	hb2->pack_start (*p_spbTimeOfDay, TRUE, TRUE, s_HBOff);
	p_spbTimeOfDay->show ();
	p_spbNorthDir->set_digits (0);
	hb2->pack_start (*p_spbNorthDir, TRUE, TRUE, s_HBOff);
	p_spbNorthDir->show ();
	p_hb2_4->pack_start (*hb2, TRUE, TRUE, s_HBOff);
	hb2->show ();
	p_vb2->pack_start (*p_hb2_4, TRUE, TRUE, s_VBOff);
	p_hb2_4->show ();

	p_lblWaterClarity->set_justify (GTK_JUSTIFY_LEFT);
	p_hb2_3->pack_start (*p_lblWaterClarity, TRUE, TRUE, s_HBOff);
	p_lblWaterClarity->show ();
        p_hsWaterClarity->set_update_policy (GTK_UPDATE_DELAYED);
        p_hsWaterClarity->set_digits (2);
        p_hsWaterClarity->set_draw_value (TRUE);
	p_hb2_3->pack_start (*p_hsWaterClarity, TRUE, TRUE, s_HBOff);
        p_hsWaterClarity->show ();
	p_vb2->pack_start (*p_hb2_3, TRUE, TRUE, s_VBOff);
	p_hb2_3->show ();

	d_frame2.add (*p_vb2);
	p_vb2->show ();
	d_frame2.show ();
}


/*
 *  buildPOVOptionsFrame: build the frame for the POVRay options
 */
void TFDialogRender::buildPOVOptionsFrame ()
{
	p_lblfileName->set_justify (GTK_JUSTIFY_LEFT);
	p_hb3_1->pack_start (*p_lblfileName, TRUE, TRUE, s_HBOff);
	p_lblfileName->show ();

	p_glstTemplateFile = g_list_append (p_glstTemplateFile, const_cast<char*>("tf_land1.pov"));
	p_glstTemplateFile = g_list_append (p_glstTemplateFile, const_cast<char*>("tf_land2.pov"));
	p_glstTemplateFile = g_list_append (p_glstTemplateFile, const_cast<char*>("tf_land3.pov"));
	p_glstTemplateFile = g_list_append (p_glstTemplateFile, const_cast<char*>("custom"));
	p_cmbTemplateFile->set_popdown_strings (p_glstTemplateFile);
	p_cmbTemplateFile->get_entry()->set_text (TF_DEFAULT_POV_FILE);
	p_cmbTemplateFile->get_entry()->set_editable (FALSE);
	p_cmbTemplateFile->show ();
	p_hb3_1->pack_start (*p_cmbTemplateFile, TRUE, TRUE, s_HBOff);
	p_vb3->pack_start (*p_hb3_1, TRUE, TRUE, s_VBOff);
	p_hb3_1->show ();

	p_lbllightning->set_justify (GTK_JUSTIFY_LEFT);
	p_hb3_2->pack_start (*p_lbllightning, TRUE, TRUE, s_HBOff);
	p_lbllightning->show ();

	p_glstLighting = g_list_append (p_glstLighting, const_cast<char*>(_(LIGHTING_PLASTIC)));
	p_glstLighting = g_list_append (p_glstLighting, const_cast<char*>(_(LIGHTING_REALSKY)));
	p_glstLighting = g_list_append (p_glstLighting, const_cast<char*>(_(LIGHTING_SUNBEAMS_FAST)));
	p_glstLighting = g_list_append (p_glstLighting, const_cast<char*>(_(LIGHTING_SUNBEAMS_SLOW)));
	p_cmbLighting->set_popdown_strings (p_glstLighting);
	p_cmbLighting->get_entry()->set_text (_(LIGHTING_PLASTIC));
	p_cmbLighting->get_entry()->set_editable (FALSE);
	p_cmbLighting->show ();
	p_hb3_2->pack_start (*p_cmbLighting, TRUE, TRUE, s_HBOff);
	p_vb3->pack_start (*p_hb3_2, TRUE, TRUE, s_VBOff);
	p_hb3_2->show ();

	p_lblwidth->set_justify (GTK_JUSTIFY_LEFT);
	p_hb3_3->pack_start (*p_lblwidth, TRUE, TRUE, s_HBOff);
	p_lblwidth->show ();
	p_adjwidth->set_value (400);
	p_hb3_3->pack_start (*p_hswidth, TRUE, TRUE, s_HBOff);
        p_hswidth->set_digits (0);
	p_hswidth->show ();
	p_vb3->pack_start (*p_hb3_3, TRUE, TRUE, s_VBOff);
	p_hb3_3->show ();

	Gtk_HBox *hb = new Gtk_HBox (TRUE, 5);

	p_lblkeepFiles->set_justify (GTK_JUSTIFY_LEFT);
	p_hb3_4->pack_start (*p_lblkeepFiles, TRUE, TRUE, s_HBOff);
	p_lblkeepFiles->show ();
	d_cbKeepFiles.set_active (FALSE);
	hb->pack_start (d_cbKeepFiles, TRUE, TRUE, s_HBOff);
	d_cbKeepFiles.show ();
	d_cbAutoRender.set_active (TRUE);
	hb->pack_start (d_cbAutoRender, TRUE, TRUE, s_HBOff);
	d_cbAutoRender.show ();
	p_hb3_4->pack_start (*hb, TRUE, TRUE, s_HBOff);
	hb->show();
	p_vb3->pack_start (*p_hb3_4, TRUE, TRUE, s_VBOff);
	p_hb3_4->show ();

	d_frame3.add (*p_vb3);
	p_vb3->show ();
	d_frame3.show ();

}


/*
 *  getValues: get all the RenderOptions values from the widgets 
 */
void TFDialogRender::getValues ()
{
	RenderOptions	*rOpt = p_HF->getRenderOptions ();
	char		*s;
	string 		entry;
	int		t;

	if (!rOpt)
		{
		rOpt = new RenderOptions (p_HF->getName());
		p_HF->setRenderOptions (rOpt);
		}

	rOpt->setSmooth (d_cbSmooth.get_active());
	rOpt->setHalfYScale (d_cbhalfYscale.get_active());
	rOpt->setKeepFiles (d_cbKeepFiles.get_active());
	rOpt->setHaveClouds (d_cbHaveClouds.get_active());
	rOpt->setHaveFog (d_cbHaveFog.get_active());
	rOpt->setSize ((UI)p_adjwidth->get_value(), 
		       (UI)(p_adjwidth->get_value()*0.75));

	entry = p_cmbLighting->get_entry()->get_text();
	if (entry == _(LIGHTING_PLASTIC))
		t = 0;
	else 
	if (entry == _(LIGHTING_REALSKY))
		t = 1;
	else
	if (entry == _(LIGHTING_SUNBEAMS_FAST))
		t = 2;
	else
	if (entry == _(LIGHTING_SUNBEAMS_SLOW))
		t = 3;
	else
		SanityCheck::bailout ("Lighting Level out of range", "TFDialogRender::getValues");
	rOpt->setLightingLevel (t);

	//rOpt->setquality ();
	rOpt->setTimeOfDay (p_adjTimeOfDay->get_value());
	rOpt->setNorthDir (p_adjNorthDir->get_value());
	rOpt->setCameraLocation (atof(d_enPosX.get_text().c_str()), 
				atof(d_enPosY.get_text().c_str()),
				atof(d_enPosZ.get_text().c_str()));
	rOpt->setLookAtLocation (atof(d_enLookAtX.get_text().c_str()), 
				atof(d_enLookAtY.get_text().c_str()),
				atof(d_enLookAtZ.get_text().c_str()));

	entry = p_cmbTemplateFile->get_entry()->get_text();
	s = (char *)malloc (256*sizeof(char));
	sprintf (s, "%s/%s", TF_DATADIR, entry.c_str());
	//printf ("%s\n", s);
	rOpt->setPOVFile (s);
	rOpt->setHFFile (p_HF->getName());

	// taken from global options (TFOptions.h)
	rOpt->setSealevel (TFOptions::s_sealevel);
	rOpt->setWaterClarity (p_adjWaterClarity->get_value());
}


/*
 *  setValues: set the widget values from the RenderOptions calss
 */
void TFDialogRender::setValues ()
{
	RenderOptions	*rOpt = p_HF->getRenderOptions ();
	UI		uit, t; 
	float		x, y, z;
	string		s;
	char		buf[80];

	if (!rOpt)
		{
		rOpt = new RenderOptions (p_HF->getName());
		p_HF->setRenderOptions (rOpt);
		}

	d_cbSmooth.set_active (rOpt->getSmooth());
	d_cbhalfYscale.set_active (rOpt->getHalfYScale());
	d_cbKeepFiles.set_active (rOpt->getKeepFiles());
	d_cbHaveClouds.set_active (rOpt->getHaveClouds());
	d_cbHaveFog.set_active (rOpt->getHaveFog());
	rOpt->getSize (&uit, NULL);
	p_adjwidth->set_value ((float)uit);

	t = rOpt->getLightingLevel ();
	if (t == 0)
		p_cmbLighting->get_entry()->set_text (_(LIGHTING_PLASTIC));
	else if (t == 1)
		p_cmbLighting->get_entry()->set_text (_(LIGHTING_REALSKY));
	else if (t == 2)
		p_cmbLighting->get_entry()->set_text (_(LIGHTING_SUNBEAMS_FAST));
	else if (t == 3)
		p_cmbLighting->get_entry()->set_text (_(LIGHTING_SUNBEAMS_SLOW));
	else 
		SanityCheck::bailout ("Lighting Level out of range", "TFDialogRender::getValues");

	p_adjWaterClarity->set_value (rOpt->getWaterClarity());
	p_adjTimeOfDay->set_value (rOpt->getTimeOfDay());
	p_adjNorthDir->set_value (rOpt->getNorthDir());

	rOpt->getCameraLocation (&x, &y, &z);
	sprintf (buf, "%f", x);
	string s1 (buf);
	d_enPosX.set_text (s1);
	sprintf (buf, "%f", y);
	string s2 (buf);
	d_enPosY.set_text (s2);
	sprintf (buf, "%f", z);
	string s3 (buf);
	d_enPosZ.set_text (s3);

	rOpt->getLookAtLocation (&x, &y, &z);
	sprintf (buf, "%f", x);
	string s4 (buf);
	d_enLookAtX.set_text (s4);
	sprintf (buf, "%f", y);
	string s5 (buf);
	d_enLookAtY.set_text (s5);
	sprintf (buf, "%f", z);
	string s6 (buf);
	d_enLookAtZ.set_text (s6);
}


/*
 *  buttonCallbackApply: update what's been changed and redraw if necessary
 */
void TFDialogRender::buttonCallbackApply ()
{
	HeightFieldExport	*HFE = new HeightFieldExport (p_HF);

	getValues ();
	if (d_cbAutoRender.get_active())
		{
		HFE->renderPOV ();
		connect_to_function (Gtk_Main::timeout(1000),
                                &TFWindowHandler::waitForChildrenCallback);
		}

	if (b_applyHit)
		this->setHFobjs (p_HF, p_HFD);

	this->b_applyHit = TRUE;
}


gint TFDialogRender::delete_event_impl (GdkEventAny *e)
{
	this->hide ();
	return (0);
}




