/* Terraform - (C) 1997-2000 Robert Gasch (r.gasch@chello.nl)
 *  - http://212.187.12.197/RNG/terraform/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "FileIO.h"
#include "HeightFieldIO.h"
#include "HeightFieldOps.h"
#include "GlobalTrace.h"
#include "GlobalSanityCheck.h"
#include "MathFunctions.h"
#include "MathGauss.h"
#include "MathRandom.h"
#include "MathTrig.h"
#include "Timer.h"
#include "tf_flexarrCOORD2.h"


#define NO_FLOW		-987654321		// marker for 'undefined'
#ifndef MIN
# define MIN(a,b)	(a<b ? a : b)
#endif
#ifndef MAX
# define MAX(a,b)	(a>b ? a : b)
#endif 


/*
 *  constructor: initialize
 */
HeightFieldOps::HeightFieldOps (HeightField *HF) 
{
	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "+++ HeightFieldOps\n");

	SanityCheck::bailout ((!HF), "HF==NULL", "HeightFieldOps::HeightFieldOps");
	//SanityCheck::bailout ((!HF->d_hf), "HF->d_hf==NULL", "HeightFieldOps::HeightFieldOps");

	p_HF = HF;
}


/*
 *  destructor: nothing to clean up
 */
HeightFieldOps::~HeightFieldOps ()
{
	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "--- HeightFieldOps\n");
}


/*
 *  merge: provide some basic HF operators to combine 2 HFs 
 */
HeightField *HeightFieldOps::merge (char operation, HeightField *HF1, 
				HeightField *HF2, float weight1, float weight2)
{
	SanityCheck::bailout ((!HF1), "HF1==NULL", "HeightFieldOps::merge");
	SanityCheck::bailout ((!HF2), "HF2==NULL", "HeightFieldOps::merge");
	SanityCheck::bailout ((weight1 <= 0 || weight1> 1), "weight1 out of range", "HeightFieldOps::merge");
	SanityCheck::bailout ((weight2 <= 0 || weight2> 1), "weight2 out of range", "HeightFieldOps::merge");
	if (HF1->getWidth() != HF2->getWidth() ||
	    HF1->getHeight() != HF2->getHeight() )
		{
		SanityCheck::warning ((TRUE), "different size Height Fields", "HeightFieldOps::merge");
		return NULL;
		}

	int	i,
		lim = HF1->getSize(),
		width = HF1->getWidth(),
		height = HF1->getHeight();	
	PTYPE	*hf = new PTYPE[HF1->getSize()];

	if (operation == '+')
		{
		for (i=0; i<lim; i++)
			hf[i] = HF1->getEl(i)*weight1 + HF2->getEl(i)*weight2;
		}
	else
	if (operation == '-')
		{
		for (i=0; i<lim; i++)
			hf[i] = HF1->getEl(i)*weight1 - HF2->getEl(i)*weight2;
		}
	else
	if (operation == '*')
		{
		for (i=0; i<lim; i++)
			hf[i] = (1+(HF1->getEl(i)*weight1)) * (1+(HF2->getEl(i)*weight2));
		}
	else
	if (operation == '/')
		{
		for (i=0; i<lim; i++)
			hf[i] = (1+(HF1->getEl(i)*weight1)) / (1+(HF2->getEl(i)*weight2));
		}
	else
	if (operation == 'M')		// max
		{
		for (i=0; i<lim; i++)
			if (HF1->getEl(i)*weight1 > HF2->getEl(i)*weight2)
				hf[i] = HF1->getEl(i)*weight1;
			else 
				hf[i] = HF2->getEl(i)*weight2;
		}
	if (operation == 'm')		// min
		{
		for (i=0; i<lim; i++)
			if (HF1->getEl(i)*weight1 < HF2->getEl(i)*weight2)
				hf[i] = HF1->getEl(i)*weight1;
			else 
				hf[i] = HF2->getEl(i)*weight2;
		}

	HeightField	*HF = new HeightField (hf, width, height, "preview");
	HF->setSaved (FALSE);
	return HF;
}


/*
 *  upslopeSFD: find the number of single-flow-direction 
 *	upslope cells for a given cell. Cache is used to 
 * 	to store computed SFD values.  
 */
int HeightFieldOps::upslopeSFD (int x, int y, PTYPE *cache)
{
	int	ux=x, uy=y, 		// upflow x & y coordinates
		found, count=0,
		iLow, iHigh, 
		jLow, jHigh;

	SanityCheck::bailout ((!cache), "cache==NULL", "HeightFieldOps::upslopeSFD");

	if (cache[p_HF->Pos2Off(x,y)] == NO_FLOW)
		{
		found=FALSE;
		iLow = (ux > 0 ? ux-1 : 0);
		iHigh = (ux < (p_HF->getWidth()-1) ? ux+1 : p_HF->getWidth()-1);
		jLow = (uy > 0 ? uy-1 : 0);
		jHigh = (uy < (p_HF->getHeight()-1) ? uy+1 : p_HF->getHeight()-1);
		// find the highest downflow neighbour
		for (int i=iLow; i<=iHigh; i++)
		    for (int j=jLow; j<=jHigh; j++)
			{
			if (i != x || j!= y)
				{
				if (p_HF->getEl(i,j) > p_HF->getEl(ux,uy))
					{
					found=TRUE;
					ux=i;
					uy=j;
					}
				}
			}

		// downflow
		if (found)
		    count += upslopeSFD (ux,uy,cache)+1;
		else
		    count = 0;

		cache[p_HF->Pos2Off(x,y)] = count;
		d_cMiss++;
		}
	else
		{
		count = (int)cache[p_HF->Pos2Off(x,y)];
		d_cHit++;
		}

	return (count);
}


/*
 *  upslopeMFD: find the number of multiple-flow-direction
 *	upslope cells for a given cell. countedFlag should be 
 * 	re-initialized every time flowmap() calls this 
 *	function since it's being used as an indicator 
 * 	that a cell has been counted in the current MFD 
 *	pass for this cell. Since the 
 */
float HeightFieldOps::upslopeMFD (int x, int y, PTYPE *countedFlag)
{
	int	iLow, iHigh, 
		jLow, jHigh;
	float	count=0;

	SanityCheck::bailout ((!countedFlag), "countedFlag==NULL", "HeightFieldOps::upslopeMFD");

	iLow = (x > 0 ? x-1 : 0);
	iHigh = (x < (p_HF->getWidth()-1) ? x+1 : p_HF->getWidth()-1);
	jLow = (y > 0 ? y-1 : 0);
	jHigh = (y < (p_HF->getHeight()-1) ? y+1 : p_HF->getHeight()-1);
	for (int i=iLow; i<=iHigh; i++)
	    for (int j=jLow; j<=jHigh; j++)
		{
		if (i != x || j!= y)
			{
			if ( (p_HF->getEl(i,j) > p_HF->getEl(x,y))  &&
			     (countedFlag[p_HF->Pos2Off(i,j)] == NO_FLOW) )
				{
				countedFlag[p_HF->Pos2Off(i,j)] = TRUE;
				count += upslopeMFD (i,j,countedFlag) +
					 ((i==x || j==y) ? 0.6 : 0.4);
				}
			}
		}

	return (count);
}


/*
 *  flowpap: calculate the flowmap for the height field as detailed in 
 *    "Comparison of single and mulitple flow direction algorithms for 
 *    computing topographic parameters in TOPMODEL", Wolock & McCabe Jr., 
 *    Water Resources Research, Vol. 31, No. 5, Pages 1315-1324, May 1995
 *  The flowmap is essentially   ln(a/tan(b))   where 
 *	a = A/C 
 *	C = contour length
 * 	A = (number of upslope cells + 1) * (grid cell area)
 * 	tab(b) = ( change in elevation with neighboring cell / 
 * 		   		cell center distance )
 *	ln = Napierian Log = 
 *			 	   log (n/10^7)
 *		ln (n) =     -	-------------------
 *				log (10^7/(10^7-1))
 * 
 *  Interestingly enough, the paper uses the Napierian Log, which 
 *  in my case gives huge numbers at the top of the scale and zero 
 *  at the bottom. I find that using the natural log I get a 
 *  range which closely resembles the ones shown in the paper. 
 *  Does anybody out there (who has found this) know more about this?
 *  I've probably goofed up somewhere but don't know where. 
 */
HeightField *HeightFieldOps::flowmap (bool doSFD, bool ignoreSealevel,
				float maxElvErode) 	// above this no erosion
{
	char	buf[256];
	PTYPE	ELV_RT = 0.33;
	int	i, j,
		offij,
		dfx, dfy, 			// upflow x & y coordinates
		found,	
		iLow, iHigh, 
		jLow, jHigh,
		dotint;				// dot intervals 
	double 	C=10,				// contour length XXXXXXX
		gca=C*C, 			// Grid Cell Area
		a, A,
		tanb;
	Timer	fTimer;
	float	eTime;				// elapsed time
	PTYPE	fmin=-NO_FLOW, fmax=NO_FLOW;	// min and max flowmap values
	PTYPE	*cache=NULL,		// SFD/MFD cache
		*hfFlow=NULL,			// flowmap 
		*initCache=NULL;		// cache used for memcpy source
	HeightField *HFF=NULL;		// resulting flowmap HF

	fTimer.start ();
	dotint = p_HF->getSize()/50;
	d_cHit=d_cMiss=0;

	sprintf (buf, "Generating %s flowmap (%s) for %s (1 dot = %d cells) ...\n", 
		  (doSFD ? "SFD" : "MFD"), (ignoreSealevel ? "full" : "partial"), 
		  p_HF->getName(), dotint); 
	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, buf);

	hfFlow = new PTYPE[p_HF->getSize()];	// resulting flowmap 
	cache = new PTYPE[p_HF->getSize()];	// chache

	if (!doSFD)
		initCache = new PTYPE[p_HF->getSize()];	// init values

	for (i=0; i<p_HF->getSize(); i++)
		if (doSFD)
			cache[i] = NO_FLOW;
		else
			initCache[i] = NO_FLOW;
			
	if (!doSFD)
		memcpy (cache, initCache, sizeof(PTYPE)*p_HF->getSize());

	for (j=0; j<p_HF->getHeight(); j++)
		for (i=0; i<p_HF->getWidth(); i++)
		    if (ignoreSealevel ||
			(!ignoreSealevel && 
			  (p_HF->getEl(i,j) > p_HF->getSealevel()) &&
			  (p_HF->getEl(i,j) < maxElvErode) ) ) 
			{
			//printf ("%d, %d = ", i, j); fflush (stdout);

			if (doSFD)
				A = (upslopeSFD(i,j,cache)) * gca;
			else
				{
				memcpy (cache, initCache, 
					sizeof(PTYPE)*p_HF->getSize());
				A = (upslopeMFD(i,j,cache)) * gca;
				}
			a = A/C;

			// find lowest downslope square 
			found=FALSE;	
			dfx=i;
			dfy=j;
			iLow = (dfx > 0 ? dfx-1 : 0);
			iHigh = (dfx < (p_HF->getWidth()-1) ? dfx+1 : p_HF->getWidth()-1);
			jLow = (dfy > 0 ? dfy-1 : 0);
			jHigh = (dfy < (p_HF->getHeight()-1) ? dfy+1 : p_HF->getHeight()-1);
			for (int ii=iLow; ii<=iHigh; ii++)
			    for (int jj=jLow; jj<=jHigh; jj++)
				{
				if (ii != i || jj != j)
				    {
				    if ( p_HF->getEl(ii,jj) < p_HF->getEl(dfx,dfy) )
						{
						found = TRUE;
						dfx = ii;
						dfy = jj;
						}
				    }
				}

			// there is a downflow 
			offij = p_HF->Pos2Off(i,j);
			if (found && a)
				{
				tanb = (((p_HF->getEl(offij) - p_HF->getEl(dfx,dfy))*
					     (p_HF->getWidth()*ELV_RT)) / C);

				hfFlow[offij] = 
					//MathFunctions::napierianLog (a);
					//MathFunctions::napierianLog (1/tanb);
					//MathFunctions::napierianLog (a/tanb);
					log (a/tanb);	
					//log (a);	
				}
			// no downflow 
			else
				hfFlow[offij] = NO_FLOW;

			//printf ("Flow %d,%d, A=%f, flow=%f\n", i,j,A, hfFlow[offij]);
			if (hfFlow[offij] > fmax && hfFlow[offij]!=NO_FLOW)
				fmax = hfFlow[offij];
			if (hfFlow[offij] < fmin && hfFlow[offij]!=NO_FLOW)
				fmin = hfFlow[offij];

			if (!(offij%dotint) &&
			    GlobalTrace::trace(GlobalTrace::TRACE_VERBOSE, "."))
				;
			}

	// make 0 lowest value so that NO_FLOW==0 
	for (int i=0; i<p_HF->getSize(); i++)
		{
		if (hfFlow[i] == NO_FLOW)
			hfFlow[i] = 0; 
		else
			hfFlow[i] -= fmin; 
		}
			

	fTimer.stop ();
	eTime = fTimer.getElapsedSeconds ();
	sprintf (buf, "\nDone. Elapesed time: %.2f seconds\n", eTime);
	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, buf);

	sprintf (buf, "Flowmap max, min = %f, %f\nCache hit, miss = %ld, %ld\n", 
		fmax, fmin, d_cHit, d_cMiss);
	GlobalTrace::trace (GlobalTrace::TRACE_DEBUG, buf);

	delete [] cache;
	if (!doSFD)
		delete [] initCache;

	HFF = new HeightField (hfFlow, p_HF->getWidth(), p_HF->getHeight());
	HFF->setSaved (FALSE);

	return HFF;
}


/* 
 * erode: erode the HF using the flowmap to determine the terrain delta
 */
int HeightFieldOps::erode (int 	nTurns, 	// # of erodsion turns
			   int 	maxFlowAge, 	// new flowmap every n turns
			   float maxElvErode, 	// above this no erosion
			   char *fName,		// if fName = save every maxFlowAge turns
			   bool trimLocalPeaks,	// trim if (flowmap==0)
			   bool doSFD, 		// flowmap param
			   bool ignoreSealevel, // flowmap param
			   HeightFieldDraw *HFD)
{
	char		buf[256];
	HeightField 	*HFF;			// flowmap HF
	int		nFlowmap=0,		// number of flowmaps so far
			flowAge=0,		// age of flowmap
			dotint;
	double		errate = 2e-4;		// max erosion per turn

	dotint = int(maxFlowAge/50.0);

	SanityCheck::bailout ((!HFD), "HFD==NULL", "HeightFieldOps::erode");
	// fName should be formatted as a sprintf format string and contain %d
	if (fName)
		{
		SanityCheck::bailout ((!strstr(fName,"%d")), "bad string format", "HeightFieldOps::erode");
		SanityCheck::bailout ((!strchr(fName, '.')), "bad string format", "HeightFieldOps::erode");
		}

	HFF = flowmap (doSFD, ignoreSealevel);
	for (int i=0; i<nTurns; i++, flowAge++)
		{
		if (!(i%dotint) &&
		    GlobalTrace::trace(GlobalTrace::TRACE_FLOW, "."))
			;

		p_HF->setEl(0, p_HF->getEl(0) - errate*HFF->getEl(0));
		for (int n=1; n<p_HF->getSize(); n++)
			{
			if (HFF->getEl(n) > 0)
				p_HF->setEl(n, p_HF->getEl(n) - errate*HFF->getEl(n));
			else
				{
				// average flowmap vale with neighbours
				if (trimLocalPeaks)
					p_HF->setEl (n, p_HF->getEl(n) - errate *
						( HFF->getEl(n-1) + 
						  HFF->getEl(n+1) + 
						  HFF->getEl(n-p_HF->getWidth()) + 
						  HFF->getEl(n+p_HF->getWidth()) )/4);
				else
					p_HF->setEl (n, p_HF->getEl(n) - errate *
						HFF->getEl(n));
				}

			if (p_HF->getEl(n) < 0)
				p_HF->setEl (n, 0);
			}

		if ((maxFlowAge>0) && (flowAge == maxFlowAge))
			{
			GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "\n");
			//roughSmooth (false, true, 1.0);
			p_HF->gatherStatistics ();
			if (HFD)
				HFD->draw ();

			if (fName)
				{
				sprintf (buf, fName, nFlowmap);
				HeightFieldIO           *HFIO;

				HFIO = new HeightFieldIO (buf);
				HFIO->write (p_HF);
				delete HFIO;  
				}

			delete HFF;
			HFF = flowmap (doSFD, ignoreSealevel);
			flowAge = 0;
			nFlowmap++;

			sprintf (buf, "\nEroding %d/%d\n", i, nTurns); 
			GlobalTrace::trace (GlobalTrace::TRACE_FLOW, buf);
			}
		}

	if (HFF)
		delete HFF;

	return 0;
}



/*
 * hfDouble()  double resolution using modified-midpoint-displacement
 *	       algorithm; different versions for tiling and non-tiling
 *	       matrices. <f> is local fractal scaling, <f2> is overall
 *	       fractal scaling.
 * 	       This routine is adapted from John Beals's HF-Lab.
 */
int HeightFieldOps::hfDouble (double f, double f2) 	
{
	char		buf[80];
	int 		x, y;
	int 		xsize=0, ysize=0;
	int 		xsize1, ysize1;   	// old1 HF 
	int 		xsize2, ysize2;   	// old2 HF
  	int 		tile=FALSE;
  	D 		sfac;
  	D 		a, b, c, d;
	MathGauss	*mGauss = new MathGauss (time(NULL));
	HeightField	*HF2; 

	xsize1 = p_HF->getWidth();
	ysize1 = p_HF->getHeight();

	if (!tile) 			// non-tiling: don't wrap at edges 
		{  
		xsize2--;
		ysize2--;
		}

	xsize2 = xsize1*2;
	ysize2 = ysize1*2;

	sprintf (buf, "doubling to %dx%d ... ", xsize2, ysize2);  
	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, buf);

	HF2 = new HeightField (xsize2, ysize2);

	f2 /= sqrt((D)p_HF->getSize());  // normalize to matrix size 

	for (y=0;y<ysize1;y++) 
     		for (x=0;x<xsize1;x++)
      			HF2->setEl (x*2,y*2, p_HF->getEl(x,y));

	if (!tile) 			// fill in edges 
		{
		for (y=1;y<ysize2;y+=2) 
			{
			HF2->setEl (0,y, HF2->getEl(0,y-1));
			HF2->setEl (xsize2-1,y, HF2->getEl(xsize2-1,y-1));
			}
		for (x=1;x<xsize2;x+=2) 
			{
			HF2->setEl (x,0, HF2->getEl(x-1,0));
			HF2->setEl (x,ysize2-1, HF2->getEl(x-1,ysize2-1));
			}
		}

	for (y=1; y<ysize2; y+=2) 	// tiling and non-tiling interpolation
	    {
	    for (x=1; x<xsize2; x+=2) 
		{
		if (tile) 
			{
			a = HF2->getElmod(x+1,y+1); b = HF2->getElmod(x-1,y+1);
			c = HF2->getElmod(x+1,y-1); d = HF2->getElmod(x-1,y-1);
			} 
		else 
			{
			a = HF2->getElclip(x+1,y+1); b = HF2->getElclip(x-1,y+1);
			c = HF2->getElclip(x+1,y-1); d = HF2->getElclip(x-1,y-1);
			}
	        sfac = f2+f*(MAX(MAX(a,b),MAX(c,d)) - MIN(MIN(a,b),MIN(c,d)));
		HF2->setEl (x,y, (a+b+c+d)/4.0+sfac*(mGauss->rnd()-0.5));
		}
	    }

	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, "cent1 ...");

	for (y=0; y<ysize2; y+=2) 
	    {      
	    for (x=1; x<xsize2; x+=2) 
		{
		if (tile) 
			{
			a = HF2->getElmod(x+1,y); b = HF2->getElmod(x-1,y);
			c = HF2->getElmod(x,y+1); d = HF2->getElmod(x,y-1);
			} 
		else 
			{
			a = HF2->getElclip(x+1,y); b = HF2->getElclip(x-1,y);
			c = HF2->getElclip(x,y+1); d = HF2->getElclip(x,y-1);
			}
		sfac = f2+f*(MAX(MAX(a,b),MAX(c,d)) - MIN(MIN(a,b),MIN(c,d)));
		HF2->setEl (x,y,  (a+b+c+d)/4.0+sfac*(mGauss->rnd()-0.5));
		}
	    }

	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, "edge1 ...");

	for (y=1; y<ysize2; y+=2) 
	    {      
	    for (x=0; x<xsize2; x+=2) 
		{
		if (tile) 
			{
			a = HF2->getElmod(x+1,y); b = HF2->getElmod(x-1,y);
			c = HF2->getElmod(x,y+1); d = HF2->getElmod(x,y-1);
			} 
		else 
			{
			a = HF2->getElclip(x+1,y); b = HF2->getElclip(x-1,y);
			c = HF2->getElclip(x,y+1); d = HF2->getElclip(x,y-1);
			}
		sfac = f2+f*(MAX(MAX(a,b),MAX(c,d)) - MIN(MIN(a,b),MIN(c,d)));
		HF2->setEl (x,y, (a+b+c+d)/4.0+sfac*(mGauss->rnd()-0.5));
		}
	    }

	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, "edge2 ...");

	for (y=0;y< ysize2; y+=2) 	// backtrack and adjust original data 
	    {
	    for (x=0; x<xsize2; x+=2) 
		{
		if (tile) 
			{
			a = HF2->getElmod(x+1,y+1); b = HF2->getElmod(x-1,y+1);
			c = HF2->getElmod(x+1,y-1); d = HF2->getElmod(x-1,y-1);
			} 
		else 
			{
			a = HF2->getElclip(x+1,y+1); b = HF2->getElclip(x-1,y+1);
			c = HF2->getElclip(x+1,y-1); d = HF2->getElclip(x-1,y-1);
			}
		sfac = 0.5*(f2+f*(MAX(MAX(a,b),MAX(c,d)) - MIN(MIN(a,b),MIN(c,d))));
		HF2->setEl (x,y, (a+b+c+d)/4.0+sfac*(mGauss->rnd()-0.5));
		}
	    }

	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, "done\n");

	delete mGauss;

	p_HF->init (HF2->releaseData(), xsize2, ysize2, p_HF->getName());
	p_HF->setSaved (FALSE);

	delete HF2; 
	return (0);
}  



/*
 * hfHalf ()   half resolution by averaging 4 points into 1
 * 	       This routine is adapted from John Beals's HF-Lab.
 */
int HeightFieldOps::hfHalf() 
{
	int 		x,y,xx,yy;
	int 		xsize1, ysize1;
	int 		xsize2, ysize2;
	char		buf[80];
	HeightField	*HF2;

	if (p_HF->getWidth()*p_HF->getHeight() < 25)
		return 1;

	xsize1 = p_HF->getWidth();
	ysize1 = p_HF->getHeight();
	xsize2 = xsize1/2;
	ysize2 = ysize1/2;

	sprintf(buf, "Diminishing to %dx%d ... ", xsize2,ysize2); 
	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, buf);

	HF2 = new HeightField (xsize2, ysize2);

	for (y=0; y<ysize2; y++) 
	    {      		// copy over initial values 
	    yy = y * 2;
	    for (x=0; x<xsize2; x++) 
		{
		xx = x * 2;
		HF2->setEl (x,y, ( p_HF->getEl(xx,yy) + p_HF->getEl(xx+1,yy) +
			  	p_HF->getEl(xx,yy+1) + p_HF->getEl(xx+1,yy+1) ) /4.0);
		}
	    }

	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, "done\n");

	p_HF->init (HF2->releaseData(), xsize2, ysize2, p_HF->getName());
	p_HF->setSaved (FALSE);

	delete HF2; 
	return (0);
}  


/*
 * hfResize()   half resolution by averaging 4 points into 1
 * 	       This routine is adapted from John Beals's HF-Lab.
 */
HeightField *HeightFieldOps::hfResize (int newx, int newy)
{
	float 		incx = ((float)p_HF->getWidth())/((float)newx),
			incy = ((float)p_HF->getHeight())/((float)newy);
	int		x, 
			y,
			offSrc,
			offDst;
	HeightField	*HF=NULL;

	// return a new copy of the HF if size is too small
	if (p_HF->getWidth()<=newx || p_HF->getHeight()<=newy)
		return new HeightField (p_HF->getCopyOfData(), p_HF->getWidth(), 
					p_HF->getHeight(), "Preview");

	HF = new HeightField (newx, newy, "resize");

	for (y=0; y<newy; y++)
		for (x=0; x<newx; x++)
			{
			offSrc = p_HF->Pos2Off ((int)(x*incx),(int)(y*incy));
			offDst = (int)(y*newx+x);
			HF->setEl (offDst, p_HF->getEl (offSrc));
			//printf ("HFResize: %d  ->  %f\n", offDst, hf[offDst]);
			}

	HF->setSaved (FALSE);

	return HF;
}



/*
 *  placePOVObject: place a PovRay object on the landscape
 * 	We use the following method:
 *	- calculate a basic square size dependant upon object placement density and HF scale size in POV units.
 *	- calculate the size of an inner square proportional to placeVar 
 *	- place an object of randomized size at the correct offset for the inner square
 */
int HeightFieldOps::placePOVObject (
			char	*outfile,	// output filename
			char 	*object, 	// object name
			int	hfScaleX,	// height field x scaling factor
			int	hfScaleY,	// height field y scaling factor
			int	HFScaleZ,	// height field y scaling factor
			PTYPE 	floor, 		// low elevation for object placement
			PTYPE	ceil, 		// high elevation for object placement
			float	density,	// density factor for object placement
			float	placeVar,	// placement variance ratio 
			int	objScaleX, 	// object x scaling factor
			int	objScaleY, 	// object y scaling factor
			int	objScaleZ, 	// object y scaling factor
			PTYPE	objScaleXvar, 	// object x scaling factor variance
			PTYPE	objScaleYvar, 	// object z scaling factor variance
			PTYPE	objScaleZvar, 	// object z scaling factor variance
			int	seed)		// random number seed
{
	FILE	*fout, *ft;
	float	e,				// elevation
		xOff, yOff,			// offset for inner placement square
		nGridX, nGridY,			// numbers of x and y size of a grid
		cellSizeX, cellSizeY,			// x and y size of a grid
		ixpov, izpov, 			// x and y increment POVray scale
		ixhf, iyhf, 			// x and y increment p_HF subscript scale
		hsx, hsy, hsz, lsx, lsy, lsz;	// high and low scaling factors
	int	nobs=0;
	char	buf[80];
    	MathRandom	*rnd = NULL;

	//printf ("\n%s, %s, hfs[xyz]=%d, %d, %d\n", outfile, object, hfScaleX, hfScaleY, HFScaleZ);
	//printf ("floor=%f, ceil=%f, dens=%f, placeVar=%f,\n", floor, ceil, density, placeVar);
	//printf ("os[xyz]=%d,%d,%d ", objScaleX, objScaleY, objScaleZ);
	//printf ("os[xyz]var=%f, %f, %f, seed=%d\n\n", objScaleXvar, objScaleYvar, objScaleZvar, seed);

	// Sanity Checks
	SanityCheck::bailout ((!outfile), "outfile==NULL", "HeightFieldOps::PlacePOVObject");
	SanityCheck::bailout ((!strlen(outfile)), "strlen(outfile)==0", "HeightFieldOps::PlacePOVObject");
	SanityCheck::bailout ((!object), "object==NULL", "HeightFieldOps::PlacePOVObject");
	SanityCheck::bailout ((!strlen(object)), "strlen(object)==0", "HeightFieldOps::PlacePOVObject");
	SanityCheck::bailout ((floor < p_HF->getMin() || floor > p_HF->getMax()), 
			"floor out of range", "HeightFieldOps::PlacePOVObject");
	SanityCheck::bailout ((ceil < p_HF->getMin() || ceil > p_HF->getMax()), 
			"ceil out of range", "HeightFieldOps::PlacePOVObject");
	SanityCheck::bailout ((density <= 0), "density<=0", "HeightFieldOps::PlacePOVObject");
	SanityCheck::bailout ((placeVar < 0 || placeVar > 1), "placeVar out of range", "HeightFieldOps::PlacePOVObject");
	SanityCheck::bailout ((objScaleXvar < 0 || objScaleXvar > 1), "objScaleXvar out of range", "HeightFieldOps::PlacePOVObject");
	SanityCheck::bailout ((objScaleYvar < 0 || objScaleYvar > 1), "objScaleYvar out of range", "HeightFieldOps::PlacePOVObject");
	SanityCheck::bailout ((objScaleZvar < 0 || objScaleZvar > 1), "objScaleZvar out of range", "HeightFieldOps::PlacePOVObject");

	if (!(fout=fopen (outfile, "w"))) 		// open output file
		{
		cerr << "HeightFieldOps::PlacePOVObject(): can't open file " << outfile << "\n";
		return (-1);
		}

    	rnd = new MathRandom (seed);

	sprintf (buf, "Placing PovObjects on %s ... ", p_HF->getName());
	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, buf);

	// find percentage scaling ranges (0 .. 2)
	hsx=1+objScaleXvar; hsy=1+objScaleYvar; hsz=1+objScaleZvar;
	lsx=1-objScaleXvar; lsy=1-objScaleYvar; lsz=1-objScaleZvar;

	// calculate number of cells 
	nGridX=(hfScaleX/objScaleX)*density;		// high density == more cells
	nGridY=(HFScaleZ/objScaleZ)*density;		// povray dim z == program dim y

	// calculate cell size 
	cellSizeX=(hfScaleX/nGridX);
	cellSizeY=(HFScaleZ/nGridY);

	// calculate max offset from center
	xOff=(cellSizeX-(cellSizeX-(cellSizeX*placeVar)));
	yOff=(cellSizeY-(cellSizeY-(cellSizeY*placeVar)));

	printf ("%f, %f, %f, %f\n", nGridX, nGridY, cellSizeX, cellSizeY); 
	printf ("%f, %f\n", xOff, yOff);

	ft=stdout;
	for (int c=0; c<2; c++)
	   {
	   if ((c==0 && GlobalTrace::isSet (GlobalTrace::TRACE_DEBUG)) || c==1)
		{
		if (c==1)
			fprintf (ft, "// File created by terraform ...\n");
		fprintf (ft, "// HeightFieldOps::PlacePOVObject Settings\n");
		fprintf (ft, "// Output Filename: 		%s\n", outfile);
		fprintf (ft, "// Object name:	 		%s\n", object);
		fprintf (ft, "// Height Field Scales (x,y,z):	%d, %d, %d\n", hfScaleX, hfScaleY, HFScaleZ);
		fprintf (ft, "// Floor, Ceil, Density:	%f, %f, %f\n", floor, ceil, density);
		fprintf (ft, "// x scale & variance:		%d, %f\n", objScaleX, objScaleXvar);
		fprintf (ft, "// y scale & variance:		%d, %f\n", objScaleY, objScaleYvar);
		fprintf (ft, "// z scale & variance:		%d, %f\n", objScaleZ, objScaleZvar);
		fprintf (ft, "// seed:			%d\n", seed);
		fprintf (ft, "// hsx, hsy, hsz:			%f, %f, %f\n", hsx, hsy, hsz);
		fprintf (ft, "// lsx, lsy, lsz:			%f, %f, %f\n", lsx, lsy, lsz);
		fprintf (ft, "// nGridX, nGridY:		%f, %f\n", nGridX, nGridY);
		fprintf (ft, "// xOff, yOff:			%f, %f\n\n", xOff, yOff);
		}
	   ft=fout;
	   }

	ixpov=cellSizeX;			// increment on POVray scale
	izpov=cellSizeY;			// povray dim z == array dim y
	ixhf=p_HF->getWidth()/nGridX;		// increment on p_HF subscript scale
	iyhf=p_HF->getHeight()/nGridY;		// increment on p_HF subscript scale

	
	printf ("xOff=%f, yOff=%f\nnGridX=%f, nGridY=%f\n", xOff, yOff, nGridX, nGridY);
	printf ("ixpov=%f, izpov=%f, ixhf=%f, iyhf=%f\n", ixpov, izpov, ixhf, iyhf);

	// loop through the squares on Pov and Hf offsets
	for (float jp=0, jh=0; jp<HFScaleZ; jp+=izpov, jh+=iyhf)
		for (float ip=0, ih=0; ip<hfScaleX; ip+=ixpov, ih+=ixhf)
			{
			// variables holding placement info for current object 
			float	aoffx, aoffz,	// actual object placement offset in grid (from grid center)
				aobjScaleX=0, 	// actual object scales 
				aobjScaleY=0, 
				aobjScaleZ=0,	
				aopx=0, 	// actual object position 
				aopy=0, 
				aopz=0;	

			//printf ("X:: %f, %f, %f, %f, %d", ip, jp, ih, jh, off);fflush(stdout);

			// reverse Y/Z coordinate system for POV 
			e=p_HF->getEl(ih,p_HF->getHeight()-jh);	// reverse drawing direction from terraform to povray

			if (e >= floor && e <= ceil)
				{
				nobs++;

#ifdef TEST_CODE
				// this will place objects with no variance
				// was useful for testing 
				aopx = ip;
				aopz = jp;
				aopy = e*hfScaleY;
				fprintf (fout, "object { %s  scale < %f, %f, %f > translate < %f, %f, %f > }\n",
					object, 1.0, 1.0, 1.0, aopx, aopy, aopz);
#else

				// get an offset for the inner placement square
				aoffx = xOff*rnd->rnd1();
				aoffz = yOff*rnd->rnd1();

				// calculate the object position in the scene
				if (rnd->rnd1() > 0.5)
					aopx = ip+ixpov/2+aoffx;
				else
					aopx = ip+ixpov/2-aoffx;

				if (rnd->rnd1() > 0.5)
					aopz = jp+izpov/2+aoffz;
				else
					aopz = jp+izpov/2-aoffz;

				aopy = e*hfScaleY;
	
				// calculate the object scaling factors
				aobjScaleX = ((hsx-lsx)*rnd->rnd1()+lsx);
				aobjScaleY = ((hsy-lsy)*rnd->rnd1()+lsy);
				aobjScaleZ = ((hsz-lsz)*rnd->rnd1()+lsz);

				fprintf (fout, "object { %s  scale < %f, %f, %f > translate < %f, %f, %f > }\n",
					object, objScaleX*aobjScaleX, 
					objScaleY*aobjScaleY, objScaleZ*aobjScaleZ, 
					aopx, aopy, aopz);
#endif
				}
			}
	fclose (fout); 			// close output file

	sprintf (buf, "%d %ss placed\n", nobs, object);
	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, buf);

	delete rnd;
	return (0);
}


/*
 *  roughSmooth: either roughen or smooth the height field. 
 */
int HeightFieldOps::roughSmooth (bool roughen, bool bigGrid, float factor)
{
	SanityCheck::bailout ((factor<0), "factor<0", "HeightFieldOps::smooth");
	SanityCheck::bailout ((factor>1), "factor>1", "HeightFieldOps::smooth");

	PTYPE	o, t;
	int	limx=p_HF->getWidth()-1, limy=p_HF->getHeight()-1;

	for (int x=1; x<limx; x++)
		for (int y=1; y<limy; y++)
			{
			o = p_HF->getEl (x, y);
			if (bigGrid == 0)
				{
				t = p_HF->getEl(x-1,y) + p_HF->getEl(x+1,y) +
				    p_HF->getEl(x,y-1) + p_HF->getEl(x,y+1);
				t /= 4;
				}
			else
				{
				t = p_HF->getEl(x-1,y) + p_HF->getEl(x+1,y) +
				    p_HF->getEl(x,y-1) + p_HF->getEl(x,y+1) + 
				    p_HF->getEl(x-1,y-1) + p_HF->getEl(x-1,y+1) +
				    p_HF->getEl(x+1,y-1) + p_HF->getEl(x+1,y+1);
				t /= 8;
				}
			if (roughen)
				p_HF->setEl (x,y, o-factor*(t-o));
			else
				p_HF->setEl (x,y, o+factor*(t-o));
			}

	p_HF->setSaved (FALSE);
	return (0);
}


/*
 *  transform: transform the elevation by some power factor
 */
int HeightFieldOps::transform (float factor, int adjust_sea_level)
{
	//SanityCheck::bailout ((!p_HF->getData()), "p_HF->getData()==NULL", "HeightFieldOps::transform");

	char	buf[80];

	sprintf (buf, "Transforming with power %f ...\n", factor);
	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, buf);

	for (int i=0; i<p_HF->getSize(); i++)
		p_HF->setEl(i, pow(p_HF->getEl(i),factor));
	if (adjust_sea_level)
		p_HF->setSealevel (pow (p_HF->getSealevel(), factor));;
	p_HF->setSaved (FALSE);
	return (0);
}


/*
 *  terrace: make terraces into the terrain
 */
int HeightFieldOps::terrace (int nLevels, float factor, bool adjustSealevel)
{
	//SanityCheck::bailout ((!p_HF->d_hf), "p_HF->d_hf==NULL", "HeightFieldOps::transform");

	// increase nLevels by one so that the highest elevation (1.0) gets 
	// the last level (which is 1 point high).
	nLevels++;
	factor=1-factor;		// convert tightness scale 

	char	buf[80];
	int	level;
	float	range = p_HF->getMax() - p_HF->getMin(),
		lRange = range/nLevels,
		base,
		diff;


	sprintf (buf, "Terracing %d levels with tightness %f ...\n", nLevels, factor);
	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, buf);

	for (int i=0; i<p_HF->getSize(); i++)
		{
		level = (int)(p_HF->getEl(i)/lRange);
		base = level*lRange;
		diff = p_HF->getEl(i) - base;
		p_HF->setEl(i, base + diff*factor);
		}

	p_HF->setSaved (FALSE);
	return (0);
}


/*
 *  gaussianHill: create a gaussian hill according to paramters 
 */
int HeightFieldOps::gaussianHill (float cx, float cy, float radius, 
				float radfac, float hscale, 
				float smooth, float deltascale)
{
	int 	xsize = p_HF->getWidth(), 
			ysize = p_HF->getHeight();
	float	d, 								// distance
			sf, 							// smoothing factor
			fxmin, fxmax, fymin, fymax, 	// min, max working floats
			sdist, sdistr;					// smoothing distances 
 	int 	xmin, xmax, ymin, ymax, xcent, ycent;
 	int 	x, y;
	D 		xa,ya,tmp,rad;
	char	buf[80];

	SanityCheck::bailout ((cx<0), "cx<0", "HeightFieldOps::gaussianHill");
	SanityCheck::bailout ((cx>1), "cx>1", "HeightFieldOps::gaussianHill");
	SanityCheck::bailout ((cy<0), "cy<0", "HeightFieldOps::gaussianHill");
	SanityCheck::bailout ((cy>1), "cy>1", "HeightFieldOps::gaussianHill");
	SanityCheck::bailout ((smooth<0), "smooth<0", "HeightFieldOps::gaussianHill");
	SanityCheck::bailout ((smooth>1), "smooth>0.5", "HeightFieldOps::gaussianHill");
	SanityCheck::bailout ((deltascale<0), "deltascale<0", "HeightFieldOps::gaussianHill");
	SanityCheck::bailout ((deltascale>1), "deltascale>1", "HeightFieldOps::gaussianHill");

	// calculate max and min coordinates on 0 .. 1 scale
	fxmin = cx - radius;
	fxmax = cx + radius;
	fymin = cy - radius;
	fymax = cy + radius;
	if (fxmin < 0) fxmin = 0;
	if (fxmax > 1) fxmax = 1;
	if (fymin < 0) fymin = 0;
	if (fymax > 1) fymax = 1;

	// translate everything to subscript scale (0 .. n)
	xmin = (int)(fxmin*xsize);	
	xmax = (int)(fxmax*xsize);
	ymin = (int)(fymin*ysize);
	ymax = (int)(fymax*ysize);
	xcent = (int)(cx*xsize);
	ycent = (int)(cy*xsize);
	radius *= ((xsize+ysize)/2);
	sdist = radius*smooth;
	sdistr = radius-sdist;

	sprintf(buf, "Gaussian Hill: %f, %f: radfac=%f, hscale=%f\n", 
		cx, cy, radfac, hscale);
	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, buf);
	sprintf(buf, "Gaussian Hill: xrange=%d,%d, yrange=: %d,%d\n", 
		xmin, xmax, ymin, ymax);
	GlobalTrace::trace (GlobalTrace::TRACE_DEBUG, buf);

 	radfac = radfac * radfac;
	for (y=ymin; y<ymax; y++) 
		{
		ya = (((D)y+0.5)/ysize) - cy;
		ya = ya*ya;

		for (x=xmin; x<xmax; x++) 
			{
			d = MathTrig::distance (xcent, ycent, x, y);
			if (d <= radius)
				{
    				tmp = p_HF->getEl(x,y); 
				xa = (((D)x+0.5)/xsize) - cx;
				xa = xa * xa;
				rad = xa+ya;

				// see if we should be smoothing
				if (d > sdistr)
					{
					d -= sdistr;
					sf = 1-d/sdist;
					tmp += deltascale * sf * 
						hscale * exp(-rad/radfac);
					}
				else
					tmp += deltascale * hscale * 
						exp(-rad/radfac);
				p_HF->setEl (x,y, tmp);
    				//printf ("%d  -->  %f\n", p_HF->Pos2Off(x,y), p_HF->getEl(x,y)); 
				}
			} 
		} 

	p_HF->setSaved (FALSE);
	return (0);
}


/*
 *  rotate90: counterclockwise rotate in 90, 180 or 270 degrees
 */
int HeightFieldOps::rotate90 (int degrees)
{
	SanityCheck::bailout ((!p_HF), "p_HF==NULL", "HeightFieldOps::rotate90");
	//SanityCheck::bailout ((!p_HF->d_hf), "p_HF->d_hf==NULL", "HeightFieldOps::rotate90");

	PTYPE	*hft=NULL;
	int	w = p_HF->getWidth(), 
		h = p_HF->getHeight(),
		s = p_HF->getSize();
	char	buf[80];

	sprintf (buf, "Rotating %d degrees ...\n", degrees); 
	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, buf);

	if (degrees < 0)
		degrees = (-degrees)%360;
	else
	if (degrees > 360)
		degrees = degrees%360;

	if (degrees == 0)
		{
		}
	else
	if (degrees == 90)
		{
		hft=new PTYPE [s];
		for (int x=0; x<w; x++)
			for (int y=0; y<h; y++)
				hft[p_HF->Pos2Off(x,y)] = p_HF->getEl(h-1-y, x);
		p_HF->init (hft, w, h, p_HF->getName(), TRUE);
		}
	else
	if (degrees == 180)
		{
		PTYPE	pt;
		int	idx=s-1, 
			lim=s/2, 
			c=0;

		while (c < lim)
			{
			pt=p_HF->getEl(c);
			p_HF->setEl(c, p_HF->getEl(idx-c));
			p_HF->setEl(idx-c, pt);
			c++;
			}
		}
	else
	if (degrees == 270)
		{
		hft=new PTYPE [s];
		for (int x=0; x<w; x++)
			for (int y=0; y<h; y++)
				hft[p_HF->Pos2Off(x,y)] = p_HF->getEl(y, h-1-x);
		p_HF->init (hft, w, h, p_HF->getName(), TRUE);
		}
	else
		{
		printf ("Warning: Rotate90() paramter (%d) out of range ... Cancelling \n", degrees);
		return (-1);
		}

	p_HF->setSaved (FALSE);
	return (0);
}


/*
 *  Invert: invert the height field 
 */
int HeightFieldOps::invert (int adjust_sea_level)
{
	char 	buf[80];
	//SanityCheck::bailout ((!p_HF->d_hf), "p_HF->d_hf==NULL", "HeightFieldOps::invert");

	sprintf (buf, "Inverting height field ... \n");
	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, buf);

	for (int i=0; i<p_HF->getSize(); i++)
		p_HF->setEl (i, 1-p_HF->getEl(i));
	p_HF->gatherStatistics ();
	if (adjust_sea_level)
		p_HF->setSealevel (1-p_HF->getSealevel());
	p_HF->setSaved (FALSE);
	return (0);
}


/*
 *  hf_sort: compare 2 PTYPE values; used by qsort()
 */
inline int HeightFieldOps::hf_sort (const void *_v1_, const void *_v2_)
{
	PTYPE	*v1 = static_cast<PTYPE*>(_v1_),
		*v2 = static_cast<PTYPE*>(_v2_);

	if (*v1 < *v2)
		return (-1);
	else
	if (*v1 > *v2)
		return (1);
	else
		return (0);
}


/*
 *  fold: fold height field edges under sealevel.
 *  How: Take a slice 'margin' long, fill it with the ABS 
 *  difference between it's height and the HF, sort it and 
 *  then subtract it from the HF elevation at that point 
 *  (starting with the largest values on the HF outside), 
 *  scaling the value subtracted in a linear fashion (0..1).
 */
int HeightFieldOps::fold (int margin)
{
	int	i=0, lim=0, off, offlim, 
		(*hf_cmp_func) (const void *, const void *) = hf_sort;
	float	pstep; 
	PTYPE	slice[margin]; 
	char 	buf[80];

	//SanityCheck::bailout ((!p_HF->d_hf), "p_HF->d_hf==NULL", "HeightFieldOps::Fold");
	SanityCheck::bailout ((margin<0), "margin<0", "HeightFieldOps::Fold");
	SanityCheck::bailout ((margin>p_HF->getWidth()/2 || margin>p_HF->getHeight()/2), "margin>x/2 || margin>y/2", "HeightFieldOps::Fold");

	sprintf (buf, "Folding with margin %d ...\n", margin); 
	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, buf);

	if (!margin)
		return (1);

	// top and bottom 
	off = 0;
	offlim = p_HF->getWidth();
	for (int x=off; x<offlim; x++)
	    {
		//for (lim=0; lim<margin && p_HF->d_hf[p_HF->Pos2Off(x,lim)]>p_HF->d_sealevel; lim++)
		for (lim=0; lim<margin; lim++)
			{
			slice[lim] = p_HF->getEl(x,lim) - p_HF->getSealevel();
			if (slice[lim] < 0)
				slice[lim] = 0;
			}
		qsort ((void *)slice, (size_t)lim, (size_t)sizeof(PTYPE), hf_cmp_func);
		pstep = (float)1/(float)lim;
		for (i=0; i<lim; i++)
			{
			p_HF->setEl (x, i, p_HF->getEl(x,i) - (slice[lim-1-i]*pstep*(lim-i)));
			if (p_HF->getEl(x,i) < 0)
				p_HF->setEl (x, i, 0);
			}
				

		//for (lim=0; lim<margin && p_HF->getEl(x,p_HF->getHeight()-1-lim)>p_HF->d_sealevel; lim++)
		for (lim=0; lim<margin; lim++)
			{
			slice[lim] = p_HF->getEl(x,p_HF->getHeight()-1-lim) - p_HF->getSealevel();
			if (slice[lim] < 0)
				slice[lim] = 0;
			}
		qsort ((void *)slice, (size_t)lim, (size_t)sizeof(PTYPE), hf_cmp_func);
		pstep = (float)1/(float)lim;
		for (i=0; i<lim; i++)
			{
			p_HF->setEl (x, p_HF->getHeight()-1-i, 
				p_HF->getEl(x, p_HF->getHeight()-1-i) - 
					(slice[lim-1-i]*pstep*(lim-i)));
			if (p_HF->getEl(x,p_HF->getHeight()-1-i) < 0)
				p_HF->setEl(x, p_HF->getHeight()-1-i, 0);
			}
	    off++;
	    offlim = p_HF->getWidth();
	    }

	// left and right sides 
	for (int y=0; y<p_HF->getHeight(); y++)
	    {
		//for (lim=0; lim<margin && p_HF->getEl(lim,y)>p_HF->getSealevel(); lim++)
		for (lim=0; lim<margin; lim++)
			{
			slice[lim] = p_HF->getEl(lim,y) - p_HF->getSealevel();
			if (slice[lim] < 0)
				slice[lim] = 0;
			}
		qsort ((void *)slice, (size_t)lim, (size_t)sizeof(PTYPE), hf_cmp_func);
		pstep = (float)1/(float)lim;
		for (i=0; i<lim; i++)
			{
			p_HF->setEl (i,y, p_HF->getEl(i,y) - (slice[lim-1-i]*pstep*(lim-i)));
			if (p_HF->getEl(i,y) < 0)
				p_HF->setEl (i,y, 0);
			}

		//for (lim=0; lim<margin && p_HF->getEl(p_HF->getWidth()-1-lim,y)>p_HF->getSealevel(); lim++)
		for (lim=0; lim<margin; lim++)
			{
			slice[lim] = p_HF->getEl(p_HF->getWidth()-1-lim,y) - p_HF->getSealevel();
			if (slice[lim] < 0)
				slice[lim] = 0;
			}
		qsort ((void *)slice, (size_t)lim, (size_t)sizeof(PTYPE), hf_cmp_func);
		pstep = (float)1/(float)lim;
		for (i=0; i<lim; i++)
			{
			p_HF->setEl(p_HF->getHeight()-1-i,y, 
				p_HF->getEl(p_HF->getHeight()-1-i,y) - 
					(slice[lim-1-i]*pstep*(lim-i)));
			if (p_HF->getEl(p_HF->getHeight()-1-i,y) < 0)
				p_HF->setEl (p_HF->getHeight()-1-i,y, 0);  
			}
	    }

	p_HF->setSaved (FALSE);
	return (0);
}


/*
 *  linearScale: pick a point (centerx, centery) and scale the points 
 *  where distance is mindist<=distance<=maxdist linarly from either 
 *  1..0 if scale_factor is < 1; otherwise was scale from 1..2. 
 */
int HeightFieldOps::linearScale (	float 	centerx, 
					float	centery, 
					float 	scale_factor, 
					float 	mindist,
					float	maxdist, 
					float 	smoothFactor,
					bool 	invertScale,
					bool	scaleFullDist)
{
	float	p,		// percentage change multiplication factor
		smoothFloor,
		smoothCeil;
	double  d, 		// distance
		md;		// max distance
	char	buf[256];
		

	//SanityCheck::bailout ((!p_HF->d_hf), "p_HF->d_hf==NULL", "HeightFieldOps::LinearScale");
	SanityCheck::bailout ((centerx<0), "centerx < 0", "HeightFieldOps::LinearScale");
	SanityCheck::bailout ((centerx>1), "centerx > 1", "HeightFieldOps::LinearScale");
	SanityCheck::bailout ((centery<0), "centery < 0", "HeightFieldOps::LinearScale");
	SanityCheck::bailout ((centery>1), "centery > 1", "HeightFieldOps::LinearScale");
	SanityCheck::bailout ((scale_factor<0), "scale_factor < 0", "HeightFieldOps::LinearScale");
	SanityCheck::bailout ((mindist<0), "mindist < 0", "HeightFieldOps::LinearScale");
	SanityCheck::bailout ((mindist>1), "mindist > 1", "HeightFieldOps::LinearScale");
	SanityCheck::bailout ((maxdist<0), "maxdist < 0", "HeightFieldOps::LinearScale");
	SanityCheck::bailout ((maxdist>1), "maxdist > 1", "HeightFieldOps::LinearScale");
	SanityCheck::bailout ((smoothFactor<0), "smoothFactor < 0", "HeightFieldOps::LinearScale");
	SanityCheck::bailout ((smoothFactor>1), "smoothFactor > 1", "HeightFieldOps::LinearScale");

	if (SanityCheck::warning ((maxdist==mindist), "maxdist==mindist - linear scale cancelled", NULL))
		return 1;

	sprintf (buf, "Linear Scaling (%.2f, %.2f, %.2f, %.2f, %.2f, %s, %s) ...\n",
		centerx, centery, scale_factor, mindist, maxdist, 
		invertScale==TRUE ? "InvertScale" : "NoInvertScale",
		scaleFullDist?"ScaleFullDist":"NoScaleFullDist");
	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, buf);

	// convert to real (subscript) scale
	centerx *= p_HF->getWidth();
	centery *= p_HF->getHeight();
	mindist *= (centerx+centery)/2;
	maxdist *= (centerx+centery)/2;
	//smoothFloor = mindist*(1-smoothFactor);
	//smoothCeil = maxdist*(1+smoothFactor);
	smoothFloor = mindist*(1+smoothFactor);
	smoothCeil = maxdist*(1-smoothFactor);

	md=MathTrig::distance (0, 0, p_HF->getWidth(), p_HF->getHeight());
	//md=MathTrig::distance (centerx, centery, 0, 0);
	//if ((d=MathTrig::distance(centerx,centery,p_HF->getWidth(),0)) > md)
	//	md=d;
	//if ((d=MathTrig::distance(centerx,centery,0,p_HF->getHeight())) > md)
	//	md=d;
	//if ((d=MathTrig::distance(centerx,centery,p_HF->getWidth(),p_HF->getHeight())) > md)
	//	md=d;

	if (!scaleFullDist)
		md = maxdist;
		//md = (maxdist>md ? md-mindist : maxdist-mindist);

	for (int x=0; x<p_HF->getWidth(); x++)
		for (int y=0; y<p_HF->getHeight(); y++)
			{
			PTYPE	pt=0;
			d=MathTrig::distance (centerx, centery, x, y);

			if (scale_factor >= 1)
				p=(d/md)*scale_factor;	// 1 .. scale_factor
			else
				p=(d/md)*(1-scale_factor);  // 0 .. 1

			if (smoothFactor > 0 && d>mindist && d<smoothFloor)
				{
				PTYPE	pp;

				pp = (d-mindist)/(smoothFloor-mindist);
				pt += p_HF->getEl(x,y)*pp;
				pt += p_HF->getEl(x,y)*p*(1-pp);
				p_HF->setEl (x,y, pt);
				}
			else
			if (smoothFactor > 0 && d<maxdist && d>smoothCeil)
				{
				PTYPE	pp;

				pp = (maxdist-d)/(maxdist-smoothCeil);
				pt += p_HF->getEl(x,y)*p*pp;
				pt += p_HF->getEl(x,y)*(1-pp);
				p_HF->setEl (x,y, pt);
				}
			else
			if (d >= smoothFloor && d <= smoothCeil)
				p_HF->setEl (x,y, p_HF->getEl(x,y)*pt);

			//printf ("%f, %f\n", p, p_HF->getEl(x,y));
			//fflush (stdout);
			}

	p_HF->setSaved (FALSE);
	return (0);
}


/*
 *  mirror: mirror the height field on one of it's axis.
 *  Type can be 1, 2, 3, 4 for x-axis, y-axis, top (left
 *  to bottom right) or bottom (left to top right)
 */
int HeightFieldOps::mirror (int type)
{
	int	mid, 
		x, y,
		w = p_HF->getWidth(), 
		h = p_HF->getHeight();
	PTYPE	t;
	

	// Mirror around X axis
	if (type == 1)
		{
		GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, 
			"Mirroring around X axis ...\n");

		mid = h/2;
		for (y=0; y<mid; y++)
		   for (x=0; x<w; x++)
			{
			t = p_HF->getEl (x,y);
			p_HF->setEl (x, y, p_HF->getEl(x,h-y-1));
			p_HF->setEl (x, h-y-1, t);
			}
		}
	else
	// Mirror around Y axis
	if (type == 2)
		{
		GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, 
			"Mirroring around Y axis ...\n");

		mid = w/2;
		for (x=0; x<mid; x++)
		    for (y=0; y<h; y++)
			{
			t = p_HF->getEl (x,y);
			p_HF->setEl (x,y, p_HF->getEl(w-x-1,y));
			p_HF->setEl (w-x-1,y, t);
			}
		}
	else
	// Mirror diagonally: this code will only work on square height fields
	if (type == 3 || type == 4)
		{
		if (w != h)
			{
			printf ("HeightFieldOps::mirror(): HF is not square ... cancelling\n");
			return (-1);
			}

		if (type == 3)
		    {
		    GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, 
			"Mirroring around TL-BR axis ...\n");
		    for (y=0; y<h; y++)
				for (x=y; x<w; x++)
					{
					t = p_HF->getEl (x,y);
					p_HF->setEl (x,y, p_HF->getEl(y,x));
					p_HF->setEl (y,x, t);
					}
		    }
		else
		    {
		    GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, 
			"Mirroring around BL-TR axis ...\n");
		    for (y=0; y<h; y++)
				for (x=0; x<w-y; x++)
					{
					t = p_HF->getEl (x,y);
					p_HF->setEl (x,y, p_HF->getEl(w-y-1,h-x-1));
					p_HF->setEl (w-y-1,h-x-1, t);
					}
		    }
		}
	else
		SanityCheck::bailout ((1), "Uncaught mirror type operation requested", "HeightFieldOps::mirror");

	return (0);
}


/*
 *  fill: fill the Height Field up to elv
 */
int HeightFieldOps::fill (PTYPE	elv, float factor, bool keepMin)
{
	int	n=0;
	float	p;
	char	buf[80];

	//if (!p_HF->d_hf)
	//	{
	//	printf ("Warning: hf==NULL in HeightFieldOps::Fill ... operation cancelled\n");
	//	return (-1);
	//	}

	if (elv < 0 || elv > 1)
		{
		cerr << "Error: elevation out of range (" << 
			elv << ") in HeightFieldOps::Fill() ...\n";
		return (-1);
		}

	sprintf (buf, "Filling to elevation %f, delta %f%% ... ", elv, (1-factor)*100);
	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, buf);

	if (keepMin)
		{
		for (int i=0; i<p_HF->getSize(); i++)
			if (p_HF->getEl(i)<elv && p_HF->getEl(i)>p_HF->getMin())
				{
				p = elv-p_HF->getEl(i);
				p -= (p*factor);
				p_HF->setEl(i, elv - p);
				n++;
				}
	    	}
	else
		{
		for (int i=0; i<p_HF->getSize(); i++)
			if (p_HF->getEl(i)<elv)
				{
				p = elv-p_HF->getEl(i);
				p -= (p*factor);
				p_HF->setEl(i, elv - p);
				n++;
				}
	   	}

	sprintf (buf, "filled %d points\n", n);
	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, buf);

	p_HF->setSaved (FALSE);
	return (0);
}

//////////////////////////////////////////////////////////////////////////////

/*
 *  contour lines: generate contour lines (also called Isograms) and export 
 *	them to postscript. 
 * 
 * 	The return value is a pointer to a TFGListFlexArr* of FlexArray items 
 * 	which each holds a list of pointers to COORD2 structures (defined 
 * 	in MathTrig.h). 
 */
TFGListFlexArr* HeightFieldOps::contourLines (int nElvLevels, float epsilon, 
					int squareSize)
{
	SanityCheck::bailout ((!nElvLevels), "nElvLevels==0", "HeightFieldOps::HeightFieldOps");
	SanityCheck::bailout ((epsilon<=0), "epsolin<=0", "HeightFieldOps::HeightFieldOps");
	SanityCheck::bailout ((epsilon>=1), "epsolin>=1", "HeightFieldOps::HeightFieldOps");

	int		x, y, n, off;
	char		buf[80];
	COORD2		*coord2=NULL;		// 2D coordinate
	TFGListFlexArr	*listMain = NULL;	// list of flexarrarys
	TFFlexArrayCoord2 *faCLine = NULL;	// flexarray of 2d coordinates
	PTYPE           levelElv = 1.0/(PTYPE)nElvLevels,
	                halfLevelElv=levelElv/2.0, // half an elevation level
	                el;

	// init variables
	p_clBuffer = new bool[p_HF->getSize()];
	memset (p_clBuffer, 0, sizeof(bool)*p_HF->getSize());
	d_clPoints = 0;
	d_clSquareSize = squareSize;
	listMain = new TFGListFlexArr ();
        d_clTargetElv = -levelElv/2.0;		// cE + lE = center of band
	//d_clLastRWidth = 0;


	sprintf (buf, "Finding %d contour lines with epsilon %.2f ... ", nElvLevels, epsilon);
	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, buf);


	// start looking for points on a contour line
	for (n=0; n<nElvLevels; n++)
	    {
	    d_clTargetElv += levelElv;
	    d_clLevelLow = d_clTargetElv - halfLevelElv*epsilon;
	    d_clLevelHigh = d_clTargetElv + halfLevelElv*epsilon;
	    for (y=0; y<p_HF->getHeight(); y++)
	      for (x=0; x<p_HF->getWidth(); x++)
		    {
		    off = p_HF->Pos2Off(x,y);
		    if (!p_clBuffer[off])
			  {
			  el = p_HF->getEl (off);
			  if (el >= d_clLevelLow && el <= d_clLevelHigh)
				{
				coord2 = new COORD2;
				coord2->x = x;
				coord2->y = y;
				faCLine = new TFFlexArrayCoord2 (100, 2);
				faCLine->append (coord2);
				p_clBuffer[off] = TRUE;
				clFollow (faCLine, x, y);

				if (faCLine->getSize() >= 3)
					{ 
					faCLine->trim ();
					listMain->append (faCLine);
					d_clPoints += faCLine->getSize();
					}
				else
					delete faCLine;
				}
			  }
	    	}
	     }

	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, "done\n");
	sprintf (buf, "Found %d contour lines, %d points (avg. length = %.2f)\n", 
		listMain->length(), d_clPoints, ((float)d_clPoints)/((float)listMain->length()));
	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, buf);

	delete [] p_clBuffer;
	p_clBuffer = NULL;

	return listMain;
}


/*
 *  clFollow: follow the contour line (Isogram) as if sniffing a 
 * 	trail. Check the cell's neighbours for and cells that could be 
 * 	on the countour line and recurse down this path, updating the 
 * 	coordinate cache (which is used as a marker to indicate that 
 * 	a certain point has already been used in a contour line). 
 */
void HeightFieldOps::clFollow (TFFlexArrayCoord2 *faCLine, int x, int y)
{
	SanityCheck::bailout ((!p_clBuffer), "p_clBuffer==NULL", "HeightFieldOps::clFollow");
	SanityCheck::bailout ((!faCLine), "faCLine==NULL", "HeightFieldOps::clFollow");

	PTYPE	*bestFit=NULL;
	COORD2	*coord2=NULL;		// 2D coordinate

	bestFit = clCheckSquareR (x, y, d_clSquareSize);

	// store coordinates and continue recursing
	if (bestFit)
		{
		int	d = bestFit - p_HF->getData();

		coord2 = new COORD2;
		coord2->x = p_HF->Off2Posx(d);
		coord2->y = p_HF->Off2Posy(d);
		p_clBuffer[d] = TRUE;
		faCLine->append (coord2);

		// since contour lines could get pretty long, recursion depth 
		// becomes an issue. Thus, it is important that this function 
		// is tail-recursive.
		clFollow (faCLine, coord2->x, coord2->y);
		}
}


/*
 *  clCheckSqaureR: check the maxWidth sized square around x,y to find the 
 * 	vertex closest to el==levelElv.  the previous code gave preference
 *      to vertices that were closer to the starting vertex.  this one just
 *      finds the closest thing within the user's setting for "Square Radius".
 */
PTYPE *HeightFieldOps::clCheckSquareR (int x, int y, int width)
{
	SanityCheck::bailout ((width<1), "width<1", "HeightFieldOps::clCheckSqaureR");
	// SanityCheck::bailout ((width>maxWidth), "width>maxWidth", "HeightFieldOps::clCheckSqaureR");

	int 	i, j, off, 
		iLow = (x >= width ? x-width : 0),
		iHigh = (x < (p_HF->getWidth()-width) ? x+width : p_HF->getWidth()-1),
		jLow = (y >= width ? y-width : 0),
		jHigh = (y < (p_HF->getHeight()-width) ? y+width : p_HF->getHeight()-1);
	PTYPE	*bestFit=NULL, 
	        el; 

	//printf ("C: %d, %d,    %d, %d,    %d, %d\n", x, y, iLow, iHigh, jLow, jHigh);
	//fflush (stdout);

	for (i=iLow; i<=iHigh; i++)
	    for (j=jLow; j<=jHigh; j++) 
		{
	        off = p_HF->Pos2Off(i,j);
	        el  = p_HF->getEl(off);
	        if (i != x || j!= y)  		// if it isn't the starting point
		  if (!p_clBuffer[off] &&   	// if we haven't looked at it before,
		      el >= d_clLevelLow && 	// and if it's in range, then
		      el <= d_clLevelHigh)  	// see if it's a new best fit.
		      {
		      bestFit = clCheckBestFit (bestFit, p_HF->getData()+off, off);

		      // if we've looked at a cell, never consider it again:
		      p_clBuffer[off] = TRUE;
		      }
		}

	// this new conception of clCheckSquareR isn't recursive.
	//	d_clLastRWidth = width;
	//	if (width < maxWidth && !bestFit)
	//		bestFit = clCheckSquareR (x, y, width+1, maxWidth);
	//	else
	//	if (width >= maxWidth)
	//		d_clLastRWidth = 0;

	return bestFit;
}



/*
 *  clCheckBestFit: check if a new cell is better than the best so far.
 *      this routine doesn't need to worry about checking p_clBuffer because
 *      previous routines have already done so.  besides, semantically speaking
 *      it isn't this function's job to do so; this function is simply passed
 *      two cells and told to return whichever one is better.
 */
PTYPE *HeightFieldOps::clCheckBestFit (PTYPE *bestFit, PTYPE *cell, int off)
{
	PTYPE	deltaElv,
		el = p_HF->getEl(off); 

	if (!bestFit)  // if we don't have a best fit yet, then *cell is it.
		{
		bestFit = cell;
		d_clCurBestElv = ABS(d_clTargetElv - el);
		}
	else  // if we do have a best fit already, then do further checking...
		{
		deltaElv = ABS (el-d_clTargetElv);
		if (deltaElv < d_clCurBestElv)
			{
			bestFit = cell;
			d_clCurBestElv = deltaElv;
			}
	    	}

	return bestFit;
}


//////////////////////////////////////////////////////////////////////////////


/* Add craters to a heightfield array 
 * Copyright (C) 1995 by Heiko Eissfeldt
 * heiko@colossus.escape.de
 * modified JAN 1996 John Beale for use with HF-Lab  
 * modified JUN 1999 Robert Gasch for use with Terraform
 */


#define alpha   (35.25/180.0*M_PI)      /* sphere segment cutoff angle */
#define crater_depth 0.85  /* 0.9 */


static double a1 = crater_depth, b, d;


/* 
 * crProfile: method to determine the z-elevation dependent on normalized 
 * 	squared radial coordinate 
 */
double HeightFieldOps::crProfile (double nsq_rad)
{
	static 	double c = 50.0; // 100.0;	// controls the wall thickeness 
	double 	radialcoord;	
	radialcoord = sqrt(fabs(nsq_rad));

	if (radialcoord > b) 
     		// outer region gauss distribution 
		return d*exp(-c*(radialcoord-b)*(radialcoord-b));
	else 
		// inner region sphere segment 
		return a1 - sqrt(1.0 - nsq_rad);
}



#define pow4(x) ((x)*(x)*(x)*(x))
#define border 0.63
#define inner_scale 1
#define outer_scale (1.0-inner_scale*pow4(border))/pow4(1.0-border)

/* 
 * crDissolve: function to control the weight of the crater profile as opposed
 * to the underlying terrain elevation 
 */
double HeightFieldOps::crDissolve (double nsq_rad)
{
#if 0
	if (nsq_rad > border*border) 
		{
		double 	temp = (1-sqrt(nsq_rad));
		return 1.0-outer_scale*pow4(temp);
		} 
	else 
		return inner_scale*nsq_rad*nsq_rad;
#else
	if (nsq_rad > 0.6*0.6) 
		return 1.0-0.9*exp(-30.0*(sqrt(nsq_rad) - 0.6)*(sqrt(nsq_rad) - 0.6));
	else 
		return 0.1*exp(-25.0*(sqrt(nsq_rad) - 0.6)*(sqrt(nsq_rad) - 0.6));
# if 0
	if (nsq_rad > border*border) 
		{
		double 	temp = (1-sqrt(nsq_rad));
		return 1.0-outer_scale*pow4(temp);
		} 
	else 
		{
		return 1.0- 4.0*(sqrt(nsq_rad) - 0.17)*(sqrt(nsq_rad) - 0.17);
		return 0.04+1.7*(sqrt(nsq_rad) - 0.38)*(sqrt(nsq_rad) - 0.38);
		}
# endif
#endif
}



//#define craterCoverage 0.15
//#define craterCoverage 0.50
#define X_WRAP(x)  x=(((x)>=0)?(x)%p_HF->getWidth():(((x)%p_HF->getWidth())+p_HF->getWidth())%p_HF->getWidth())
#define Y_WRAP(y)  y=(((y)>=0)?(y)%p_HF->getHeight():(((y)%p_HF->getHeight())+p_HF->getHeight())%p_HF->getHeight())


/* 
 * distributeCraters:  doing some damage to the surface 
 */
int HeightFieldOps::distributeCraters(unsigned int how_many, 
        bool wrap, D ch_scale, D cr_scale, D craterCoverage, int seed,
	float cx, float cy)
{
    int 	i, j, k, xloc, yloc, cratersize;
    int 	sq_radius;
    double 	nsq_rad, weight;
    int 	lev_samples, samples;
    double 	level_with, level_pure;
    double 	craterscale;		// vertical crater scaling factor 
    double 	c,d2, shift;
    double 	b1,b2,b3;		// radius scaling params 
    int 	meshsize;		// mean of xsize and ysize ? 
    int		lim = p_HF->getSize();
    PTYPE	*hfOrig = new PTYPE[lim];
    MathRandom	*rnd, *rndSeed;
    char	buf[80];

    SanityCheck::bailout ((how_many>1 && (cx!=-1||cy!=-1)), "impossible parameter combination", "HeightFieldOps::HeightFieldOps");
    SanityCheck::bailout ((how_many==1 && (cx<0||cy<0)), "impossible parameter combination", "HeightFieldOps::HeightFieldOps");

    sprintf (buf, "Distributing %d craters (%f, %f)\n", how_many,
		ch_scale, cr_scale);
    GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, buf);

    rnd = new MathRandom (seed);
    rndSeed = new MathRandom (rnd->rnd());
    b1 = 10.0;
    b2 = 1.0/b1/b1/b1/b1; 
    b3 = 1.0/b1;
    meshsize = (int)(sqrt(p_HF->getWidth() * p_HF->getHeight()));
    b = sin(alpha);
    d = a1 - cos(alpha);

    // copy the terrain 
    memcpy (hfOrig, p_HF->getData(), lim*sizeof(PTYPE)); 

    k = how_many;
    while (k > 0) 
	{
	int 		ii, jj;
	MathRandom	*rndLoop = new MathRandom (rndSeed->rnd());

	/* pick a cratersize according to a power law distribution */
	/* greatest first. So great craters never eliminate small ones */
	/* c = (double)(how_many-k+1)/how_many + b3; */
	/* craters appear in random order */

	c = rnd->rnd1() + b3;	/* c is in the range b3 ... b3+1 */
	d2 = b2/c/c/c/c;	/* d2 is in the range 0 ... 1 */

	//if (how_many == 1) d2 = 1.0;   /* single craters set to max. size */

	if (how_many == 1)
		{
		xloc = (int)(cx*p_HF->getWidth());
		yloc = (int)(cy*p_HF->getWidth());
		cratersize = (int)(cr_scale*((p_HF->getWidth()+p_HF->getHeight())/2));
		}
	else
		{
		xloc = (int)(rnd->rnd1()*p_HF->getWidth());  /* pick a random location for the crater */
		yloc = (int)(rnd->rnd1()*p_HF->getHeight());
		cratersize = 3 + (int)(d2 * craterCoverage * meshsize * cr_scale);
		}

/* macro to determine the height dependent on crater size */
#define CRATER_SCALE (((ch_scale*pow((cratersize/(3+craterCoverage*meshsize)),0.9)) \
		       /256*pow(meshsize/256.0,0.1))/craterCoverage*80)

	craterscale = CRATER_SCALE;     /* vertical crater scaling factor */
	
	/* what is the mean height of this plot */
	samples = lev_samples = MAX((cratersize*cratersize)/5, 1);
	level_with = level_pure = 0.0;
	while (samples) 
	    {
	    i = (int)(rndLoop->rnd1()*(2*cratersize) - cratersize);
	    j = (int)(rndLoop->rnd1()*(2*cratersize) - cratersize);

	    if (i*i+j*j > cratersize*cratersize) continue;
	    ii = xloc + i; jj = yloc + j;

	    // handle wrapping details... 
	    if ((wrap) || ((ii >= 0) && (ii < p_HF->getWidth()) && (jj >= 0) && (jj < p_HF->getHeight()))) 
		{
		X_WRAP(ii);
		Y_WRAP(jj);
		level_with += p_HF->getEl(ii, jj);
		level_pure += hfOrig[p_HF->Pos2Off(ii, jj)];
		samples--;
	    	}

	    }
	level_with /= lev_samples;
	level_pure /= lev_samples;

	/* Now lets create the crater. */

	/* In order to do it efficiently, we calculate for one octant
	 * only and use eightfold symmetry, if possible.
	 * Main diagonals and axes have a four fold symmetry only.
	 * The center point has to be treated single.
	 */

#define SQUEEZE 1.3

/* this macro calculates the coordinates, does clipping and modifies
 * the elevation at this spot due to shift and weight.
 * hfOrig() contains the crater-free underground. p_HF() contains the result.
 * level_with: average altitude of cratered surface in region
 * level_pure: average altitude of uncratered surface
 */

#define SHIFT(x,y) { \
    ii = xloc + (x); jj = yloc + (y); \
    if (wrap || ((ii >= 0) && (ii < p_HF->getWidth()) && \
		 (jj >= 0) && (jj < p_HF->getHeight()))) {\
      X_WRAP(ii);  Y_WRAP(jj); \
      p_HF->setEl (ii,jj, (shift  + (p_HF->getEl(ii,jj))*weight + \
      (level_with + (hfOrig[p_HF->Pos2Off(ii, jj)]-level_pure)/SQUEEZE) * (1.0-weight))); \
    } \
  }

/* macro to do four points at once. Points are rotated by 90 degrees. */
#define FOURFOLD(i,j)   SHIFT(i,j) SHIFT(-j,i) SHIFT(-i,-j) SHIFT(j,-i)

/* get eightfold symmetry by mirroring fourfold symmetry along the x==y axe */
#define EIGHTFOLD       FOURFOLD(i,j) FOURFOLD(j,i)


	/* The loop covers a triangle (except the center point)
	 * Eg cratersize is 3, coordinates are shown as i,j
	 *
	 *              3,3 j
	 *         2,2  3,2 |
	 *    1,1  2,1  3,1 v
	 * x  1,0  2,0  3,0
	 * ^          <-i
	 * |
	 * center point
	 *
	 * 2,1 , 3,2 and 3,1 have eightfold symmetry.
	 * 1,0 , 2,0 , 3,0 , 1,1 , 2,2 and 3,3 have fourfold symmetry.
	 */

	for (i = cratersize; i > 0; i--) {
	    for (j = i; j >= 0; j--) {

		/* check if outside */
		sq_radius = i*i+j*j;
		nsq_rad = (double)sq_radius/cratersize/cratersize;
		if (nsq_rad > 1) continue;

		/* inside the crater area */
		shift = craterscale*crProfile(nsq_rad);
		weight = crDissolve(nsq_rad);

		if (i==j || j==0) {
		    FOURFOLD(i,j)
		} else {
		    EIGHTFOLD
		}
	    }
	}
	/* the center point */
	shift =  craterscale*crProfile(0.0);
	weight = crDissolve(0.0);
	SHIFT(0,0)

	/* one crater added */
	k--;

	if ((how_many - k) % 1000 == 0)
    		GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, ".");
	delete rndLoop;
	}

    delete hfOrig;
    delete rndSeed;
    delete rnd;
    p_HF->setSaved (FALSE);
    GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, "done\n");
    return(0);
} 

//////////////////////////////////////////////////////////////////////////////
