/*
  Module       : image.c
  Purpose      : Routines dealing with image display
  More         : see qiv README
  Policy       : GNU GPL
  Homepage     : http://www.klografx.de/
*/	

#include <stdio.h>
#include <string.h>
#include <gdk/gdkx.h>
#include <sys/time.h>
#include "qiv.h"

/*
 *	Load & display image
 */

void qiv_load_image()
{
    if (im) {
      gdk_imlib_kill_image(im); /* Discard previous image */
      im = NULL;
    }

    im=gdk_imlib_load_image(image_names[image_idx]);

    if (!im) { /* error */
      im=NULL;
      image_error = 1;
      w=org_w=500;
      h=org_h=400;
    }
    else { /* Retrieve image properties */
      image_error = 0;
      w=org_w=im->rgb_width;
      h=org_h=im->rgb_height;
    }

    reset_display_settings(&win_x,&win_y,&w,&h);

      /* desktop-background -> exit */


    if (to_root==1||to_root_t==1||to_root_s==1) {
      if (!im) {
	fprintf(stderr, "qiv: cannot load background_image\n");
        gdk_exit(1);
      }  
      set_desktop_image();
      gdk_exit(0);
    }

    win_x=(screen_x - w) / 2;
    win_y=(screen_y - h) / 2;

    if (first) {
      if (!fullscreen) {
	attr.window_type=GDK_WINDOW_TOPLEVEL;
	attr.wclass=GDK_INPUT_OUTPUT;
	attr.event_mask=GDK_ALL_EVENTS_MASK;
	attr.x=0;
	attr.y=0;
	attr.width=org_w;
	attr.height=org_h;
	win=gdk_window_new(NULL,&attr,ATTRIBUTES_MASK);
	gdk_window_set_cursor(win,cursor);
	gdk_window_set_hints(win,win_x,win_y,w,h,w,h,GDK_HINT_MIN_SIZE |
			     GDK_HINT_MAX_SIZE | GDK_HINT_POS);
			     
	if (!wm_handle) {
	  gdk_window_hide(win);
	  if (center) {
	    gdk_window_move_resize(win,win_x,win_y,w,h);
	  }else{
	    gdk_window_resize(win,w,h);
	  }
	  gdk_window_show(win);
	}
	else {
	  gdk_window_show(win);
	  if (center) {
	    gdk_window_move_resize(win,win_x,win_y,w,h);
	  }else{
	    gdk_window_resize(win,w,h);
	  }
	}

      }
      else {
	attr.window_type=GDK_WINDOW_TEMP;
	attr.wclass=GDK_INPUT_OUTPUT;
	attr.event_mask=GDK_ALL_EVENTS_MASK;
	attr.x = attr.y = 0;
	attr.width=screen_x;
	attr.height=screen_y;
	parent=NULL;
	win=gdk_window_new(parent,&attr,ATTRIBUTES_MASK);
	gdk_window_set_cursor(win,cursor);
	gdk_window_show(win);
	gc = gdk_gc_new(win);
      }
      first = 0;
    }
    if (!im) {
      gdk_window_set_background(win,&color_blue);
    }else{
      gdk_window_set_background(win,&color_bg);
    }
    if(do_grab || fullscreen) {
      gdk_keyboard_grab(win,FALSE,CurrentTime);
      gdk_pointer_grab(win,FALSE,event_mask,NULL,cursor,CurrentTime);
    }

    update_image(image_error,win_x,win_y,w,h,mod);

}

/* set image as background */
void set_desktop_image(void)
{
  GdkVisual         *gvis;
  GdkWindow         *rwin;
  GdkPixmap         *temp;
  GdkGC             *rootGC;
  GdkPixmap         *m = NULL;
  gchar             *buffer;
  
  gint w=screen_x,h=screen_y;
  gint win_x=0, win_y=0;


  if(to_root || to_root_t) {
    w = org_w;
    h = org_h;
  }
  if(to_root) {
    win_x=(screen_x - w) / 2;
    win_y=(screen_y - h) / 2;
  }
  
  gdk_imlib_set_image_modifier(im,&mod);
  gdk_imlib_render(im,w,h);
  p=gdk_imlib_move_image(im);

  if (transparency)
    m=gdk_imlib_move_mask(im);

  rwin = (GdkWindow *) & gdk_root_parent;
  gvis = gdk_window_get_visual(rwin);
  if (gvis != gdk_imlib_get_visual()) {
    fprintf(stderr, "qiv: **WARNING** Your root window's visual is not the visual Imlib chose\nqiv cannot set the background currently\n");
    return;
  }
  
  if (to_root_t) {
    gdk_window_set_back_pixmap(rwin,p,FALSE);
  }
  else {
    buffer = malloc(screen_x*screen_y);
    memset(buffer,0x00,screen_x*screen_y);
    rootGC = gdk_gc_new(rwin);
    temp = gdk_pixmap_create_from_data(rwin, buffer, screen_x, 
				       screen_y, gvis->depth, &color_bg,
				       &color_bg);
    gdk_draw_pixmap(temp,rootGC,p,0,0,win_x,win_y,w,h);
    gdk_window_set_back_pixmap(rwin,temp,FALSE);
  }
  
    gdk_window_clear(rwin);
    gdk_flush();
}

void zoom_in(gint *win_x,gint *win_y,gint *w,gint *h)		/* zoom in */
{
  *w += (gint)(org_w * 0.1);
  *h += (gint)(org_h * 0.1);
  *win_x = (screen_x - *w) / 2;
  *win_y = (screen_y - *h) / 2;
}

void zoom_out(gint *win_x,gint *win_y,gint *w,gint *h)		/* zoom out */
{
  *w -= (gint)(org_w * 0.1);
  *h -= (gint)(org_h * 0.1);
  *win_x = (screen_x - *w) / 2;
  *win_y = (screen_y - *h) / 2;
}

void zoom_maxpect(gint *win_x,gint *win_y,gint *w,gint *h)
{
  double zx, zy;
  zx = (double)screen_x / (double)org_w;
  zy = (double)screen_y / (double)org_h;
  *w = (gint)(org_w * MIN(zx,zy));
  *h = (gint)(org_h * MIN(zx,zy));
  *win_x = (screen_x - *w) / 2;
  *win_y = (screen_y - *h) / 2;
}

/*
  Set display settings to startup values
  which are used whenever a new image is loaded.
*/
void reset_display_settings(gint *win_x,gint *win_y,gint *w,gint *h)
{
  if(rotate_press && return_press) {
    gdk_imlib_destroy_image(im);
    im=gdk_imlib_load_image(image_names[image_idx]);
    *w=org_w=im->rgb_width;
    *h=org_h=im->rgb_height;
    *win_x = (screen_x - *w) / 2;
    *win_y = (screen_y - *h) / 2;
    rotate_press = return_press = 0;
  }

  if (maxpect || ((*w > screen_x || *h > screen_y) && scale_down == 1))
    zoom_maxpect(win_x,win_y,w,h);
  else {
    *w = org_w;
    *h = org_h;
    *win_x = (screen_x - *w) / 2;
    *win_y = (screen_y - *h) / 2;
  }
}

/* Something changed the image.  Redraw it. */
void update_image(char image_error, gint win_x, gint win_y, gint w, gint h, GdkImlibColorModifier mod)
{
  gint length;			/* Length of the Title-String */
  GdkPixmap *m = NULL;
  gchar *win_title[150];
  double elapsed;		/* for the render time  */
  struct timeval before, after;
  GdkGC *statusbar_black_gc = NULL;	/* Background for the statusbar-background ;) (black) */
  GdkGC *statusbar_background_gc = NULL;	/* Statusbar background GC (orange) */
  GdkFont *text_font;
  
  text_font = gdk_font_load(STATUSBAR_FONT);

  if (!image_error) {
    gdk_imlib_set_image_modifier(im,&mod);
    gettimeofday(&before, 0);	/* getting start_render_time */
    gdk_imlib_render(im,w,h);	/* Imlib render ... */
    gettimeofday(&after, 0);	/* getting end_render_time */

    /* calculate elapsed time */
    elapsed = ((after.tv_sec  + after.tv_usec  / 1.0e6) -
	       (before.tv_sec + before.tv_usec / 1.0e6));

    gdk_imlib_free_pixmap(p);	/* killing old pixmap */
    p=gdk_imlib_move_image(im);	/* creating new */
  
    if (transparency)
      m=gdk_imlib_move_mask(im);	/* creating transparency */

    sprintf((char *)win_title, "%s (%dx%d) %1.01fs %d%% [%d/%d] b%d/c%d/g%d",
	    image_names[image_idx], org_w, org_h, elapsed,
	    (int)((1.0-(org_w - w)/(double)org_w)*100), image_idx+1, images, mod.brightness/8-32,
	    mod.contrast/8-32, mod.gamma/8-32);
  }
  else {
    sprintf((char *)win_title, "ERROR! cannot load image: %s",
	    image_names[image_idx]);
    gdk_beep();
  }

  length = strlen((const char *)win_title);

  gdk_window_set_title(win,(const gchar*)win_title);
  gdk_window_set_hints(win,win_x,win_y,w,h,w,h,GDK_HINT_MIN_SIZE |
		       GDK_HINT_MAX_SIZE | GDK_HINT_POS );
  if (!fullscreen) {
    if (center) {
      gdk_window_move_resize(win,win_x,win_y,w,h);
    }else{
      gdk_window_resize(win,w,h);
    }
    if (!image_error) {
      gdk_window_set_back_pixmap(win,p,FALSE);
      if (transparency && m)
        gdk_window_shape_combine_mask(win,m,0,0);
    }
    gdk_window_clear(win);
  }
  else {
    gdk_window_clear(win);
    if (!image_error) 			  
      gdk_draw_pixmap(win,gc,p,0,0,win_x,win_y,w,h);
    
    if (statusbar) {	/* draw stausbar */
      statusbar_background_gc = gdk_gc_new(win);
      statusbar_black_gc = gdk_gc_new(win);
      gdk_gc_set_foreground(statusbar_background_gc,&text_background_color);
      gdk_draw_rectangle (win, statusbar_black_gc, 1, screen_x-11-length*7+2,
			  screen_y-19, length*7+4, 16);
      gdk_draw_rectangle (win, statusbar_background_gc, 1, screen_x-10-length*7+2,
			  screen_y-18, length*7+2, 14);
      gdk_draw_text (win, text_font, gc, screen_x-5-length*7-2, screen_y-7,
		     (const char *)win_title, length);
    }
  }
  gdk_window_show(win);		/* display image */
  gdk_flush();
}
