// This file is part of Moonlight Creator
//   Copyright (C) 1996-1998  Stephane Rehel
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/*
   strgcls.cc

   String Class

   Fri Aug 21 1992 /SCFR::Kepler.
   Reviewed: Wed Sep 16 1992

   Revision: June 1994
   Stephane Rehel, SCFR::Kepler, rehel@ensisun.imag.fr

   Revision: July 28th, 1995
   Stephane Rehel [SCFR::Kepler], Stephane.Rehel@imag.fr

   August 9th, 1995 /SR
     add insert() and remove()
*/

#include <assert.h>
#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>

#include "OString.h"


/////////////////////////////////////////////////////////////////////////////

// static

OString OString::nil; // unassigned string

/////////////////////////////////////////////////////////////////////////////

// static
const OString& OString::empty()
{
  static OString* emptyString= 0;
  if( emptyString == 0 )
    emptyString= new OString("");

  return *emptyString;
}

/////////////////////////////////////////////////////////////////////////////
// OString Class
//

void OString::destroy()
{
  if( theString != 0 )
    {
    free( (void*) theString );
    theString= 0;
    }
}

/////////////////////////////////////////////////////////////////////////////

void OString::alloc( size_t _bufferLength )
{
  destroy();
  _alloc( _bufferLength );
  theLength= 0;
  *theString= 0;
}

/////////////////////////////////////////////////////////////////////////////

// This "inline" is due to a bug of pgcc-2.90.23 980102 (egcs-1.0.1)
// To be followed!
inline void OString::realloc( size_t newLength )
{
  assert( assigned() );
  assert( newLength >= theLength );

  if( newLength+1 <= bufferLength )
    return;
  char* newString= (char*) ::realloc( (void*) theString,
                                      (newLength+1) * sizeof(char) );

//assert( newString != 0 );
//  if( newString == 0 )
//    ThrowException( ERR_OUTOFMEMORY );

  theString= newString;
  bufferLength= newLength+1;
}

/////////////////////////////////////////////////////////////////////////////

void OString::set( const char* initStrg, size_t strgLength )
{
  if( initStrg == theString )
    return; // user is doing: strg= strg.get(); !!!

  // careful: user could do: strg= strg.get() + 1

  OString temp;
  temp._alloc(strgLength);
  if( strgLength > 0 )
    memcpy( (void*) temp.theString, (void*) initStrg, strgLength );

  temp.theString [ strgLength ] = 0;

  destroy();

  theString= temp.theString;
  theLength= strgLength;
  bufferLength= temp.bufferLength;
  temp.theString= 0;
}

/////////////////////////////////////////////////////////////////////////////

void OString::_alloc( size_t length )
{
  assert( theString == 0 );
  theString= (char*) malloc( (length+1) * sizeof(char) );

assert( theString != 0 );
//  if( theString == 0 )
//    ThrowException( ERR_OUTOFMEMORY );

  bufferLength= length+1;
}

/////////////////////////////////////////////////////////////////////////////

OString operator + ( const OString& os1, const OString& os2 )
{
  assert( os1.assigned() );
  assert( os2.assigned() );

  OString os( os1.theLength + os2.theLength );

  strcpy( os.theString, os1.theString );
  strcpy( os.theString+os1.theLength, os2.theString );
  os.theLength= os1.theLength + os2.theLength;

  return os;
}

/////////////////////////////////////////////////////////////////////////////

OString OString::operator + ( char ch )
{
  assert( assigned() );
  assert( ch != 0 );

  OString os( theLength + 1 );

  strcpy( os.theString, theString );
  os.theLength= theLength;

  os.theString[os.theLength++]= ch;
  os.theString[os.theLength]= 0;

  return os;
}

/////////////////////////////////////////////////////////////////////////////

OString& OString::operator += ( const char* s )
{
  assert( assigned() );
  assert( s != 0 );

  size_t len= strlen(s);
  int totalLength= theLength + len;

  // careful: user could do: strg += strg.get() + 1
  OString temp;
  temp._alloc(totalLength);
  strcpy( temp.theString, theString );
  strcpy( temp.theString+theLength, s );

  destroy();
  theString= temp.theString;
  theLength= totalLength;
  bufferLength= temp.bufferLength;
  temp.theString= 0;

//  realloc( theLength + len );
//  strcat( theString, s );
//  theLength+= len;

  return *this;
}

/////////////////////////////////////////////////////////////////////////////

OString& OString::operator += ( const OString& os )
{
  assert( assigned() );
  assert( os.assigned() );

  return operator += ( os.get() );
}

/////////////////////////////////////////////////////////////////////////////

OString& OString::operator += ( char ch )
{
  assert( assigned() );
  assert( ch != 0 );

  realloc( theLength + 5 );
  theString[theLength++]= ch;
  theString[theLength]= 0;
  return *this;
}

/////////////////////////////////////////////////////////////////////////////

OString OString::getSub( int i1, int i2 ) const
{
  assert( assigned() );
  if( i1 < 0 )
    i1= 0;
  if( i2 >= int(theLength) )
    i2= theLength - 1;
  if( i1 > i2 )
    return OString("");

  OString os( i2 - i1 + 1 );
  os.theLength= i2 - i1 + 1;
  size_t i;
  for( i= 0; i< os.theLength; ++i )
    os.theString[i]= theString[i1+i];
  os.theString[i]= 0;

//  strncpy( os.theString, theString + i1, i2 - i1 + 1 );
//  os.theLength= i2 - i1 + 1;

  return os;
}

/////////////////////////////////////////////////////////////////////////////

// 0 <= i <= length()
void OString::insert( size_t i, char ch )
{
  assert( assigned() );
  assert( ch != 0 );
  assert( i <= theLength );

  realloc( theLength + 5 );
  assert( theString != 0 );
  int j= int(theLength);
  while( j >= int(i) )
    {
    assert( j+1 < int(bufferLength) );
    assert( j >= 0 );
    theString[j+1]= theString[j];
    --j;
    }
  theString[i]= ch;
  ++theLength;
}

/////////////////////////////////////////////////////////////////////////////

// 0 <= i
void OString::remove( size_t i )
{
  assert( assigned() );
  if( i >= theLength )
    return;

  int j= int(i);
  while( j < int(theLength) )
    {
    assert( j+1 < int(bufferLength) );
    assert( j >= 0 );
    theString[j]= theString[j+1];
    ++j;
    }
  --theLength;
}

/////////////////////////////////////////////////////////////////////////////

void OString::fill( size_t length, int ch )
{
  if( ! assigned() )
    _alloc(length);
   else
    {
    theLength= 0;
    realloc( length );
    }

  assert( assigned() );

  if( length != 0 )
    memset( (void*) theString, ch, length );

  theString [ length ] = 0;

  theLength= (ch==0) ? 0 : length;
}

/////////////////////////////////////////////////////////////////////////////

// 0 <= i < theLength
void OString::put( size_t i, char ch )
{
  assert( assigned() );

  assert( i < theLength );
  theString [ i ] = ch;
}

/////////////////////////////////////////////////////////////////////////////
// From ImageMagick 3.6.5: magick/utility.c: GlobExpression(...)
//
// static
boolean OString::glob( const char* expression, const char* pattern )
{
  if( pattern == 0 )
    return true;

  if( strlen(pattern) == 0 )
    return true;
  if( strcmp( pattern, "*" ) == 0 )
    return true;

  boolean done= false;
  while( (*pattern != '\0') && !done )
    {
    if( *expression == '\0' )
      if( (*pattern != '{') && (*pattern != '*') )
        break;
    switch( *pattern )
      {
      case '\\':
        {
        pattern++;
        if( *pattern != '\0' )
          pattern++;
        break;
        }

      case '*':
        {
        pattern++;
        boolean status= false;
        while( (*expression != '\0') && !status )
          status= glob( expression++, pattern );
        if( status )
          {
          while (*expression != '\0')
            expression++;
          while (*pattern != '\0')
            pattern++;
          }
        break;
        }

      case '[':
        {
        pattern++;
        for(;;)
          {
          if( (*pattern == '\0') || (*pattern == ']') )
            {
            done= true;
            break;
            }
          if( *pattern == '\\' )
            {
            pattern++;
            if( *pattern == '\0' )
              {
              done= true;
              break;
              }
            }
          if( *(pattern+1) == '-' )
            {
            char c= *pattern;
            pattern+= 2;
            if( *pattern == ']' )
              {
              done= true;
              break;
              }
            if( *pattern == '\\' )
              {
              pattern++;
              if( *pattern == '\0' )
                {
                done= true;
                break;
                }
              }
            if( (*expression < c) || (*expression > *pattern) )
              {
              pattern++;
              continue;
              }
            }
           else
            if( *pattern != *expression )
              {
              pattern++;
              continue;
              }
          pattern++;
          while( (*pattern != ']') && (*pattern != '\0') )
            {
            if( (*pattern == '\\') && (*(pattern+1) != '\0') )
              pattern++;
            pattern++;
            }
          if( *pattern != '\0' )
            {
            pattern++;
            expression++;
            }
          break;
          } // end for(;;)
        break;
        } // end case '['

      case '?':
        {
        pattern++;
        expression++;
        break;
        }

      case '{':
        {
        pattern++;
        while( (*pattern != '}') && (*pattern != '\0') )
          {
          const char* p= expression;
          boolean match= true;
          while( (*p != '\0') && (*pattern != '\0') &&
                 (*pattern != ',') && (*pattern != '}') && match )
            {
            if( *pattern == '\\' )
              pattern++;
            match= (*pattern == *p);
            p++;
            pattern++;
            }
          if( *pattern == '\0' )
            {
            match= false;
            done= true;
            break;
            }
           else
            if( match )
              {
              expression=p;
              while ((*pattern != '}') && (*pattern != '\0'))
                {
                pattern++;
                if( *pattern == '\\' )
                  {
                  pattern++;
                  if( *pattern == '}' )
                    pattern++;
                  }
                }
              }
             else
              {
              while( (*pattern != '}') && (*pattern != ',') &&
                     (*pattern != '\0') )
                {
                pattern++;
                if( *pattern == '\\' )
                  {
                  pattern++;
                  if( (*pattern == '}') || (*pattern == ',') )
                    pattern++;
                  }
                }
              }
          if( *pattern != '\0' )
            pattern++;
          } // end while
        break;
        } // end case '{'

      default:
        {
        if( *expression != *pattern )
          done= true;
         else
          {
          expression++;
          pattern++;
          }
        }
      } // end switch
    } // end white

  while( *pattern == '*' )
    pattern++;

  return (*expression == '\0') && (*pattern == '\0');
}


/////////////////////////////////////////////////////////////////////////////

// return first index if char is found
// if not found return -1
int OString::locateChar( char ch ) const
{
  assert( assigned() );

  const char* strg= get();
  int i= 0;

  while( strg[i] != 0 && strg[i] != ch )
    ++i;

  return (strg[i] == 0) ? -1 : i;
}

/////////////////////////////////////////////////////////////////////////////

boolean OString::glob( const char* pattern ) const
{
  assert( assigned() );
  return OString::glob( get(), pattern );
}

/////////////////////////////////////////////////////////////////////////////

boolean OString::glob( const OString& pattern ) const
{
  assert( assigned() );
  assert( pattern.assigned() );

  return OString::glob( get(), pattern.get() );
}

/////////////////////////////////////////////////////////////////////////////

boolean OString::isPattern() const
{
  assert( assigned() );
  const char* s= get();
  while( *s != 0 )
    {
    if( *s == '\\'|| *s == '*' || *s == '?' ||
        *s == '[' || *s == ']' ||
        *s == '{' || *s == '}' )
      return true;

    ++s;
    }

  return false;
}

/////////////////////////////////////////////////////////////////////////////

OString OString::getFirstWord() const
{
  assert( assigned() );

  int len= 0;

  while( (*this)[len] != ' ' && (*this)[len] != '\0' )
    ++len;

  return getFirst(len+1);
}

/////////////////////////////////////////////////////////////////////////////

// remove spaces at the beginning and at the end
OString OString::trim() const
{
  int first= 0;

  while( (*this)[first] == ' ' )
    ++first;

  if( (*this)[first] == '\0' )
    return OString("");

  int last= length()-1;

  while( (*this)[last] == ' ' )
    --last;

  return getSub(first,last);
}

/////////////////////////////////////////////////////////////////////////////

OString operator + ( const OString& os, int i )
{
  static char buffer[32];
  sprintf( buffer, "%d", i );
  return os + OString(buffer);
}

/////////////////////////////////////////////////////////////////////////////

OString operator + ( int i, const OString& os )
{
  static char buffer[32];
  sprintf( buffer, "%d", i );
  return OString(buffer) + os;
}

/////////////////////////////////////////////////////////////////////////////

OString& OString::operator += ( int i )
{
  static char buffer[32];
  sprintf( buffer, "%d", i );
  return operator += ( buffer );
}

/////////////////////////////////////////////////////////////////////////////

// static
OString OString::itoa( int i )
{
  static char buffer[32];
  sprintf( buffer, "%d", i );
  return OString(buffer);
}

/////////////////////////////////////////////////////////////////////////////

void OString::toUpper()
{
  assert( assigned() );

  for( int i= 0; i < int(theLength); ++i )
    theString[i]= (char) toupper( theString[i] );
}

/////////////////////////////////////////////////////////////////////////////

void OString::toLower()
{
  for( int i= 0; i < int(theLength); ++i )
    theString[i]= (char) tolower( theString[i] );
}

/////////////////////////////////////////////////////////////////////////////
