// This file is part of Moonlight Creator
//   Copyright (C) 1996-1998  Stephane Rehel
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/*
   Array3.cc

   Creation: SR, August 24th, 1995
   Revisions:

*/

#ifndef __Array3_h
#define __Array3_h

#include <assert.h>
#include "boolean.h"

/////////////////////////////////////////////////////////////////////////////

template<class T>
class Array3
{
protected:
  int nX, nY, nZ;
  T*** array;

public:
  Array3();
  virtual ~Array3()
    {
    destroy();
    }

  void destroy();
  void create( int _nX, int _nY, int _nZ );
  boolean allocated() const
    {
    return array != 0;
    }

  int getI() const { return nX; }
  int getJ() const { return nY; }
  int getK() const { return nZ; }

  // 0 <= i < nX
  // 0 <= j < nY
  // 0 <= k < nZ
  T get( int i, int j, int k ) const;

  // 0 <= i < nX
  // 0 <= j < nY
  // 0 <= k < nZ
  T& operator () ( int i, int j, int k );

  boolean inArray( int i, int j, int k ) const
    {
    return i >= 0 && i < nX &&
           j >= 0 && j < nY &&
           k >= 0 && k < nZ ;
    }

  void fillRegion( int ni1, int ni2,
                   int nj1, int nj2,
                   int nk1, int nk2,
                   const T t );

  void fill( const T t )
    {
    assert( array != 0 );
    fillRegion( 0, nX-1, 0, nY-1, 0, nZ-1, t );
    }
};

/////////////////////////////////////////////////////////////////////////////

template<class T>
inline
Array3<T>::Array3()
{
  array= 0;
  nX= nY= nZ= 0;
}

/////////////////////////////////////////////////////////////////////////////

template<class T>
inline
void Array3<T>::destroy()
{
  if( array == 0 )
    return;

  for( int k= 0; k < nZ; ++k )
    {
    assert( array[k] != 0 );
    for( int j= 0; j < nY; ++j )
      delete [] array[k][j];
    delete array[k];
    }
  delete array;
  array= 0;
}

/////////////////////////////////////////////////////////////////////////////

template<class T>
inline
void Array3<T>::create( int _nX, int _nY, int _nZ )
{
  assert( array == 0 );
  assert( _nX >= 1 );
  assert( _nY >= 1 );
  assert( _nZ >= 1 );

  nX= _nX;
  nY= _nY;
  nZ= _nZ;

  array= new T** [ nZ ];
  for( int k= 0; k < nZ; ++k )
    {
    array[k]= new T* [ nY ];
    T** array_k= array[k];
    for( int j= 0; j < nY; ++j )
      array_k[j]= new T [ nX ];
    }
}

/////////////////////////////////////////////////////////////////////////////

// 0 <= i < nX
// 0 <= j < nY
// 0 <= k < nZ
template<class T>
inline
T Array3<T>::get( int i, int j, int k ) const
{
  assert( i >= 0 );
  assert( i < nX );
  assert( j >= 0 );
  assert( j < nY );
  assert( k >= 0 );
  assert( k < nZ );
  assert( array != 0 );

  return array[k][j][i];
}

/////////////////////////////////////////////////////////////////////////////

// 0 <= i < nX
// 0 <= j < nY
// 0 <= k < nZ
template<class T>
inline
T& Array3<T>::operator () ( int i, int j, int k )
{
  assert( i >= 0 );
  assert( i < nX );
  assert( j >= 0 );
  assert( j < nY );
  assert( k >= 0 );
  assert( k < nZ );
  assert( array != 0 );

  return array[k][j][i];
}

/////////////////////////////////////////////////////////////////////////////

template<class T>
inline
void Array3<T>::fillRegion( int ni1, int ni2,
                            int nj1, int nj2,
                            int nk1, int nk2,
                            const T t )
{
  assert( array != 0 );

  assert( ni1 >= 0 );
  assert( ni1 < nX );
  assert( ni2 >= 0 );
  assert( ni2 < nX );
  assert( ni1 <= ni2 );

  assert( nj1 >= 0 );
  assert( nj1 < nY );
  assert( nj2 >= 0 );
  assert( nj2 < nY );
  assert( nj1 <= nj2 );

  assert( nk1 >= 0 );
  assert( nk1 < nZ );
  assert( nk2 >= 0 );
  assert( nk2 < nZ );
  assert( nk1 <= nk2 ); // nom de diou

  T*** array_k= array + nk1;
  for( int k= nk1; k <= nk2; ++k, ++array_k )
    {
    T** array_kj= (*array_k)+nj1;
    for( int j= nj1; j <= nj2; ++j, ++array_kj )
      {
      T* array_kji= (*array_kj)+ni1;
      for( int i= ni1; i <= ni2; ++i, ++array_kji )
        *array_kji= t;
      }
    }
}


#endif // ifndef __Array3_h

