/*
 * GTK See -- a image viewer based on GTK+
 * Copyright (C) 1998 Hotaru Lee <jkhotaru@mail.sti.com.cn> <hotaru@163.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <gtk/gtk.h>

#include "gtypes.h"

#ifdef HAVE_LIBJPEG
#include <jpeglib.h>
#include "im_jpeg.h"
#endif

#ifdef HAVE_LIBTIFF
#include <tiffio.h>
#include "im_tiff.h"
#endif

#ifdef HAVE_LIBPNG
#include <png.h>
#include "im_png.h"
#endif

#include "im_gif.h"
#include "im_xpm.h"
#include "im_bmp.h"
#include "im_ico.h"
#include "im_pcx.h"
#include "im_pnm.h"
#include "im_psd.h"
#include "im_xbm.h"
#include "im_xcf.h"

#include "scale.h"
#include "scanline.h"
#include "detect.h"

#define RANDOM_SEED 314159265

static gint bgcolor_red, bgcolor_green, bgcolor_blue;
static gint bits_red, bits_green, bits_blue;
static gint mask_red, mask_green, mask_blue;
static guint32 bgcolor;

static GtkWidget *image_wid;
static GdkImage *scan_image;
static guchar *scan_image_src;
static gint orig_image_width;
static gint left_pos, top_pos;
static RotateType rotate_type;
static gint current_scanflags;

static gboolean scanline_cancelled = FALSE;

gboolean	jpeg_scanline	(guchar *buffer,
				 gint width,
				 gint left,
				 gint scanline,
				 gint components,
				 gint pass,
				 LayerMode mode);
gboolean	jpeg_scanline_buffered
				(guchar *buffer,
				 gint width,
				 gint left,
				 gint scanline,
				 gint components,
				 gint pass,
				 LayerMode mode);
gboolean	gif_scanline	(guint *buffer,
				 guchar *cmap[],
				 gint transparent,
				 gint width,
				 gint scanline,
				 gint frame);
gboolean	gif_scanline_buffered
				(guint *buffer,
				 guchar *cmap[],
				 gint transparent,
				 gint width,
				 gint scanline,
				 gint frame);
gchar*		find_preprocess_extension
				(gchar *filename,
				 gchar **command,
				 gchar **args);
gchar*		find_filename	(gchar *filename);
/* src1, src2 and dest are pointers to one pixel
 * b1 and b2 are num_bytes values for src1 and src2
 * ha1 and ha2 tell me whether src1 and src2 has alpha
 */
void		pixel_apply_layer_mode
				(guchar *src1,
				 guchar *src2,
				 guchar *dest,
				 gint b1,
				 gint b2,
				 gboolean ha1,
				 gboolean ha2,
				 LayerMode mode);
void		rgb_to_hsv	(gint *r,
				 gint *g,
				 gint *b);
void		hsv_to_rgb	(gint *h,
				 gint *s,
				 gint *v);
void		rgb_to_hls	(gint *r,
				 gint *g,
				 gint *b);
void		hls_to_rgb	(gint *h,
				 gint *l,
				 gint *s);
gint		hls_value	(gfloat n1,
				 gfloat n2,
				 gfloat hue);

gchar*
find_preprocess_extension(gchar *filename, gchar **command, gchar **args)
{
	gchar *ext;
	
	ext = strrchr(filename, '.');
	if (ext == NULL) return NULL;
	if (strcmp(ext, ".gz") == 0)
	{
		*command = "gzip";
		*args = "-cfd";
		return ext;
	} else if (strcmp(ext, ".bz2") == 0)
	{
		*command = "bzip2";
		*args = "-cfd";
		return ext;
	} else if (strcmp(ext, ".bz") == 0)
	{
		*command = "bzip";
		*args = "-cd";
		return ext;
	} else
	{
		return NULL;
	}
}

gchar*
find_filename(gchar *filename)
{
	gchar *ext;
	
	if ((ext = strrchr(filename, '/')) == NULL)
	{
		return NULL;
	} else
	{
		ext++;
		if (*ext == '\0') return NULL;
		return ext;
	}
}

void
scanline_init(GdkColorContext *gcc, GtkStyle *style)
{
	bits_red = gcc->bits.red +
		gcc->bits.green +
		gcc->bits.blue - 8;
	mask_red = ((1 << gcc->bits.red) - 1) <<
		(gcc->bits.green + gcc->bits.blue);
	bits_green = gcc->bits.green + gcc->bits.blue - 8;
	mask_green = ((1 << gcc->bits.green) - 1) <<
		gcc->bits.blue;
	bits_blue = 8 - gcc->bits.blue;
	mask_blue = ((1 << gcc->bits.blue) - 1);
	
	bgcolor_red = (style->bg[GTK_STATE_NORMAL]).red >> 8;
	bgcolor_green = (style->bg[GTK_STATE_NORMAL]).green >> 8;
	bgcolor_blue = (style->bg[GTK_STATE_NORMAL]).blue >> 8;
	
	bgcolor = (bgcolor_red << bits_red) & mask_red;
	bgcolor |= (bgcolor_green << bits_green) & mask_green;
	bgcolor |= (bgcolor_blue >> bits_blue);

	srand(RANDOM_SEED);
}

void
scanline_start()
{
	scanline_cancelled = FALSE;
}

void
scanline_cancel()
{
	scanline_cancelled = TRUE;
}

gboolean
jpeg_scanline(
	guchar *buffer,
	gint width,
	gint left,
	gint scanline,
	gint components,
	gint pass,
	LayerMode mode)
{
	register guint32 color;
	register gint i, j;
	static gint pos, pos2;
	static gint alpha, cur_red, cur_green, cur_blue;
	static guchar src1[4], src2[4], dest[4];
	
	switch (rotate_type)
	{
		case ROTATE_90:
			pos = scan_image->width - scanline - 1;
			break;
		case ROTATE_180:
			pos = scan_image->height - scanline - 1;
			pos2 = scan_image->width - left - width;
			break;
		case ROTATE_270:
			pos2 = scan_image->height - left - width;
		default:
			break;
	}
	if (pass == 0)
	{
		cur_red = bgcolor_red;
		cur_green = bgcolor_green;
		cur_blue = bgcolor_blue;
	}
	for (i=0, j=0; i<width; i++)
	{
		if (pass > 0)
		{
			switch (rotate_type)
			{
				case ROTATE_90:
					color = gdk_image_get_pixel(scan_image, pos, left+i);
					break;
				case ROTATE_180:
					color = gdk_image_get_pixel(scan_image, width-left-i-1, pos);
					break;
				case ROTATE_270:
					color = gdk_image_get_pixel(scan_image, scanline, scan_image->height-left-i-1);
					break;
				case ROTATE_NONE:
				default:
					color = gdk_image_get_pixel(scan_image, left+i, scanline);
				break;
			}
			cur_red = (color & mask_red) >> bits_red;
			cur_green = (color & mask_green) >> bits_green;
			cur_blue = (color & mask_blue) << bits_blue;
		}
		switch (components)
		{
		  case 1:
			color = (buffer[j] << bits_red) & mask_red;
			color |= (buffer[j] << bits_green) & mask_green;
			color |= (buffer[j++] >> bits_blue);
			break;
		  case 2:
			if (pass > 0 && mode > 0)
			{
				src1[0] = cur_red;
				src1[1] = cur_green;
				src1[2] = cur_blue;
				src1[3] = 255;
				src2[0] = buffer[j];
				src2[1] = buffer[j];
				src2[2] = buffer[j];
				src2[3] = buffer[j+1];
				pixel_apply_layer_mode(src1, src2, dest, 4, 4, TRUE, TRUE, mode);
				alpha = dest[3];
				color = ((dest[0] * alpha / 255 + cur_red * (255-alpha) / 255) << bits_red) & mask_red;
				color |= ((dest[1] * alpha / 255 + cur_green * (255-alpha) / 255) << bits_green) & mask_green;
				color |= ((dest[2] * alpha / 255 + cur_blue * (255-alpha) / 255) >> bits_blue);
			} else
			{
				alpha = buffer[j+1];
				color = ((buffer[j] * alpha / 255 + cur_red * (255-alpha) / 255) << bits_red) & mask_red;
				color |= ((buffer[j] * alpha / 255 + cur_green * (255-alpha) / 255) << bits_green) & mask_green;
				color |= ((buffer[j] * alpha / 255 + cur_blue * (255-alpha) / 255) >> bits_blue);
			}
			j += 2;
			break;
		  case 3:
			color = (buffer[j++] << bits_red) & mask_red;
			color |= (buffer[j++] << bits_green) & mask_green;
			color |= (buffer[j++] >> bits_blue);
			break;
		  case 4:
			if (pass > 0 && mode > 0)
			{
				src1[0] = cur_red;
				src1[1] = cur_green;
				src1[2] = cur_blue;
				src1[3] = 255;
				src2[0] = buffer[j];
				src2[1] = buffer[j+1];
				src2[2] = buffer[j+2];
				src2[3] = buffer[j+3];
				pixel_apply_layer_mode(src1, src2, &buffer[j], 4, 4, TRUE, TRUE, mode);
			}
			alpha = buffer[j+3];
			color = ((buffer[j++] * alpha / 255 + cur_red * (255-alpha) / 255) << bits_red) & mask_red;
			color |= ((buffer[j++] * alpha / 255 + cur_green * (255-alpha) / 255) << bits_green) & mask_green;
			color |= ((buffer[j++] * alpha / 255 + cur_blue * (255-alpha) / 255) >> bits_blue);
			j++;
			break;
		  default:
			break;
		}
		switch (rotate_type)
		{
			case ROTATE_90:
				gdk_image_put_pixel(scan_image, pos, left+i, color);
				break;
			case ROTATE_180:
				gdk_image_put_pixel(scan_image, scan_image->width-i-left-1, pos, color);
				break;
			case ROTATE_270:
				gdk_image_put_pixel(scan_image, scanline, scan_image->height-left-i-1, color);
				break;
			case ROTATE_NONE:
			default:
				gdk_image_put_pixel(scan_image, left+i, scanline, color);
				break;
		}
	}
	if (current_scanflags & SCANLINE_INTERACT)
	{
		while (gtk_events_pending()) gtk_main_iteration();
	}
	if (image_wid == NULL) return FALSE;
	if (!(current_scanflags & SCANLINE_DISPLAY)) return FALSE;
	switch (rotate_type)
	{
		case ROTATE_90:
			gdk_draw_image(
				image_wid->window,
				image_wid->style->black_gc,
				scan_image,
				pos, left,
				left_pos + pos, left + top_pos,
				1, width
			);
			break;
		case ROTATE_180:
			gdk_draw_image(
				image_wid->window,
				image_wid->style->black_gc,
				scan_image,
				pos2, pos,
				pos2 + left_pos, top_pos + pos,
				width, 1
			);
			break;
		case ROTATE_270:
			gdk_draw_image(
				image_wid->window,
				image_wid->style->black_gc,
				scan_image,
				scanline, pos2,
				left_pos + scanline, pos2 + top_pos,
				1, width
			);
			break;
		case ROTATE_NONE:
		default:
			gdk_draw_image(
				image_wid->window,
				image_wid->style->black_gc,
				scan_image,
				left, scanline,
				left + left_pos, top_pos + scanline,
				width, 1
			);
			break;
	}
	return scanline_cancelled;
}

gboolean
jpeg_scanline_buffered(
	guchar *buffer,
	gint width,
	gint left,
	gint scanline,
	gint components,
	gint pass,
	LayerMode mode)
{
	register gint i, j, k;
	static guint ptr, alpha, cur_red, cur_green, cur_blue;
	static guchar src1[4], src2[4], dest[4];
	
	if (pass == 0)
	{
		cur_red = bgcolor_red;
		cur_green = bgcolor_green;
		cur_blue = bgcolor_blue;
	}
	
	switch (components)
	{
	  case 2:
		/* turning grayscale-alpha into RGB */
		ptr = (scanline * orig_image_width + left) * 3;
		for (i = 0, j = 0, k = 0; i < width; i++)
		{
			if (pass > 0)
			{
				cur_red = scan_image_src[ptr + k];
				cur_green = scan_image_src[ptr + k + 1];
				cur_blue = scan_image_src[ptr + k + 2];
			}
			if (pass > 0 && mode > 0)
			{
				src1[0] = cur_red;
				src1[1] = cur_green;
				src1[2] = cur_blue;
				src1[3] = 255;
				src2[0] = buffer[j];
				src2[1] = buffer[j];
				src2[2] = buffer[j];
				src2[3] = buffer[j+1];
				pixel_apply_layer_mode(src1, src2, dest, 4, 4, TRUE, TRUE, mode);
				alpha = dest[3];
				scan_image_src[ptr + k++] = (dest[0] * alpha / 255 + cur_red * (255-alpha) / 255);
				scan_image_src[ptr + k++] = (dest[1] * alpha / 255 + cur_green * (255-alpha) / 255);
				scan_image_src[ptr + k++] = (dest[2] * alpha / 255 + cur_blue * (255-alpha) / 255);
			} else
			{
				alpha = buffer[j+1];
				scan_image_src[ptr + k++] = (buffer[j] * alpha / 255 + cur_red * (255-alpha) / 255);
				scan_image_src[ptr + k++] = (buffer[j] * alpha / 255 + cur_green * (255-alpha) / 255);
				scan_image_src[ptr + k++] = (buffer[j] * alpha / 255 + cur_blue * (255-alpha) / 255);
			}
			j += 2;
		}
		break;
	  case 4:
		/* turning RGBA into RGB, using bgcolor */
		ptr = (scanline * orig_image_width + left) * 3;
		for (i = 0, j = 0, k = 0; i < width; i++)
		{
			if (pass > 0)
			{
				cur_red = scan_image_src[ptr + k];
				cur_green = scan_image_src[ptr + k + 1];
				cur_blue = scan_image_src[ptr + k + 2];
				if (mode > 0)
				{
					src1[0] = cur_red;
					src1[1] = cur_green;
					src1[2] = cur_blue;
					src1[3] = 255;
					src2[0] = buffer[j];
					src2[1] = buffer[j+1];
					src2[2] = buffer[j+2];
					src2[3] = buffer[j+3];
					pixel_apply_layer_mode(src1, src2, &buffer[j], 4, 4, TRUE, TRUE, mode);
				}
			}
			alpha = buffer[j+3];
			scan_image_src[ptr + k++] = (buffer[j++] * alpha / 255 + cur_red * (255-alpha) / 255);
			scan_image_src[ptr + k++] = (buffer[j++] * alpha / 255 + cur_green * (255-alpha) / 255);
			scan_image_src[ptr + k++] = (buffer[j++] * alpha / 255 + cur_blue * (255-alpha) / 255);
			j++;
		}
		break;
	  default:
		memcpy(scan_image_src + (scanline * width + left) * components, buffer, width * components);
		break;
	}
	return scanline_cancelled;
}

gboolean
gif_scanline(guint *buffer, guchar *cmap[], gint transparent, gint width, gint scanline, gint frame)
{
	register guint32 color;
	register gint i, c;
	
	for (i=0; i<width; i++)
	{
		if ((c = buffer[i]) == transparent)
		{
			color = bgcolor;
		} else
		{
			color = (cmap[0][c] << bits_red) & mask_red;
			color |= (cmap[1][c] << bits_green) & mask_green;
			color |= (cmap[2][c] >> bits_blue);
		}
		switch (rotate_type)
		{
			case ROTATE_90:
				gdk_image_put_pixel(scan_image, scan_image->width - scanline - 1, i, color);
				break;
			case ROTATE_180:
				gdk_image_put_pixel(scan_image, width - i - 1, scan_image->height - scanline - 1, color);
				break;
			case ROTATE_270:
				gdk_image_put_pixel(scan_image, scanline, scan_image->height - i - 1, color);
				break;
			case ROTATE_NONE:
			default:
				gdk_image_put_pixel(scan_image, i, scanline, color);
				break;
		}
	}
	if (current_scanflags & SCANLINE_INTERACT)
	{
		while (gtk_events_pending()) gtk_main_iteration();
	}
	if (image_wid == NULL) return FALSE;
	if (!(current_scanflags & SCANLINE_DISPLAY)) return FALSE;
	switch (rotate_type)
	{
		case ROTATE_90:
			gdk_draw_image(
				image_wid->window,
				image_wid->style->black_gc,
				scan_image,
				scan_image->width - scanline - 1, 0,
				left_pos + scan_image->width - scanline - 1, top_pos,
				1, width
			);
			break;
		case ROTATE_180:
			gdk_draw_image(
				image_wid->window,
				image_wid->style->black_gc,
				scan_image,
				0, scan_image->height - scanline - 1,
				left_pos, top_pos + scan_image->height - scanline - 1,
				width, 1
			);
			break;
		case ROTATE_270:
			gdk_draw_image(
				image_wid->window,
				image_wid->style->black_gc,
				scan_image,
				scanline, 0,
				left_pos + scanline, top_pos,
				1, width
			);
			break;
		case ROTATE_NONE:
		default:
			gdk_draw_image(
				image_wid->window,
				image_wid->style->black_gc,
				scan_image,
				0, scanline,
				left_pos, top_pos + scanline,
				width, 1
			);
			break;
	}
	return scanline_cancelled;
}

gboolean
gif_scanline_buffered(guint *buffer, guchar *cmap[], gint transparent, gint width, gint scanline, gint frame)
{
	register gchar *j;
	register gint i, c;
	
	for (i=0, j=scan_image_src+width*scanline*3; i<width; i++)
	{
		if ((c = buffer[i]) == transparent)
		{
			*(j++) = bgcolor_red;
			*(j++) = bgcolor_green;
			*(j++) = bgcolor_blue;
		} else
		{
			*(j++) = cmap[0][c];
			*(j++) = cmap[1][c];
			*(j++) = cmap[2][c];
		}
	}
	return scanline_cancelled;
}

GdkImage*
load_scaled_image(
	gchar *orig_filename,
	ImageInfo *info,
	gint max_width,
	gint max_height,
	RotateType r_type,
	GtkWidget *window,
	GtkWidget *container,
	gint scanflags)
{
	gint width, height, temp, pid, status, comp;
	gboolean no_scale, is_tempfile;
	gchar *d, *ext, *tempfile;
	gchar filename[256];
	gchar *cmd, *args;
	ImageType type;
	gint orig_width, orig_height, ncolors;
	
	current_scanflags = scanflags;
	
	/* Whether the image file should be preprocessed */
	if ((ext = find_preprocess_extension(orig_filename, &cmd, &args)) == NULL)
	{
		strcpy(filename, orig_filename);
		is_tempfile = FALSE;
	} else
	{
		/* try to preprocess it into /tmp */
		if ((tempfile = find_filename(orig_filename)) == NULL) return NULL;
		strcpy(filename, "/tmp/");
		strncat(filename, tempfile, ext - tempfile);
		is_tempfile = TRUE;
		if ((pid = fork()) < 0) return NULL;
		if (pid == 0)
		{
			/* child process */
			FILE* f;
			if (!(f = fopen(filename,"w")))
			{
				_exit(127);
			}
			if (-1 == dup2(fileno(f),fileno(stdout)))
			{
				_exit(127);
			}
			execlp (cmd, cmd, args, orig_filename, NULL);
			_exit(127);
		} else
		{
			/* parent process */
			waitpid(pid, &status, 0);
			/* we need to get the header field for gzipped images */
			if (!detect_image_type(filename, info))
			{
				unlink(filename);
				info->valid = FALSE;
				return NULL;
			}
		}
	}
	
	if (info->width < 0)
	{
		if (is_tempfile) unlink(filename);
		info->valid = FALSE;
		return NULL;
	}

	type = info->type;
	orig_width = info->width;
	orig_height = info->height;
	ncolors = info->ncolors;
	
	orig_image_width = orig_width;

	info->desc = NULL;
	info->has_desc = FALSE;
	rotate_type = r_type;
	
	no_scale = max_width < 0 || max_height < 0 ||
		(r_type%2 == 0 && orig_width <= max_width && orig_height <= max_height) ||
		(r_type%2 == 1 && orig_width <= max_height && orig_height <= max_width);
	if (no_scale)
	{
		switch (r_type)
		{
			case ROTATE_90:
			case ROTATE_270:
				width = orig_height;
				height = orig_width;
				break;
			case ROTATE_NONE:
			case ROTATE_180:
			default:
				width = orig_width;
				height = orig_height;
				break;
		}
		scan_image = gdk_image_new(
			GDK_IMAGE_NORMAL,
			gdk_window_get_visual(window -> window),
			width,
			height);
		if (container == NULL)
		{
			image_wid = NULL;
		} else
		{
			if (scanflags & SCANLINE_CONTAINER)
			{
				image_wid = gtk_image_new(scan_image, NULL);
				gtk_container_add(GTK_CONTAINER(container), image_wid);
				gtk_widget_show(image_wid);
			} else
			{
				/* the container itself should be a GtkImage */
				image_wid = container;
				gtk_image_set(GTK_IMAGE(image_wid), scan_image, NULL);
			}
			if (current_scanflags & SCANLINE_DISPLAY)
			{
				while (image_wid->allocation.x < 0 || gtk_events_pending())
					gtk_main_iteration();
			}
			left_pos = image_wid->allocation.x +
				(image_wid->allocation.width - width + 1) / 2;
			top_pos = image_wid->allocation.y +
				(image_wid->allocation.height - height + 1) / 2;
		}

		switch (type)
		{
		  case JPG:
#ifdef HAVE_LIBJPEG
			jpeg_load(filename, (JpegLoadFunc)jpeg_scanline);
#else
			return NULL;
#endif
			break;
		  case TIF:
#ifdef HAVE_LIBTIFF
			tiff_load(filename, (TiffLoadFunc)jpeg_scanline);
#else
			return NULL;
#endif
			break;
		  case PNG:
#ifdef HAVE_LIBPNG
			png_load(filename, (PngLoadFunc)jpeg_scanline);
#else
			return NULL;
#endif
			break;
		  case BMP:
			bmp_load(filename, (BmpLoadFunc)jpeg_scanline);
			break;
		  case GIF:
			d = gif_load(filename, (GifLoadFunc)gif_scanline);
			if (d != NULL && strlen(d) > 0)
			{
				info->desc = d;
				info->has_desc = TRUE;
			}
			break;
		  case XPM:
			xpm_load(filename, (XpmLoadFunc)gif_scanline);
			break;
		  case ICO:
			ico_load(filename, (IcoLoadFunc)gif_scanline);
			break;
		  case PCX:
			pcx_load(filename, (PcxLoadFunc)jpeg_scanline);
			break;
		  case PNM:
			pnm_load(filename, (PnmLoadFunc)jpeg_scanline);
			break;
		  case PSD:
			d = psd_load(filename, (PsdLoadFunc)jpeg_scanline);
			if (d != NULL && strlen(d) > 0)
			{
				info->desc = d;
				info->has_desc = TRUE;
			}
			break;
		  case XBM:
			xbm_load(filename, (XbmLoadFunc)jpeg_scanline);
			break;
		  case XCF:
			xcf_load(filename, (XcfLoadFunc)jpeg_scanline);
			break;
		  default:
			return NULL;
		}
	} else
	{
		switch (r_type)
		{
			case ROTATE_90:
			case ROTATE_270:
				if (max_width * orig_width < max_height * orig_height)
				{
					width = max_width;
					height = width * orig_width / orig_height;
				} else
				{
					height = max_height;
					width = height * orig_height / orig_width;
				}
				temp = max_width;
				max_width = max_height;
				max_height = temp;
				break;
			case ROTATE_NONE:
			case ROTATE_180:
			default:
				if (max_width * orig_height < max_height * orig_width)
				{
					width = max_width;
					height = width * orig_height / orig_width;
				} else
				{
					height = max_height;
					width = height * orig_width / orig_height;
				}
				break;
		}
		
		scan_image = gdk_image_new(
			GDK_IMAGE_NORMAL,
			gdk_window_get_visual(window -> window),
			width,
			height);
		if (container == NULL)
		{
			image_wid = NULL;
		} else
		{
			if (scanflags & SCANLINE_CONTAINER)
			{
				image_wid = gtk_image_new(scan_image, NULL);
				gtk_container_add(GTK_CONTAINER(container), image_wid);
				gtk_widget_show(image_wid);
			} else
			{
				/* the container itself should be a GtkImage */
				image_wid = container;
				gtk_image_set(GTK_IMAGE(image_wid), scan_image, NULL);
			}
			if (current_scanflags & SCANLINE_DISPLAY)
			{
				while (image_wid->allocation.x < 0 || gtk_events_pending())
					gtk_main_iteration();
			}
			left_pos = image_wid->allocation.x +
				(image_wid->allocation.width - width + 1) / 2;
			top_pos = image_wid->allocation.y +
				(image_wid->allocation.height - height + 1) / 2;
		}
		
		if (type == JPG)
		{
#ifdef HAVE_LIBJPEG
			comp = ncolors / 8;
			scan_image_src = g_malloc(orig_width * orig_height * comp);
			jpeg_load(filename, (JpegLoadFunc)jpeg_scanline_buffered);
#else
			return NULL;
#endif
		} else
		if (type == TIF)
		{
#ifdef HAVE_LIBTIFF
			comp = 3;
			scan_image_src = g_malloc(orig_width * orig_height * 3);
			tiff_load(filename, (TiffLoadFunc)jpeg_scanline_buffered);
#else
			return NULL;
#endif
		} else
		if (type == PNG)
		{
#ifdef HAVE_LIBPNG
			if (info->bpp == 1) comp = 1;
			else comp = 3;
			scan_image_src = g_malloc(orig_width * orig_height * comp);
			png_load(filename, (PngLoadFunc)jpeg_scanline_buffered);
#else
			return NULL;
#endif
		} else
		if (type == PNM)
		{
			comp = ncolors / 8;
			if (comp < 1) comp = 1;
			scan_image_src = g_malloc(orig_width * orig_height * comp);
			pnm_load(filename, (PnmLoadFunc)jpeg_scanline_buffered);
		} else
		if (type == XBM)
		{
			comp = 1;
			scan_image_src = g_malloc(orig_width * orig_height);
			xbm_load(filename, (XbmLoadFunc)jpeg_scanline_buffered);
		} else
		if (type == BMP)
		{
			comp = 3;
			scan_image_src = g_malloc(orig_width * orig_height * 3);
			bmp_load(filename, (BmpLoadFunc)jpeg_scanline_buffered);
		} else
		if (type == GIF)
		{
			comp = 3;
			scan_image_src = g_malloc(orig_width * orig_height * 3);
			d = gif_load(filename, (GifLoadFunc)gif_scanline_buffered);
			if (d != NULL && strlen(d) > 0)
			{
				info->desc = d;
				info->has_desc = TRUE;
			}
		} else
		if (type == XPM)
		{
			comp = 3;
			scan_image_src = g_malloc(orig_width * orig_height * 3);
			xpm_load(filename, (XpmLoadFunc)gif_scanline_buffered);
		} else
		if (type == ICO)
		{
			comp = 3;
			scan_image_src = g_malloc(orig_width * orig_height * 3);
			ico_load(filename, (IcoLoadFunc)gif_scanline_buffered);
		} else
		if (type == PCX)
		{
			comp = 3;
			scan_image_src = g_malloc(orig_width * orig_height * 3);
			pcx_load(filename, (PcxLoadFunc)jpeg_scanline_buffered);
		} else
		if (type == PSD)
		{
			comp = 3;
			scan_image_src = g_malloc(orig_width * orig_height * 3);
			d = psd_load(filename, (PsdLoadFunc)jpeg_scanline_buffered);
			if (d != NULL && strlen(d) > 0)
			{
				info->desc = d;
				info->has_desc = TRUE;
			}
		} else
		if (type == XCF)
		{
			comp = 3;
			scan_image_src = g_malloc(orig_width * orig_height * 3);
			xcf_load(filename, (XcfLoadFunc)jpeg_scanline_buffered);
		}
		/* Important!!!
		 * When the GIF image was read to buffer, it has
		 * been turned into RGB, but not indexed.
		 * So, we just use jpeg_scanline.
		 * It's always 3 bytes per pixel.
		 * Don't worry about the transparent color. :)
		 */
		if (!scanline_cancelled)
		{
			if (current_scanflags & SCANLINE_PREVIEW)
			{
				scale_region_preview(scan_image_src,
					orig_width, orig_height,
					max_width, max_height,
					comp, (ScaleFunc)jpeg_scanline);
			} else
			{
				scale_region(scan_image_src,
					orig_width, orig_height,
					max_width, max_height,
					comp, (ScaleFunc)jpeg_scanline);
			}
		}
		g_free(scan_image_src);
	}
	
	/* If the filename is decompressed one, unlink it */
	if (is_tempfile) unlink(filename);

	return scan_image;
}

void
pixel_apply_layer_mode(
	guchar *src1,
	guchar *src2,
	guchar *dest,
	gint b1,
	gint b2,
	gboolean ha1,
	gboolean ha2,
	LayerMode mode)
{
	static gint b, alpha, t, t2, red1, green1, blue1, red2, green2, blue2;
	
	alpha = (ha1 || ha2) ? max(b1, b2) - 1 : b1;
	switch (mode)
	{
	  case DISSOLVE_MODE:
		for (b = 0; b < alpha; b++)
			dest[b] = src2[b];
		/*  dissolve if random value is > opacity  */
		t = (rand() & 0xFF);
		dest[alpha] = (t > src2[alpha]) ? 0 : src2[alpha];
		break;
	  case MULTIPLY_MODE:
		for (b = 0; b < alpha; b++)
			dest[b] = (src1[b] * src2[b]) / 255;
		if (ha1 && ha2)
			dest[alpha] = min(src1[alpha], src2[alpha]);
		else if (ha2)
			dest[alpha] = src2[alpha];
		break;
	  case SCREEN_MODE:
		for (b = 0; b < alpha; b++)
			dest[b] = 255 - ((255 - src1[b]) * (255 - src2[b])) / 255;
		if (ha1 && ha2)
			dest[alpha] = min(src1[alpha], src2[alpha]);
		else if (ha2)
			dest[alpha] = src2[alpha];
		break;
	  case OVERLAY_MODE:
		for (b = 0; b < alpha; b++)
		{
			/* screen */
			t = 255 - ((255 - src1[b]) * (255 - src2[b])) / 255;
			/* mult */
			t2 = (src1[b] * src2[b]) / 255;
			dest[b] = (t * src1[b] + t2 * (255 - src1[b])) / 255;
		}
		if (ha1 && ha2)
			dest[alpha] = min(src1[alpha], src2[alpha]);
		else if (ha2)
			dest[alpha] = src2[alpha];
		break;
	  case DIFFERENCE_MODE:
		for (b = 0; b < alpha; b++)
		{
			t = src1[b] - src2[b];
			dest[b] = (t < 0) ? -t : t;
		}
		if (ha1 && ha2)
			dest[alpha] = min(src1[alpha], src2[alpha]);
		else if (ha2)
			dest[alpha] = src2[alpha];
		break;
	  case ADDITION_MODE:
		for (b = 0; b < alpha; b++)
		{
			t = src1[b] + src2[b];
			dest[b] = (t > 255) ? 255 : t;
		}
		if (ha1 && ha2)
			dest[alpha] = min(src1[alpha], src2[alpha]);
		else if (ha2)
			dest[alpha] = src2[alpha];
		break;
	  case SUBTRACT_MODE:
		for (b = 0; b < alpha; b++)
		{
			t = (gint)src1[b] - (gint)src2[b];
			dest[b] = (t < 0) ? 0 : t;
		}
		if (ha1 && ha2)
			dest[alpha] = min(src1[alpha], src2[alpha]);
		else if (ha2)
			dest[alpha] = src2[alpha];
		break;
	  case DARKEN_ONLY_MODE:
		for (b = 0; b < alpha; b++)
		{
			t = src1[b];
			t2 = src2[b];
			dest[b] = (t < t2) ? t : t2;
		}
		if (ha1 && ha2)
			dest[alpha] = min(src1[alpha], src2[alpha]);
		else if (ha2)
			dest[alpha] = src2[alpha];
		break;
	  case LIGHTEN_ONLY_MODE:
		for (b = 0; b < alpha; b++)
		{
			t = src1[b];
			t2 = src2[b];
			dest[b] = (t < t2) ? t2 : t;
		}
		if (ha1 && ha2)
			dest[alpha] = min(src1[alpha], src2[alpha]);
		else if (ha2)
			dest[alpha] = src2[alpha];
		break;
	  case HUE_MODE:
	  case SATURATION_MODE:
	  case VALUE_MODE:
		red1 = src1[0]; green1 = src1[1]; blue1 = src1[2];
		red2 = src2[0]; green2 = src2[1]; blue2 = src2[2];
		rgb_to_hsv (&red1, &green1, &blue1);
		rgb_to_hsv (&red2, &green2, &blue2);
		switch (mode)
		{
		  case HUE_MODE:
			red1 = red2;
			break;
		  case SATURATION_MODE:
			green1 = green2;
			break;
		  case VALUE_MODE:
			blue1 = blue2;
			break;
		  default:
			break;
		}
		hsv_to_rgb (&red1, &green1, &blue1);
		dest[0] = red1; dest[1] = green1; dest[2] = blue1;
		if (ha1 && ha2)
			dest[3] = min(src1[3], src2[3]);
		else if (ha2)
			dest[3] = src2[3];
		break;
	  case COLOR_MODE:
		red1 = src1[0]; green1 = src1[1]; blue1 = src1[2];
		red2 = src2[0]; green2 = src2[1]; blue2 = src2[2];
		rgb_to_hls (&red1, &green1, &blue1);
		rgb_to_hls (&red2, &green2, &blue2);
		red1 = red2;
		blue1 = blue2;
		hls_to_rgb (&red1, &green1, &blue1);
		dest[0] = red1; dest[1] = green1; dest[2] = blue1;
		if (ha1 && ha2)
			dest[3] = min(src1[3], src2[3]);
		else if (ha2)
			dest[3] = src2[3];
		break;
	  case NORMAL_MODE:
	  default:
		for (b = 0; b < b2; b++) dest[b] = src2[b];
		break;
	}
}

void
rgb_to_hsv (gint *r,
            gint *g,
            gint *b)
{
  gint red, green, blue;
  gfloat h, s, v;
  gint min, max;
  gint delta;

  h = 0.0;

  red = *r;
  green = *g;
  blue = *b;

  if (red > green)
    {
      if (red > blue)
        max = red;
      else
        max = blue;

      if (green < blue)
        min = green;
      else
        min = blue;
    }
  else
    {
      if (green > blue)
        max = green;
      else
        max = blue;

      if (red < blue)
        min = red;
      else
        min = blue;
    }

  v = max;

  if (max != 0)
    s = ((max - min) * 255) / (gfloat) max;
  else
    s = 0;

  if (s == 0)
    h = 0;
  else
    {
      delta = max - min;
      if (red == max)
        h = (green - blue) / (gfloat) delta;
      else if (green == max)
        h = 2 + (blue - red) / (gfloat) delta;
      else if (blue == max)
        h = 4 + (red - green) / (gfloat) delta;
      h *= 42.5;

      if (h < 0)
        h += 255;
      if (h > 255)
        h -= 255;
    }

  *r = h;
  *g = s;
  *b = v;
}


void
hsv_to_rgb (gint *h,
            gint *s,
            gint *v)
{
  gfloat hue, saturation, value;
  gfloat f, p, q, t;

  if (*s == 0)
    {
      *h = *v;
      *s = *v;
      *v = *v;
    }
  else
    {
      hue = *h * 6.0 / 255.0;
      saturation = *s / 255.0;
      value = *v / 255.0;

      f = hue - (gint) hue;
      p = value * (1.0 - saturation);
      q = value * (1.0 - (saturation * f));
      t = value * (1.0 - (saturation * (1.0 - f)));

      switch ((gint) hue)
        {
        case 0:
          *h = value * 255;
          *s = t * 255;
          *v = p * 255;
          break;
        case 1:
          *h = q * 255;
          *s = value * 255;
          *v = p * 255;
          break;
        case 2:
          *h = p * 255;
          *s = value * 255;
          *v = t * 255;
          break;
        case 3:
          *h = p * 255;
          *s = q * 255;
          *v = value * 255;
          break;
        case 4:
          *h = t * 255;
          *s = p * 255;
          *v = value * 255;
          break;
        case 5:
          *h = value * 255;
          *s = p * 255;
          *v = q * 255;
          break;
        }
    }
}

void
rgb_to_hls (gint *r,
            gint *g,
            gint *b)
{
  gint red, green, blue;
  gfloat h, l, s;
  gint min, max;
  gint delta;

  red = *r;
  green = *g;
  blue = *b;

  if (red > green)
    {
      if (red > blue)
        max = red;
      else
        max = blue;

      if (green < blue)
        min = green;
      else
        min = blue;
    }
  else
    {
      if (green > blue)
        max = green;
      else
        max = blue;

      if (red < blue)
        min = red;
      else
        min = blue;
    }

  l = (max + min) / 2.0;

  if (max == min)
    {
      s = 0.0;
      h = 0.0;
    }
  else
    {
      delta = (max - min);

      if (l < 128)
        s = 255 * (gfloat) delta / (gfloat) (max + min);
      else
        s = 255 * (gfloat) delta / (gfloat) (511 - max - min);

      if (red == max)
        h = (green - blue) / (gfloat) delta;
      else if (green == max)
        h = 2 + (blue - red) / (gfloat) delta;
      else
        h = 4 + (red - green) / (gfloat) delta;

      h = h * 42.5;

      if (h < 0)
        h += 255;
      if (h > 255)
        h -= 255;
    }

  *r = h;
  *g = l;
  *b = s;
}

gint
hls_value (gfloat n1,
           gfloat n2,
           gfloat hue)
{
  gfloat value;

  if (hue > 255)
    hue -= 255;
  else if (hue < 0)
    hue += 255;
  if (hue < 42.5)
    value = n1 + (n2 - n1) * (hue / 42.5);
  else if (hue < 127.5)
    value = n2;
  else if (hue < 170)
    value = n1 + (n2 - n1) * ((170 - hue) / 42.5);
  else
    value = n1;

  return (gint) (value * 255);
}

void
hls_to_rgb (gint *h,
            gint *l,
            gint *s)
{
  gfloat hue, lightness, saturation;
  gfloat m1, m2;

  hue = *h;
  lightness = *l;
  saturation = *s;

  if (saturation == 0)
    {
      /*  achromatic case  */
      *h = lightness;
      *l = lightness;
      *s = lightness;
    }
  else
    {
      if (lightness < 128)
        m2 = (lightness * (255 + saturation)) / 65025.0;
      else
        m2 = (lightness + saturation - (lightness * saturation)/255.0) / 255.0;

      m1 = (lightness / 127.5) - m2;

      /*  chromatic case  */
      *h = hls_value (m1, m2, hue + 85);
      *l = hls_value (m1, m2, hue);
      *s = hls_value (m1, m2, hue - 85);
    }
}
