// copyright (c) 1997,1998 stephen f. white
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; see the file COPYING.  If not, write to
// the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

import java.util.*;
// import java.net.*;		// for URL stuff

public class Dispatcher implements ClientThreadObserver, VRMLObjectObserver
{
    TalkerPanel		talker;
    ClientThread	thread;
    Hashtable		objects = new Hashtable(); // objid's -> VRMLObjects
    SceneInterface	scene;

    // VID of the player object
    int			userVid;

    // a VRML representation of the player's avatar
    VRMLObject		self;

    // mode for having an out-of-body experience
    boolean		oobe = false;

    // relative position for next OOBE
    VSFVec3f		oobeOffset = new VSFVec3f(0.0F, 0.75F, -3.0F);
    VSFRotation		oobeOrientation = new VSFRotation(0.0F, 1.0F, 0.1F,
							  3.14159F);

    public Dispatcher(SceneInterface scene) {
	this.scene = scene;
    }

    public void setTalker(TalkerPanel talker) {
	this.talker = talker;
    }

    public VRMLObject getObject(int vid) {
	return (VRMLObject) objects.get(new Integer(vid));
    }

    public void send(int vid, short field, VField value) {
	if (thread != null) thread.send(vid, field, value);
    }

    // sendPosition:  called from scene when user moves viewpoint

    public void sendPosition(VSFVec3f value)
    {
	if (oobe) {
//	    oobeOffset = value.minus((VSFVec3f) self.getField(VIP.POSITION));
	} else {
	    if (thread != null) thread.send(userVid, VIP.POSITION, value);
	    if (self != null) self.setField(VIP.POSITION, value);
	}
    }

    // sendOrientation:  called from scene when user turns viewpoint

    public void sendOrientation(VSFRotation value)
    {
	if (oobe) {
//	    oobeOrientation = value;
	} else {
	    if (thread != null) thread.send(userVid, VIP.ORIENTATION, value);
	    if (self != null) self.setField(VIP.ORIENTATION, value);
	}
    }

    // Talker notifications
    public void onLogin(String hostname, int port,
			String username, String avatarURL) {
	talker.disable();
	scene.onStatus("Connecting to VNet server at " + hostname + "...");
	thread = new ClientThread(this, hostname, port, username, avatarURL);
	thread.start();
    }

    public void sendMessage(String txt) {
	if (thread != null) thread.send(userVid, VIP.MESSAGE,
					new VSFString(txt));
    }

    public void sendPrivateMessage(int vid, String txt) {
	if (thread != null) thread.send(vid, VIP.PRIVATE_MESSAGE,
					new VSFString(txt));
    }

    public void scaleAvatar(float scaleValue) {
	if (thread != null) thread.send(userVid, VIP.SCALE,
		    new VSFVec3f(scaleValue, scaleValue, scaleValue));
    }

    public void onQuit() {
	if (thread != null) {
	    thread.send(userVid, VIP.QUIT, null);
	    talker.disable();
	}
    }

    public void onError(String msg) {
	scene.onError(msg);
	talker.enable();
    }

    // ClientThread notifications
    public void onNetConnect(int vid) {
	talker.userList.removeAllUsers();
	if (vid >= 0) {
	    scene.onStatus("Connected.");
	    userVid = vid;
	    talker.setConnected(vid);
	    scene.onConnect();
	} else {
	    onError("That name is already in use, or invalid.  Please choose another.");
	    thread = null;
	}
    }

    public void setField(int vid, short field, VField value)
    {
	VRMLObject	obj = (VRMLObject) objects.get(new Integer(vid));

	if (obj != null) obj.setField(field, value);
	if (obj == self && !oobe) {
	    if (field == VIP.POSITION) {
		scene.setViewpointPosition((VSFVec3f) value);
	    } else if (field == VIP.ORIENTATION) {
		scene.setViewpointOrientation((VSFRotation) value);
	    }
	}
    }

    public void onNetInput(int vid, short field, VField value) {
	if (field >= 0) {
	    setField(vid, field, value);
	} else switch (field) {
	  case VIP.ADD_OBJECT:
	    String url = ((VSFString) value).getValue();
	    VRMLObject obj = (VRMLObject) objects.get(new Integer(vid));
	    if (obj != null) {
		obj.URL = url;
	    } else {
		obj = scene.createObject(vid, url, this);
		objects.put(new Integer(obj.id), obj);
		if (vid == userVid) self = obj;
		if (vid != userVid || oobe) addObject(obj);
	    }
	    log("loading " + obj + ", URL " + obj.URL);
	    obj.load();
	    break;
	  case VIP.REMOVE_OBJECT:
	    talker.userList.removeUser(vid);
	    removeObject((VRMLObject) objects.get(new Integer(vid)));
	    objects.remove(new Integer(vid));
//	    if (talker.builder.selection == vid) {
//		talker.builder.setSelection(-1);
//	    }
	    break;
	  case VIP.MESSAGE:
	  case VIP.PRIVATE_MESSAGE:
	    talker.appendText(((VSFString) value).getValue());
	    break;
	  case VIP.CREATE_OBJECT:
	    obj = scene.createObject(vid, "", this);
	    objects.put(new Integer(obj.id), obj);
	    addObject(obj);
	    talker.setSelection(vid);
	    scene.onStatus("Created.");
	    break;
	  case VIP.USER_INFO:
	    talker.userList.addUser(vid, ((VSFString) value).getValue());
	    break;
	}
    }

    public void onNetDisconnect() {
	scene.onStatus("vnet v1.1b1 (c) 1998 stephen f. white, jeff sonstein");
	removeAllObjects();
	talker.setConnected(-1);
	scene.onDisconnect();
    }

    private void addObject(VRMLObject obj)
    {
	log("adding " + obj + " to scene");
	scene.addObject(obj);
    }

    private void removeObject(VRMLObject obj)
    {
	log("removing " + obj + " from scene");
	scene.removeObject(obj);
    }

    private void removeAllObjects()
    {
	log("removing all objects from scene");
	scene.removeAllObjects();
	objects.clear();
    }

    public void log(String str) {
	System.err.println(userVid + ":  " + str);
    }

    public void createObject() {
	scene.onStatus("Creating new object...");
	if (thread != null) thread.send(userVid, VIP.CREATE_OBJECT, null);
    }

    public void deleteObject(int vid) {
	if (thread != null) thread.send(vid, VIP.REMOVE_OBJECT, null);
    }

    public void changeURL(int vid, String url) {
	if (thread != null) thread.send(vid, VIP.ADD_OBJECT, new VSFString(url));
    }

    // VRMLObjectObserver callback
    public void onClicked(VRMLObject obj)
    {
	talker.setSelection(obj.id);
    }

    // VRMLObjectObserver callback
    public void onLoaded(VRMLObject obj)
    {
	log("loaded " + obj + ", URL " + obj.URL);
	if (obj == self) {
	    talker.addGestures(obj.gestures);
	}
    }

    public void toggleOOBE()
    {
	VSFVec3f	position = (VSFVec3f) self.getField(VIP.POSITION);
	VSFRotation	orientation = (VSFRotation)
					      self.getField(VIP.ORIENTATION);
	oobe = !oobe;

	if (oobe) {
	    // add an avatar for myself
	    addObject(self);

	    // place the user at some distance from the avatar
	    scene.setViewpointPosition(position.plus(oobeOffset));
	    scene.setViewpointOrientation(oobeOrientation);
	} else {

	    // remove my avatar from the scene
	    removeObject(self);

	    // now place user where the avatar was
	    scene.setViewpointPosition(position);
	    scene.setViewpointOrientation(orientation);
	}
    }

    void gesture(int i) {
	if (thread != null) thread.send(userVid, (short) (VIP.NUM_FIELDS + i), new VSFBool(true));
    }
}
