/*
**  Net Tamagotchi v0.9 by Milos Glisic, mglisic@lagged.net
**
**  Usage: tama [port]
**
**  Greets go to: fredsan, m3lt, printf1, snowman
**  Xtra special thanx to Gopher for finding 78923734 bugs and making
**  it possible for Net Tamagotchi to be at least semi-stable.  
**
**
**  History:
**
**  0.1
**	- Fully functional
**	- Commands: status, feed, quit, about, help + command aliases
**	- One client at a time + 3 client queue
**	- Compatible with both \n and \n\r clients
**
**  0.2
**	- Improved error checking and status reports
**	- Added 'list' function
**	- Added extra command aliases
**	- Added command line port option
**	- Added loneliness factor and 'pet' function
**	- Fixed denial-of-service problem involving belated alarm()
**	- Removed "last feeding/petting time" - more fun this way
**	- Added grace period if Tamagotchi is starving but not lonely
**
**  0.3
**	- Optimized a few routines
**	- Removed deceased Tamagotchis from 'list'
**	- Much enhanced code compatibility - POSIX, ANSI, X/Open
**	- Added client hostname resolution
**	- Added username/password format checking to prevent corruption
**
**  0.4
**	- Added weight, feeding control, relative health patterns
**	- Completely restructured the effect of loneliness on diet/health
**	- Improved and expanded the Net Tamagotchi API (tm)
**	- Fixed possible corruption in 'passwd' and its handler, setpass()
**	- Optimized and improved status()
**	- Fixed fatal flaw in setweight() - this was bad
**	- Added more error checking
**	- Fixed feature in feed() that allowed infinite feeding if lonely
**
**  0.5
**	- Added time and date to log reports
**	- Divided source code into modules for easier maintenance
**	- Made the main socket (s) a global variable (more optimal)
**	- Added error message and appropriate log entry on timeout
**	- Added enhanced bounds checking to the API
**	- First public Net Tamagotchi release
**
**  0.6
**	- Added SIGSEGV signal handler for better error reports
**	- Added ability to change Tamagotchi's name - chname
**	- Added limit to the number of listings in list()
**	- Added MOTD file capability
**	- Fixed lousy death checking in list()
**	- Fixed denial-of-service problem if "no route to host"
**	- Fixed major problem in status() - now computes only on login
**
**  0.7
**	- Added argument parsing and ability to feed/pet/see other pets
**	- Added ability to change other pets' passwords/names
**	- Separated command interpreter into separate module
**	- Put together tama.h, making code maintenance much easier
**	- Fixed tabs in list()
**
**  0.8
**	- Added 'kill' command
**	- Added password specification on the command line
**	- Added 'motd' command
**	- Added multiple client support (1-MAXCLIENTS)
**
**  0.9
**	- Added error message if Tamagotchi is full
**	- Added session purging and other checks at specified intervals
*/

/* includes */
#include <stdio.h>
#include <sys/wait.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <netinet/in.h>
#include <netdb.h>
#include <fcntl.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <time.h>

#include "tama.h"

	int s;

/* Return ASCII string of current time */
char *logtime()
{
	struct tm *thetime;
	time_t utime;
	char *string;

	utime = time(NULL);
	thetime = localtime(&utime);
	string = asctime(thetime);

	string[strlen(string)-6]=0;
	return string;
}

/* SIGINT handler */
void term(int sig)
{
	/* On ^C, flush all streams and exit. */

	printf("%s Received INT signal. Exiting.\n", logtime());
	fflush(NULL);
	sleep(1);
	exit(0);
}

/* SIGALRM handler */
void timeout(int sig)
{
	/* On timeout, display error, flush all streams and exit. */

	put("\nConnection timed out. Disconnecting.\n");
	printf("%s Connection timed out. Disconnecting.\n", logtime());
	fflush(NULL);
	exit(0);
}

/* SIGSEGV handler */
void segv(int sig)
{
	put("An error has occured in the Net Tamagotchi server.\n");
	put("Please report the circumstances which caused this to ff@lagged.net\n");
	put("Thank you.\n");
	printf("%s Segmentation violation. Client handler exiting.\n", logtime());
	fflush(NULL);	
	exit(0);
}

/* read a string from the client - added compatibility */
/* for clients that use \r\n for newline, like win95 telnet */
void get(char *buf)
{
	int ctr;

	for(ctr=0; ctr<BUFLEN; ctr++) {
		if(recv(s, buf+ctr, 1, 0)<0) {
			close(s);
			exit(0);
		}
		if(buf[ctr]=='\r') ctr--;
		else if(buf[ctr]=='\n') break;
	}

	buf[ctr]='\0';
}

/* sends output to client - extended client support */
void put(char *buf)
{
	int ctr;

	for(ctr=0; ctr<strlen(buf); ctr++) {
		send(s, buf+ctr, 1, 0);
		if(buf[ctr]=='\n')
			send(s, "\r", 1, 0);
	}
}

void main(int argc, char **argv)
{
	fd_set input;
	pid_t pid;
	int rs, ns, fromlen, port, fd, opt = 1, flags, clients = 0;
	struct timeval to;
	struct sockaddr_in sin, fsin;
	struct hostent *hp;
	char buf[BUFLEN], name[MAXNAME+1], *host, arg[BUFLEN], *ptr; 

	if(argc>1) {
		if(argc>2 || atoi(argv[1])==0) {
			fprintf(stderr, COMMANDLINE);
			return;
		} else port = atoi(argv[1]);
	} else port = PORT;

	/* Hook signals */
	(void) signal(SIGINT, term);
	(void) signal(SIGALRM, timeout);
	(void) signal(SIGSEGV, segv);

	printf("%s [%d] Starting %s", logtime(), getpid(), VER);
	if((ns = socket(AF_INET, SOCK_STREAM, IPPROTO_IP)) < 0) {
		perror("socket()");
		return;
	}
	printf("%s Created socket: s=%d\n", logtime(), ns);

	setsockopt(ns, SOL_SOCKET, SO_REUSEADDR, (char *)&opt, sizeof(opt));

	memset(&sin, 0, sizeof(struct sockaddr_in));
	memset(&fsin, 0, sizeof(struct sockaddr_in));

	sin.sin_family=AF_INET;
	sin.sin_port=htons(port);
	sin.sin_addr.s_addr=htonl(inet_addr(LOCAL));

	if(bind(ns, (struct sockaddr *)&sin, sizeof(sin)) < 0) {
		printf("%s ", logtime());
		fflush(stdout);
		perror("bind()");
		return;
	}
	printf("%s Bound socket to port %d\n", logtime(), port);

	if(listen(ns, MAXQUEUE) < 0) {
		perror("listen()");
		return;
	}


	printf("%s Listening for connections...\n", logtime());
	while(1) {
		/* Set time interval in which to perform zombie checks...
		** We have to do this every time it loops because select()
		** under Linux clears the timeout struct... lame. */
		to.tv_sec = CHECKTIME * 60;
		to.tv_usec = 0;

		/* Clear input */
		FD_ZERO(&input);
		FD_SET(ns, &input);
		if(select(ns + 1, &input, NULL, NULL, &to) > 0) {
			if((rs = accept(ns, (struct sockaddr *)&fsin, &fromlen)) < 0) {
				printf("%s ", logtime());
				fflush(stdout);
				perror("accept()");
				continue;
			}
		} else {
			/* Kill off zombies */
			while((pid = waitpid(0, NULL, WNOHANG)) > 0) {
				printf("%s [%d] Connection closed - purging session\n", logtime(), pid);
				clients--;
			}
			continue;
		}

		if(clients >= MAXCLIENTS) {
			s = rs;
			put("\nSorry, Net Tamagotchi is full right now.\nTry logging in later.\n\n");
			close(rs);
			continue;
		}

		clients++;
		flags = fcntl(ns, F_GETFL);
		flags |= O_NONBLOCK;

		if(fcntl(ns, F_SETFL, flags) < 0) {
			printf("%s ", logtime());
			fflush(stdout);
			perror("fcntl()");
		}

	/* Fork a child to handle the session and wait for it to terminate */
		if((pid=fork()) > 0) {

	/* Resolve remote hostname */
			hp = gethostbyaddr((char *)&fsin.sin_addr, sizeof(struct in_addr), fsin.sin_family);
			if (hp)
				host = hp->h_name;
			else
				host = inet_ntoa(fsin.sin_addr);	
	
			printf("%s [%d] Accepted connection from %s\n", logtime(), pid, host);
			close(rs);
			continue;
		}

		/* Login */
		s = rs;

		if((fd=open(MOTD, O_RDONLY)) > 0)
			putmotd(fd);

		put(INTRO);
		alarm(TIMELIMIT);	/* Set timeout alarm */

		get(buf);
		strncpy(name, buf, MAXNAME);
		if(exist(buf)<0) {
			/* Check username format */
			if(check(name)<0) {
				put("That name is invalid.\n");
				put(STRINGRULE);
				close(s);
				exit(0);
			}
			put("That Tamagotchi doesn't exist. Would you like to create it? ");
			get(buf);
			if(buf[0]!='y' && buf[0]!='Y') {
				put("Fine, but you're missing out!\n");
				close(s);
				exit(0);
			}
			while(1) {
				put("Please choose a password: ");
				get(buf);
	
			/* Check password format validity */
				if(check(buf)==0) break;
				put(STRINGRULE);
			}
			if(new(name, buf) < 0) {
				put(NOACCESS );
				close(s);
				exit(0);
			}
			put("\nNew Tamagotchi \"");
			put(name);
			put("\", created.\n");
			printf("%s Created %s\n", logtime(), name);
		} else {
			put("Tamagotchi found. Please enter password: ");
			get(buf);
			if(checkpass(name, buf)<0) {
				printf("%s Incorrect password for %s\n", logtime(), name);
				put("Password incorrect.\n");
				return;
			}
		}

		printf("%s [%d] `%s` logged in\n", logtime(), getpid(), name);
		put("Hi! The time limit for this session is 5 minutes\n");
		status(name, 1);

		while(1) {
			do {
				buf[0]='\0';
				put("> ");
				get(buf);
			} while(strlen(buf)==0);
	
			printf("%s Got command from %s: %s\n", logtime(), name, buf);

			/* parse argument */
			if(strstr(buf, " ")!=NULL) {
				ptr = buf;
				while(isalnum(ptr[0])) ptr++;
				ptr[0] = 0;
				while(!isalnum(ptr[0])) ptr++;
				strncpy(arg, ptr, BUFLEN);
			} else arg[0] = 0;

			if(strstr(arg, " ")!=NULL) {
				ptr = arg;
				while(isalnum(ptr[0])) ptr++;
				ptr[0] = 0;
				while(!isalnum(ptr[0])) ptr++;
			} else ptr = NULL;

			if(exec(buf, arg, ptr, name) < 0) {
				put(BYE);
				close(s);
				return;
			}
		}
	}
}
