## vim:ts=4:et:nowrap
##
##---------------------------------------------------------------------------##
##
## PySol -- a Python Solitaire game
##
## Copyright (C) 1999 Markus Franz Xaver Johannes Oberhumer
## Copyright (C) 1998 Markus Franz Xaver Johannes Oberhumer
##
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program; see the file COPYING.
## If not, write to the Free Software Foundation, Inc.,
## 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
##
## Markus F.X.J. Oberhumer
## <markus.oberhumer@jk.uni-linz.ac.at>
## http://wildsau.idv.uni-linz.ac.at/mfx/pysol.html
##
##---------------------------------------------------------------------------##


# imports
import sys, math

# PySol imports
if sys.modules.has_key("pysoltk"):                                  #bundle#
    from gamedb import registerGame, GameInfo, GI                   #bundle#
    from util import *                                              #bundle#
    from stack import *                                             #bundle#
    from game import Game                                           #bundle#
    from layout import Layout                                       #bundle#
    from hint import AbstractHint, DefaultHint, CautiousDefaultHint #bundle#
    from pysoltk import MfxCanvasText, getFont                      #bundle#


# /***********************************************************************
# //
# ************************************************************************/

class Braid_Hint(DefaultHint):
    # FIXME: demo is not too clever in this game
    pass


# /***********************************************************************
# //
# ************************************************************************/

class Braid_Foundation(AbstractFoundationStack):
    def updateText(self):
        AbstractFoundationStack.updateText(self)
        self.game.updateText()

    def acceptsPile(self, from_stack, cards):
        if not AbstractFoundationStack.acceptsPile(self, from_stack, cards):
            return 0
        # check suit
        if self.cards and cards[0].suit != self.cards[-1].suit:
            return 0
        # check rank
        card_dir = (cards[0].rank - self.game.base_card.rank) % self.cap.mod
        if len(self.cards) == 0:
            return card_dir == 0
        game_dir = self.game.getFoundationDirection()
        if game_dir == 0:
            return card_dir == 1 or card_dir == 12
        else:
            return (self.cards[-1].rank + game_dir) % self.cap.mod == cards[0].rank


class Braid_BraidStack(OpenStack):
    def __init__(self, x, y, game):
        OpenStack.__init__(self, x, y, game)
        self.CARD_YOFFSET = self.game.app.images.CARD_YOFFSET
        # use a nice sine wave for the x offsets
        self.CARD_XOFFSET = []
        n = 9
        dx = 0.4 * self.game.app.images.CARDW * (2*math.pi/n)
        last_x = 0
        for i in range(n):
            x = int(round(dx * math.sin(i + 1)))
            ##print x, x - last_x
            self.CARD_XOFFSET.append(x - last_x)
            last_x = x


class Braid_RowStack(ReserveStack):
    def fillStack(self):
        if not self.cards and self.game.s.braid.cards:
            self.game.moveMove(1, self.game.s.braid, self)

    def getBottomImage(self):
        return self.game.app.images.getBraidBottom()


class Braid_ReserveStack(ReserveStack):
    def acceptsPile(self, from_stack, cards):
        if from_stack is self.game.s.braid or from_stack in self.game.s.rows:
            return 0
        return ReserveStack.acceptsPile(self, from_stack, cards)

    def getBottomImage(self):
        return self.game.app.images.getTalonBottom()


# /***********************************************************************
# // Braid
# ************************************************************************/

class Braid(Game):
    Hint_Class = Braid_Hint

    #
    # game layout
    #

    def createGame(self):
        # create layout
        l, s = Layout(self), self.s

        # set window
        # (piles up to 20 cards are playable - needed for Braid_BraidStack)
        h = max(4*l.YS + 30, l.YS+(20-1)*l.YOFFSET)
        self.setSize(10*l.XS+l.XM, l.YM + h)

        # extra settings
        self.base_card = None

        # create stacks
        s.addattr(braid=None)      # register extra stack variable
        x, y = l.XM, l.YM
        for i in range(2):
            s.rows.append(Braid_RowStack(x + 0.5*l.XS, y, self))
            s.rows.append(Braid_RowStack(x + 4.5*l.XS, y, self))
            y = y + 3 * l.YS
        y = l.YM + l.YS
        for i in range(2):
            s.rows.append(Braid_ReserveStack(x, y, self))
            s.rows.append(Braid_ReserveStack(x + l.XS, y, self))
            s.rows.append(Braid_ReserveStack(x, y + l.YS, self))
            s.rows.append(Braid_ReserveStack(x + l.XS, y + l.YS, self))
            x = x + 4 * l.XS
        x = l.XM + l.XS * 5/2
        y = l.YM
        s.braid = Braid_BraidStack(x, y, self)
        x = l.XM + 7 * l.XS
        y = l.YM + l.YS * 3/2
        s.talon = WasteTalonStack(x, y, self, max_rounds=3)
        l.createText(s.talon, "ss")
        s.talon.texts.rounds = MfxCanvasText(self.canvas,
                               x + l.CW / 2, y - l.YM,
                               anchor="s")
        x = x - l.XS
        s.waste = WasteStack(x, y, self)
        l.createText(s.waste, "ss")
        x = l.XM + 8 * l.XS
        y = l.YM
        for i in range(4):
            s.foundations.append(Braid_Foundation(x, y, self, i, mod=13, base_rank=-1))
            s.foundations.append(Braid_Foundation(x + l.XS, y, self, i, mod=13, base_rank=-1))
            y = y + l.YS
        self.dir_info_text = MfxCanvasText(self.canvas,
                               x + l.CW + l.XM / 2, y,
                               anchor="n", font=getFont("canvas_card", cardw=l.CW))

        # define stack-groups
        self.sg.openstacks = s.foundations + s.rows
        self.sg.talonstacks = [s.talon] + [s.waste]
        self.sg.dropstacks = [s.braid] + s.rows + [s.waste]

    #
    # game extras
    #

    def getFoundationDirection(self):
        for s in self.s.foundations:
            if len(s.cards) >= 2:
                break
        else:
            return 0
        d = (s.cards[1].rank - s.cards[0].rank) % 13
        assert d == 1 or d == 12
        return d


    def updateText(self):
        if not self.base_card:
            t = ""
        else:
            game_dir = self.getFoundationDirection()
            t = RANKS[self.base_card.rank]
            if game_dir == 1:
                t = t + " Ascending"
            elif game_dir == 12:
                t = t + " Descending"
        if cmp(t, self.dir_info_text["text"]) != 0:
            self.dir_info_text["text"] = t

    #
    # game overrides
    #

    def startGame(self):
        self.base_card = None
        self.updateText()
        for i in range(20):
            self.s.talon.dealRow(rows=[self.s.braid])
        self.s.talon.dealRow()
        # deal base_card to foundations
        self.base_card = self.s.talon.getCard()
        to_stack = self.s.foundations[2 * self.base_card.suit]
        self.flipMove(self.s.talon)
        self.moveMove(1, self.s.talon, to_stack)
        # deal first card to WasteStack
        self.s.talon.dealCards()

    def shallHighlightMatch(self, stack1, card1, stack2, card2):
        return (card1.suit == card2.suit and
                ((card1.rank + 1) % 13 == card2.rank or (card2.rank + 1) % 13 == card1.rank))

    def getHighlightPilesStacks(self):
        return ()

    def _restoreGameHook(self, game):
        self.base_card = self.cards[game.loadinfo.base_card_id]
        self.updateText()

    def _loadGameHook(self, p):
        self.loadinfo.addattr(base_card_id=None)    # register extra load var.
        self.loadinfo.base_card_id = p.load()

    def _saveGameHook(self, p):
        p.dump(self.base_card.id)


# register the game
registerGame(GameInfo(12, Braid, "Braid",
                      GI.GT_2DECK_TYPE, 2, 2))


