/*C* $Id: file.cc,v 1.4 1997/09/26 17:57:57 james Exp $
 *
 * Hatman - The Game of Kings
 * Copyright (C) 1997 James Pharaoh & Timothy Fisken
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *C*/

#include "File.h"
#include "debug.h"
#include "util.h"
#include <errno.h>
#include <stdarg.h>
#include <stdlib.h>
#include <assert.h>

//--------------------------------------------------------------------------------------------------------------------------------

File::File()
{
 fp = NULL;
}

File::File(const char* filename, const char* mode)
{
 assert(filename != NULL); assert(mode != NULL);
 fp = NULL;
 open(filename, mode);
}

File::File(FILE* _fp)
{
 assert(_fp != NULL);
 fp = _fp;
 fpWantsClosing = true;
}

#ifndef NDEBUG
File::File(File& object)
 : Object(object)
{
 fatal("You did a File = File, didn't you? Well I can't allow it.\nIt's too silly, I am afraid.\n");
}
#endif

File::~File()
{
 if(fpWantsClosing) close();
}

FILE* File::open(const char* filename, const char* mode)
{
 assert(this != NULL); assert(filename != NULL); assert(mode != NULL);
 close();
 VPRINTF("<file> opening \"%s\" in mode \"%s\"\n", filename, mode);
 fp = fopen(filename, mode);
 if(!fp) setError("%s", errnoStr());
 return fp;
}

int File::close()
{
 assert(this != NULL);
 if(!fp) return 0;
 int ret = fclose(fp);
 fp = NULL;
 return ret;
}

int File::scanf(const char* format, ...)
{
 assert(this != NULL); assert(format != NULL);
 va_list ap;
 va_start(ap, format);
 int ret = vfscanf(fp, format, ap);
 va_end(ap);
 return ret;
}

bool File::read(void* data, size_t size)
{
 if(fread(data, size, 1, fp) != 1)
  {
   if(errno) setError("%s", errnoStr());
   else setError("end of file");
   return false;
  }
 return true;
}

bool File::write(const void* data, size_t size)
{
 if(fwrite(data, size, 1, fp) != 1)
  {
   setError("%s", errnoStr());
   return false;
  }
 return true;
}

//--------------------------------------------------------------------------------------------------------------------------------

File fStdOut(stdout);
File fStdIn(stdin);
File fStdErr(stderr);

//--------------------------------------------------------------------------------------------------------------------------------

Pipe::Pipe()
 : File()
{
}

Pipe::Pipe(const char* filename, const char* mode)
{
 assert(filename != NULL); assert(mode != NULL);
 fp = NULL;
 open(filename, mode);
}

FILE* Pipe::open(const char* filename, const char* mode)
{
 assert(this != NULL); assert(filename != NULL); assert(mode != NULL);
 close();
 VPRINTF("<file> opening \"%s\" as pipe in mode \"%s\"\n", filename, mode);
 fp = popen(filename, mode);
 if(!fp) setError(errnoStr());
 return fp;
}

int Pipe::close()
{
 assert(this != NULL);
 if(!fp) return 0;
 int ret = pclose(fp);
 fp = NULL;
 return ret;
}

//--------------------------------------------------------------------------------------------------------------------------------

GzipPipe::GzipPipe(const char* filename, const char* mode)
{
 assert(filename != NULL); assert(mode != NULL);
 fp = NULL;
 open(filename, mode);
}

FILE* GzipPipe::open(const char* filename, const char* mode)
{
 assert(this != NULL); assert(filename != NULL); assert(mode != NULL);
 if(!strcmp(mode, "r"))
  {
   char* s;
   asprintf(&s, "gzip -dc %s", filename);
   Pipe::open(s, "r"); // assigns fp
   free(s);
   return fp;
  }
 else if(!strcmp(mode, "w"))
  {
   char* s;
   asprintf(&s, "/bin/sh -c \"gzip - > %s\"", filename);
   Pipe::open(s, "w"); // assigns fp
   free(s);
   return fp;
  }

 setError("unknown file mode: \"%s\"", mode);
 return NULL;
}

//--------------------------------------------------------------------------------------------------------------------------------

bool save(const char* filename, Object* object, bool (Object::*write)(File& f))
{
 File* f = gzFilename(filename)? new GzipPipe(filename, "w") : new File(filename, "w");
 if(!f->ready()) { setError("loading %s: %s", filename, errStr); return false; }
 if(!(object->*write)(*f)) { delete f; setError("saving %s: %s", filename, errStr); return false; }
 delete f;
 return true;
}

Object* load(const char* filename, Object* create(File& f))
{
 Object* o = NULL;
 File* f = gzFilename(filename)? new GzipPipe(filename, "r") : new File(filename, "r");
 if(!f->ready()) { setError("loading %s: %s", filename, errStr); return NULL; }
 if((o = create(*f)) == NULL) { delete f; setError("loading %s: %s", filename, errStr); return NULL; }
 return o;
}

//--------------------------------------------------------------------------------------------------------------------------------
