/*C*
 *
 * Hatman - The Game of Kings
 * Copyright (C) 1997 James Pharaoh & Timothy Fisken
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *C*/

#include "Color.h"
#include "Console.h"
#include "Point.h"
#include "Rect.h"
#include "VgaContext.h"
#include "alphaTable.h"
#include "palette.h"
#include "../util/debug.h"
#include "../util/util.h"
#include <assert.h>
#include <string.h>

//--------------------------------------------------------------------------------------------------------------------------------

VgaContext::VgaContext()
{
 assert(bpp != 0 && bypp != 0);
 initAlphaTable();

 pFb = new unsigned char* [vgaH];
 unsigned char* p = new unsigned char[vgaW * vgaH * bypp];
 for(int y=0; y<vgaH; y++) { pFb[y] = p; p += vgaW * bypp; }

 clipping = false;
 clippingRect = screenRect;
}

VgaContext::~VgaContext()
{
 delete pFb[0];
 delete pFb;
}

//--------------------------------------------------------------------------------------------------------------------------------

void VgaContext::copyBox(VgaContext* sc, Rect sr, VgaContext* dc, Point dp)
{
 int sx = bypp * sr.left(), dx = bypp * dp.x, bytewidth = bypp * sr.w;
 for(int sy=sr.top(), dy=dp.y; sy<=sr.bottom(); sy++, dy++)
  memcpy(&dc->pFb[dy][dx], &sc->pFb[sy][sx], bytewidth);
}

//--------------------------------------------------------------------------------------------------------------------------------

Rect VgaContext::fillBox(Rect r, Color c)
{
 if(clipping) r.clipTo(clippingRect);
 for(int y=r.top(); y<=r.bottom(); y++)
  drawHLine(Point(r.left(), y), r.width(), c);
 return r;
#if 0
 switch(bpp)
  {

  case 8:
   {
    unsigned char s = rgb2pal(c);
    for(int y=r.top(); y<=r.bottom(); y++) memset(&pFb[y][r.left()], s, r.w);
   }
   break;

  case 16:
   {
    for(int y=r.top(); y<=r.bottom(); y++)
     {
      unsigned short* d = (unsigned short*) &pFb[y][2 * r.left()];
      if(c.a == 0xFF)
       {
	unsigned short c16 = c.b >> 3 | (c.g & 0xFC) << 3 | (c.r & 0xF8) << 8;
	for(int x=0; x<r.w; x++) *d++ = c16;
       }
      else drawHLine(Point(r.left(), y), r.width(), c);
     }
   }
   break;

  case 24:
   for(int y=r.top(); y<=r.bottom(); y++)
    {
     unsigned char* d = &pFb[y][3 * r.left()];
     if(c.a == 0xFF) for(int x=0; x<r.w; x++) { *d++ = c.b; *d++ = c.g; *d++ = c.r; }
     else for(int x=0; x<r.w; x++)
      {
       *d++ = alphaTable[255 - c.a][*d] + alphaTable[c.a][c.b];
       *d++ = alphaTable[255 - c.a][*d] + alphaTable[c.a][c.g];
       *d++ = alphaTable[255 - c.a][*d] + alphaTable[c.a][c.r];
      }
    }
   break;

  case 32:
   for(int y=r.top(); y<=r.bottom(); y++)
    {
     unsigned char* p = &pFb[y][4 * r.left()];
     for(int x=0; x<r.w; x++) { *p++ = c.b; *p++ = c.g; *p++ = c.r; p++; }
    }
   break;

  }
 #endif
}

void VgaContext::fill(Color c)
{
 fillBox(screenRect, c);
}

//--------------------------------------------------------------------------------------------------------------------------------

Rect VgaContext::drawHLine(Point p, int w, Color c)
{
 // FIXME all alpha support
 switch(bpp)
  {

  case 8:
   memset(&fb()[p.y][p.x], rgb2pal(c), w);
   break;

  case 16:
   {
    uint16* d = (uint16*) &fb()[p.y][2 * p.x];
    if(c.a == 0xFF)
     {
      unsigned short s = c.b >> 3 | (c.g & 0xFC) << 3 | (c.r & 0xF8) << 8;
      for(int x = 0; x < w; x++) *d++ = s;
     }
    else for(int x = 0; x < w; x++)
     {
      uint8 b = alphaTable[255 - c.a][(*d & 0x001F) << 3] + alphaTable[c.a][c.b];
      uint8 g = alphaTable[255 - c.a][(*d & 0x07E0) >> 3] + alphaTable[c.a][c.g];
      uint8 r = alphaTable[255 - c.a][(*d & 0xF800) >> 8] + alphaTable[c.a][c.r];
      *d++ = b >> 3 | (g & 0xFC) << 3 | (r & 0xF8) << 8;
     }
   }
   break;

  case 24:
   {
    unsigned char* d = &pFb[p.y][p.x * 3];
    if(c.a == 0xFF)
     for(int x = 0; x<w; x++) { *d++ = c.b; *d++ = c.g; *d++ = c.r; }
    else
     for(int x = 0; x<w; x++)
      {
       *d++ = alphaTable[255 - c.a][*d] + alphaTable[c.a][c.b];
       *d++ = alphaTable[255 - c.a][*d] + alphaTable[c.a][c.g];
       *d++ = alphaTable[255 - c.a][*d] + alphaTable[c.a][c.r];
      }
   }
   break;

  case 32:
   {
    unsigned char* d = &pFb[p.y][p.x << 2];
    if(c.a == 0xFF)
     for(int x = 0; x<w; x++) { *d++ = c.b; *d++ = c.g; *d++ = c.r; d++; }
    else
     for(int x = 0; x<w; x++)
      {
       *d++ = alphaTable[255 - c.a][*d] + alphaTable[c.a][c.b];
       *d++ = alphaTable[255 - c.a][*d] + alphaTable[c.a][c.g];
       *d++ = alphaTable[255 - c.a][*d] + alphaTable[c.a][c.r];
       d++;
      }
   }
   break;

  }
 return Rect(p, Point(w, 0));
}

Rect VgaContext::drawVLine(Point p, int h, Color c)
{
 // FIXME all alpha support
 // FIXME this could probably be faster
 switch(bpp)
  {

  case 8:
   for(int y = 0; y<h; y++) fb()[p.y + y][p.x] = rgb2pal(c);
   break;

  case 16:
   {
    if(c.a == 0xFF)
     {
      unsigned short s = c.b >> 3 | (c.g & 0xFC) << 3 | (c.r & 0xF8) << 8;
      for(int y=0; y<h; y++)
       reinterpret_cast<uint16*const*>(fb())[p.y + y][p.x] = s;
     }
    else
     for(int y=0; y<h; y++)
      {
       uint16* d = &reinterpret_cast<uint16*const*>(fb())[p.y + y][p.x];
       uint8 b = alphaTable[255 - c.a][(*d & 0x001F) << 3] + alphaTable[c.a][c.b];
       uint8 g = alphaTable[255 - c.a][(*d & 0x07E0) >> 3] + alphaTable[c.a][c.g];
       uint8 r = alphaTable[255 - c.a][(*d & 0xF800) >> 8] + alphaTable[c.a][c.r];
       *d = b >> 3 | (g & 0xFC) << 3 | (r & 0xF8) << 8;
      }
   }
   break;

  case 24:
   for(int y=0; y<h; y++)
    { 
     unsigned char* d = &pFb[p.y + y][p.x * 3];
     if(c.a == 0xFF)
      { *d++ = c.b; *d++ = c.g; *d++ = c.r; }
     else
      {
       *d++ = alphaTable[255 - c.a][*d] + alphaTable[c.a][c.b];
       *d++ = alphaTable[255 - c.a][*d] + alphaTable[c.a][c.g];
       *d++ = alphaTable[255 - c.a][*d] + alphaTable[c.a][c.r];
      }
    }
   break;

  case 32:
   for(int y=0; y<h; y++)
    {
     unsigned char* q = &pFb[p.y + y][p.x << 2];
     *q++ = c.b; *q++ = c.g; *q++ = c.r;
    }
   break;

  }
 return Rect(p, Point(0, h));
}

Rect VgaContext::drawRect(Rect r, Color c)
{
 drawHLine(r.topLeft(), r.w, c);
 drawHLine(r.bottomLeft(), r.w, c);
 drawVLine(r.topLeft(), r.h, c);
 drawVLine(r.topRight(), r.h, c);
 return r;
}

//--------------------------------------------------------------------------------------------------------------------------------
