/*C*
 *
 * Hatman - The Game of Kings
 * Copyright (C) 1997 James Pharaoh & Timothy Fisken
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *C*/

#include "Console.h"
#include "Keyboard.h"
#include "Mouse.h"
#include "VgaBlur.h"
#include "../util/String.h"
#include "../util/debug.h"
#include "../util/error.h"
#include <assert.h>

#ifdef GGI
ggi_visual_t consoleVisual;
#endif

#ifdef SVGALIB
#include <vga.h>
#endif

//--------------------------------------------------------------------------------------------------------------------------------

VgaBlur* screen;
Keyboard* keyboard;
Mouse* mouse;

TT_Engine ttEngine;

//--------------------------------------------------------------------------------------------------------------------------------

int bpp = 0, bypp = 0;

//--------------------------------------------------------------------------------------------------------------------------------

static bool consoleInitGraphics()
{
 bool have8bpp = false, have16bpp = false, have24bpp = false, have32bpp = false;

#ifdef GGI
 VPRINTF("<console> calling ggiInit()\n");
 if(ggiInit() < 0) { setError("ggiInit() failed"); return false; }
 
 VPRINTF("<console> calling ggiOpen()\n");
 if((consoleVisual = ggiOpen(NULL)) == NULL)
  { setError("ggiOpen(NULL) failed"); ggiExit(); return false; }
 
 ggiSetEventMask(consoleVisual, (gii_event_mask)(emKey | emPointer));
 ggiSetFlags(consoleVisual, GGIFLAG_ASYNC);

 VPRINTF("<console> checking available modes\n");
 have8bpp = (ggiCheckSimpleMode(consoleVisual, vgaW, vgaH, 1, GT_8BIT, NULL) == 0)? true : false;
 have16bpp = (ggiCheckSimpleMode(consoleVisual, vgaW, vgaH, 1, GT_16BIT, NULL) == 0)? true : false;
 have24bpp = (ggiCheckSimpleMode(consoleVisual, vgaW, vgaH, 1, GT_24BIT, NULL) == 0)? true : false;
 have32bpp = (ggiCheckSimpleMode(consoleVisual, vgaW, vgaH, 1, GT_32BIT, NULL) == 0)? true : false;
#endif

#ifdef SVGALIB
 VPRINTF("<console> calling vga_init()\n");
 if(vga_init() != 0) { setError("vga_init() failed"); return false; }

 VPRINTF("<console> checking available modes\n");
 // this is really twisted but, hey, thats svgalib for you...
 String modePrefix = String("G") + String(vgaW) + String("x") + String(vgaH);
 have8bpp = vga_hasmode(vga_getmodenumber(const_cast<char*>((modePrefix + String("x256")).cs())))? true : false;
 have16bpp = vga_hasmode(vga_getmodenumber(const_cast<char*>((modePrefix + String("x64K")).cs())))? true : false;
 have24bpp = vga_hasmode(vga_getmodenumber(const_cast<char*>((modePrefix + String("x16M")).cs())))? true : false;
 have32bpp = vga_hasmode(vga_getmodenumber(const_cast<char*>((modePrefix + String("x16M32")).cs())))? true : false;
#endif

 VPRINTF("<console> has bitdepths:");
 if(have8bpp) VPRINTF(" 8bpp");
 if(have16bpp) VPRINTF(" 16bpp");
 if(have24bpp) VPRINTF(" 24bpp");
 if(have32bpp) VPRINTF(" 32bpp");
 VPRINTF("\n");

 switch(bpp)
  {
   // normally we pick the best mode ourselves

  case 0:
   if(have24bpp) { bpp = 24; bypp = 3; }
   else if(have32bpp) { bpp = 32; bypp = 4; }
   else if(have16bpp) { bpp = 16; bypp = 2; }
   else if(have8bpp) { bpp = 8; bypp = 1; }
   else
    {
     setError("no suitable graphics mode");
     goto fail;
    }
   break;
   
   // if bpp is already set, though, only try that mode

  case 8:
   if(!have8bpp) goto modeNotAvail;
   bypp = 1;
   break;

  case 16:
   if(!have16bpp) goto modeNotAvail;
   bypp = 2;
   break;

  case 24:
   if(!have24bpp) goto modeNotAvail;
   bypp = 3;
   break;

  case 32:
   if(!have32bpp) goto modeNotAvail;
   bypp = 4;
   break;

   // if bpp was set to something bizarre
  default:
   setError("no such mode %dx%dx%d", vgaW, vgaH, bpp);
   goto fail;
  }

 // we have found a mode one way or another
 VPRINTF("<console> selected mode: %dx%dx%d\n", vgaW, vgaH, bpp);
 return true;
 
modeNotAvail:
 // a mode was suggested but is not available
 setError("mode not available %dx%dx%d", vgaW, vgaH, bpp);
fail:
#ifdef GGI
 ggiClose(consoleVisual);
 ggiExit();
#endif
#ifdef SVGALIB
 vga_setmode(TEXT);
#endif
 return false;
}

static void consoleExitGraphics()
{
#ifdef GGI
  ggiClose(consoleVisual);
  ggiExit();
#endif

#ifdef SVGALIB
 vga_setmode(TEXT);
#endif

 // reset this just in case we are restarted...
 bpp = 0;
}

//--------------------------------------------------------------------------------------------------------------------------------

static int consoleInitCount = 0;

bool consoleInit()
{
 if(consoleInitCount++ == 0)
  {
   VPRINTF("<console> initialising\n");

   if(!consoleInitGraphics()) return false;

   VPRINTF("<console> Initialising freetype library\n");
   if(TT_Init_FreeType(&ttEngine))
    { setError("initialising freetype"); consoleExitGraphics(); return false; }

   VPRINTF("<console> Creating screen, keyboard, mouse\n");
   screen = new VgaBlur; 
   keyboard = new Keyboard;
   mouse = new Mouse;

   VPRINTF("<console> Initialised Ok\n");
  }
 return true;
}

void consoleExit()
{
 assert(consoleInitCount > 0);
 if(--consoleInitCount == 0)
  {
   delete mouse;
   delete keyboard;
   delete screen;

   consoleExitGraphics();
   TT_Done_FreeType(ttEngine);

   VPRINTF("<console> Exited\n");
  }
}

//--------------------------------------------------------------------------------------------------------------------------------
