/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *\
 * This is GNU GO, a Go program. Contact gnugo@gnu.org, or see   *
 * http://www.gnu.org/software/gnugo/ for more information.      *
 *                                                               *
 * Copyright 1999 by the Free Software Foundation.               *
 *                                                               *
 * This program is free software; you can redistribute it and/or *
 * modify it under the terms of the GNU General Public License   *
 * as published by the Free Software Foundation - version 2.     *
 *                                                               *
 * This program is distributed in the hope that it will be       *
 * useful, but WITHOUT ANY WARRANTY; without even the implied    *
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR       *
 * PURPOSE.  See the GNU General Public License in file COPYING  *
 * for more details.                                             *
 *                                                               *
 * You should have received a copy of the GNU General Public     *
 * License along with this program; if not, write to the Free    *
 * Software Foundation, Inc., 59 Temple Place - Suite 330,       *
 * Boston, MA 02111, USA                                         *
\* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */




/* This tells Alpha OSF/1 not to define a getopt prototype in <stdio.h>.
 *    Ditto for AIX 3.2 and <stdlib.h>.  
 */

#ifndef _NO_PROTO
#define _NO_PROTO
#endif

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <time.h>
#include <signal.h>

#include "getopt.h"

#include "main.h"
#include "interface.h"
#include "gmp.h"
#include "sgf.h"
#include "sgf_utils.h"
#include "ttsgf_read.h"
#include "ttsgf_write.h"
#include "ttsgf.h"
#include "sgfana.h"
#include "liberty.h"
#include "hash.h"


/* 
 * Show usage of GNU Go.
 *
 * This string is modelled on the gnu tar --help output.
 */

#define USAGE "\n\
Usage : gnugo [-opts]\n\
\n\
Main Options:\n\
       --mode <mode>     Force the playing mode ('ascii', 'test' or 'gmp').\n\
                         Default is ASCII.\n\
                         If no terminal is detected GMP (Go Modem Protocol)\n\
                         will be assumed.\n\
       --quiet           Don't print copyright and other informational messages\n\
   -l, --infile <file>   Load name sgf file\n\
   -L, --until <move>    Stop loading just before move is played. <move>\n\
                         can be the move number or location (eg L10).\n\
   -o, --outfile file    Write sgf output to file\n\
   -p, --playstyle <style>    style of play, use --help playstyle for usage\n\
\n\
Game Options: (--mode ascii)\n\
       --boardsize num   Set the board size to use (%d..%d)\n\
       --color <color>   Choose your color ('black' or 'white')\n\
       --handicap <num>  Set the number of handicap stones (0..%d)\n\
       --komi <num>      Set the komi\n\
\n\
Informative Output:\n\
   -v, --version         Display the version of GNU Go\n\
   -h, --help            Display this help message\n\
       --help analyze    Display help about analyzer options\n\
       --help debug      Display help about debugging options\n\
       --help playstyle  Display help about playstyle options\n\
       --copyright       Display copyright notice\n\
\n\
    "

#define COPYRIGHT "\n\n\
This is GNU GO, a Go program. Contact gnugo@gnu.org, or see\n\
http://www.gnu.org/software/gnugo/ for more information.\n\n\
Copyright 1999 by the Free Software Foundation.\n\n\
This program is free software; you can redistribute it and/or\n\
modify it under the terms of the GNU General Public License\n\
as published by the Free Software Foundation - version 2.\n\n\
This program is distributed in the hope that it will be\n\
useful, but WITHOUT ANY WARRANTY; without even the implied\n\
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR\n\
PURPOSE.  See the GNU General Public License in file COPYING\n\
for more details.\n\n\
You should have received a copy of the GNU General Public\n\
License along with this program; if not, write to the Free\n\
Software Foundation, Inc., 59 Temple Place - Suite 330,\n\
Boston, MA 02111, USA\n\n\
"
#define USAGE_DEBUG "\n\
Debugging Options:\n\
\n\
       --testmode <mode> set the test mode to one of the following:\n\
                         (requires -l, implies --mode test)\n\
	             move: test at move node only\n\
		     annotation: test at annotation node only\n\
		     both: test at move and annotation nodes\n\
		     game: test to see if gnugo considered each move made\n\
                 This overrides a testmode=... comment in the sgf file\n\
   -a, --allpats                test all patterns\n\
   -T, --printboard             colored display of dragons\n\
   -E                           colored display of eye spaces\n\
   -d, --debug [level]          debugging output (see liberty.h for bits)\n\
   -D, --depth [depth]          deep reading cutoff (default %d; higher=stronger, slower)\n\
   -B, --backfill_depth [depth] deep reading cutoff (default %d; higher=stronger, slower)\n\
   -H, --hash [level]		hash (see liberty.h for bits)\n\
   -w, --worms                  worm debugging\n\
   -m, --moyo [level]           moyo debugging, show moyo board\n\
   -b, --benchmark num          benchmarking mode - can be used with -l\n\
   -s, --stack                  stack trace (for debugging purposes)\n\n\
   -S, --statistics             print statistics (for debugging purposes)\n\n\
   -t, --trace                  verbose tracing (use twice or more to trace reading)\n\
   -r, --seed number            set random number seed\n\
       --decidestring string    give full read tracing when studying this string\n\
       --score [end|last|move]  count or estimate territory of the input file\n\
       --printsgf outfile       load SGF file, output final position (requires -l)\n\
"
    

/* define globals */

board_t p[MAX_BOARD][MAX_BOARD];  /* go board */
int ko_i;
int ko_j;


Hash_item  hashitem;
unsigned char potential_moves[MAX_BOARD][MAX_BOARD];

int black_captured, white_captured;                /* num. of black and white stones captured */
int black_captured_backup, white_captured_backup; 

int lib;                              /* current stone liberty written by count() */
int libi[MAXLIBS], libj[MAXLIBS];     /* array of liberties found : filled by count() */
int size;                  /* cardinality of a group : written by count() */

int stackp;                /* stack pointer */
int movenum;               /* movenumber */
int depth;                 /* deep reading cut off */
int backfill_depth;        /* deep reading cut off */

int showstack;             /* debug stack pointer */
int showstatistics;        /* print statistics */
int allpats;               /* generate all patterns, even small ones */
int printworms;            /* print full data on each string */
int printmoyo;             /* print moyo board each move*/
int printboard;            /* print board each move */
int board_size=19;         /* board size */
int count_variations=0;    /* used by decide_string */
int sgf_dump=0;            /* used by decide_string*/
int loading=0;             /* TRUE indicates last loaded move comes from file*/
int style=STY_DEFAULT;     /* style of play */

struct stats_data stats;

struct worm_data worm[MAX_BOARD][MAX_BOARD];
struct dragon_data dragon[MAX_BOARD][MAX_BOARD];
struct half_eye_data half_eye[MAX_BOARD][MAX_BOARD];
struct eye_data black_eye[MAX_BOARD][MAX_BOARD];
struct eye_data white_eye[MAX_BOARD][MAX_BOARD];

int distance_to_black[MAX_BOARD][MAX_BOARD];
int distance_to_white[MAX_BOARD][MAX_BOARD];
int strategic_distance_to_black[MAX_BOARD][MAX_BOARD];
int strategic_distance_to_white[MAX_BOARD][MAX_BOARD];
int black_domain[MAX_BOARD][MAX_BOARD];
int white_domain[MAX_BOARD][MAX_BOARD];

Hashtable  * movehash;

int debug=0;
int hashflags=HASH_BASICNET3 | HASH_READLAD2;
int verbose=0;     /* trace level                                           */
int showstack=0;   /* print the stack pointer (for debugging)               */
int showstatistics=0;
int printworms=0;  /* print full data about each string on the board        */
int printmoyo=0;   /* print moyo board each move                            */
int allpats=0;     /* compute and print value of patterns, even small ones  */
int color_has_played=0;    /* whether color has placed a stone yet */

int last_move_i;           /* The position of the last move */
int last_move_j;           /* -""-                          */
SGFNodeP sgf_root = NULL;
char *analyzerfile = NULL;

/* long options which have no short form */
enum {OPT_BOARDSIZE=1,
      OPT_HANDICAPSTONES,
      OPT_COLOR,
      OPT_KOMI,
      OPT_MODE,
      OPT_INFILE,
      OPT_OUTFILE, 
      OPT_QUIET,
      OPT_SHOWCOPYRIGHT,
      OPT_TESTMODE,
      OPT_DECIDE_STRING,
      OPT_SCORE,
      OPT_PRINTSGF,
      OPT_ANALYZER_FILE,
      OPT_ANALYZE,
      OPT_HELPANALYZE,
};

/* names of playing modes */

enum mode {
  MODE_UNKNOWN=0,
  MODE_ASCII,
  MODE_GMP,
  MODE_SGF,
  MODE_LOAD_AND_ANALYZE,
  MODE_LOAD_AND_SCORE,
  MODE_LOAD_AND_PRINT,
  MODE_SOLO,
  MODE_TEST,
  MODE_DECIDE_STRING
};


/* Definitions of the --long options. Final column is
 * either an OPT_ as defined in the enum above, or it
 * is the equivalent single-letter option.
 * It is useful to keep them in the same order as the
 * help string, for maintenance purposes only.
 */

static struct option const long_options[] =
{
  {"mode",     required_argument, 0, OPT_MODE},
  {"testmode", required_argument, 0, OPT_TESTMODE},
  {"quiet", no_argument, 0, OPT_QUIET},

  {"infile",         required_argument, 0, 'l'},
  {"until",          required_argument, 0, 'L'},
  {"outfile",        required_argument, 0, 'o'},

  {"boardsize",      required_argument, 0, OPT_BOARDSIZE},
  {"color",          required_argument, 0, OPT_COLOR},
  {"handicap",       required_argument, 0, OPT_HANDICAPSTONES},
  {"komi",           required_argument, 0, OPT_KOMI},

  {"help",           optional_argument, 0, 'h'},
  {"helpanalyze",    no_argument,       0, OPT_HELPANALYZE },
  {"copyright",      no_argument,       0, OPT_SHOWCOPYRIGHT},
  {"version",        no_argument,       0, 'v'},

  {"allpats",        no_argument,       0, 'a'},
  {"printboard",     no_argument,       0, 'T'},
  {"debug",          required_argument, 0, 'd'},
  {"depth",          required_argument, 0, 'D'},
  {"backfill_depth", required_argument, 0, 'B'},
#if HASHING
  {"hash",           required_argument, 0, 'H'},
#endif
  {"worms",          no_argument,       0, 'w'},
  {"moyo",           required_argument, 0, 'm'},
  {"benchmark",      required_argument, 0, 'b'},
  {"stack",          no_argument,       0, 's'},
  {"statistics",     no_argument,       0, 'S'},
  {"trace",          no_argument,       0, 't'},
  {"seed",           required_argument, 0, 'r'},
  {"playstyle",      required_argument, 0, 'p'},
  {"decidestring",   required_argument, 0, OPT_DECIDE_STRING},
  {"score",          required_argument, 0, OPT_SCORE},
  {"printsgf",       required_argument, 0, OPT_PRINTSGF},
  {"analyzerfile",   required_argument, 0, OPT_ANALYZER_FILE},
  {"analyze",        required_argument, 0, OPT_ANALYZE},
  {NULL, 0, NULL, 0}
};



/* 
 * Cgoban sends us a sigterm when it wants us to die. But it doesn't
 * close the pipe, so we cannot rely on gmp to pick up an error.
 * 
 * We want to keep control so we can close the output sgf file
 * properly, so we trap the signal.
 */

volatile int  time_to_die = 0;   /* set by signal handlers */

void 
sigterm_handler(int sig)
{
  time_to_die = 1;
  if (!get_opt_quiet)
    write(2, "*SIGTERM*\n", 10);  /* bad to use stdio in a signal handler */

  close(0);                  /* this forces gmp.c to return an gmp_err */

  /* I thought signal handlers were one-shot, yet on my linux box it is kept.
   * Restore the default behaviour so that a second signal has the
   * original effect - in case we really are stuck in a loop
   */
  signal(sig, SIG_DFL);

  /* schedule a SIGALRM in 5 seconds, in case we haven't cleaned up by then
   * - cgoban sends the SIGTERM only once 
   */
  alarm(5);
}


void
sigint_handler(int sig)
{
  time_to_die = 1;
  write(2, "*SIGINT*\n", 9);  /* bad to use stdio in a signal handler */
  signal(sig, SIG_DFL);

  /* Don't bother with an alarm - user can just press ^c a second time */
}


static int
show_version(void)
{
  fprintf(stderr, "GNU Go Version %s\n", VERSION);

  return 1;
}

static int
show_help(void)
{
  fprintf(stderr, USAGE, 
	  MIN_BOARD, MAX_BOARD,MAX_HANDICAP);
  return 1;
}


static int
show_copyright(void)
{
  fputs(COPYRIGHT, stderr);

  return 1;
}


#define USAGE_STYLE "\n\
Usage : gnugo [-opts]\n\
\n\
Play Style Options;\n\
   -p, --playstyle <style>      style of play\n\
   <style> can be:
              standard          default opening\n\
              no_fuseki         minimal opening\n\
              tenuki            often plays tenuki in the opening\n\
              fearless          risky style of play\n\
              aggressive        both style tenuki and fearless\n\
"

static void set_style(char *arg)
{
  if (strncmp(arg, "standard", 5) == 0)
    style = STY_DEFAULT;
  else if (strncmp(arg, "no_fuseki", 5) == 0)
    style = STY_NO_FUSEKI;
  else if (strncmp(arg, "tenuki", 5) == 0)
    style = STY_TENUKI|STY_DEFAULT;
  else if (strncmp(arg, "fearless", 5) == 0)
    style = STY_FEARLESS|STY_DEFAULT;
  else if (strncmp(arg, "aggressive", 5) == 0)
    style = STY_FEARLESS|STY_TENUKI|STY_DEFAULT;
  else if (strncmp(arg, "help", 5) == 0) {
    fprintf(stderr, USAGE_STYLE);
    exit(EXIT_SUCCESS);
  } else {
    fprintf(stderr, "Invalid play style selection: %s\n", arg);
    fprintf(stderr, "Try `gnugo --help' for more information.\n");

    exit( EXIT_FAILURE);
  }
}


int
main(int argc, char *argv[])
{
   int i, umove;
   enum mode playmode = MODE_UNKNOWN;
   enum testmode testmode = UNKNOWN_TESTMODE;

   char *outfile = NULL;
   char *until = NULL;

   char *printsgffile = NULL;

   char *infile = NULL;

   char decidestring[4];

   int benchmark = 0;  /* benchmarking mode (-b) */
   float komi;

   int seed=0;           /* If seed is zero, Gnu Go will play a different game 
			  each time. If it is set using -r, Gnu Go will play the
			  same game each time. (Change seed to get a different
			  game). */


   /* Initialize the entire hashing system. */
   hash_init();

   /* Initialize the board. */
   init_board();
   
     /* Initialize the game info structure. */
   init_ginfo();
   init_gopt();

   /* Set some standard options. */
   umove=BLACK;
   
   /* Now weed through all of the command line options. */
   while ((i=getopt_long(argc, argv, "-ab:B:d:D:Egh::H:l:L:o:p:r:fsStTvwm:",
			 long_options, NULL)) != EOF)
     {
       switch (i) {
       case 'T' : printboard++; set_opt_display_board(1); break;
       case 't' : ++verbose; break;
       case 'a' : allpats = 1; break;
	 
       case 'l' : infile = optarg; 
	 if ((sgf_root = readsgffile(infile)) == NULL) {
	   fprintf(stderr, "Cannot open or parse '%s'\n", infile);
	   exit(EXIT_FAILURE);
	 }
	 break;
	 
       case 'b' : benchmark = atoi(optarg); playmode=MODE_SOLO; break;
       case 'r' : seed = atoi(optarg); break;
       case 'p' : set_style(optarg); break;
       case 's' : showstack = 1; break;
       case 'S' : showstatistics = 1; break;
       case 'w' : printworms = 1; break;
       case 'm' : printmoyo = atoi(optarg); break;
       case 'd' : debug = strtol(optarg, NULL, 0);  /* allows 0x... */ break;
       case 'D' : depth=atoi(optarg); break;
#if HASHING
       case 'H' : hashflags = strtol(optarg, NULL, 0);  /* allows 0x... */ break;
#endif
       case 'E' : printboard=2; set_opt_display_board(1); break;
       case 'B' : backfill_depth=atoi(optarg); break;
       case 'L' : until = optarg;
	 if (playmode == MODE_UNKNOWN)
	   playmode = MODE_LOAD_AND_ANALYZE;
	 break;
       case 'o' : outfile = optarg;
	 set_outfile(optarg);
	 break;
	 
       case OPT_QUIET :
	 set_opt_quiet(1);
	 break;


       case OPT_HANDICAPSTONES :
	 {
	   int handicap = atoi(optarg);

	   if (handicap < 0 || handicap > MAX_HANDICAP) {
	       fprintf(stderr, "Illegal handicap: %d.\n",handicap);
	       fprintf(stderr, "Try `gnugo --help' for more information.\n");

	       exit(EXIT_FAILURE);
	     }
	   set_handicap(handicap);
	 }
	 break;

       case OPT_BOARDSIZE : 
         if(!set_boardsize(atoi(optarg)))
         {
	   fprintf(stderr,"Illegal board size: %d.\n", atoi(optarg));
	   fprintf(stderr,"Try `gnugo --help' for more information.\n");
	   exit(EXIT_FAILURE);
	 }
	 board_size=get_boardsize();
	 break;
	 
       case OPT_KOMI : 
	 if (sscanf(optarg,"%f",&komi) !=1 ) {
	   fprintf(stderr,"Invalid komi selection: %s\n",optarg);
	   fprintf(stderr, "Try `gnugo --help' for more information.\n");
	   
	   exit(EXIT_FAILURE);
	 }
	 set_komi(komi);
	 break;

       case OPT_MODE : 
	 if (strcmp(optarg,"ascii")==0) playmode = MODE_ASCII;
	 else if (strcmp(optarg,"gmp")==0) playmode = MODE_GMP;
	 else if (strcmp(optarg,"test")==0) playmode = MODE_TEST;
	 else {
	   fprintf(stderr,"Invalid mode selection: %s\n",optarg);
	   fprintf(stderr, "Try `gnugo --help' for more information.\n");

	   exit(EXIT_FAILURE);
	 }
	 break;

       case OPT_DECIDE_STRING :
	 strcpy(decidestring, optarg);
	 playmode=MODE_DECIDE_STRING;
	 break;

       case OPT_TESTMODE : 
	 playmode = MODE_TEST;
	 if (strcmp(optarg,"move")==0)
	   testmode=MOVE_ONLY;
	 else if (strcmp(optarg,"annotation")==0)
	   testmode=ANNOTATION_ONLY;
	 else if (strcmp(optarg,"both")==0)
	   testmode=BOTH;
	 else if (strcmp(optarg,"game")==0)
	   testmode=GAME;
	 else {
	   fprintf(stderr,"Invalid test mode selection: %s\n",optarg);
	   fprintf(stderr, "Try `gnugo --help' for more information.\n");

	   exit(EXIT_FAILURE);
	 }
	 break;

       case OPT_SCORE: until = optarg;
	 if (playmode == MODE_UNKNOWN)
	   playmode = MODE_LOAD_AND_SCORE;
	 break;

       case OPT_PRINTSGF:
	 playmode =  MODE_LOAD_AND_PRINT;
	 printsgffile = optarg;
	 break;

       case OPT_ANALYZER_FILE:
	 analyzerfile = optarg;
	 break;
       case OPT_ANALYZE:
	 parse_analyzer_options(optarg);
	 break;
       case OPT_HELPANALYZE:
	 sgfHelp();
	 return EXIT_SUCCESS;
	 break;
       case OPT_COLOR : 
	 if (strcmp(optarg,"white")==0) 
	   umove = WHITE;
	 else if (strcmp(optarg,"black")==0) 
	   umove = BLACK;
	 else {
	   fprintf(stderr,"Invalid color selection: %s\n",optarg);
	   fprintf(stderr, "Try `gnugo --help' for more information.\n");

	   exit(EXIT_FAILURE);
	 }
	 break;

       case OPT_SHOWCOPYRIGHT : 
	 show_copyright();
	 return EXIT_SUCCESS;
	 break;
       case 'v':
	 show_version();
	 return EXIT_SUCCESS;
	 break;
       case 'h' : 
         show_version();
         if(optarg)
         {           /*in the default behavior of getopt_long with optional args
                       you need to type "-hdebug"
                       I can't get long options "--helpdebug" to work at all
                         */
           if(!strncmp(optarg,"playstyle",7)) fprintf(stderr, USAGE_STYLE);
           if(!strncmp(optarg,"analyze",7)) sgfHelp();
           if(!strncmp(optarg,"debug",5)) fprintf(stderr,USAGE_DEBUG,DEPTH, BACKFILL_DEPTH);
         }else
           /*this is the trick to get "--help debug" and "-h debug" to work*/
           if (optind < argc)
           {
               if(!strncmp(argv[optind],"playstyle",7)) fprintf(stderr, USAGE_STYLE);
               if(!strncmp(argv[optind],"analyze",7)) sgfHelp();
               if(!strncmp(argv[optind],"debug",5)) fprintf(stderr,USAGE_DEBUG,DEPTH, BACKFILL_DEPTH);
           }else
	     show_help();
	 return EXIT_SUCCESS;
	 break;

	 /* NOTE : getopt returns '?' if an illegal option is supplied */
       case '?' :
       default :
	 fprintf(stderr, "Try `gnugo --help' for more information.\n");

	 exit(EXIT_FAILURE);
       }
     }

   /* Figure out a default mode if there was no explicit one. */

   if (playmode == MODE_UNKNOWN) {
     if (infile) {
       /* if -L was provided, mode would already be set... */
       if ( (testmode = guess_mode_from_sgf_comment(sgf_root)) != UNKNOWN_TESTMODE)
	 playmode = MODE_TEST;
       else
	 playmode = MODE_LOAD_AND_ANALYZE;
     }
     else
       playmode = (isatty(0)) ? MODE_ASCII : MODE_GMP;
   }

   /* start random number seed */
   if (!seed) seed=time(0);
   srand(seed);
   set_seed(seed);

#if 0
Done in init_board()
   /* initialize moyo structure */
   init_moyo();
#endif

#if HASHING
   /* Initialize hash table */
   movehash = hashtable_new(50000, 30000, 100000);
#endif

   /* clear some caches */
   clear_wind_cache();
   clear_safe_move_cache();
   
   if(!get_opt_quiet())
     {
       fprintf(stderr, "\n");
       show_version();
       show_copyright();
     }
   
   if (outfile)
   {

   if ((playmode != MODE_DECIDE_STRING) &&  (outfile))
     if (!sgf_open_file(outfile))
       {
	 fprintf(stderr, "Error: could not open '%s'\n", optarg);
	 exit(EXIT_FAILURE);
       }
   }

   switch(playmode) 
   {
     case MODE_GMP:     
       if(!sgf_root)
         sgfCreateHeaderNode(komi);
       play_gmp();
       break;
     case MODE_SOLO:
       if(!sgf_root)
         sgfCreateHeaderNode(komi);
       play_solo(benchmark);
       break;
     case MODE_TEST:    
       if (!sgf_root) {
	 fprintf(stderr, "You must use -l infile with test mode.\n");
	 exit(EXIT_FAILURE);
       }
       play_test(sgf_root, testmode);
       break;

     case MODE_LOAD_AND_ANALYZE:
       load_and_analyze_sgf_file(sgf_root, until, benchmark);
       break;

   case MODE_LOAD_AND_SCORE:
       if (!sgf_root) {
         fprintf(stderr, "gnugo: --score must be used with -l\n");
	 exit(EXIT_FAILURE);
       }
       load_and_score_sgf_file(sgf_root,until);
       break;
   case MODE_LOAD_AND_PRINT:
     if (!sgf_root) {
       fprintf(stderr, "gnugo: --printsgf must be used with -l\n");
       exit(EXIT_FAILURE);
     } else {
       int next;
       load_sgf_header(sgf_root);
       sgf_write_game_info(board_size, get_handicap(),get_komi(),get_seed(), "load and print"); 
       next = load_sgf_file(sgf_root, NULL);
       sgf_close_file();
       sgf_open_file(printsgffile);
       sgf_write_game_info(board_size, get_handicap(),get_komi(),get_seed(), "load and print"); 
       sgf_printboard(next);
     }
     break;
   case MODE_DECIDE_STRING:
     {
       int m, n;
       
       load_sgf_file(sgf_root, until);
       
       n = tolower(*decidestring) - 'a';
       if (tolower(*decidestring) >= 'i')
	 --n;
       m = get_boardsize() - atoi(decidestring+1);
       
       if (!infile) {
	 fprintf(stderr, "gnugo: --decidestring must be used with -l\n");
	 return (EXIT_FAILURE);
       }
       sgf_decidestring(m, n, outfile);
     }
     break;

   case MODE_ASCII :  
   default :     
     set_computer_player(OTHER_COLOR(umove));
     if(!sgf_root)
       sgfCreateHeaderNode(komi);
     else
       {
         load_sgf_header(sgf_root);
         set_computer_player(NONE);
       }
     play_ascii(infile);
     break;
   }
   
   sgf_close_file();
   
   if (analyzerfile&&sgf_root)
     writesgf(sgf_root,analyzerfile,seed);
   
   return 0;
}  /* end main */


/*
 * Local Variables:
 * tab-width: 8
 * c-basic-offset: 2
 * End:
 */
