// Copyright (c) 1997 by Jim Lynch.
// This software comes with NO WARRANTY WHATSOEVER.
//
// This program is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; version 2 dated June, 1991, or, at your
//    option, any LATER version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program;  if not, write to the Free Software
//    Foundation, Inc., 675 Mass Ave., Cambridge, MA 02139, USA.
//
// On Debian Linux systems, the complete text of the GNU General
// Public License can be found in `/usr/doc/copyright/GPL'.

#ifndef Question_h
#define Question_h

#include "Record.h"

#include <strstream.h>

class Question : public Record
{
private:
  string *question;
  string *yesKey;
  string *noKey;

  void Set
    (
      const char *aQuestion,
      const char *aYesKey,
      const char *aNoKey
    )
    {
      question = new string(aQuestion);
      yesKey = new string(aYesKey);
      noKey = new string(aNoKey);
    }

  void Copy(const Question &orig)
    {
      question = yesKey = noKey = 0;

      if(orig.question)
	question = new string(*orig.question);

      if(orig.yesKey)
	yesKey = new string(*orig.yesKey);

      if(orig.noKey)
	noKey = new string(*orig.noKey);
    }

  void Default(void)
    {
      question = new string;
      yesKey = new string;
      noKey = new string;
    }

protected:
  virtual void Destroy(void)
    {
      if(question)    delete question;
      if(yesKey)      delete yesKey;
      if(noKey)       delete noKey;

      question = yesKey = noKey = 0;
    }

  // Warning:  You must call Destroy() before calling 
  // any method starting with "Set" in this class,
  // -unless- you are initializing the object.
  virtual void SetNull(void)
    {
      question = new string;
      noKey = new string;
      yesKey = new string;
    }
  
  virtual void SetFromDataString(const char *theDataString)
    {
      const char *data = theDataString;

      if(data[0] == 'q')
      {
	istrstream it(theDataString + 2);
	ostrstream theQuestion;
	ostrstream theYesKey;
	ostrstream theNoKey;

	// grab the question text
	char nextCh = it.get();
	
	while(nextCh != '|')
	{
	  theQuestion << nextCh;

	  nextCh = it.get();
	}

	question = new string(MakeCStringCopy(theQuestion));

	// grab the yesKey text
	nextCh = it.get();
	
	while(nextCh != '|')
	{
	  theYesKey << nextCh;

	  nextCh = it.get();
	}

	yesKey = new string(MakeCStringCopy(theYesKey));
	
	// grab the noKey text
	nextCh = it.get();
	
	while(nextCh != '|')
	{
	  theNoKey << nextCh;

	  nextCh = it.get();
	}

	noKey = new string(MakeCStringCopy(theNoKey));
      }
      else
      {
	SetNull();
      }
    }
  
public:
  Question()
    {
      Default();
    }
  
  Question(const Question &orig)
    {
      Copy(orig);
    }

  Question
    (
      const char *aQuestion,
      const char *aYesKey,
      const char *aNoKey
    )
    {
      Set(aQuestion, aYesKey, aNoKey);
    }

  virtual ~Question() 
    {
      Destroy();
    }

  Question &operator=(const Question &orig)
    {
      Destroy();
      Copy(orig);
    }
  
  Question(Db &handle, const string &theKey)
    {
      Default();
      
      Read(handle, theKey);
    }

  string GetYesKey(void)
    {
      return *yesKey;
    }

  string GetNoKey(void)
    {
      return *noKey;
    }
  
  virtual const char *FormNewDataCString() const
    {
      string temp = "q|";

      if(question)
	temp += (*question);

      temp += "|";

      if(yesKey)
	temp += (*yesKey);

      temp += "|";

      if(noKey)
	temp += (*noKey);

      temp += "|";

      return MakeCStringCopy(temp);
    }

  virtual void Print(ostream &out) const
    {
      out << (question ? (*question) : string());
    }
};

#endif
