#include "StandardPackage.hh"

#include "IIR_EnumerationSubtypeDefinition.hh"
#include "IIR_IntegerSubtypeDefinition.hh"
#include "IIR_FloatingSubtypeDefinition.hh"
#include "IIR_PhysicalSubtypeDefinition.hh"
#include "IIR_ArrayTypeDefinition.hh"
#include "IIR_AccessTypeDefinition.hh"
#include "IIR_RangeTypeDefinition.hh"
#include "IIR_AttributeDeclaration.hh"
#include "IIR_TypeDeclaration.hh"
#include "IIR_LibraryDeclaration.hh"
#include "IIR_SubtypeDeclaration.hh"
#include "IIR_FunctionDeclaration.hh"
#include "IIR_Identifier.hh"
#include "IIR_CharacterLiteral.hh"
#include "IIR_EnumerationLiteral.hh"
#include "IIR_IntegerLiteral32.hh"
#include "IIR_FloatingPointLiteral64.hh"
#include "hash_table.hh"
#include "IIRScram_String.hh"
#include "IIR_PhysicalUnit.hh"
#include "IIR_PhysicalLiteral.hh"
#include "IIR_UnitList.hh"
#include "IIR_EnumerationLiteralList.hh"

// This has to be here to avoid a static initialization ordering problem
hash_table<IIRScram_String>
IIRScram_TextLiteral::string_table;


int 
StandardPackage::STD_INT_MAX = 2147483647;

int 
StandardPackage::STD_INT_MIN = -2147483647;

double 
StandardPackage::REAL_MAX = 0.179769313486231e+309;

double 
StandardPackage::REAL_MIN = -0.179769313486231e+309;

char *
StandardPackage::file_string = "scram built-in standard library";

IIR_Identifier *
StandardPackage::file_info = IIR_Identifier::get( file_string, strlen( file_string ) );


// Boolean has to be defined before anything else
IIR_EnumerationTypeDefinition *
StandardPackage::boolean_type = init_boolean_type();

IIR_TypeDeclaration *
StandardPackage::boolean_decl = init_type_decl( "boolean", boolean_type );

// Here are the builtin universal types.
IIR_IntegerSubtypeDefinition *
StandardPackage::savant_universal_integer = init_integer_type();

IIR_FloatingSubtypeDefinition *
StandardPackage::savant_universal_real = init_real_type();

// These types represent those declared in std.standard.
IIR_EnumerationTypeDefinition *
StandardPackage::bit_type = init_bit_type();

IIR_TypeDeclaration *
StandardPackage::bit_decl = init_type_decl( "bit", bit_type );

IIR_IntegerSubtypeDefinition *
StandardPackage::integer_type = init_integer_type();

IIR_TypeDeclaration *
StandardPackage::integer_decl = init_type_decl( "integer", integer_type );

IIR_EnumerationTypeDefinition *
StandardPackage::character_type = init_character_type();

IIR_TypeDeclaration *
StandardPackage::character_decl = init_type_decl( "character", character_type );

IIR_IntegerSubtypeDefinition *
StandardPackage::positive_type = init_positive_type();

IIR_SubtypeDeclaration *
StandardPackage::positive_decl = init_subtype_decl( "positive", positive_type );

IIR_ArrayTypeDefinition *
StandardPackage::string_type = init_string_type();

IIR_TypeDeclaration *
StandardPackage::string_decl = init_type_decl( "string", string_type );

IIR_EnumerationTypeDefinition *
StandardPackage::severity_level_type = init_severity_level_type();

IIR_TypeDeclaration *
StandardPackage::severity_level_decl = init_type_decl( "severity_level", severity_level_type );

IIR_FloatingSubtypeDefinition *
StandardPackage::real_type = init_real_type();

IIR_TypeDeclaration *
StandardPackage::real_decl = init_type_decl( "real", real_type );

IIR_PhysicalSubtypeDefinition *
StandardPackage::time_type = init_time_type();

IIR_TypeDeclaration *
StandardPackage::time_decl = init_type_decl( "time", time_type );

IIR_PhysicalSubtypeDefinition *
StandardPackage::delay_length_type = init_delay_length_type();

IIR_SubtypeDeclaration *
StandardPackage::delay_length_decl = init_subtype_decl( "delay_length", delay_length_type );

IIR_IntegerSubtypeDefinition *
StandardPackage::natural_type = init_natural_type();

IIR_SubtypeDeclaration *
StandardPackage::natural_decl = init_subtype_decl( "natural", natural_type );

IIR_ArrayTypeDefinition *
StandardPackage::bit_vector_type = init_bit_vector_type();

IIR_TypeDeclaration *
StandardPackage::bit_vector_decl = init_type_decl( "bit_vector", bit_vector_type );

IIR_EnumerationTypeDefinition *
StandardPackage::file_open_kind_type = init_file_open_kind_type();

IIR_TypeDeclaration *
StandardPackage::file_open_kind_decl = init_type_decl( "file_open_kind", file_open_kind_type );

IIR_EnumerationTypeDefinition *
StandardPackage::file_open_status_type = init_file_open_status_type();

IIR_TypeDeclaration *
StandardPackage::file_open_status_decl = init_type_decl( "file_open_status", file_open_status_type );

IIR_AttributeDeclaration *
StandardPackage::foreign_decl = init_foreign_decl();

IIR_FunctionDeclaration *
StandardPackage::now_decl = init_now_decl();

IIR_LibraryDeclaration *
StandardPackage::std_decl = init_std_decl();

IIR_AccessTypeDefinition *
StandardPackage::savant_null_type_definition = init_null_type();

void 
StandardPackage::fill_list( IIR_DeclarationList *list ){
  list->append( boolean_decl );
  list->_append( &boolean_type->enumeration_literals );
  list->append( bit_decl );
  list->_append( &bit_type->enumeration_literals );
  list->append( integer_decl );
  list->append( character_decl );
  list->_append( &character_type->enumeration_literals );
  list->append( positive_decl );
  list->append( string_decl );
  list->append( severity_level_decl );
  list->_append( &severity_level_type->enumeration_literals );
  list->append( real_decl );
  list->append( time_decl );
  list->append( time_type->get_primary_unit() );
  list->_append( &time_type->units );
  list->append( delay_length_decl );
  list->append( natural_decl );
  list->append( bit_vector_decl );
  list->append( file_open_kind_decl );
  list->_append( &file_open_kind_type->enumeration_literals );
  list->append( file_open_status_decl );
  list->_append( &file_open_status_type->enumeration_literals );
  list->append( foreign_decl );
  list->append( now_decl );
}

IIR_IntegerSubtypeDefinition *
StandardPackage::init_integer_type(){
  return init_integer_type( STD_INT_MIN, STD_INT_MAX );
}

IIR_IntegerSubtypeDefinition *
StandardPackage::init_positive_type(){
  IIR_IntegerSubtypeDefinition *retval = new IIR_IntegerSubtypeDefinition();
  retval->set_base_type( integer_type->get_base_type() );
  retval->_set_type_mark( integer_type );

  IIR_IntegerLiteral32 *left = IIR_IntegerLiteral32::get( 1 );
  IIR_IntegerLiteral32 *right = IIR_IntegerLiteral32::get( STD_INT_MAX );
  
  left->_set_subtype( retval );
  right->_set_subtype( retval );
  
  retval->set_left( left );
  retval->set_right( right );
  retval->set_direction( boolean_type->get_right() );

  set_locator_info( retval );

  return retval;
}

IIR_IntegerSubtypeDefinition *
StandardPackage::init_natural_type(){
  IIR_IntegerSubtypeDefinition *retval = new IIR_IntegerSubtypeDefinition();
  retval->set_base_type( integer_type->get_base_type() );
  retval->_set_type_mark( integer_type );

  IIR_IntegerLiteral32 *left = IIR_IntegerLiteral32::get( 0 );
  IIR_IntegerLiteral32 *right = IIR_IntegerLiteral32::get( STD_INT_MAX );
  
  left->_set_subtype( retval );
  right->_set_subtype( retval );
  
  retval->set_left( left );
  retval->set_right( right );

  retval->set_direction( boolean_type->get_right() );

  set_locator_info( retval );

  return retval;
}

IIR_FloatingSubtypeDefinition *
StandardPackage::init_real_type(){
  return init_real_type( REAL_MIN, REAL_MAX );
}

IIR_EnumerationTypeDefinition *
StandardPackage::init_boolean_type(){
  char *literals[] = {
    "false",
    "true",
    0
  };
  
  boolean_type = new IIR_EnumerationTypeDefinition();

  return init_enumeration_type( literals, boolean_type );
}

IIR_EnumerationTypeDefinition *
StandardPackage::init_bit_type(){
  char *literals[] = { "'0'", "'1'", 0 };
  return init_enumeration_type( literals );
}

IIR_EnumerationTypeDefinition *
StandardPackage::init_character_type(){
  char *literals[] = {
    "nul",  "soh",  "stx",  "etx",  "eot",  "enq",  "ack",  "bel",
    "bs",   "ht",   "lf",   "vt",   "ff",   "cr",   "so",   "si",
    "dle",  "dc1",  "dc2",  "dc3",  "dc4",  "nak",  "syn",  "etb",
    "can",  "em",   "sub",  "esc",  "fsp",  "gsp",  "rsp",  "usp", 
    "' '",  "'!'",  "'\"'", "'#'",  "'$'",  "'%'",  "'&'",  "'''",  
    "'('",  "')'",  "'*'",  "'+'",  "','",  "'-'",  "'.'",  "'/'",
    "'0'",  "'1'",  "'2'",  "'3'",  "'4'",  "'5'",  "'6'",  "'7'",
    "'8'",  "'9'",  "':'",  "';'",  "'<'",  "'='",  "'>'",  "'?'",
    "'@'",  "'A'",  "'B'",  "'C'",  "'D'",  "'E'",  "'F'",  "'G'",
    "'H'",  "'I'",  "'J'",  "'K'",  "'L'",  "'M'",  "'N'",  "'O'",
    "'P'",  "'Q'",  "'R'",  "'S'",  "'T'",  "'U'",  "'V'",  "'W'",
    "'X'",  "'Y'",  "'Z'",  "'['",  "'\\'",  "']'",  "'^'",  "'_'",
    "'`'",  "'a'",  "'b'",  "'c'",  "'d'",  "'e'",  "'f'",  "'g'",
    "'h'",  "'i'",  "'j'",  "'k'",  "'l'",  "'m'",  "'n'",  "'o'",
    "'p'",  "'q'",  "'r'",  "'s'",  "'t'",  "'u'",  "'v'",  "'w'",
    "'x'",  "'y'",  "'z'",  "'{'",  "'|'",  "'}'",  "'~'",  "del",
    "c128", "c129", "c130", "c131", "c132", "c133", "c134", "c135",
    "c136", "c137", "c138", "c139", "c140", "c141", "c142", "c143",
    "c144", "c145", "c146", "c147", "c148", "c149", "c150", "c151",
    "c152", "c153", "c154", "c155", "c156", "c157", "c158", "c159",
    "''",  "''",  "''",  "''",  "''",  "''",  "''",  "''", 
    "''",  "''",  "''",  "''",  "''",  "''",  "''",  "''", 
    "''",  "''",  "''",  "''",  "''",  "''",  "''",  "''", 
    "''",  "''",  "''",  "''",  "''",  "''",  "''",  "''", 
    "''",  "''",  "''",  "''",  "''",  "''",  "''",  "''", 
    "''",  "''",  "''",  "''",  "''",  "''",  "''",  "''", 
    "''",  "''",  "''",  "''",  "''",  "''",  "''",  "''", 
    "''",  "''",  "''",  "''",  "''",  "''",  "''",  "''",
    "''",  "''",  "''",  "''",  "''",  "''",  "''",  "''",
    "''",  "''",  "''",  "''",  "''",  "''",  "''",  "''", 
    "''",  "''",  "''",  "''",  "''",  "''",  "''",  "''",
    "''",  "''",  "''",  "''",  "''",  "''",  "''",  "''", 
    0 };

  return init_enumeration_type( literals );
}

IIR_EnumerationTypeDefinition *
StandardPackage::init_severity_level_type(){
  char *literals[] = {
    "note", 
    "warning", 
    "error", 
    "failure",
    0
  };

  return init_enumeration_type( literals );
}

IIR_EnumerationTypeDefinition *
StandardPackage::init_file_open_kind_type(){
  char *literals[] = {
    "open_kind",	
    "status_error",   
    "name_error",	
    "mode_error",
    0
  };
  return init_enumeration_type( literals );
}

IIR_EnumerationTypeDefinition *
StandardPackage::init_file_open_status_type(){
  char *literals[] = {
    "open_kind",	
    "status_error",   
    "name_error",	
    "mode_error",
    0
  };
  return init_enumeration_type( literals );
}

IIR_ArrayTypeDefinition *
StandardPackage::init_string_type(){
  return init_unconstrained_array_type( positive_type,  character_type );
}

IIR_ArrayTypeDefinition *
StandardPackage::init_bit_vector_type(){
  return init_unconstrained_array_type( natural_type, bit_type );
}

IIR_PhysicalSubtypeDefinition *
StandardPackage::init_time_type(){
  struct unit units[] = {
    { "ps",  1000, "fs" },
    { "ns",  1000, "ps" },
    { "us",  1000, "ns" },
    { "ms",  1000, "us" },
    { "sec", 1000, "ms" },
    { "min", 60,   "sec" },
    { "hr",  60,   "min" },
    { 0, 0, 0 }
  };

  return init_physical_type( (int)STD_INT_MIN, (int)STD_INT_MAX, (char *)"fs", units );
}

IIR_PhysicalSubtypeDefinition *
StandardPackage::init_delay_length_type(){
  IIR_PhysicalSubtypeDefinition *retval = new IIR_PhysicalSubtypeDefinition();
  retval->set_base_type( time_type->get_base_type() );

  retval->set_left( IIR_FloatingPointLiteral64::get_value( 0.0 ) );
  retval->set_right( time_type->get_right() );
  retval->set_direction( boolean_type->get_right() );
    
  set_locator_info( retval );

  return retval;
}

IIR_AttributeDeclaration *
StandardPackage::init_foreign_decl(){
  IIR_AttributeDeclaration *retval;

  retval = new IIR_AttributeDeclaration();
  retval->set_declarator( IIR_Identifier::get( "foreign", strlen("foreign") ) );
  retval->set_subtype( (IIR_TypeDefinition *)string_type );

  set_locator_info( retval );

  return retval;
}

IIR_FunctionDeclaration *
StandardPackage::init_now_decl(){
   IIR_FunctionDeclaration *retval;

   retval = new IIR_FunctionDeclaration();
   retval->set_declarator( IIR_Identifier::get( "now", strlen( "now" ) ) );
   retval->set_return_type( (IIR_TypeDefinition *)time_type );
   retval->set_pure( IIR_IMPURE_FUNCTION );

   set_locator_info( retval );

   return retval;
}

IIR_TypeDeclaration *
StandardPackage::init_type_decl( char *id, IIR_TypeDefinition *subtype ){
  ASSERT( id != NULL );
  ASSERT( subtype != NULL );

  IIR_TypeDeclaration *retval = new IIR_TypeDeclaration();
  retval->set_declarator( IIR_Identifier::get( (IIR_Char *)id, strlen( id ) ) );
  retval->set_type( (IIR_TypeDefinition *)subtype );
  subtype->_set_declaration( retval );

  set_locator_info( retval );

  return retval;
}

IIR_SubtypeDeclaration *
StandardPackage::init_subtype_decl( char *id, IIR_TypeDefinition *subtype ){
  ASSERT( id != NULL );
  ASSERT( subtype != NULL );

  IIR_SubtypeDeclaration *retval = new IIR_SubtypeDeclaration();
  retval->set_declarator( IIR_Identifier::get( id, strlen( id ) ) );
  retval->set_subtype( subtype );
  subtype->_set_declaration( retval );

  set_locator_info( retval );

  return retval;
}


IIR_EnumerationTypeDefinition *
StandardPackage::init_enumeration_type( char *literals[], 
					IIR_EnumerationTypeDefinition *supplied_type ){
  
  IIR_EnumerationTypeDefinition *retval;
  if( supplied_type == 0 ){
    retval = new IIR_EnumerationTypeDefinition();
  }
  else{
    retval = supplied_type;
  }

  int i = 0;
  while( literals[i] != 0 ){
    IIR_TextLiteral *literal_declarator;
    if( literals[i][0] == '\'' ){
      literal_declarator = IIR_CharacterLiteral::get( (char*)literals[i], strlen( literals[i] ) );
    }
    else{
      literal_declarator = IIR_Identifier::get( (char*)literals[i], strlen( literals[i] ) );
    }

    IIR_EnumerationLiteral *lit = new IIR_EnumerationLiteral();
    set_locator_info( lit );
    lit->set_declarator( literal_declarator );
    lit->set_subtype( retval );
    lit->set_position( IIR_IntegerLiteral32::get( i ) );
    retval->enumeration_literals.append( lit );
    
    i++;
  }
  
  retval->set_left( retval->enumeration_literals.first() );
  retval->set_right( retval->enumeration_literals.last() );
  retval->set_direction( boolean_type->get_right() );
  
  set_locator_info( retval );

  return retval;
}

IIR_IntegerSubtypeDefinition *
StandardPackage::init_integer_type( int left_bound, int right_bound ){
  IIR_IntegerTypeDefinition *base_type = new IIR_IntegerTypeDefinition();
  IIR_IntegerSubtypeDefinition *retval = new IIR_IntegerSubtypeDefinition();

  retval->set_base_type( base_type );

  IIR_IntegerLiteral32 *left = IIR_IntegerLiteral32::get( left_bound );
  IIR_IntegerLiteral32 *right = IIR_IntegerLiteral32::get( right_bound );
  
  left->_set_subtype( retval );
  right->_set_subtype( retval );

  base_type->set_left( left );
  base_type->set_right( left );
  base_type->set_direction( boolean_type->get_right() );

  retval->set_left( left );
  retval->set_right( right );
  retval->set_direction( boolean_type->get_right() );

  set_locator_info( retval );
  set_locator_info( base_type );

  return retval;
}

IIR_FloatingSubtypeDefinition *
StandardPackage::init_real_type( double left_bound, double right_bound ){
  IIR_FloatingTypeDefinition *base_type = new IIR_FloatingTypeDefinition();
  IIR_FloatingSubtypeDefinition *retval = new IIR_FloatingSubtypeDefinition();

  retval->set_base_type( base_type );
  IIR_FloatingPointLiteral64 *left = IIR_FloatingPointLiteral64::get_value( left_bound );
  IIR_FloatingPointLiteral64 *right = IIR_FloatingPointLiteral64::get_value( right_bound );
  
  left->_set_subtype( retval );
  right->_set_subtype( retval );

  base_type->set_left( left );
  base_type->set_right( left );
  base_type->set_direction( boolean_type->get_right() );


  retval->set_left( left );
  retval->set_right( right );
  retval->set_direction( boolean_type->get_right() );

  set_locator_info( retval );
  set_locator_info( base_type );

  return retval;
}

IIR_PhysicalSubtypeDefinition *
StandardPackage::init_physical_type( int left_bound, 
				     int right_bound, 
				     char *base_unit_string, 
				     struct unit units[] ){

  IIR_PhysicalTypeDefinition *base_type = new IIR_PhysicalTypeDefinition();
  IIR_PhysicalSubtypeDefinition *retval = new IIR_PhysicalSubtypeDefinition();

  IIR_IntegerLiteral32 *left = IIR_IntegerLiteral32::get( left_bound );
  IIR_IntegerLiteral32 *right = IIR_IntegerLiteral32::get( right_bound );
  left->_set_subtype( retval );
  right->_set_subtype( retval );

  retval->set_base_type( base_type );
  retval->set_left( left );
  retval->set_right( right );
  retval->set_direction( boolean_type->get_right() );

  base_type->set_left( left );
  base_type->set_right( left );
  base_type->set_direction( boolean_type->get_right() );

  set_locator_info( retval );
  set_locator_info( base_type );

  IIR_PhysicalUnit *base_unit = new IIR_PhysicalUnit();
  set_locator_info( base_unit );

  base_unit->set_declarator( IIR_Identifier::get( base_unit_string, strlen(base_unit_string) ));
  base_unit->_set_physical_type( retval );

  IIR_IntegerLiteral32 *unity = IIR_IntegerLiteral32::get( 1 );
  base_unit->set_multiplier( unity );
  unity->_set_subtype( retval );
  retval->set_primary_unit( base_unit );


  IIR_PhysicalUnit *current_unit;
  int i = 0;
  while( units[i].unit_name != 0 ){
    current_unit = new IIR_PhysicalUnit();
    set_locator_info( current_unit );
    current_unit->set_declarator( IIR_Identifier::get( units[i].unit_name, 
						       strlen( units[i].unit_name ) ) );

    IIR_PhysicalLiteral *multiplier = new IIR_PhysicalLiteral();
    set_locator_info( multiplier );
    multiplier->set_abstract_literal( IIR_IntegerLiteral32::get( units[i].multiplier ) );
    multiplier->set_unit_name( find_unit( units[i].base_unit, retval ) );
    multiplier->set_subtype( retval );
    current_unit->set_multiplier( multiplier );

    // Set subtypes of the current unit/multiplier.
    current_unit->_set_physical_type( retval );
    ASSERT( current_unit->get_multiplier() != NULL &&
	    current_unit->get_multiplier()->get_kind() == IIR_PHYSICAL_LITERAL );
    ((IIR_PhysicalLiteral *)current_unit->get_multiplier())->set_subtype( retval );

    retval->units.append( current_unit );
    i++;
  }

  return retval;
}

IIR_ArrayTypeDefinition *
StandardPackage::init_unconstrained_array_type( IIR_ScalarTypeDefinition *index_type,
						IIR_TypeDefinition *element_type ){
  IIR_ArrayTypeDefinition *retval;
  IIR_RangeTypeDefinition *empty_range = new IIR_RangeTypeDefinition();

  set_locator_info( empty_range );
  empty_range->set_base_type( index_type->get_base_type() );

  empty_range->_set_type_mark( index_type );

  IIR_ScalarTypeDefinition *unconstrained_index;
  unconstrained_index = 
    (IIR_ScalarTypeDefinition *)index_type->_construct_new_subtype( 0, empty_range );
  // So, we the subtype constructed we're done with this.
  delete empty_range;

  retval = (IIR_ArrayTypeDefinition*)
    IIR_ArrayTypeDefinition::_construct_unconstrained( unconstrained_index, element_type );
  set_locator_info( retval );

  return retval;
}

void
StandardPackage::set_locator_info( IIR *to_set_in ){
  to_set_in->set_file_name( file_info );
  to_set_in->set_line_number( 0 );
}

IIR_PhysicalUnit *
StandardPackage::find_unit( char *name, IIR_PhysicalSubtypeDefinition *physical_type ){
  IIR_PhysicalUnit *retval = 0;

  IIR_Identifier *temp = IIR_Identifier::get( name, strlen( name ) );
  IIR_PhysicalUnit *current = physical_type->units.first();
  while( current != NULL ){
    if( IIR_TextLiteral::_cmp( current->_get_declarator(), name ) == 0 ){
      retval = current;
      break;
    }
    current = physical_type->units.successor( current );
  }

  delete temp;

  return retval;
}

IIR_LibraryDeclaration *
StandardPackage::init_std_decl(){
  IIR_LibraryDeclaration *retval = new IIR_LibraryDeclaration();
  retval->set_declarator( IIR_Identifier::get( "std", strlen("std") ) );
  set_locator_info( retval );
  return retval;
}

StandardPackage::StandardPackage(){
  char *id = "standard";
  set_declarator( IIR_Identifier::get( id, strlen( id ) ) );
  fill_list( &package_declarative_part );
}

StandardPackage *
StandardPackage::init_std_standard(){
  StandardPackage *retval = new StandardPackage();
  set_locator_info( retval );
  return retval;
}

IIR_AccessTypeDefinition *
StandardPackage::init_null_type(){
  IIR_AccessTypeDefinition *retval = new IIR_AccessTypeDefinition();

  set_locator_info( retval );

  return retval;
}

StandardPackage *
StandardPackage::std_standard_decl = init_std_standard();
