// Copyright (c) 1996-1999 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.


// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	phil.wilsey@uc.edu
//          Dale E. Martin	dmartin@ece.uc.edu
//          Malolan Chetlur     mal@ece.uc.edu

//---------------------------------------------------------------------------
// 
// $Id: IIRScram_Operator.cc,v 1.4 1999/10/26 16:44:44 dmartin Exp $
// 
//---------------------------------------------------------------------------
#include "IIRScram_Operator.hh"
#include "IIR_AssociationElement.hh"
#include "IIR_FunctionCall.hh"
#include "IIR_FunctionDeclaration.hh"
#include "IIR_Identifier.hh"
#include "error_func.hh"
#include "resolution_func.hh"
#include "symbol_table.hh"
#include "StandardPackage.hh"
#include "IIR_EnumerationTypeDefinition.hh"

IIRScram_Operator::~IIRScram_Operator(){}


IIR_Boolean 
IIRScram_Operator::_type_check_user_declared( set<IIR_TypeDefinition> *context_set ){  
  IIR_Boolean retval = FALSE;

  set<IIR_Declaration> *function_declarations = _symbol_lookup();
  if( function_declarations == NULL ){
    return FALSE;
  }

  // Build an argument list for use with the function we have for
  // resolution.
  IIR_AssociationList *argument_list = _build_argument_list();
  
  IIR_Declaration *current_declaration = function_declarations->get_element();
  while( current_declaration != NULL ){

    if( current_declaration->_num_required_args() != argument_list->num_elements() ){
      function_declarations->remove( current_declaration );
    }

    current_declaration = function_declarations->get_next_element();
  }

  if( function_declarations != NULL ){
    resolve_subprogram_decls( function_declarations, argument_list,
			      context_set);
    
    // We only need to complain if it's ambiguous.
    switch( function_declarations->num_elements() ){
    case 0:{
      retval = FALSE;
      break;
    }
    case 1:{
      retval = TRUE;

      ASSERT( function_declarations->get_element() != NULL );
      ASSERT( function_declarations->get_element()->get_kind() == IIR_FUNCTION_DECLARATION );

      IIR_FunctionDeclaration *my_decl;
      my_decl = (IIR_FunctionDeclaration *)function_declarations->get_element();
      
      set_implementation( my_decl );
      _type_check_operands( );

      break;
    }
    default:{
      // This will guarantee that other type check methods aren't applied
      // to this node.
      retval = TRUE;
      report_ambiguous_error( this, function_declarations );
    }
    }
    
  }
  delete function_declarations;

  return retval;
}

void 
IIRScram_Operator::_type_check( set<IIR_TypeDefinition> * ){
  // At the moment, don't to anything at all.
}


IIR *
IIRScram_Operator::_rval_to_decl( IIR_TypeDefinition *my_rval ){
  
  set_subtype( my_rval );

  ASSERT( _is_resolved() == TRUE );

  return this;
}


set<IIR_Declaration> *
IIRScram_Operator::_symbol_lookup(){
  set<IIR_Declaration> *retval = NULL;
  
  char *name = _get_function_name();
  
  IIR_Identifier *to_lookup = IIR_Identifier::get( name, strlen(name) );
  retval = new set<IIR_Declaration>(*_get_symbol_table()->find_set( to_lookup ));

  to_lookup->release();

  return retval;
}

set<IIR_TypeDefinition> *
IIRScram_Operator::_get_rval_set(IIR_Boolean (IIR::*constraint_function)() ){
  set<IIR_TypeDefinition> *retval = NULL;

  // We're caching rval sets here, for performance.
  if( has_been_type_checked == FALSE ){
    // First look for the user overloaded operators.
    retval = _get_user_overloaded_rvals();
    
    if( retval != NULL ){
      my_rvals = new set<IIR_TypeDefinition>( *retval );
    }
    
    has_been_type_checked = TRUE;
  }
  else{
    retval = new set<IIR_TypeDefinition>( *my_rvals );
  }

  return retval;
}

set<IIR_TypeDefinition> *
IIRScram_Operator::_get_user_overloaded_rvals(){
  set<IIR_TypeDefinition> *retval = NULL;

  set<IIR_Declaration> *my_decls = _symbol_lookup();
  
  if( my_decls == NULL ){
    return NULL;
  }
  else{
    int num_arguments = _get_num_args();
    IIR_Declaration *current_decl = my_decls->get_element();
    while( current_decl != NULL ){
      if( current_decl->_num_required_args() != num_arguments ){
	my_decls->remove( current_decl );
      }
      else{
	ASSERT( current_decl->get_kind() == IIR_FUNCTION_DECLARATION );
	IIR_FunctionDeclaration *as_function = (IIR_FunctionDeclaration *)current_decl;
	IIR_AssociationList *arg_list = _build_argument_list();
	IIR_Boolean valid_call = 
	  arg_list->_check_valid_arguments( &as_function->interface_declarations, NULL );
	
	if( valid_call == FALSE ){
	  my_decls->remove( current_decl );
	}
	arg_list->_destroy_list();
	delete arg_list;
      }

      current_decl = my_decls->get_next_element();
    }    

    retval = decl_set_to_typedef_set( my_decls );
    delete my_decls;
  }

  return retval;
}

ostream & 
IIRScram_Operator::_print( ostream &os ){
  os << _get_function_name();

  return os;
}

IIR *
IIRScram_Operator::_semantic_transform( set<IIR_TypeDefinition> *context_set ){
  IIR *retval = this;

  if( _type_check_user_declared( context_set ) == TRUE ){
    ASSERT( get_implementation() != NULL );
    ASSERT( get_implementation()->_is_resolved() == TRUE );
    
    // Only transform this to a function call if the user _explicitly_
    // declared this operator.
    if( get_implementation()->_is_implicit_declaration() == FALSE ){
      IIR_FunctionCall *function_call = new IIR_FunctionCall();
      copy_location( this, function_call );
      function_call->set_implementation( get_implementation() );
      function_call->parameter_association_list = *_build_argument_list();
      function_call->parameter_association_list._resolve_and_order( &get_implementation()->interface_declarations, NULL );
      
#ifdef DEVELOPER_ASSERTIONS
      IIR_AssociationElement *current =  function_call->parameter_association_list.first();
      while( current != NULL ){
	ASSERT( current->_is_resolved() == TRUE );
	current =  function_call->parameter_association_list.successor( current );
      }
#endif

      retval = function_call;
    }
    // else retval = this already.
  }

  return retval;
}


