
// Copyright (c) 1996-1999 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.


// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	phil.wilsey@uc.edu
//          Dale E. Martin	dmartin@ece.uc.edu
//          Malolan Chetlur     mal@ece.uc.edu
//          Umesh Kumar V. Rajasekaran urajasek@ece.uc.edu
//          Krishnan Subramani  skrish@ece.uc.edu
//          Narayanan Thondugulam nthondug@ece.uc.edu
//          Swaminathan Subramanian ssubrama@ececs.uc.edu

//---------------------------------------------------------------------------
// 
// $Id: IIRScram_BlockStatement.cc,v 1.6 1999/10/10 19:09:02 dmartin Exp $
// 
//---------------------------------------------------------------------------

#include "IIRScram_BlockStatement.hh"
#include "IIRScram_ProcessStatement.hh"
#include "IIR_Identifier.hh"
#include "resolution_func.hh"
#include "IIR_Label.hh"
#include "IIR_TextLiteral.hh"
#include "IIR_List.hh"
#include "IIR_GenericList.hh"
#include "IIR_AssociationList.hh"
#include "IIR_PortList.hh"
#include "IIR_DeclarationList.hh"
#include "IIR_ConcurrentStatementList.hh"
#include "IIR_AssociationElement.hh"
#include "symbol_table.hh"
#include <strstream.h>
#include "IIR_TypeDefinition.hh"
#include "IIR_ObjectDeclaration.hh"
#include "IIR_ComponentDeclaration.hh"
#include "IIR_ConstantInterfaceDeclaration.hh"
#include "IIR_ComponentConfiguration.hh"
#include "IIR_ComponentInstantiationStatement.hh"
#include "IIR_SignalInterfaceDeclaration.hh"
#include "IIR_SignalDeclaration.hh"
#include "IIR_AliasDeclaration.hh"
#include "IIR_BlockStatement.hh"
#include "IIR_ConfigurationSpecification.hh"
#include "IIR_Designator.hh"
#include "IIR_LibraryUnit.hh"
#include "IIR_EntityDeclaration.hh"
#include "IIR_EnumerationLiteral.hh"
#include "IIR_IntegerLiteral.hh"
#include "IIR_ConcurrentGenerateForStatement.hh"
#include "IIR_ConcurrentGenerateIfStatement.hh"
#include "IIR_FunctionCall.hh"
#include "IIR_IndexedName.hh"
#include "IIR_SubprogramDeclaration.hh"
#include "set.hh"

IIR*
IIRScram_BlockStatement::guard_expr = NULL;

IIRScram_BlockStatement::IIRScram_BlockStatement() {
  my_implicit_guard_signal = NULL;
}

IIRScram_BlockStatement::~IIRScram_BlockStatement(){}

void 
IIRScram_BlockStatement::_set_implicit_guard_signal(IIR_SignalDeclaration *implicit_guard_signal ){
  my_implicit_guard_signal = implicit_guard_signal;
}

IIR_SignalDeclaration *
IIRScram_BlockStatement::_get_implicit_guard_signal(  ){
  return my_implicit_guard_signal;
}

IIR_GenericList *
IIRScram_BlockStatement::_get_generic_list(){
  return &generic_clause;
}

void 
IIRScram_BlockStatement::_publish_vhdl(ostream &_vhdl_out) {
  if (get_label() != NULL) {
    get_label()->_publish_vhdl(_vhdl_out);
    _vhdl_out << ":\n";
  }

  _vhdl_out << "  block ";
  if( get_guard_expression() != NULL) {
    _vhdl_out << "(";
    get_guard_expression()->_publish_vhdl(_vhdl_out);
    _vhdl_out << ")";
    _vhdl_out << " is\n";
  }

  if(generic_clause.num_elements() != 0) {
    _vhdl_out << "   generic ( ";
    generic_clause._publish_vhdl(_vhdl_out);
    _vhdl_out << " );\n";

    if(generic_map_aspect.num_elements() != 0) {
      _vhdl_out << "   generic map ( ";
      generic_map_aspect._publish_vhdl(_vhdl_out);
      _vhdl_out << " );\n";
    }
  }

  if(port_clause.num_elements() != 0) {
    _vhdl_out << "   port ( ";
    port_clause._publish_vhdl(_vhdl_out);
    _vhdl_out << " );\n";

    if(port_map_aspect.num_elements() != 0) {
      _vhdl_out << "   port map ( ";
      port_map_aspect._publish_vhdl(_vhdl_out);
      _vhdl_out << " );\n";
    }
  }

  block_declarative_part._publish_vhdl_decl(_vhdl_out);

  _vhdl_out << " begin\n";
  block_statement_part._publish_vhdl(_vhdl_out);
  _vhdl_out << " end block ";

  if (get_label() != NULL) {
    get_label()->_publish_vhdl(_vhdl_out);
  }  
  _vhdl_out << ";\n";
}

void 
IIRScram_BlockStatement::_type_check(){
  // Things we need to check...
  // if we have a guard expression, it must be boolean valued.
  if( get_guard_expression() != NULL ){
    set_guard_expression( _type_check_and_resolve_boolean_condition( get_guard_expression() ) );

    // Now that we've resolved this, we'll set the default value of the implicit signal to
    // the resolved expression.
    _get_implicit_guard_signal()->set_value( get_guard_expression() );
  }
  
#ifdef DEVELOPER_ASSERTIONS
  if( get_guard_expression() == NULL ){
    ASSERT( _get_implicit_guard_signal() == NULL );
  }
  else{
    ASSERT( _get_implicit_guard_signal() != NULL );
  }
#endif

  // We need to type check the port map and generic map if they have stuff in them.

  generic_map_aspect._resolve_and_order( &generic_clause, NULL );
  port_map_aspect._resolve_and_order( &port_clause, NULL );  

  _get_symbol_table()->open_scope( this );
  _make_interface_visible( _get_symbol_table() );
  block_declarative_part._type_check_disconnection_specifications();
  block_declarative_part._type_check_configuration_specifications( block_statement_part );
  block_declarative_part._type_check_attribute_specifications( block_statement_part );
  _get_symbol_table()->close_scope( this );


}

void 
IIRScram_BlockStatement::_type_check_instantiate_statements(){ 
  _get_symbol_table()->open_scope( this );
  _make_interface_visible( _get_symbol_table() );
  block_statement_part._type_check_instantiate_statements();
  _get_symbol_table()->close_scope( this );
}


set<IIR_Declaration> *
IIRScram_BlockStatement::_find_declarations( IIR_Name *to_find ){
  set<IIR_Declaration> *retval = new set<IIR_Declaration>;
  
  set<IIR_Declaration> *current_set = port_clause._find_declarations(to_find );
  
  if( current_set != NULL ){
    retval->add( current_set );
    delete current_set;
  }

  current_set = generic_clause._find_declarations(to_find );
  if( current_set != NULL ){
    retval->add( current_set );
    delete current_set;
  }

  current_set = block_declarative_part._find_declarations(to_find );
  if( current_set != NULL ){
    retval->add( current_set );
    delete current_set;
  }

  if( retval->num_elements() == 0 ){
    delete retval;
    retval = NULL;
  }

  return retval;
}

void 
IIRScram_BlockStatement::_make_interface_visible( symbol_table *sym_tab ){
  sym_tab->make_visible( &generic_clause );
  sym_tab->make_visible( &port_clause );
  sym_tab->make_visible( &block_declarative_part );
  block_statement_part._make_interface_visible( sym_tab );
}

IIR_List *
IIRScram_BlockStatement::_get_statement_list(){
  return &block_statement_part;
}

void
IIRScram_BlockStatement::_publish_cc_blocks_elaborate()
{
  IIR_ConcurrentStatement *conc_stmt;
  char *temp = _current_architecture_name;
  char *old_current_name = _current_publish_name;
  ostrstream newName;
  
  _current_publish_name      = "SB";
  
  conc_stmt = block_statement_part.first();
  
  while (conc_stmt != NULL) {
    if (conc_stmt->get_kind() != IIR_COMPONENT_INSTANTIATION_STATEMENT &&
	conc_stmt->get_kind() != IIR_PROCESS_STATEMENT) {
      conc_stmt->_publish_cc_elaborate();
    }
    conc_stmt = block_statement_part.successor(conc_stmt);
  }
  _current_architecture_name = temp;
  _current_publish_name      = old_current_name;
}

void
IIRScram_BlockStatement::_publish_cc() {
  IIR *temp = _current_publish_node;
  ostrstream ent_arch_block;
  char *old_architecture_name = _current_architecture_name;
  char *old_current_name      = _current_publish_name;
  ostrstream arch_name;
  IIR *old_guard_expr;

  PublishedUnit _saved_publishing_unit = _get_currently_publishing_unit();
  _set_currently_publishing_unit(BLOCK);
  _current_publish_name = "SB";
  _current_publish_node = this;

  _publish_cc_declarations();
  old_guard_expr = guard_expr;
  // Set the static variable so that it can be used in the process using guards
  guard_expr = get_guard_expression();
  block_statement_part._publish_cc();
  guard_expr = old_guard_expr;

  _current_publish_name = old_current_name;
  _current_architecture_name = old_architecture_name;
  

  _current_publish_node = temp;

  _set_currently_publishing_unit(_saved_publishing_unit);

}

void
IIRScram_BlockStatement::_publish_cc_declarations() {
  IIR_Char *block_unit_name = NULL;
  ostrstream ent_arch_block;
  ent_arch_block << _current_entity_name << "_"
		 << _current_architecture_name;
  _publish_cc_enclosing_stmt_to_architecture_path(ent_arch_block);
  ent_arch_block << "_" << *_get_label()
		 << ends;
  block_unit_name = ent_arch_block.str();

  _cc_out.set_file(TRUE, block_unit_name, "_decls.hh");
  _cc_out << "#ifndef " << block_unit_name << "_DECLS_HH" << endl;
  _cc_out << "#define " << block_unit_name << "_DECLS_HH" << endl << endl;
  _cc_out << "#include \"";
  _cc_out << _current_entity_name << "_" << _current_architecture_name;
  _publish_cc_enclosing_stmt_to_architecture_path();
  _cc_out << "_decls.hh\"\n\n";
  // If this block has an implicit guard signal then add it to the list
  if (_get_implicit_guard_signal() != NULL) {
    block_declarative_part.append(_get_implicit_guard_signal());
  }

  generic_clause._publish_cc_elaborate_as_pointers(FALSE);
  block_declarative_part._publish_cc();
  
  _cc_out.set_file(TRUE, block_unit_name, "_decls.hh");
  _cc_out << "#endif\n";

  _cc_out.set_file(TRUE, block_unit_name, "_decls.cc");
  _cc_out << "#include \"" << block_unit_name << "_decls.hh\""
	  << endl << endl;

  generic_clause._publish_cc_elaborate_as_pointers(TRUE);
  block_declarative_part._publish_cc_decl();

  delete [] block_unit_name;
}

void
IIRScram_BlockStatement::_publish_cc_elaborate() {
  IIR* old_publish_node = _current_publish_node;

  _current_publish_node = this;
  
  _publish_cc_headerfile();
  _publish_cc_ccfile();

  _publish_cc_blocks_elaborate();

  _current_publish_node      = old_publish_node;
}

void
IIRScram_BlockStatement::_publish_cc_headerfile() {
  char *buf = NULL;
  ostrstream filename;

  _publish_cc_binding_name(filename);
  filename << ends;
  buf = filename.str();
  _cc_out.set_file(TRUE, buf, "_elab.hh");
  _cc_out << "#ifndef " << buf << "_HH\n";
  _cc_out << "#define " << buf << "_HH\n\n";
  
  _cc_out << "#include \"_savant_entity_elab.hh\"\n";

  _cc_out << "#include \"";
  _cc_out << _current_publish_name << _current_entity_name << "_"
	  << _current_architecture_name ;
  _publish_cc_enclosing_stmt_to_architecture_path();
  _cc_out << "_elab.hh\"\n";
  // Include the declarations file 
  _cc_out << "#include \"";
  _cc_out << _current_entity_name << "_" << _current_architecture_name;
  _publish_cc_enclosing_stmt_to_architecture_path();
  _cc_out << "_" << *_get_label();
  _cc_out << "_decls.hh\"\n";

  _cc_out << "\n";
  _publish_cc_class_includes();
  _cc_out << "\n";
  _publish_cc_class();

  _cc_out << "\n#endif" << endl;
  delete [] buf;
}

void
IIRScram_BlockStatement::_publish_cc_class_includes() {
  char *temp = _current_architecture_name;
  char *old_current_name = _current_publish_name;
  ostrstream newName;
  
  _current_publish_name      = "SB";
  _cc_out << "class ";
  _publish_cc_binding_name();
  _cc_out << "_elab;\n";

  IIRScram::_publish_cc_class_includes(&block_statement_part);

  _current_architecture_name = temp;
  _current_publish_name      = old_current_name;
}

void
IIRScram_BlockStatement::_publish_cc_class() {
  _cc_out << "class ";
  this->_publish_cc_binding_name();
  _cc_out << "_elab ";
  
  _cc_out << " : public _savant_entity_elab {\n";
  _cc_out << "public:\n\n";

  this->_publish_cc_binding_name();
  _cc_out << "_elab(" << _current_publish_name << _current_entity_name << "_"
	  << _current_architecture_name;
  _publish_cc_enclosing_stmt_to_architecture_path();
  _cc_out << "_elab *);\n";

  _cc_out << _current_publish_name << _current_entity_name << "_"
	  << _current_architecture_name;
  _publish_cc_enclosing_stmt_to_architecture_path();
  _cc_out  << "_elab *enclosingScope;\n\n";
  
  if (generic_clause.num_elements() > 0) {
    this->_publish_cc_binding_name();
    _cc_out << "_elab(";
    _cc_out << _current_publish_name << _current_entity_name << "_"
	    << _current_architecture_name;
    _publish_cc_enclosing_stmt_to_architecture_path();
    _cc_out  << "_elab *enclosingScope, \n";
    generic_clause._publish_generic_parameter_list();
    _cc_out << ");\n";
  }

  _cc_out << "~";
  this->_publish_cc_binding_name();
  _cc_out << "_elab();\n";

  _cc_out << "void instantiate();\n";
  _cc_out << "void createNetInfo();\n";
  _cc_out << "void connect(int, int, ...);\n\n";

  _publish_cc_signals(&block_declarative_part);
  generic_clause._publish_cc_elaborate();
  port_clause._publish_cc_elaborate();
  
  _publish_cc_object_pointers(&block_statement_part);

  if (generic_clause.first() != NULL) {
    _cc_out << "  void copyGenericsToGlobals();\n";
  }
  
  _cc_out << "};\n";
}

void
IIRScram_BlockStatement::_publish_cc_ccfile()
{
  char *buf = NULL;
  ostrstream filename;

  _publish_cc_binding_name(filename);
  filename << ends;
  buf = filename.str();
  _cc_out.set_file(TRUE, buf, "_elab.cc");
  _cc_out << "#include \"" << buf << "_elab.hh\"\n\n";
  _cc_out << "#ifdef SEQUENTIAL" << endl;
  _cc_out << "#include \"BaseSequential.hh\" " << endl;
  _cc_out << "#define OBJTYPE BaseSequential" << endl;
  _cc_out << "#else" << endl;
  _cc_out << "#include \"BasicTimeWarp.hh\" " << endl;
  _cc_out << "#define OBJTYPE BasicTimeWarp" << endl;
  _cc_out << "#endif\n" << endl;

  _publish_cc_headerfiles_for_cc();
  _cc_out << "extern OBJTYPE *proc_array[];\n";

  _publish_cc_constructor();
  _publish_cc_destructor(&block_statement_part);
  if (generic_clause.first() != NULL) {
    _publish_cc_copy_generics_to_globals();
  }
  _publish_cc_instantiate();
  _publish_cc_createNetInfo();
  _publish_cc_connect();
  //_publish_cc_partition();
  delete [] buf;
}

void
IIRScram_BlockStatement::_publish_cc_headerfiles_for_cc(){
  char *temp = _current_architecture_name;
  char *old_current_name = _current_publish_name;

  _current_publish_name = "SB";

  _publish_cc_headerfiles_for_cc_default();

  IIR_Declaration* decl = block_declarative_part.first();
  while(decl != NULL) {
    if(decl->get_kind() == IIR_CONFIGURATION_SPECIFICATION) {
      if (decl->_get_entity_aspect() != NULL) {
	decl->_get_entity_aspect()->_publish_cc_headerfiles_for_cc_default();
      }
    }
    // Nothing for other nodes.
    decl =  block_declarative_part.successor(decl);
  }

  IIR_ConcurrentStatement *conc_stmt = block_statement_part.first();
  while (conc_stmt != NULL) {
    switch( conc_stmt->get_kind() ){
    case IIR_PROCESS_STATEMENT:
    case IIR_BLOCK_STATEMENT:
    case IIR_CONCURRENT_GENERATE_FOR_STATEMENT:
    case IIR_CONCURRENT_GENERATE_IF_STATEMENT:
      conc_stmt->_publish_cc_headerfiles_for_cc_default( );
      break;

    case IIR_COMPONENT_INSTANTIATION_STATEMENT:
      conc_stmt->_publish_cc_headerfiles_for_cc_default();
      if (((IIR_ComponentInstantiationStatement *) conc_stmt)->_get_configuration() != NULL) {
	((IIR_ComponentInstantiationStatement *) conc_stmt)->_get_configuration()->_publish_cc_headers();
      }
      break;
      
    default:
      // do nothing.
      ;
    }

    conc_stmt = block_statement_part.successor( conc_stmt );
  }
  
  _current_architecture_name = temp;
  _current_publish_name      = old_current_name;
}

void
IIRScram_BlockStatement::_publish_cc_constructor() {
  _publish_cc_constructor_with_no_arguments();
  if (generic_clause.num_elements() > 0) {
    _publish_cc_constructor_with_arguments();
  }
}

void
IIRScram_BlockStatement::_publish_cc_constructor_with_no_arguments()
{
  int numGenericClause = generic_clause.num_elements();
  int numPortClause = port_clause.num_elements();
  this->_publish_cc_binding_name();
  _cc_out << "_elab::";
  
  this->_publish_cc_binding_name();
  _cc_out << "_elab("
	  << _current_publish_name << _current_entity_name << "_"
	  << _current_architecture_name;
  _publish_cc_enclosing_stmt_to_architecture_path();
  _cc_out << "_elab *outerScope)";
    
  if(numGenericClause > 0) {
    _cc_out << ":\n";
    generic_clause._publish_generic_init();
    if(numPortClause > 0) {
      _cc_out << ",\n";
      port_clause._publish_cc_port_init();
    }
  }
  else {
    if(numPortClause > 0) {
      _cc_out << ":\n";
      port_clause._publish_cc_port_init();
    }
  }
  
  _publish_cc_signal_objects_init(1);
  _cc_out << " {\n";
  _cc_out << "  enclosingScope = outerScope;\n";
  // before newing the elab guys down the heirarchy copy generics of this block 
  // to the generic pointers in the block_decls file
  if (numGenericClause > 0) {
    _cc_out << "  copyGenericsToGlobals();\n";
  }
  _publish_cc_object_pointers_init(&block_statement_part, &block_declarative_part);
  _cc_out << "}\n";
}

void
IIRScram_BlockStatement::_publish_cc_constructor_with_arguments()
{
  int numGenericClause = generic_clause.num_elements();
  int numPortClause = port_clause.num_elements();

  this->_publish_cc_binding_name();
  _cc_out << "_elab::";
  this->_publish_cc_binding_name();
  _cc_out << "_elab(\n";
  
  _cc_out << _current_publish_name << _current_entity_name << "_"
	  << _current_architecture_name;
  _publish_cc_enclosing_stmt_to_architecture_path();
  _cc_out << "_elab *outerScope, ";
    
  generic_clause._publish_generic_parameter_list();
  _cc_out << " )";
  
  if(numGenericClause > 0) {
    _cc_out << ":\n";
    generic_clause._publish_generic_init_by_arguments();
    if(numPortClause > 0) {
      _cc_out << ",\n";
      port_clause._publish_cc_port_init();
    }
  }
  else {
    if(numPortClause > 0) {
      _cc_out << ":\n";
      port_clause._publish_cc_port_init();
    }
  }
  
  _publish_cc_signal_objects_init(1);
  _cc_out << " {\n";
  _cc_out << "  enclosingScope = outerScope;\n";
  _publish_cc_object_pointers_init(&block_statement_part, &block_declarative_part);
  _cc_out << "}\n";
}

void
IIRScram_BlockStatement::_publish_cc_signal_objects_init(int flag)
{
  IIR_Declaration* decl;
  int first = 0;
  decl = block_declarative_part.first();
  int numGenericClause = generic_clause.num_elements();
  int numPortClause    = port_clause.num_elements();
  
  while (decl != NULL) {
    if(decl->get_kind() == IIR_SIGNAL_DECLARATION) {
      if((first == 0) && (flag == 1) && (numGenericClause == 0) && (numPortClause == 0)) {
	_cc_out << ":\n";
	first = 1;
      }
      else {
	_cc_out << ",\n";
      }
      decl->_publish_cc_elaborate();
      _cc_out << "(ObjectBase::SIGNAL_NETINFO";
      if(((IIR_SignalDeclaration*)decl)->get_subtype()->_is_array_type()) {
	_cc_out << ", ";
	((IIR_SignalDeclaration*)decl)->get_subtype()->_publish_cc_range();
      }      
      _cc_out << ")";
      //If the Signals has implicit signals
      //They are also needed for elaboration info
      if( decl->_get_implicit_declarations() != NULL && 
	  decl->_get_implicit_declarations()->num_elements() != 0) {
	IIR_Declaration* imp_decl = decl->_get_implicit_declarations()->get_element();
	while(imp_decl != NULL) {
	  if(imp_decl->get_kind() == IIR_SIGNAL_DECLARATION) {
	    _cc_out << ",\n";
	    imp_decl->_publish_cc_elaborate();
	    _cc_out << "(ObjectBase::SIGNAL_NETINFO";
	    if(((IIR_SignalDeclaration*)imp_decl)->get_subtype()->_is_array_type()) {
	      _cc_out << ", ";
	      ((IIR_SignalDeclaration*)imp_decl)->get_subtype()->_publish_cc_range();
	    }
	    _cc_out << ")";
	  }
	  imp_decl = decl->_get_implicit_declarations()->get_next_element();
	}
      }
    }
    else if(decl->get_kind() == IIR_ALIAS_DECLARATION) {
      if(((IIR_AliasDeclaration*)decl)->get_name()->_is_signal()) {
	if((first == 0) && (flag == 1) && (numGenericClause == 0) && (numPortClause == 0)) {
	  _cc_out << ":\n";
	  first = 1;
	}
	else {
	  _cc_out << ",\n";
	}
	((IIR_AliasDeclaration*)decl)->_publish_cc_elaborate_alias_init();
      }
    }
    decl = block_declarative_part.successor(decl);    
  }
}

void
IIRScram_BlockStatement::_publish_cc_instantiate()
{
  IIR_ConcurrentStatement *conc_stmt;

  _cc_out << "void\n";
  this->_publish_cc_binding_name();
  _cc_out << "_elab::instantiate()  {\n";

  conc_stmt = block_statement_part.first();
  while (conc_stmt != NULL) {
    conc_stmt->_publish_cc_instantiate_call();
    conc_stmt = block_statement_part.successor(conc_stmt);
  }
  _cc_out << "createNetInfo();\n";
  _cc_out << "}\n";
}

void
IIRScram_BlockStatement::_publish_cc_createNetInfo()
{
  IIR_ConcurrentStatement *conc_stmt;
  IIR_Boolean found = false;
  int wanted_instantiation = 1;
  IIR_Char* tmp  = _current_elab_name;
  IIR_Char* tmp2 = _current_publish_name;
  IIR* tmpNode   = _current_publish_node;
  PublishedUnit oldunit = _get_currently_publishing_unit();
  
  _cc_out << "void\n";
  this->_publish_cc_binding_name();
  _cc_out << "_elab::createNetInfo() {\n";

  _set_currently_publishing_unit(BLOCK);
  
  conc_stmt = block_statement_part.first();
  while (conc_stmt!=NULL) {
    _current_publish_node = this;
    if (conc_stmt->get_kind() == IIR_PROCESS_STATEMENT) {
      ostrstream objectname;
      if(conc_stmt->_get_label() != NULL) {
	objectname << *(conc_stmt->_get_label()->_get_declarator()) << ends;
      }
      else {
	objectname << "ANON_PROCESS" << conc_stmt << ends;
      }
      objectname << "_elab_obj" << ends;
      _current_elab_name = objectname.str();
      conc_stmt->_publish_createNetInfo();
      delete [] _current_elab_name;
      _current_elab_name = NULL;
    }
    else if ((conc_stmt->_is_block_statement() == TRUE) ||
	     (conc_stmt->_is_concurrent_generate_statement() == TRUE )){
    }
    else if ( conc_stmt->get_kind() == IIR_COMPONENT_INSTANTIATION_STATEMENT ){
      conc_stmt->_publish_createNetInfo();
    }
    else {
      cerr << "ERROR! IIRScram_BlockStatement::"
	   << "_publish_cc_createNetInfo(): unknown conc_statement "
	   << "type |" << conc_stmt->get_kind_text() << "| in arch" << endl;
    }
    found = FALSE;
    wanted_instantiation = 1;
    conc_stmt = block_statement_part.successor(conc_stmt);
  }
  _cc_out << "}\n\n";

  _set_currently_publishing_unit(oldunit);
  
  _current_elab_name    = tmp;
  _current_publish_name = tmp2;
  _current_publish_node = tmpNode;
}

void
IIRScram_BlockStatement::_publish_cc_connect()
{
  IIR_AssociationElement* actual_clause;
  _cc_out << "void\n";
  this->_publish_cc_binding_name();
  _cc_out << "_elab::connect(int inputsignals, int outputsignals, ...) {\n";
  _cc_out << "int NoofSignals = inputsignals + outputsignals;";
  _cc_out << "va_list ap;\n";
  _cc_out << "opFanoutinfo = (VHDLType**)new char[NoofSignals * sizeof(VHDLType*)];\n";
  _cc_out << "va_start(ap, outputsignals);\n";
  _cc_out << "for(int i=0; i <NoofSignals; i++) {\n";
  _cc_out << "  opFanoutinfo[i] = va_arg(ap, VHDLType*);\n";
  _cc_out << "}\n";
  _cc_out << "va_end(ap);\n";
  
  //Pass on the output connection  inforamtion to its output signals
  IIR_SignalInterfaceDeclaration* portelement;
  IIR_Mode mode;
  int index = 0;

  _cc_out << "if(inputsignals > 0) {\n";

  portelement = port_clause.first();
  actual_clause = port_map_aspect.first();
  for(; portelement != NULL; ) {
    mode = portelement->get_mode();
    if(mode == IIR_IN_MODE ) {
      if (!(actual_clause->_get_actual()->_is_static_expression())){
      _cc_out << "setSourceInfo(";
      portelement->_publish_cc_elaborate(); 
      _cc_out << ",*(opFanoutinfo[";
      _cc_out << index << "]));\n";
      index++;
      }
    }
    portelement = port_clause.successor(portelement);
    if (port_map_aspect.successor(actual_clause)!=NULL)
	actual_clause = port_map_aspect.successor(actual_clause);
  }
  
  _cc_out << "}\n";

  _cc_out << "if(outputsignals > 0) {\n";

  portelement = port_clause.first();
  for(; portelement != NULL; ) {
    mode = portelement->get_mode();
    if(mode == IIR_INOUT_MODE || mode == IIR_OUT_MODE) {
      if (!(actual_clause->_get_actual()->_is_static_expression())){
      _cc_out << "Add(";
      portelement->_publish_cc_elaborate(); 
      _cc_out << ",*(opFanoutinfo[";
      _cc_out << index << "]));\n";
      if (mode != IIR_OUT_MODE){
      _cc_out << "setSourceInfo(";
      portelement->_publish_cc_elaborate(); 
      _cc_out << ",*(opFanoutinfo[";
      _cc_out << index << "]));\n";
      }
      index++;
      }
    }
    portelement = port_clause.successor(portelement);
  }

  _cc_out << "}\n";

  IIR_ConcurrentStatement *conc_stmt;
  IIR_Char* objectname_ptr = NULL;
  conc_stmt = block_statement_part.first();
  while (conc_stmt != NULL) {
    if (conc_stmt->get_kind() == IIR_COMPONENT_INSTANTIATION_STATEMENT) {
      ostrstream objectname;
      //###yet to check for label
      objectname << *(conc_stmt->_get_label()) << "_elab_obj" << ends;
      _current_elab_name = objectname.str();
      objectname_ptr = _current_elab_name;
      ((IIR_ComponentInstantiationStatement*)conc_stmt)->_publish_connect();
      delete [] objectname_ptr;
      _current_elab_name = NULL;
      objectname_ptr = NULL;
    }
    else if (conc_stmt->_is_concurrent_generate_statement() == TRUE) {
      conc_stmt->_get_label()->_publish_cc_elaborate();
      _cc_out << "_elab_obj->connect(0, 0);\n";
    }
    else if (conc_stmt->_is_block_statement() == TRUE) {
      ((IIR_BlockStatement *) conc_stmt)->_publish_connect();
    }

    conc_stmt = block_statement_part.successor(conc_stmt);
  }
  _cc_out << "}\n";
}

void
IIRScram_BlockStatement::_publish_connect()
{
  IIR_AssociationElement* actual_clause;
  int                     noofinputsignals      = 0;
  int                     noofoutputsignals     = 0;
  IIR_Boolean             first                 = false;
  IIR_FunctionCall        *downTypeConversionFn = NULL;
  IIR_Declaration         *tempDeclaration      = NULL;
  IIR_Label               *label                = _get_label();
  int                     literal_count         = 0;

  _get_label()->_publish_cc_elaborate();

  if (_get_currently_publishing_unit() == GENERATE_FOR) {
    _cc_out << "_elab_obj[i - generateLeft].connect(";
  }
  else {
    _cc_out << "_elab_obj->connect(";
  }

  actual_clause = port_map_aspect.first();
  
  if (actual_clause == NULL) {
    _cc_out << "0, 0);\n";
    return;
  }

  while(actual_clause != NULL) {
    ASSERT(actual_clause->get_formal() != NULL);
    switch(actual_clause->get_formal()->_get_mode()) {
    case IIR_IN_MODE:
      noofinputsignals++;
      first = true;
      break;
    case IIR_OUT_MODE:
    case IIR_INOUT_MODE:
      noofoutputsignals++;
      first = true;
      break;
    default:
      // do nothing
      break;
    }
    if (actual_clause->_get_actual()->_is_static_expression() == TRUE){ 
        literal_count++;
    }
    actual_clause = port_map_aspect.successor(actual_clause);
  }
  
   if (literal_count == noofinputsignals + noofoutputsignals) {
    _cc_out << "0, 0);\n";
    return;
  }

  actual_clause = port_map_aspect.first();

  while(actual_clause != NULL) {
    ASSERT(actual_clause->get_formal() != NULL);
    switch(actual_clause->get_formal()->_get_mode()) {
    case IIR_IN_MODE:
      if (actual_clause->_get_actual()->_is_static_expression()){
	  noofinputsignals--;
      }
      first = true;
      break;
    case IIR_OUT_MODE:
    case IIR_INOUT_MODE:
      if (actual_clause->_get_actual()->_is_static_expression()){
	  noofoutputsignals--;
      }
      first = true;
      break;
    default:
      // do nothing
      break;
    }
    actual_clause = port_map_aspect.successor(actual_clause);
  }
  
  _cc_out << noofinputsignals << ", " << noofoutputsignals;

  if(first == true) {
    actual_clause = port_map_aspect.first();  
    while(actual_clause != NULL) {
      if (actual_clause->_get_actual()->_is_static_expression() == FALSE){
	if( actual_clause->get_formal()->_get_mode() == IIR_IN_MODE ){
	  _cc_out << ", &";
	  if(actual_clause == NULL) {
	    ASSERT(actual_clause != NULL);
	  }
	  ASSERT(actual_clause->_get_actual() != NULL);
	  if (actual_clause->_get_actual()->get_kind() ==IIR_FUNCTION_CALL) {
	    ((IIR_FunctionCall *) actual_clause->_get_actual())->_publish_cc_elaborate_arg();
	  }
	  else {
	    actual_clause->_get_actual()->_publish_cc_elaborate();
	  }
	}
       }
       actual_clause = port_map_aspect.successor(actual_clause);
     }
    
    actual_clause = port_map_aspect.first();  
    for(;actual_clause != NULL; ) {
      switch(actual_clause->get_formal()->_get_mode()) {
      case IIR_OUT_MODE:
      case IIR_INOUT_MODE:
	_cc_out << ", ";
	_cc_out << "&";
	if (actual_clause->_get_actual()->get_kind() == IIR_FUNCTION_CALL) {
	  ((IIR_FunctionCall *) actual_clause->_get_actual())->_publish_cc_elaborate_arg();
	}
	else {
	  if (actual_clause->_get_actual()->_is_iir_declaration() == TRUE) {
	    tempDeclaration = (IIR_Declaration *) actual_clause->_get_actual();
	    ASSERT (tempDeclaration->_is_iir_declaration() == TRUE );
	    _publish_cc_scoping_prefix( tempDeclaration->_get_declarative_region(),this->_get_enclosing_scope());
	  }
	  else if (actual_clause->_get_actual()->get_kind() == IIR_INDEXED_NAME) {
	    IIR_IndexedName *tempIdxName = (IIR_IndexedName *) actual_clause->_get_actual();
	    ASSERT ( tempIdxName->get_kind() == IIR_INDEXED_NAME );
	    if (tempIdxName->get_prefix()->_is_iir_declaration() == TRUE) {
	      tempDeclaration = (IIR_Declaration *) tempIdxName->get_prefix();

	      ASSERT ( tempDeclaration->_is_iir_declaration() == TRUE );
	      _publish_cc_scoping_prefix( tempDeclaration->_get_declarative_region(),this->_get_enclosing_scope());
	    }
	  }
	  
	  actual_clause->_get_actual()->_publish_cc_elaborate();
	}
	break;
      default:
	// do nothing;
	break;
      }
      actual_clause = port_map_aspect.successor(actual_clause);
    }
  }
  _cc_out << ");\n";
  
  actual_clause = port_map_aspect.first();  
  while(actual_clause != NULL) {
    switch(actual_clause->get_formal()->_get_mode()) {
    case IIR_IN_MODE:
    case IIR_INOUT_MODE:
      ASSERT(actual_clause->_get_actual() != NULL);
      if(actual_clause->_get_actual()->get_kind() == IIR_FUNCTION_CALL) {
	downTypeConversionFn = (IIR_FunctionCall*)actual_clause->_get_actual();
	_cc_out << "addDownTypeConversionFn(";
	if(actual_clause->get_formal()->get_kind() == IIR_FUNCTION_CALL) {
	  label->_publish_cc_elaborate();
	  _cc_out << "_elab_obj->";
	  ((IIR_FunctionCall *)actual_clause->get_formal())->_publish_cc_elaborate_arg();
	}
	else {
	  label->_publish_cc_elaborate();
	  
	  if (_get_currently_publishing_unit() == GENERATE_FOR) {
	    _cc_out << "_elab_obj[i - generateLeft].";
	  }
	  else {
	    _cc_out << "_elab_obj->";
	  }
	  actual_clause->get_formal()->_publish_cc_elaborate();
	}
	_cc_out << ", ";
	((IIR_FunctionCall *)actual_clause->_get_actual())->_publish_cc_elaborate_arg();
	_cc_out << ", ";
	downTypeConversionFn->get_implementation()->_publish_cc_type_conversion_function_name();
	_cc_out << ");\n";
      }
      break;
    default:
      // do nothing
      break;
    }
    actual_clause = port_map_aspect.successor(actual_clause);
  }
}
  

void
IIRScram_BlockStatement::_publish_createNetInfo() {    
  IIR_AssociationElement* actual_clause = NULL;
  IIR* formal = NULL;
  IIR_Label *label = _get_label();

  actual_clause = port_map_aspect.first();
  if(actual_clause == NULL){
    return;
  }
  
  while(actual_clause != NULL) {
    formal = actual_clause->get_formal();
    ASSERT(formal != NULL);
    ASSERT(formal->_is_resolved() == TRUE);
    switch(formal->_get_mode()) {
    case IIR_IN_MODE:
      _publish_cc_elaborate_Add(actual_clause, label);
      break;
    case IIR_OUT_MODE:
      // Code for adding the resolution information tree
      _publish_cc_elaborate_addChild(actual_clause, label);
      break;
    case IIR_INOUT_MODE:
      _publish_cc_elaborate_Add(actual_clause, label);
      // Code for adding the resolution information tree
      _publish_cc_elaborate_addChild(actual_clause, label);
      break;
    default:
      cerr << "Buffer and linkage mode not yet supported in "
	   << "Component instantiation statements\n";
      break;
    }
    actual_clause = port_map_aspect.successor(actual_clause);
  }
}


void 
IIRScram_BlockStatement::
_publish_cc_elaborate_Add(IIR_AssociationElement *actual_clause,
			  IIR_Label *label) {
  IIR* decl = actual_clause->get_formal();
  IIR_FunctionCall *upTypeConversionFn = NULL;
  IIR_FunctionCall *downTypeConversionFn = NULL;
  IIR_Declaration* tempDeclaration = NULL;
    
  if (actual_clause->_get_actual()->get_kind()!=IIR_FUNCTION_CALL &&
      actual_clause->get_formal()->get_kind()!=IIR_FUNCTION_CALL){
    _cc_out << "Add(";
    ASSERT(actual_clause->_get_actual() != NULL);
    if(actual_clause->_get_actual()->get_kind() == IIR_FUNCTION_CALL) {
      downTypeConversionFn = (IIR_FunctionCall *)actual_clause->_get_actual();
      downTypeConversionFn->_publish_cc_elaborate_arg();
    } else {
      if(actual_clause->_get_actual()->_is_iir_declaration() == TRUE && 
	 actual_clause->_get_actual()->_is_static_expression() == FALSE) {
	ASSERT ( actual_clause->_get_actual()->_is_iir_declaration() == TRUE );
	tempDeclaration = (IIR_Declaration *) actual_clause->_get_actual();

	ASSERT ( tempDeclaration->_is_iir_declaration() == TRUE );
	_publish_cc_scoping_prefix(tempDeclaration->_get_declarative_region(),this->_get_enclosing_scope());
      }
      else if(actual_clause->_get_actual()->_is_name() == TRUE) {
	tempDeclaration = (IIR_Declaration *) actual_clause->_get_actual()->_get_prefix_declaration();
	ASSERT ( tempDeclaration->_is_iir_declaration() == TRUE );
	_publish_cc_scoping_prefix(tempDeclaration->_get_declarative_region(),this->_get_enclosing_scope());
      }
      else if (actual_clause->_get_actual()->_is_static_expression()){
	actual_clause->_get_actual()->_get_subtype()->_publish_cc_type_string();
	_cc_out << "(ObjectBase::SIGNAL_NETINFO, ";
	actual_clause->_get_actual()->_publish_cc_value();
	actual_clause->_get_actual()->_get_subtype()->_publish_cc_object_type_info();
	_cc_out << ")";
      }
      else{
	cerr << "Error : IIRScram_BlockStatement::_publish_cc_elaborate_addChild() - declaration kind not handled yet.\n";
      }
      
      if (actual_clause->_get_actual()->_is_static_expression() == FALSE){
	actual_clause->_get_actual()->_publish_cc_elaborate();
      }
      
      _cc_out << ", ";
      label->_publish_cc_elaborate();
      
      if (_get_currently_publishing_unit() == GENERATE_FOR) {
	_cc_out << "_elab_obj[i - generateLeft].";
      }
      else {
	_cc_out << "_elab_obj->";
      }
      
      if(decl->get_kind() == IIR_FUNCTION_CALL) {
	upTypeConversionFn = (IIR_FunctionCall *)decl;
	upTypeConversionFn->_publish_cc_elaborate_arg();
      } else {
	decl->_publish_cc_elaborate();
      }    
      _cc_out << ");\n";
      
      if(upTypeConversionFn != NULL) {
	_cc_out << "setUpConversionFunctionId(";
	label->_publish_cc_elaborate();
	
	if (_get_currently_publishing_unit() == GENERATE_FOR) {
	  _cc_out << "_elab_obj[i - generateLeft].";
	}
	else {
	  _cc_out << "_elab_obj->";
	}
	
	upTypeConversionFn->_publish_cc_elaborate_arg();
	_cc_out << ", ";
	upTypeConversionFn->get_implementation()->_publish_cc_type_conversion_function_name();
	_cc_out << ");\n";
      }
    }
  }
}


void 
IIRScram_BlockStatement::
_publish_cc_elaborate_addChild(IIR_AssociationElement *actual_clause,
			       IIR_Label *label) {
  IIR* decl = actual_clause->get_formal();
  IIR_Declaration* tempDeclaration;
 
  if (actual_clause->_get_actual()->get_kind() != IIR_FUNCTION_CALL &&
      actual_clause->get_formal()->get_kind() != IIR_FUNCTION_CALL){
     _cc_out << "addChild(";
  if(actual_clause->_get_actual()->get_kind() == IIR_FUNCTION_CALL) {
    //((IIR_FunctionCall *)actual_clause->_get_actual())->_publish_cc_elaborate_arg();
  } else {
    if(actual_clause->_get_actual()->_is_iir_declaration() == TRUE) {
      ASSERT ( actual_clause->_get_actual()->_is_iir_declaration() == TRUE );
      tempDeclaration = (IIR_Declaration *) actual_clause->_get_actual();
      ASSERT ( tempDeclaration->_is_iir_declaration() == TRUE );
      _publish_cc_scoping_prefix(tempDeclaration->_get_declarative_region(),this->_get_enclosing_scope());
    }
    else if(actual_clause->_get_actual()->_is_name() == TRUE) {
      tempDeclaration = (IIR_Declaration *) actual_clause->_get_actual()->_get_prefix_declaration();
      ASSERT ( tempDeclaration->_is_iir_declaration() == TRUE );
      _publish_cc_scoping_prefix(tempDeclaration->_get_declarative_region(),this->_get_enclosing_scope());
    }
    else if (actual_clause->_get_actual()->_is_literal()){
             
      cerr << "Error : IIRScram_BlockStatement::_publish_cc_elaborate_addChid() - declaration kind not handled yet.\n";
    }
    actual_clause->_get_actual()->_publish_cc_elaborate();
  }
  
  
  _cc_out << ", ";
  label->_publish_cc_elaborate();

  if (_get_currently_publishing_unit() == GENERATE_FOR) {
    _cc_out << "_elab_obj[i - generateLeft].";
  }
  else {
    _cc_out << "_elab_obj->";
  }
  
  if(decl->get_kind() == IIR_FUNCTION_CALL) {
    //upTypeConversionFn = (IIR_FunctionCall *)decl;
    //upTypeConversionFn->_publish_cc_elaborate_arg();
  } else {
    if ( decl->_is_iir_declaration() == TRUE ) {
      tempDeclaration = (IIR_Declaration *) decl;
      ASSERT ( tempDeclaration->_is_iir_declaration() == TRUE );
    }
    else if ( decl->_is_name() == TRUE ) {
      tempDeclaration = (IIR_Declaration *) decl->_get_prefix_declaration();
      ASSERT ( tempDeclaration->_is_iir_declaration() == TRUE );
      _publish_cc_scoping_prefix(tempDeclaration->_get_declarative_region(),this->_get_enclosing_scope());
    }
    else {
      cerr << "Error : IIRScram_BlockStatement::_publish_cc_elaborate_addChid() - declaration kind not handled yet.\n";
    }
    
    decl->_publish_cc_elaborate();
  }    
  _cc_out << ");\n";
  }
}


void
IIRScram_BlockStatement::_publish_cc_type_info(){
  block_declarative_part._publish_cc_type_info();
  block_statement_part._publish_cc_type_info();
}

void
IIRScram_BlockStatement::_publish_cc_extern_type_info(){
  block_declarative_part._publish_cc_extern_type_info();
  block_statement_part._publish_cc_extern_type_info();
}

#ifdef PROCESS_GRAPH
void
IIRScram_BlockStatement::_publish_cc_driver_info() {
  IIR_SignalInterfaceDeclaration* portelement;
  IIR_AssociationElement* actual_clause;
  
  portelement = port_clause.first();
  actual_clause = port_map_aspect.first();
  for(; portelement != NULL; ) {
    _cc_out << "fp << \"";
    portelement->_publish_cc_elaborate();
    _cc_out << "\" << endl;" << endl;
    portelement->_publish_cc_elaborate();
    _cc_out << ".dump_connectivity_info(fp);" << endl;
    portelement = port_clause.successor(portelement);
  }

  IIR_ConcurrentStatement *conc_stmt;
  conc_stmt = block_statement_part.first();
  while (conc_stmt != NULL) {
    if (conc_stmt->get_kind() == IIR_COMPONENT_INSTANTIATION_STATEMENT) {
      ((IIRScram_ComponentInstantiationStatement*)conc_stmt)->_publish_cc_driver_info();
    }
    else if (conc_stmt->get_kind() == IIR_PROCESS_STATEMENT) {
      ((IIRScram_ProcessStatement*)conc_stmt)->_publish_cc_driver_info();
    }
    else if (conc_stmt->_is_block_statement() == TRUE) {
      ((IIRScram_BlockStatement *) conc_stmt)->_publish_cc_driver_info();
    }
    conc_stmt = block_statement_part.successor(conc_stmt);
  }
}
#endif
