// Copyright (c) 1996-1999 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.


// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	phil.wilsey@uc.edu
//          Dale E. Martin	dmartin@ece.uc.edu
//          Timothy J. McBrayer tmcbraye@ece.uc.edu
//          Malolan Chetlur     mal@ece.uc.edu
//          Krishnan Subramani  skrish@ece.uc.edu
//          Umesh Kumar V. Rajasekaran urajasek@ece.uc.edu
//          Narayanan Thondugulam nthondug@ece.uc.edu
//          Radharamanan Radhakrishnan  ramanan@ece.uc.edu
//          Swaminathan Subramanian ssubrama@ececs.uc.edu

//---------------------------------------------------------------------------
// 
// $Id: IIRScram.cc,v 1.13 1999/10/26 16:44:36 dmartin Exp $
// 
//---------------------------------------------------------------------------

#include <fstream.h>
#include <strstream.h>
#include <string.h>

#include "IIRScram.hh"
#include "error_func.hh"
#include "savant.hh"
#include "switch_file.hh"
#include "symbol_table.hh"
#include "library_manager.hh"
#include "set.hh"
#include "IIR_DesignFile.hh"
#include "IIR_AssociationList.hh"
#include "IIR_ArchitectureDeclaration.hh"
#include "IIR_EntityDeclaration.hh"
#include "IIR_ComponentDeclaration.hh"
#include "IIR_ConfigurationDeclaration.hh"
#include "IIR_PackageDeclaration.hh"
#include "resolution_func.hh"
#include "IIR_ConcurrentStatementList.hh"
#include "IIR_ConcurrentStatement.hh"
#include "IIR_Label.hh"
#include "IIR_DeclarationList.hh"
#include "IIR_AliasDeclaration.hh"
#include "IIR_RangeTypeDefinition.hh"
#include "IIR_SignalDeclaration.hh"
#include "IIR_SignalInterfaceDeclaration.hh"
#include "IIR_BlockStatement.hh"
#include "IIR_ConstantDeclaration.hh"

extern bool parse_error;
extern bool debug_symbol_table;

IIRScram::IIRScram(){
  _my_design_file = NULL;
  _cc_out.set_create_via_temporaries(true);
}

IIRScram::~IIRScram() {}

#ifndef DEVELOPER_ASSERTIONS
switch_file
IIRScram::_vhdl_out(1);  //file descriptor 1 is stdout
switch_file
IIRScram::_cc_out(1);
#endif

#ifdef GLOBAL_DATA
IIR_Char  *_current_publish_name = NULL;
IIR       *_current_publish_node = NULL;
IIR_Char  *_current_configuration_name = NULL;
IIR_Char  *_current_architecture_name = NULL;
IIR_Char  *_current_entity_name = NULL;
IIR_Char  *_current_elab_name = NULL;
IIR_Int32 _number_of_processes = 0;
IIR_Char  *_current_package_name = NULL;
IIR_Char  *_current_package_body_name = NULL;
IIR_Char  *_current_suffix_name = NULL;
IIR_Int32 _index_level = 0;
IIR_Char  *_current_another_name = NULL;
IIR_Boolean _allocate_return_value = FALSE;
IIR_Char* _publish_prefix_string = NULL;
IIR_TypeDefinition *_aggregate_iterator_subtype = NULL;
int _aggregate_iterator_counter = 0;
IIRScram::PublishedUnit _currently_publishing_unit = IIRScram::NONE_PARSING;
IIRScram::PublishedUnit _currently_publishing_vhdl_unit = IIRScram::NONE;
symbol_table *sym_tab = NULL;
library_manager *libMgr = NULL;
IIR_Boolean _publish_object_without_prefix = FALSE; 
IIR_ProcessStatement* _current_process_stmt = NULL;
#else
IIR_Char *
IIRScram::_current_publish_name = NULL;

IIR*
IIRScram::_current_publish_node = NULL;

IIR_Char *
IIRScram::_current_configuration_name = NULL;

IIR_Char *
IIRScram::_current_architecture_name = NULL;

IIR_Char *
IIRScram::_current_entity_name = NULL;

IIR_Char *
IIRScram::_current_elab_name = NULL;

IIR_Int32
IIRScram::_number_of_processes = 0;

IIR_Char*
IIRScram::_current_package_name = NULL;

IIR_Char*
IIRScram::_current_package_body_name = NULL;

IIR_Char*
IIRScram::_current_suffix_name = NULL;

IIR_Int32
IIRScram::_index_level = 0;

IIR_Char *
IIRScram::_current_another_name = NULL;

IIR_Boolean
IIRScram::_allocate_return_value = FALSE;

IIR_Char* 
IIRScram::_publish_prefix_string = NULL;

IIR_TypeDefinition* 
IIRScram::_aggregate_iterator_subtype = NULL;

int 
IIRScram::_aggregate_iterator_counter = 0;

IIRScram::PublishedUnit
IIRScram::_currently_publishing_unit = NONE;

IIRScram::PublishedUnit
IIRScram::_currently_publishing_vhdl_unit = NONE;

symbol_table *
IIRScram::sym_tab = NULL;

library_manager *
IIRScram::libMgr = NULL;

IIR_Boolean
IIRScram::_publish_object_without_prefix = FALSE;

IIR_ProcessStatement*
IIRScram::_current_process_stmt = NULL;

#endif

IIR_TypeDefinition* 
IIRScram::_get_name_type() {
  _report_undefined_scram_fn("_get_name_type()");
  return NULL;
}

IIR_Boolean
IIRScram::_is_attribute(){ return FALSE; }

IIR_Boolean
IIRScram::_is_longest_static_prefix() { 
  return _is_static_expression();
}

IIR_Boolean
IIRScram::_is_numeric_literal() {
  if( _is_integer_literal() == TRUE ||  _is_floating_literal() == TRUE ){
    return TRUE;
  }
  else {
    return FALSE;
  }
}

void 
IIRScram::_publish_vhdl_type_decl(ostream &_vhdl_out) {
  _publish_vhdl_decl(_vhdl_out);
}

void
IIRScram::_publish_vhdl_subtype_decl(ostream &_vhdl_out) {
  _publish_vhdl_decl(_vhdl_out);
}

void 
IIRScram::_type_check( IIR_TypeDefinition *input_decl ){
  set<IIR_TypeDefinition> temp_set;
  temp_set.add( input_decl );
  _type_check( &temp_set );
}


IIR *
IIRScram::_semantic_transform( set<IIR_TypeDefinition> * ){
#ifdef SEMANTIC_DEBUG
  cout << "Semantic transform called for node " << get_kind_text() <<
    " - returning \"this\". " << endl;
#endif
  return (IIR *)this;
}


IIR *
IIRScram::_semantic_transform( IIR_TypeDefinition *input_decl ){
  set<IIR_TypeDefinition> temp_set;
  temp_set.add( input_decl );
  return _semantic_transform( &temp_set );
}


IIR *
IIRScram::_rval_to_decl( IIR_TypeDefinition *resolved_rval ){
#ifdef SEMANTIC_DEBUG
  cout << "_rval_to_decl called for node " << get_kind_text() <<
    " - returning \"this\". " << endl;
#endif

  return (IIR *)this;
}


void 
IIRScram::_publish_cc_prefix_string() {
  if(_publish_prefix_string != NULL) {
    _cc_out << _publish_prefix_string;
  }
}


void
IIRScram::_publish_cc_scoping_prefix( IIR *, IIR *, ostream& ){
  _report_undefined_scram_fn("_publish_cc_scoping_prefix(IIR *, IIR *, ostream& = _cc_out)");
}

IIR_Char*
IIRScram::_get_publish_prefix_string(void) {
  return _publish_prefix_string;
}


void
IIRScram::_set_publish_prefix_string(IIR_Char* str) {
  _publish_prefix_string = str;
}


IIRScram::PublishedUnit
IIRScram::_get_currently_publishing_unit() {
  return _currently_publishing_unit;
}

void
IIRScram::_set_current_process_statement(IIR_ProcessStatement* p_stmt) {
  _current_process_stmt = p_stmt;
}

IIR_ProcessStatement*
IIRScram::_get_current_process_statement() {
  return _current_process_stmt;
}

void
IIRScram::_set_currently_publishing_unit(IIRScram::PublishedUnit unit) {
  _currently_publishing_unit = unit;
}


IIR_Boolean
IIRScram::_is_currently_publishing_subprogram() {
  return (_currently_publishing_unit == FUNCTION ||
	  _currently_publishing_unit == PROCEDURE);
}

IIR_Boolean
IIRScram::_is_currently_publishing_generate_for() {
  return (_currently_publishing_unit == GENERATE_FOR);
}

void 
IIRScram::_set_library_manager( library_manager *l_m ){
  libMgr = l_m;
}

library_manager *
IIRScram::_get_library_manager( ){
  return libMgr;
}

symbol_table *
IIRScram::_get_symbol_table( ){
  ASSERT( _my_design_file != NULL );
  ASSERT( _my_design_file->_get_symbol_table() != NULL );

  return _my_design_file->_get_symbol_table();
}

void 
IIRScram::_report_undefined_scram_fn(char *fn_name) {
  ostrstream os;
  os << fn_name << " not defined for IIR node: " << get_kind_text() << ends;
  report_error( this, os );
}

IIR_Boolean
IIRScram::_is_readable() {
  _report_undefined_scram_fn("_is_readable()");
  return FALSE;
}

IIR_Boolean
IIRScram::_is_writable() {
  _report_undefined_scram_fn("_is_writable()");
  return FALSE;
}

void
IIRScram::_set_passed_through_out_port(IIR_Boolean) {
  _report_undefined_scram_fn("_set_passed_through_out_port(IIR_Boolean)");
}

IIR_Boolean 
IIRScram::_is_locally_static_primary(){
  _report_undefined_scram_fn("_is_locally_static_primary()");
  return FALSE;
}

IIR_Boolean 
IIRScram::_is_globally_static_primary(){
  // If no overrided version of this got called, we're going to
  // fall back to calling "_is_locally_static_primary".  See 7.4.2 in the
  // '93 LRM for more info.
  return _is_locally_static_primary();
}

IIR_Boolean 
IIRScram::_is_textio() {
  return FALSE;
}

IIR_Boolean 
IIRScram::_is_standard() {
  return FALSE;
}

IIR_Boolean 
IIRScram::_is_relational_operator() {
  return FALSE;
}

IIR_Boolean 
IIRScram::_is_resolved_type() {
  return FALSE;
}

void 
IIRScram::_publish_vhdl(ostream &_vhdl_out) {
  _report_undefined_scram_fn("_publish_vhdl(ostream &)");
}

void 
IIRScram::_publish_vhdl_range(ostream &_vhdl_out) {
  _report_undefined_scram_fn("_publish_vhdl_range(ostream &)");
}
void 
IIRScram::_publish_vhdl_decl(ostream &_vhdl_out) {
  _report_undefined_scram_fn("_publish_vhdl_decl()");
}

void 
IIRScram::_publish_cc() {
  _report_undefined_scram_fn("_publish_cc()");
}

void 
IIRScram::_publish_cc_condition() {
  _report_undefined_scram_fn("_publish_cc_condition()");
}

void 
IIRScram::_publish_cc_name() {
  _report_undefined_scram_fn("_publish_cc_name()");
}

void 
IIRScram::_publish_cc_name_elaborate() {
  _report_undefined_scram_fn("_publish_cc_name_elaborate()");
}

void 
IIRScram::_publish_cc_state_object_init() {
  _report_undefined_scram_fn("_publish_cc_state_object_init()");
}

void 
IIRScram::_publish_cc_init() {
  _report_undefined_scram_fn("_publish_cc_init()");
}

void 
IIRScram::_publish_cc_object_type() {
  _report_undefined_scram_fn("_publish_cc_object_type()");
}

void 
IIRScram::_publish_cc_initialization_value() {
  _report_undefined_scram_fn("_publish_cc_initialization_value()");
}

void
IIRScram::_publish_cc_universal_type() {
  _report_undefined_scram_fn("_publish_cc_universal_type()");  
}

void
IIRScram::_publish_cc_kernel_type() {
  _report_undefined_scram_fn("_publish_cc_kernel_type()");
}

void
IIRScram::_publish_cc_first_objectParameter()
{
  ASSERT ( _is_object() == TRUE );

  _publish_cc();
}

void 
IIRScram::_publish_cc_wait_data() {
  _report_undefined_scram_fn("_publish_cc_wait_data()");
}

void
IIRScram::_publish_cc_data() {
  _report_undefined_scram_fn("_publish_cc_data()");
}

void 
IIRScram::_publish_cc_decl() {
  _report_undefined_scram_fn("_publish_cc_decl()");
}

void
IIRScram::_publish_cc_bounds() {
  _report_undefined_scram_fn("_publish_cc_bounds()");
}

void
IIRScram::_publish_cc_range() {
  _report_undefined_scram_fn("_publish_cc_range()");
}

void
IIRScram::_publish_cc_value() {
  _report_undefined_scram_fn("_publish_cc_value()");
}

void
IIRScram::_publish_cc_universal_value() {
  _report_undefined_scram_fn("_publish_cc_universal_value()");
}

void
IIRScram::_publish_cc_binding_name(ostream& outstream) {
  _report_undefined_scram_fn("_publish_cc_binding_name()");
}

void
IIRScram::_publish_cc_declarator() {
  _report_undefined_scram_fn("_publish_cc_declarator()");
}

void
IIRScram::_publish_cc_type_name() {
  _report_undefined_scram_fn("_publish_cc_type_name()");
}

void
IIRScram::_publish_cc_for_index() {
  _report_undefined_scram_fn("_publish_cc_for_index()");
}

IIR_TextLiteral *
IIRScram::_get_declarator() {
  ASSERT(_is_iir_declaration() == TRUE || get_kind() == IIR_SIMPLE_NAME);
  _report_undefined_scram_fn("_get_declarator()");
  return NULL;
}

IIR_GenericList *
IIRScram::_get_generic_list(){
  _report_undefined_scram_fn("_get_generic_list()");
  return NULL;
}

IIR_PortList *
IIRScram::_get_port_list(){
  _report_undefined_scram_fn("_get_port_list()");
  return NULL;
}


IIR *
IIRScram::get_value() {
  ASSERT(_is_signal() == TRUE);
  _report_undefined_scram_fn("get_value()");
  return NULL;
}

IIR_List *
IIRScram::_get_statement_list(){
  _report_undefined_scram_fn("_get_statement_list()");

  return NULL;
}

IIR_Mode
IIRScram::_get_mode() {
  _report_undefined_scram_fn("_get_mode()");
  return IIR_UNKNOWN_MODE;
}

IIR_Boolean
IIRScram::_is_published_attribute_in_state(SignalAttribute) {
  _report_undefined_scram_fn("_is_published_attribute_in_state()");
  return FALSE;
}

void 
IIRScram::_add_published_attribute_in_state(SignalAttribute) {
  _report_undefined_scram_fn("_add_published_attribute_in_state()");
}

IIR_Boolean
IIRScram::_is_published_attribute_in_constructor(SignalAttribute) {
  _report_undefined_scram_fn("_is_published_attribute_in_constructor()");
  return FALSE;
}

void 
IIRScram::_add_published_attribute_in_constructor(SignalAttribute) {
  _report_undefined_scram_fn("_add_published_attribute_in_constructor()");
}

IIR_Boolean
IIRScram::_is_published_attribute_in_initstate(SignalAttribute) {
  _report_undefined_scram_fn("_is_published_attribute_in_initstate()");
  return FALSE;
}

void 
IIRScram::_add_published_attribute_in_initstate(SignalAttribute) {
  _report_undefined_scram_fn("_add_published_attribute_in_initstate()");
}

IIR_Boolean
IIRScram::_is_ascending_range() {
  _report_undefined_scram_fn("_is_ascending_range()");
  return false;
}

IIR * 
IIRScram::_transmute() {
  _report_undefined_scram_fn("_transmute()");
  return NULL;
}

ostream&
IIRScram::_print(ostream& os) {
  _report_undefined_scram_fn("ostream& _print(ostream&)");
  return os;
}

set<IIR_Declaration> *
IIRScram::_symbol_lookup(){
  _report_undefined_scram_fn("_symbol_lookup( )");
  return NULL;
}

set<IIR_Declaration> *
IIRScram::_symbol_lookup( IIR_Declaration * ){
  _report_undefined_scram_fn("_symbol_lookup( IIR_Declaration * )");
  return NULL;
}

set<IIR_Declaration> *
IIRScram::_symbol_lookup( set<IIR_Declaration> * ){
  _report_undefined_scram_fn("_symbol_lookup( set<IIR_Declaration> * )");
  return NULL;
}

set<IIR_Declaration> *
IIRScram::_symbol_lookup( IIR_Boolean (IIR::*constraint_function)() ){
  set<IIR_Declaration> *retval = _symbol_lookup();

  if( retval != NULL ){
    retval->reduce_set( constraint_function );
  }

  return retval;
}

IIR_Label *
IIRScram::_lookup_label( IIR_Boolean complain_on_error ){
  IIR_Label *retval = NULL;
  
  ASSERT( complain_on_error == TRUE ||  complain_on_error == FALSE );
  
  set<IIR_Declaration> *decls = _symbol_lookup( &IIR::_is_iir_label );
  ASSERT( decls != NULL );
  if( decls->num_elements() == 0 && complain_on_error == TRUE ){
    report_undefined_symbol( (IIR *)this );
  }

  switch( decls->num_elements() ){
  case 0:{
    ostrstream err;
    err << "No label |" << *this << "| declared in this scope." << ends;
    report_error( this, err );
    break;
  }
  case 1:{
    IIR_Declaration *temp = decls->get_element();
    ASSERT( temp->_is_iir_label() == TRUE );
    retval = (IIR_Label *)temp;
    break;
  }
  default:{
    report_ambiguous_error( (IIR *)this, decls );
  }
  }

  return retval;
}

set<IIR_TypeDefinition> *
IIRScram::_get_rval_set( IIR_Boolean (IIR::*constraint_function)() ){
  _report_undefined_scram_fn("_get_rval_set(),\nIIR_Boolean (IIR::*constraint_function)()");
  return NULL;
}

set<IIR_TypeDefinition> *
IIRScram::_get_rval_set( set<IIR_TypeDefinition> *,
			 IIR_Boolean (IIR::*constraint_function)() ){
  _report_undefined_scram_fn("_get_rval_set( set<IIR_TypeDefinition> * ),\nIIR_Boolean (IIR::*constraint_function)()");
  return NULL;
}

set<IIR_TypeDefinition> *
IIRScram::_get_rval_set( set<IIR_Declaration> *, 
			 IIR_Boolean (IIR::*constraint_function)() ){
  _report_undefined_scram_fn("_get_rval_set( set<IIR_Declaration> * ,\nIIR_Boolean (IIR::*constraint_function)() )");
  return NULL;
}

// set<IIR_Declaration> *
// IIRScram::_find_declarations( IIR_Declaration * ){
//   _report_undefined_scram_fn("_find_declarations( IIR_Declaration * )");
//   return NULL;
// }

IIR_Declaration *
IIRScram::_find_in_implicit_list( char *to_find ){
  IIR_Declaration *current_decl;
  
  if( _get_implicit_declarations() != NULL ){
    current_decl = _get_implicit_declarations()->get_element();
    while( current_decl != NULL ){
      if( IIR_TextLiteral::_cmp( current_decl->get_declarator(), to_find ) ==0){
        return current_decl;
      }
      current_decl = _get_implicit_declarations()->get_next_element();
    }
  }

  return NULL;
}

set<IIR_Declaration> *
IIRScram::_get_implicit_declarations( ){
  _report_undefined_scram_fn("_get_implicit_declarations( )");
  return NULL;
}

void 
IIRScram::_set_implicit_declarations( set<IIR_Declaration> * ){
  _report_undefined_scram_fn("_set_implicit_declarations( set<IIR_Declaration> * )"); 
}


IIR_Declaration *
IIRScram::_find_formal_declaration(){
  _report_undefined_scram_fn("_find_formal_declaration( )");
  return NULL;
}


IIR_TypeDefinition *
IIRScram::_determine_rval_in_set( set<IIR_TypeDefinition> *, IIR_TypeDefinition * ){
  _report_undefined_scram_fn("_determine_rval_in_set( set<IIR_TypeDefinition> *, IIR_TypeDefinition * )");
  return NULL;  
}

IIR_Declaration *
IIRScram::_determine_decl_in_set( set<IIR_Declaration> *, IIR_TypeDefinition * ){
  _report_undefined_scram_fn("_determine_decl_in_set( set<IIR_Declaration> *, IIR_TypeDefinition *)");
  return NULL;  
}

IIR_TypeDefinition *
IIRScram::_get_subtype(){
  _report_undefined_scram_fn("_get_subtype()");
  return NULL;
}

IIR_TypeDefinition *
IIRScram::_get_type_of_element( int ){
  _report_undefined_scram_fn("_get_type_of_element( int )");
  return NULL;
}

IIR_Int32 
IIRScram::_get_num_indexes( ){
  _report_undefined_scram_fn("_get_num_indexes()");
  return -1;
}

IIR_TypeDefinition *
IIRScram::_get_port_type( int ){
  _report_undefined_scram_fn("_get_port_type");
  return NULL;
}

void 
IIRScram::_type_check( set<IIR_TypeDefinition> * ){
  _report_undefined_scram_fn("_type_check( set<IIR_TypeDefinition> * )");
}

IIR *
IIRScram::_rval_to_decl( IIR_TypeDefinition *prefix_rval, IIR_TypeDefinition *suffix_rval ){
  _report_undefined_scram_fn("_rval_to_decl( IIR_TypeDefinition *prefix_rval, IIR_TypeDefinition *suffix_rval )");  
  return NULL;
}

IIR *
IIRScram::_rval_to_decl( IIR_Declaration *prefix_decl, IIR_TypeDefinition *suffix_rval ){
  _report_undefined_scram_fn("_rval_to_decl( IIR_Declaration *prefix_decl, IIR_TypeDefinition *suffix_rval )");  
  return NULL;
}


IIR *
IIRScram::_get_component_name( ){
  _report_undefined_scram_fn("_get_component_name()");
  return NULL;
}

void 
IIRScram::_set_component_name( IIR * ){
  _report_undefined_scram_fn("_set_component_name()");
}

IIR_LibraryUnit *
IIRScram::_get_entity_aspect(){
  _report_undefined_scram_fn("_get_entity_aspect()");
  return NULL;
}


void 
IIRScram::_publish_cc_elaborate() {
  _report_undefined_scram_fn("_publish_cc_elaborate()");
}

void 
IIRScram::_publish_cc_subprogram_arg() {
  _report_undefined_scram_fn("_publish_cc_subprogram_arg()");
}

void 
IIRScram::_publish_cc_constructor_args() {
  _report_undefined_scram_fn("_publish_cc_constructor_args()");
}

void 
IIRScram::_publish_cc_sigdest() {
  _report_undefined_scram_fn("_publish_cc_sigdest()");
}

void
IIRScram::_publish_cc_headers() {
  _report_undefined_scram_fn("_publish_cc_headers()");
}

void 
IIRScram::_publish_cc_headerfiles_for_cc_default(){
  _cc_out << "#include \"";
  _publish_cc_binding_name();
  if (get_kind() != IIR_PROCESS_STATEMENT) {
    _cc_out << "_elab";
  }
  _cc_out << ".hh\"\n";
}

void
IIRScram::_publish_cc_direction() {
  _report_undefined_scram_fn("_publish_cc_direction()");
}

void
IIRScram::_publish_cc_addChild() {
  _report_undefined_scram_fn("_publish_cc_addChild()");
}  
  
void 
IIRScram::_get_list_of_input_signals(set<IIR_Declaration>* list) {
  _report_undefined_scram_fn("_get_list_of_input_signals()");
}

void 
IIRScram::_get_signal_source_info(set<IIR_Declaration>* siginfo) {
  _report_undefined_scram_fn("_get_signal_source_info()");
}

IIR_AttributeSpecification*
IIRScram::_get_attribute_specification(IIR*) {
  _report_undefined_scram_fn("_get_attribute_specification()");
  return NULL;
}

IIR_SignalKind
IIRScram::_get_signal_kind() {
  _report_undefined_scram_fn("_get_signal_kind()");
  return IIR_NO_SIGNAL_KIND;
}

IIR_Boolean
IIRScram::_is_guard_signal() {
  if( _is_signal() == TRUE && _get_signal_kind() != IIR_NO_SIGNAL_KIND ){
    return TRUE;
  }
  else{
    return FALSE;
  }
}

void 
IIRScram::_build_sensitivity_list(IIR_DesignatorList* list) {
  _report_undefined_scram_fn("_build_sensitivity_list()");
}

IIR_Declaration* 
IIRScram::_get_prefix_declaration() {
  _report_undefined_scram_fn("_get_prefix_declaration()");
  return NULL;
}

IIR_Declaration* 
IIRScram::_get_package_declaration() {
  _report_undefined_scram_fn("_get_package_declaration()");
  return NULL;
}

char *
IIRScram::_convert_to_library_name( ) {
  _report_undefined_scram_fn("_convert_to_library_name()");
  return NULL;
}

IIR_TextLiteral *
IIRScram::_get_prefix_string( ) {
  _report_undefined_scram_fn("_get_prefix_string()");
  return NULL;
}

void 
IIRScram::_make_interface_visible( symbol_table * ){
  _report_undefined_scram_fn("_make_interface_visible( symbol_table * )");
}

void 
IIRScram::_make_interface_visible( ){
  _make_interface_visible( _get_symbol_table() );
}

IIR *
IIRScram::_get_current_declarative_region(){
  ASSERT( _get_symbol_table() != NULL );

  return _get_symbol_table()->get_current_declarative_region();
}

#ifdef PROCESS_COMBINATION
void 
IIRScram::_static_elaborate(IIR_ArchitectureDeclaration*, 
			    IIR_DeclarationList*, char*) {
  _report_undefined_scram_fn("_static_elaborate()");
}
#endif

IIR*
IIRScram::_clone() {
  _report_undefined_scram_fn("IIR *_clone()");
  return NULL;
}

void
IIRScram::_clone(IIR* clone) {
  copy_location(this, clone);
}  

void 
IIRScram::_add_decl_into_cgen_symbol_table() {
  if(_is_literal() == FALSE) {
    _report_undefined_scram_fn("_add_decl_into_cgen_symbol_table()");
  }
}

IIR_Boolean
IIRScram::_get_publish_object_without_prefix() {
  return _publish_object_without_prefix;
}

void
IIRScram::_set_publish_object_without_prefix(IIR_Boolean value) {
  _publish_object_without_prefix = value;
}

IIR_IntegerLiteral *
IIRScram::_get_integer_static_value(){
  _report_undefined_scram_fn("_get_integer_static_value()");
  return NULL;
}


IIR *
IIRScram::_decl_to_decl( IIR_Declaration * ){
  _report_undefined_scram_fn("_decl_to_decl( IIR_Declaration * )");
  return NULL;
}

void
IIRScram::_publish_vhdl_operator(ostream &_vhdl_out){
  _report_undefined_scram_fn("_publish_vhdl_operator(ostream &)");
}

void
IIRScram::_set_aggregate_iterator_subtype(IIR_TypeDefinition *iterator_subtype) {
  _aggregate_iterator_subtype = iterator_subtype;
}

IIR_TypeDefinition*
IIRScram::_get_aggregate_iterator_subtype() {
  return _aggregate_iterator_subtype;
}

void
IIRScram::_set_aggregate_iterator(int new_counter_value) {
  _aggregate_iterator_counter = new_counter_value;
}

int
IIRScram::_get_aggregate_iterator(){
  return _aggregate_iterator_counter;
}

void
IIRScram::_set_currently_publishing_vhdl_unit(PublishedUnit unit){
  _currently_publishing_vhdl_unit = unit;
}

IIRScram::PublishedUnit
IIRScram::_get_currently_publishing_vhdl_unit(){
  return _currently_publishing_vhdl_unit;
}

IIRScram::OperatorPrecedenceLevel
IIRScram::_get_operator_precedence(){
  return IIRScram::OTHERS;
}

bool
IIRScram::_get_mangling_flag() {
  return no_mangling;
}

IIR_FunctionDeclaration*
IIRScram::_get_resolution_function(){
  return NULL;
}

void 
IIRScram::_set_resolution_function( IIR_FunctionDeclaration * ){
  _report_undefined_scram_fn("_set_resolution_function");
}

void 
IIRScram::_type_check_configuration( IIR_AssociationList &port_map_aspect,
				     IIR_AssociationList &generic_map_aspect ){
  
  _get_symbol_table()->open_scope( (IIR *)this );
  set<IIR_Declaration> *component_decls;
  component_decls = _get_component_name()->_symbol_lookup( &IIR::_is_iir_component_declaration );
  if( component_decls == NULL ){
    report_undefined_symbol( _get_component_name() );
    return;
  }

  IIR_ComponentDeclaration *my_component;
  switch( component_decls->num_elements() ){
  case 0:{
    ostrstream err;
    err << "|" << *_get_component_name() 
	<< "| was not declared as a component in this scope." << ends;
    report_error( this, err );
    return;
  }
  case 1:{
    my_component = (IIR_ComponentDeclaration *)component_decls->get_element();
    _set_component_name( _get_component_name()->_decl_to_decl( my_component ) );
    if( debug_symbol_table == true ){
      cerr << "Type checking configuration - about to make |" << 
	*_get_component_name() << "| visible." << endl;
    }
    my_component->_make_interface_visible( _get_symbol_table() );
    break;
  }
  default:{
    report_ambiguous_error( _get_component_name(), component_decls );
    break;
  }
  }

  // This currently gets resolved by the parser...
  IIR *aspect = _get_entity_aspect();
  if( aspect != NULL ){
    ASSERT( aspect->_is_iir_declaration() == TRUE );
    ASSERT( aspect->_is_resolved() == TRUE );
    IIR_Declaration *aspect_decl = (IIR_Declaration *)aspect;
    
    // The formal generics and ports of the entity must be visible here.
    // See section 10.2 of the LRM.  So, now we're deciding what entity
    // we're looking at, and using this info to put those formals back
    // into scope.
    IIR_EntityDeclaration *entity_decl = NULL;
    switch( aspect_decl->get_kind() ){
      
    case IIR_ARCHITECTURE_DECLARATION:{
      IIR_ArchitectureDeclaration *as_arch = (IIR_ArchitectureDeclaration *)aspect_decl;
      entity_decl = as_arch->get_entity();
      break;
    }
    case IIR_ENTITY_DECLARATION:{
      entity_decl =  (IIR_EntityDeclaration *)aspect_decl;
      break;
    }
    case IIR_CONFIGURATION_DECLARATION:{
      entity_decl = ((IIR_ConfigurationDeclaration *)aspect_decl)->get_entity();
      break;
    }
    default:{
      ostrstream err;
      err << "Internal error in IIRScram_ConfigurationSpecification::_type_check"
	  << " - got a " << aspect_decl->get_kind_text() 
	  << ", expecting architecture or entity." << ends;
      report_error( this, err );
      abort();
    }
    }
    
    // Resolve the port map
    if( aspect_decl->_get_port_list() != NULL 
	&& port_map_aspect.num_elements() != 0 ){
      port_map_aspect._resolve_and_order(  entity_decl->_get_port_list(),
					   my_component->_get_port_list()
					   );
    }
    else{
      ASSERT(_get_component_name()->get_kind() == IIR_COMPONENT_DECLARATION);
      build_default_map( entity_decl->port_clause,
			 ((IIR_ComponentDeclaration*)_get_component_name())->local_port_clause,
			 port_map_aspect);
    }
    
    // Resolve the generic map
    if( aspect_decl->_get_generic_list() != NULL 
	&& generic_map_aspect.num_elements() != 0 ){
      generic_map_aspect._resolve_and_order( entity_decl->_get_generic_list(),
					     my_component->_get_generic_list()
					     );
    }    
    else{
      ASSERT(_get_component_name()->get_kind() == IIR_COMPONENT_DECLARATION);
      build_default_map( entity_decl->generic_clause,
			 ((IIR_ComponentDeclaration*)_get_component_name())->local_generic_clause,
			 generic_map_aspect);
    }
  }
  _get_symbol_table()->close_scope( (IIR *)this );
  // else it's OPEN
  //  ASSERT( _is_resolved() == TRUE );
  delete component_decls;
}

IIR_TypeDefinition *
IIRScram::_type_check_iteration_scheme( IIR_ConstantDeclaration *iteration_scheme ){
  IIR_TypeDefinition *retval = NULL;

  ASSERT( iteration_scheme != NULL );
  IIR_TypeDefinition *range_type = iteration_scheme->get_subtype();
  ASSERT( iteration_scheme->get_subtype() != NULL );

  if( range_type->get_kind() == IIR_RANGE_TYPE_DEFINITION ){
    IIR_RangeTypeDefinition *temp = (IIR_RangeTypeDefinition *)range_type;
    IIR_TypeDefinition *constant_base_type =
      IIR_ScalarTypeDefinition::_determine_discrete_type( temp );
    if( constant_base_type != NULL ){
      retval = constant_base_type->_construct_new_subtype( NULL, temp );
    }
  }
  else{
    // else it should be resolved.
    ASSERT( iteration_scheme->get_subtype()->_is_resolved() == TRUE );
    retval = iteration_scheme->get_subtype();
  }

  return retval;
}



IIR *
IIRScram::_get_enclosing_scope() {
  _report_undefined_scram_fn("_get_enclosing_scope()");
  return NULL;
}

IIR_LibraryDeclaration *
IIRScram::_get_work_library(){
  return _get_library_manager()->get_work_library();
}

void 
IIRScram::copy_location ( IIRScram *source, IIRScram *dest ){
  ASSERT( source != NULL || parse_error == true  );
  ASSERT( dest != NULL || parse_error == true );
  ASSERT( source->_get_design_file() != NULL || parse_error == true );
  ASSERT( source->get_line_number() != -1 || parse_error == true );

#ifdef EXTRA_LOCATERS
  ASSERT( source->get_column_offset() != -1 || parse_error == true );
  ASSERT( source->get_character_offset() != -1 || parse_error == true );
#endif

#ifdef EXTRA_LOCATERS
  dest->_set_base_info( source->_get_design_file(), source->get_line_number(),
			source->get_column_offset(), source->get_character_offset() );
#else
  dest->_set_base_info( source->_get_design_file(), source->get_line_number(), 0 ,0 );
#endif
}

void 
IIRScram::_set_base_info( IIR_DesignFile *design_file,
			  int line_number,
			  int column_offset, 
			  int character_offset ){
  ASSERT( design_file != NULL );
  
  _set_design_file( design_file );
  set_line_number( line_number );

#ifdef EXTRA_LOCATERS
  set_column_offset( column_offset );
  set_character_offset( character_offset );
#endif
}

void
IIRScram::_copy_symbols_defined_in_enclosing_scope(symbol_table *current, symbol_table *outer) {
  
  IIR_Declaration *decl;

  decl = current->in_scope_by_type[IIRScram_Declaration::CONSTANT].first();
  while (decl != NULL) {
    if ((decl->_get_declarative_region() != this) && (!outer->in_scope(decl))){
      outer->add_declaration(decl);
    }
    decl = current->in_scope_by_type[IIRScram_Declaration::CONSTANT].successor(decl);
  }
  
  decl = current->in_scope_by_type[IIRScram_Declaration::SIGNAL].first();
  while (decl != NULL) {
    if ((decl->_get_declarative_region() != this) && (!outer->in_scope(decl))){
      outer->add_declaration(decl);
    }
    decl = current->in_scope_by_type[IIRScram_Declaration::SIGNAL].successor(decl);
  }

  decl = current->in_scope_by_type[IIRScram_Declaration::INTERFACE_SIGNAL].first();
  while (decl != NULL) {
    if ((decl->_get_declarative_region() != this) && (!outer->in_scope(decl))){
      outer->add_declaration(decl);
    }
    decl = current->in_scope_by_type[IIRScram_Declaration::INTERFACE_SIGNAL].successor(decl);
  }
  
  decl = current->in_scope_by_type[IIRScram_Declaration::VARIABLE].first();
  while (decl != NULL) {
    if ((decl->_get_declarative_region() != this) && (!outer->in_scope(decl))){
      outer->add_declaration(decl);
    }
    decl = current->in_scope_by_type[IIRScram_Declaration::VARIABLE].successor(decl);
  }
  
  decl = current->in_scope_by_type[IIRScram_Declaration::ALIAS].first();
  while (decl != NULL) {
    if ((decl->_get_declarative_region() != this) && (!outer->in_scope(decl))){
      outer->add_declaration(decl);
    }
    decl = current->in_scope_by_type[IIRScram_Declaration::ALIAS].successor(decl);
  }
  
  decl = current->in_scope_by_type[IIRScram_Declaration::ATTRIBUTE].first();
  while (decl != NULL) {
    if ((decl->_get_declarative_region() != this) && (!outer->in_scope(decl))){
      outer->add_declaration(decl);
    }
    decl = current->in_scope_by_type[IIRScram_Declaration::ATTRIBUTE].successor(decl);
  }

  decl = current->in_scope_by_type[IIRScram_Declaration::S_FILE].first();
  while (decl != NULL) {
    if ((decl->_get_declarative_region() != this) && (!outer->in_scope(decl))){
      outer->add_declaration(decl);
    }
    decl = current->in_scope_by_type[IIRScram_Declaration::S_FILE].successor(decl);
  }
}

void
IIRScram::_group_component_instantiations(IIR_ConcurrentStatementList* conc_stmt_list, int blockSize) {
  // This method groups (blockSize) number of component instantiations into
  // a single implicit block. This was done to reduce the size of the published
  // individual elaboration class. Just a neat trick to get things going -DJ
  int componentInstantiationCount = conc_stmt_list->_get_number_of_component_instantiations();
  int componentCounter               = 0;
  IIR_BlockStatement *newBlock       = NULL;
  IIR_ConcurrentStatement* conc_stmt = conc_stmt_list->first();
  IIR_ConcurrentStatement* next_stmt = NULL;
  
  while (componentInstantiationCount > blockSize) {
    componentCounter = 0;
    newBlock         = new IIR_BlockStatement;
    copy_location( this, newBlock );
    newBlock->_set_enclosing_scope( (IIR *) this );
    
    while ((componentCounter < blockSize) && (conc_stmt != NULL))  {
      next_stmt = conc_stmt_list->successor(conc_stmt);
      if (conc_stmt->get_kind() == IIR_COMPONENT_INSTANTIATION_STATEMENT) {
	conc_stmt_list->remove(conc_stmt);
	
	newBlock->block_statement_part.append(conc_stmt);
	conc_stmt->_set_enclosing_scope(newBlock);
	
	componentCounter++;
	componentInstantiationCount--;
	
	ASSERT ( conc_stmt_list->_get_number_of_component_instantiations() ==
		 componentInstantiationCount );
      }
      
      conc_stmt = next_stmt;
    }

    conc_stmt_list->append(newBlock);
  }
}

void
IIRScram::_publish_cc_class_includes(IIR_ConcurrentStatementList* conc_stmt_list) {
  ASSERT(conc_stmt_list != NULL);
  ASSERT((this->_is_concurrent_statement() == TRUE) ||
	 (this->_is_entity_decl() == TRUE) ||
	 (this->_is_component_decl() == TRUE) ||
	 (this->get_kind() == IIR_ARCHITECTURE_DECLARATION));
  
  IIR_ConcurrentStatement* conc_stmt = conc_stmt_list->first();
  while (conc_stmt != NULL) {
    if  (conc_stmt->get_kind() == IIR_PROCESS_STATEMENT) {
      _cc_out << "class ";
      if (conc_stmt->_get_label() != NULL) {
	_cc_out << *(conc_stmt->_get_label());
      }
      else {
	_cc_out << "ANON_PROCESS" << conc_stmt;
      }
      _cc_out << ";\n";
    }
    else {
      _cc_out << "class ";
      conc_stmt->_publish_cc_binding_name();
      _cc_out << "_elab;\n";
    }
    conc_stmt = conc_stmt_list->successor(conc_stmt);
  }
}

void
IIRScram::_publish_cc_signals(IIR_DeclarationList* decl_list) {
  ASSERT(decl_list != NULL);
  ASSERT((this->_is_concurrent_statement() == TRUE) ||
	 (this->_is_entity_decl() == TRUE) ||
	 (this->_is_component_decl() == TRUE) ||
	 (this->get_kind() == IIR_ARCHITECTURE_DECLARATION));
  
  IIR_Declaration* decl = decl_list->first();
  
  while (decl != NULL) {
    if(decl->_is_signal() == TRUE) {
      decl->_get_subtype()->_publish_cc_type_name();
      _cc_out << " ";
      decl->_publish_cc_elaborate();
      _cc_out << ";\n";
      
      //If the Signals has implicit signals
      //They are also needed for elaboration info
      if( decl->_get_implicit_declarations() != NULL &&
	  decl->_get_implicit_declarations()->num_elements() != 0) {
	IIR_Declaration* imp_decl = decl->_get_implicit_declarations()->get_element();
	while(imp_decl != NULL) {
	  if(imp_decl->_is_signal() == TRUE) {
	    imp_decl->_publish_cc_implicit_signal_type();
	    _cc_out << " ";
	    imp_decl->_publish_cc_elaborate();
	    _cc_out << ";\n";
	  }
	  imp_decl = decl->_get_implicit_declarations()->get_next_element();
	}
      }
    }
    else if(decl->get_kind() == IIR_ALIAS_DECLARATION) {
      if(decl->_is_signal() == TRUE) {
	((IIR_AliasDeclaration*)decl)->_publish_cc_elaborate_alias_definition();
      }
    }
    decl = decl_list->successor(decl);    
  }
}



void
IIRScram::_publish_cc_object_pointers(IIR_ConcurrentStatementList* conc_stmt_list) {

  IIR_ConcurrentStatement *conc_stmt = conc_stmt_list->first();
  while (conc_stmt != NULL) {
    if (conc_stmt->get_kind() == IIR_PROCESS_STATEMENT) {
      ostrstream objectname;
      
      if(conc_stmt->_get_label() != NULL) {
	(conc_stmt->_get_label())->_print(objectname);
	objectname << ends;
      }
      else {
	objectname << "ANON_PROCESS" << conc_stmt << ends;
      }
      
      char *objname = objectname.str();
      _cc_out << objname ;
      // If the concurrent statement is a concurrent generate for statement, then
      // we need to instantiate multiple process. So we declare a process_class ** array
      // so that process_class[i] contains the process instantiated seperately and
      // not deleted in the destructor. Because the processes are deleted in the
      // main.cc of the generate code.
      if ( this->get_kind() == IIR_CONCURRENT_GENERATE_FOR_STATEMENT ){
	_cc_out << "** ";
      } else {
	_cc_out << "* ";
      }
      _cc_out << objname << "_elab_obj;\n";
      delete [] objname;
    }
    else if((conc_stmt->get_kind() == IIR_COMPONENT_INSTANTIATION_STATEMENT) ||
	    (conc_stmt->_is_block_statement() == TRUE) ||
	    (conc_stmt->_is_concurrent_generate_statement() == TRUE)) {
      conc_stmt->_publish_cc_binding_name();
      _cc_out << "_elab* ";
      conc_stmt->_get_label()->_publish_cc_elaborate();
      _cc_out << "_elab_obj;\n";
    }
    else {
      cerr << "ERROR! IIRScram_BlockStatement::"
	   << "_publish_cc_object_pointers(): unknown concurrent statement "
	   << "type |" << conc_stmt->get_kind_text() << "| in arch" << endl;
    }
    conc_stmt = conc_stmt_list->successor(conc_stmt);
  }
}

void
IIRScram::_publish_cc_object_pointers_init(IIR_ConcurrentStatementList* conc_stmt_list,
					   IIR_DeclarationList* decl_list) {

  IIR_ConcurrentStatement* conc_stmt = NULL;
  // Publish pointers from entity statement part
  //  entitydecl->_publish_cc_object_pointers_init();
  // Publish pointers from architecture statement part
  conc_stmt = conc_stmt_list->first();
  while(conc_stmt != NULL) {
    conc_stmt->_publish_cc_concurrent_stmt_init(decl_list);
    conc_stmt = conc_stmt_list->successor(conc_stmt);
  }
}

void
IIRScram::_publish_cc_destructor(IIR_ConcurrentStatementList* conc_stmt_list) {

  IIR_ConcurrentStatement* conc_stmt = conc_stmt_list->first();

  this->_publish_cc_binding_name();
  _cc_out << "_elab::~";
  this->_publish_cc_binding_name();
  _cc_out << "_elab()  {\n";
  
  while (conc_stmt != NULL) {
    switch(conc_stmt->get_kind()) {
    case IIR_PROCESS_STATEMENT:
      if(this->get_kind() == IIR_CONCURRENT_GENERATE_FOR_STATEMENT) {
	_cc_out << "delete ";
	_cc_out << " [] ";
	
	if (conc_stmt->_get_label() != NULL) {
	  _cc_out << *(conc_stmt->_get_label());
	}
	else {
	  _cc_out << "ANON_PROCESS" << conc_stmt;
	}
	_cc_out << "_elab_obj;\n";
      }
      break;
      
    case IIR_COMPONENT_INSTANTIATION_STATEMENT:
    case IIR_BLOCK_STATEMENT:
    case IIR_CONCURRENT_GENERATE_FOR_STATEMENT:
    case IIR_CONCURRENT_GENERATE_IF_STATEMENT:
      _cc_out << "delete ";
      if(this->get_kind() == IIR_CONCURRENT_GENERATE_FOR_STATEMENT) {
	_cc_out << " [] ";
      }
      conc_stmt->_get_label()->_publish_cc_elaborate();
      _cc_out << "_elab_obj;\n";
      break;
      
    default:
      cerr << "ERROR!!!!! _publish_cc_destructor(): unknown conc_statement "
	   << "type |" << conc_stmt->get_kind_text() << "| in arch" << endl;
    }
    
    conc_stmt = conc_stmt_list->successor(conc_stmt);
  }
  _cc_out << "}\n";
}

void
IIRScram::_publish_cc_anonymous_drivers(IIR_DeclarationList* decl_list) {
  IIR_Declaration* tempDeclaration = decl_list->first();
  IIR_SignalDeclaration* tempSignalDecl = NULL;
  IIR_SignalInterfaceDeclaration *tempSignalInterDecl = NULL;
  IIR_Char* old_current_publish_name = _current_publish_name;

  _current_publish_name = ", ANONYMOUS_PROCESS_ID);\n";
  
  while (tempDeclaration != NULL) {
    if (tempDeclaration->get_kind() == IIR_SIGNAL_DECLARATION) {
      tempSignalDecl = (IIR_SignalDeclaration *) tempDeclaration;
      if ((tempSignalDecl->get_value() != NULL) && (tempSignalDecl->_is_passed_through_out_port() == FALSE)) {
	tempSignalDecl->_publish_cc_anonymous_addChild();
      }
    }
    else if (tempDeclaration->get_kind() == IIR_SIGNAL_INTERFACE_DECLARATION) {
      tempSignalInterDecl = (IIR_SignalInterfaceDeclaration *)tempDeclaration;
      if ( tempSignalInterDecl->get_value() != NULL ){
	tempSignalInterDecl->_publish_cc_anonymous_addChild();
      }
    }
    tempDeclaration = decl_list->successor(tempDeclaration);
  }

  _current_publish_name = old_current_publish_name;
}

void
IIRScram::_publish_cc_copy_generics_to_globals() {
  ASSERT ((_is_concurrent_statement() == TRUE) || (get_kind() == IIR_ENTITY_DECLARATION));
  
  if (_get_generic_list() != NULL) {
    if (_get_generic_list()->first() != NULL) {
      _cc_out << "void\n";
      _publish_cc_binding_name();
      _cc_out << "_elab::copyGenericsToGlobals()  {\n";
      _get_generic_list()->_publish_cc_copy_generics_to_globals();
      _cc_out << "}\n\n";
    }
  }
}

IIR_AttributeSpecificationList * 
IIRScram::_get_attribute_specification_list( ){
  return NULL;
}

IIR_Boolean 
IIRScram::_attach_attribute_specification( IIR_AttributeSpecification *to_attach ){
  IIR_AttributeSpecificationList *list = _get_attribute_specification_list();
  if( list == NULL ){
    return FALSE;
  }
  else{
    list->append( to_attach );
    return TRUE;
  }
}

IIR_Boolean 
IIRScram::_attach_disconnection_specification( IIR_AttributeSpecification * ){
  _report_undefined_scram_fn("_attach_disconnection_specification( IIR_AttributeSpecification *)");
  abort();
  return FALSE;
}

IIR_DeclarationList*
IIRScram::_get_declaration_list() {
  _report_undefined_scram_fn("_get_declaration_list()");
  return NULL;
}
